/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/render/font'
], function (Font) {

    'use strict';

    // class Font =============================================================

    describe('Toolkit class Font', function () {

        it('should exist', function () {
            expect(Font).to.be.a('function');
        });

        // constants ----------------------------------------------------------

        describe('constant "AUTO_BIDI_SUPPORT"', function () {
            it('should exist', function () {
                expect(Font.AUTO_BIDI_SUPPORT).to.be.a('boolean');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "isInstalled"', function () {
            it('should exist', function () {
                expect(Font).itself.to.respondTo('isInstalled');
            });
            it('should return true for system fonts', function () {
                expect(Font.isInstalled('serif')).to.equal(true);
                expect(Font.isInstalled('sans-serif')).to.equal(true);
                expect(Font.isInstalled('monospace')).to.equal(true);
            });
            it('should return false for a missing font', function () {
                expect(Font.isInstalled('__invalid__')).to.equal(false);
            });
        });

        describe('method "getTextDirection"', function () {
            var LTR_TEXT = 'abc';
            var RTL_TEXT = '\u05d0\u05d1\u05d2';
            var LTR = 'ltr';
            var RTL = Font.AUTO_BIDI_SUPPORT ? 'rtl' : 'ltr';
            it('should exist', function () {
                expect(Font).itself.to.respondTo('getTextDirection');
            });
            it('should return "ltr" for western text', function () {
                expect(Font.getTextDirection(LTR_TEXT)).to.equal(LTR);
                expect(Font.getTextDirection('  ' + LTR_TEXT + '  ')).to.equal(LTR);
            });
            it('should return "' + RTL + '" for complex text', function () {
                expect(Font.getTextDirection(RTL_TEXT)).to.equal(RTL);
                expect(Font.getTextDirection('  ' + RTL_TEXT + '  ')).to.equal(RTL);
            });
            it('should return correct direction for mixed text', function () {
                expect(Font.getTextDirection(LTR_TEXT + ' ' + RTL_TEXT)).to.equal(LTR);
                expect(Font.getTextDirection(RTL_TEXT + ' ' + LTR_TEXT)).to.equal(RTL);
            });
            it('should return "ltr" for undetermined text', function () {
                expect(Font.getTextDirection('')).to.equal(LTR);
                expect(Font.getTextDirection('  ')).to.equal(LTR);
                expect(Font.getTextDirection('.')).to.equal(LTR);
                expect(Font.getTextDirection('123')).to.equal(LTR);
            });
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should take all parameters', function () {
                var fontDesc = new Font('times', 10, false, true);
                expect(fontDesc).to.have.a.property('family', 'times');
                expect(fontDesc).to.have.a.property('size', 10);
                expect(fontDesc).to.have.a.property('bold', false);
                expect(fontDesc).to.have.a.property('italic', true);
            });
            it('should convert font name to lower-case', function () {
                var fontDesc = new Font('TiMeS', 10.25, false, true);
                expect(fontDesc).to.have.a.property('family', 'times');
            });
            it('should round font size', function () {
                var fontDesc = new Font('times', 10.25, false, true);
                expect(fontDesc).to.have.a.property('size', 10.3);
            });
            it('should accept missing flags', function () {
                var fontDesc = new Font('times', 10);
                expect(fontDesc).to.have.a.property('bold', false);
                expect(fontDesc).to.have.a.property('italic', false);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(Font).to.respondTo('clone');
            });
            it('should return a clone', function () {
                var fontDesc1 = new Font('times', 10, false, true);
                var fontDesc2 = fontDesc1.clone();
                expect(fontDesc2).to.not.equal(fontDesc1);
                expect(fontDesc2).to.deep.equal(fontDesc1);
            });
        });

        describe('method "key"', function () {
            it('should exist', function () {
                expect(Font).to.respondTo('key');
            });
            it('should return a string', function () {
                var fontDesc = new Font('times', 10, false, true);
                expect(fontDesc.key()).to.be.a.string;
            });
            it('should return different keys for different font names', function () {
                var fontDesc1 = new Font('times1', 10, false, true),
                    fontDesc2 = new Font('times2', 10, false, true);
                expect(fontDesc1.key()).to.not.equal(fontDesc2.key());
            });
            it('should return different keys for different font sizes', function () {
                var fontDesc1 = new Font('times', 10, false, true),
                    fontDesc2 = new Font('times', 11, false, true);
                expect(fontDesc1.key()).to.not.equal(fontDesc2.key());
            });
            it('should return different keys for different bold flags', function () {
                var fontDesc1 = new Font('times', 10, false, true),
                    fontDesc2 = new Font('times', 10, true, true);
                expect(fontDesc1.key()).to.not.equal(fontDesc2.key());
            });
            it('should return different keys for different italic flags', function () {
                var fontDesc1 = new Font('times', 10, false, true),
                    fontDesc2 = new Font('times', 10, false, false);
                expect(fontDesc1.key()).to.not.equal(fontDesc2.key());
            });
        });

        describe('method "getCanvasFont"', function () {
            it('should exist', function () {
                expect(Font).to.respondTo('getCanvasFont');
            });
            it('should return the canvas font style', function () {
                expect(new Font('times', 10, false, false).getCanvasFont()).to.equal('10pt times');
                expect(new Font('times,serif', 10, false, false).getCanvasFont()).to.equal('10pt times,serif');
                expect(new Font('times', 10, true, false).getCanvasFont()).to.equal('bold 10pt times');
                expect(new Font('times', 10, false, true).getCanvasFont()).to.equal('italic 10pt times');
                expect(new Font('times', 10, true, true).getCanvasFont()).to.equal('bold italic 10pt times');
                expect(new Font('Times New Roman', 10, false, false).getCanvasFont()).to.equal('10pt times new roman');
            });
        });

        describe('method "getNormalLineHeight"', function () {
            it('should exist', function () {
                expect(Font).to.respondTo('getNormalLineHeight');
            });
        });

        describe('method "getBaseLineOffset"', function () {
            it('should exist', function () {
                expect(Font).to.respondTo('getBaseLineOffset');
            });
        });

        describe('method "getTextWidth"', function () {
            it('should exist', function () {
                expect(Font).to.respondTo('getTextWidth');
            });
            it('should return the text width', function () {
                var fontDesc = new Font('times', 10, false, false);
                expect(fontDesc.getTextWidth('')).to.equal(0);
                var a = fontDesc.getTextWidth('a');
                expect(a).to.be.a('number').above(0);
                var ab = fontDesc.getTextWidth('ab');
                expect(ab).to.be.a('number').above(a);
            });
        });

        describe('method "getDigitWidths"', function () {
            it('should exist', function () {
                expect(Font).to.respondTo('getDigitWidths');
            });
            it('should return the digit measures', function () {
                var fontDesc = new Font('times', 10, false, false);
                var digitWidths = fontDesc.getDigitWidths();
                expect(digitWidths).to.be.an('array').with.length(10);
                expect(digitWidths.map(_.identity)).to.deep.equal(_.map('0123456789', fontDesc.getTextWidth.bind(fontDesc)));
                var sorted = _.sortBy(digitWidths);
                expect(digitWidths).to.have.a.property('minWidth', sorted[0]);
                expect(digitWidths).to.have.a.property('maxWidth', sorted[9]);
                expect(digitWidths).to.have.a.property('minDigit').that.is.a('number').within(0, 9);
                expect(digitWidths).to.have.a.property('maxDigit').that.is.a('number').within(0, 9);
            });
        });

        describe('method "getTextLines"', function () {
            it('should exist', function () {
                expect(Font).to.respondTo('getTextLines');
            });
        });

        describe('method "getCustomFontMetrics"', function () {
            it('should exist', function () {
                expect(Font).to.respondTo('getCustomFontMetrics');
            });
        });
    });

    // ========================================================================
});
