/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/components/pagelayout/pagelayout'
], function (AppHelper, DOM, PageLayout) {

    'use strict';

    // class PageLayout ===========================================================

    describe('Text class PageLayout', function () {

        it('should exist', function () {
            expect(PageLayout).to.be.a('function');
        });

        // private helpers ----------------------------------------------------
        var REPETITIVE_TEXT1 = 'Repetitive text in first cell';
        var REPETITIVE_TEXT2 = 'Repetitive text in second cell';

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 } } },
            { name: 'insertParagraph', start: [0] },
            { name: 'splitParagraph', start: [0, 0] },
            { name: 'insertText', text: 'Hello World', start: [1, 0] },
            { name: 'splitParagraph', start: [1, 0] },
            { name: 'setAttributes', start: [2], attrs: { paragraph: { pageBreakBefore: true } } },
            { name: 'splitParagraph', start: [2, 0] },
            { name: 'setAttributes', start: [3], attrs: { paragraph: { pageBreakBefore: true } } },

            { name: 'insertTable', start: [4], attrs: { styleId: 'TableGrid', table: { tableGrid: [7958, 7958], width: 15919, exclude: ['lastRow', 'lastCol', 'bandsVert'] } } },
            { name: 'insertRows', start: [4, 0], attrs: { row: { headerRow: true, height: 10000 } } },
            { name: 'insertCells', start: [4, 0, 0] },
            { name: 'insertParagraph', start: [4, 0, 0, 0] },
            { name: 'insertText', start: [4, 0, 0, 0, 0], text: REPETITIVE_TEXT1 },
            { name: 'insertCells', start: [4, 0, 1] },
            { name: 'insertParagraph', start: [4, 0, 1, 0] },
            { name: 'insertText', start: [4, 0, 1, 0, 0], text: REPETITIVE_TEXT2 },
            { name: 'insertRows', start: [4, 1], attrs: { row: { height: 10052 } } },
            { name: 'insertCells', start: [4, 1, 0] },
            { name: 'insertParagraph', start: [4, 1, 0, 0] },
            { name: 'insertCells', start: [4, 1, 1] },
            { name: 'insertParagraph', start: [4, 1, 1, 0] },
            { name: 'insertRows', start: [4, 2], attrs: { row: { height: 6455 } } },
            { name: 'insertCells', start: [4, 2, 0] },
            { name: 'insertParagraph', start: [4, 2, 0, 0] },
            { name: 'insertCells', start: [4, 2, 1] },
            { name: 'insertParagraph', start: [4, 2, 1, 0] }

        ];

        var model, view, pageLayout;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            view = app.getView();
            pageLayout = model.getPageLayout();
            $('#io-ox-office-temp').append(model.getNode()); // appending '.page' to the phantomjs DOM, for correct height computations
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a PageLayout class instance', function () {
                expect(new PageLayout(model, view.getContentRootNode())).to.be.an['instanceof'](PageLayout);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "getHeaderFooterPlaceHolder"', function () {
            it('should exist', function () {
                expect(pageLayout).to.respondTo('getHeaderFooterPlaceHolder');
            });
            it('should return headerFooter placeholder node', function () {
                expect(pageLayout.getHeaderFooterPlaceHolder()).to.be.an['instanceof']($);
            });
        });

        describe('method "getallHeadersAndFooters"', function () {
            it('should exist', function () {
                expect(pageLayout).to.respondTo('getallHeadersAndFooters');
            });
            it('should return all headers and footers in document', function () {
                expect(pageLayout.getallHeadersAndFooters()).to.be.an['instanceof']($);
            });
        });

        describe('group of methods with pre-requirement of initial page breaks called', function () {
            before(function () {
                pageLayout.callInitialPageBreaks();
            });

            describe('method "getPageNumber"', function () {
                it('should exist', function () {
                    expect(pageLayout).to.respondTo('getPageNumber');
                });
                it('should return the current page number of the selection', function () {
                    expect(pageLayout.getPageNumber()).to.equal(1);
                });
            });

            describe('existence of first and last header/footer wrappers', function () {
                it('should return that first header wrapper exists in document', function () {
                    expect(pageLayout.isExistingFirstHeaderWrapper()).to.equal(true);
                });
                it('should return that last footer wrapper exists in document', function () {
                    expect(pageLayout.isExistingLastFooterWrapper()).to.equal(true);
                });

            });

            describe('method "getPageAttribute"', function () {
                it('should exist', function () {
                    expect(pageLayout).to.respondTo('getPageAttribute');
                });
                it('should return that first page header is not set in document', function () {
                    expect(pageLayout.getPageAttribute('firstPage')).to.equal(false);
                });
                it('should return that even/odd pages headers are not set in document', function () {
                    expect(pageLayout.getPageAttribute('evenOddPages')).to.equal(false);
                });
                it('should return that default document orientation is set to portrait mode', function () {
                    expect(pageLayout.getPageAttribute('orientation')).to.equal('portrait');
                    expect(pageLayout.getPageAttribute('orientation')).not.to.equal('landscape');
                });
            });

            describe('method "getNumberOfDocumentPages"', function () {
                it('should exist', function () {
                    expect(pageLayout).to.respondTo('getNumberOfDocumentPages');
                });
                it('should return correct number of document pages', function () {
                    expect(pageLayout.getNumberOfDocumentPages()).to.equal(4);
                });
            });

        });

        describe('method "getPageAttribute"', function () {
            it('should exist', function () {
                expect(pageLayout).to.respondTo('getPageAttribute');
            });
            it('should return that first page header is not set in document', function () {
                expect(pageLayout.getPageAttribute('firstPage')).to.equal(false);
            });
            it('should return that even/odd pages headers are not set in document', function () {
                expect(pageLayout.getPageAttribute('evenOddPages')).to.equal(false);
            });
            it('should return that default document orientation is set to portrait mode', function () {
                expect(pageLayout.getPageAttribute('orientation')).to.equal('portrait');
                expect(pageLayout.getPageAttribute('orientation')).not.to.equal('landscape');
            });
        });

        describe('method "insertPageBreaksInTable"', function () {
            it('should render page break inside table correctly, with repeating row content over pages', function () {
                var tableNode = model.getNode().find('table');
                var firstRow = tableNode.find('tr').first();
                var repeatedRow = tableNode.find(DOM.TABLE_REPEATED_ROWNODE_SELECTOR).first();
                expect(tableNode.height()).not.to.equal(0);
                expect(firstRow.hasClass(DOM.TABLE_REPEAT_ROWNODE_CLASSNAME)).to.equal(true);
                expect(repeatedRow.length).to.equal(1);
                expect(repeatedRow.children('td').length).to.equal(2);
                expect(repeatedRow.children('td').first().text()).to.equal(REPETITIVE_TEXT1)
                    .and.to.equal(firstRow.children('td').first().text());
                expect(repeatedRow.children('td').last().text()).to.equal(REPETITIVE_TEXT2)
                    .and.to.equal(firstRow.children('td').last().text());
            });
        });
    });

    // ========================================================================
});
