/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/components/changetrack/changetrack',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/tk/utils/dateutils'
], function (AppHelper, ChangeTrack, DOM, Position, TextUtils, DateUtils) {

    'use strict';

    // class ChangeTrack ================================================

    describe('Text class ChangeTrack', function () {

        // private helpers ----------------------------------------------------

        //  the operations to be applied by the document model
        var localApp = null,
            model = null,
            pageNode = null,
            changeTrack = null,
            firstParagraph = null,
            startText = 'Hello World!',
            insertText1 = ' big',
            fullText1 = 'Hello big World!',
            getCtInfoStub = null,
            ctSelection = null, // a full change track selection object
            testAuthor = 'author1',
            testUid = '1111',

            OPERATIONS = [
                { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 }, character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' }, paragraph: { lineHeight: { type: 'percent', value: 115 }, marginBottom: 352 } } },
                { name: 'insertParagraph', start: [0] },
                { name: 'insertText', start: [0, 0], text: startText }
            ];

        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            localApp = app;
            model = app.getModel();
            pageNode = model.getNode();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(ChangeTrack).to.be.a('function');
        });

        // constructor --------------------------------------------------------
        describe('constructor', function () {
            it('should create a ChangeTrack class instance', function () {
                var changeTrackObject = new ChangeTrack(localApp, pageNode, model);
                expect(changeTrackObject).to.be.an['instanceof'](ChangeTrack); // this object cannot be used for the tests, because it is not received from the model!
            });
        });

        // public methods -----------------------------------------------------

        describe('method "model.getChangeTrack"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getChangeTrack');
            });
            it('should return the one change track object used by the application', function () {
                changeTrack = model.getChangeTrack(); // this is the change track object in the model that can be used by the tests
                expect(changeTrack).to.be.an['instanceof'](ChangeTrack);
            });
        });

        describe('method "changeTrack.isActiveChangeTracking"', function () {
            it('should exist', function () {
                expect(changeTrack).to.respondTo('isActiveChangeTracking');
            });
            it('should return that change tracking is not activated in the current context', function () {
                expect(changeTrack.isActiveChangeTracking()).to.equal(false);
            });
        });

        describe('method "changeTrack.isActiveChangeTrackingInDocAttributes"', function () {
            it('should exist', function () {
                expect(changeTrack).to.respondTo('isActiveChangeTrackingInDocAttributes');
            });
            it('should return that change tracking is not activated in document attributes', function () {
                expect(changeTrack.isActiveChangeTrackingInDocAttributes()).to.equal(false);
            });
        });

        describe('method "changeTrack.setActiveChangeTracking"', function () {
            it('should exist', function () {
                expect(changeTrack).to.respondTo('setActiveChangeTracking');
            });
            it('should activate change tracking in the document successfully', function () {
                // successfully applying operation
                expect(changeTrack.setActiveChangeTracking(true)).to.equal(true);
                // checking if change tracking is active in current context (for example not inside comments)
                expect(changeTrack.isActiveChangeTracking()).to.equal(true);
                // checking if change tracking is actived in the document attributes
                expect(changeTrack.isActiveChangeTrackingInDocAttributes()).to.equal(true);
            });
        });

        describe('method "changeTrack.getNextChangeTrack"', function () {
            it('should exist', function () {
                expect(changeTrack).to.respondTo('getNextChangeTrack');
            });
            it('should not select any change track node in the document', function () {
                var firstParagraph = Position.getParagraphElement(pageNode, [0]);
                expect($(firstParagraph).text()).to.equal(startText);
                // no change tracked nodes in the document yet
                expect(changeTrack.getNextChangeTrack()).to.equal(false);
            });
        });

        describe('method "changeTrack.updateSideBar"', function () {
            it('should exist', function () {
                expect(changeTrack).to.respondTo('updateSideBar');
            });
            it('should not draw any lines in the side bar of the document', function () {
                firstParagraph = Position.getParagraphElement(pageNode, [0]);
                expect($(firstParagraph).text()).to.equal(startText);
                // no change tracked nodes in the document yet
                expect(changeTrack.updateSideBar()).to.equal(false);
            });
        });

        describe('method "changeTrack.getChangeTrackSelection"', function () {
            it('should exist', function () {
                expect(changeTrack).to.respondTo('getChangeTrackSelection');
            });
            it('should not find any valid change track selection in the document', function () {
                firstParagraph = Position.getParagraphElement(pageNode, [0]);
                expect($(firstParagraph).text()).to.equal(startText);
                // no change tracked nodes in the document yet
                expect(changeTrack.getChangeTrackSelection()).to.equal.Null;
            });
        });

        describe('method "changeTrack.resolveChangeTracking"', function () {
            it('should exist', function () {
                expect(changeTrack).to.respondTo('resolveChangeTracking');
            });
            it('should return a resolved promise because there are no valid change track nodes in the document', function () {
                firstParagraph = Position.getParagraphElement(pageNode, [0]);
                expect($(firstParagraph).text()).to.equal(startText);
                // no change tracked nodes in the document yet
                var resolvePromise = changeTrack.resolveChangeTracking();
                expect(resolvePromise.state()).to.equal('resolved');
            });
        });

        // preparations for inserting text (changeTrack.getChangeTrackInfo requires stub)
        describe('method "changeTrack.getChangeTrackInfo"', function () {
            it('should exist', function () {
                expect(changeTrack).to.respondTo('getChangeTrackInfo');
            });
            it('should fail on direct call', function () { // problem with 'userid.toString()' because 'uid' is null
                expect(changeTrack.getChangeTrackInfo).to.throw(TypeError);
            });
        });

        // preparations for inserting text (testing used date string for change tracking)
        describe('method "DateUtils.getMSFormatIsoDateString"', function () {
            it('should exist', function () {
                expect(DateUtils).to.respondTo('getMSFormatIsoDateString');
            });
            it('should return a special change tracking date format', function () {
                var isoDateFormatString = DateUtils.getMSFormatIsoDateString({ useSeconds: false });
                var startDateString = new Date().getFullYear() + '-';
                var endDateString = ':00Z'; // ignoring seconds
                expect(isoDateFormatString.length).to.equal(20);
                expect(isoDateFormatString).to.have.string(startDateString);
                expect(isoDateFormatString).to.have.string(endDateString);
            });
        });

        // inserting some text, so that change track nodes exist
        describe('method "changeTrack.getChangeTrackSelection"', function () {

            before(function () {
                getCtInfoStub = sinon.stub(changeTrack, 'getChangeTrackInfo').callsFake(function () {
                    return { author: testAuthor, date: DateUtils.getMSFormatIsoDateString({ useSeconds: false }), uid: testUid };
                });
            });

            after(function () {
                getCtInfoStub.restore();
            });

            it('should find a valid change track selection in the document', function () {
                // using defined values for:
                // insertText1 = ' big'
                // fullText1 = 'Hello big World!'

                model.insertText(insertText1, [0, 5]);

                expect(getCtInfoStub.calledOnce).to.equal(true);

                firstParagraph = Position.getParagraphElement(pageNode, [0]);
                expect($(firstParagraph).text()).to.equal(fullText1);

                // this one change track can be selected now
                expect(changeTrack.getNextChangeTrack()).to.equal(true);
            });

            it('should return a valid change track selection object', function () {

                // the change track selection was set in the function changeTrack.getNextChangeTrack()
                ctSelection = changeTrack.getChangeTrackSelection();
                expect(ctSelection).to.be.an.object;

                // ChangeTrack selection object consists of:
                // -> activeNode: HTMLElement, the one currently node used by the iterator
                // -> activeNodeType: String, the one currently used node type, one of 'inserted', 'removed', 'modified'
                // -> activeNodeTypesOrdered: String[], ordered list of strings, corresponding to priority of node types
                // -> selectedNodes: HTMLElement[], sorted list of HTMLElements of the group belonging to 'activeNode' and 'activeNodeType
                // -> allChangeTrackNodes: jQuery, optional, cache for performance reasons, contains all change track elements in the document

                expect(ctSelection.activeNode).to.be.an.object;
                expect($(ctSelection.activeNode).text()).to.equal(insertText1);
                expect(ctSelection.activeNodeType).to.equal('inserted');
                expect(ctSelection.activeNodeTypesOrdered.length).to.equal(1);
                expect(ctSelection.activeNodeTypesOrdered[0]).to.equal('inserted');
                expect(ctSelection.selectedNodes.length).to.equal(1);
                expect(ctSelection.allChangeTrackNodes.length).to.equal(1);
                expect(ctSelection.selectedNodes[0]).to.equal(ctSelection.activeNode);
            });
        });

        // checking the selected changetrack node (must be a text span)
        describe('method "DOM.isTextSpan"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isTextSpan');
            });
            it('should identify the selected node as text span node', function () {
                expect(DOM.isTextSpan(ctSelection.activeNode)).to.equal(true);
            });
        });

        describe('method "DOM.isChangeTrackNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isChangeTrackNode');
            });
            it('should detect a selected node as change tracked node', function () {
                expect(DOM.isChangeTrackNode(ctSelection.activeNode)).to.equal(true);
                expect(DOM.isChangeTrackNode(ctSelection.activeNode.previousSibling)).to.equal(false);
                expect(DOM.isChangeTrackNode(ctSelection.activeNode.nextSibling)).to.equal(false);
            });
        });

        describe('method "DOM.isChangeTrackInsertNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isChangeTrackInsertNode');
            });
            it('should detect a selected node as inserted change tracked node', function () {
                expect(DOM.isChangeTrackInsertNode(ctSelection.activeNode)).to.equal(true);
                expect(DOM.isChangeTrackInsertNode(ctSelection.activeNode.previousSibling)).to.equal(false);
                expect(DOM.isChangeTrackInsertNode(ctSelection.activeNode.nextSibling)).to.equal(false);
            });
        });

        // side bar must contain an element
        describe('method "changeTrack.updateSideBar"', function () {
            it('should draw any lines in the side bar of the document', function () {
                // one change tracked node exists in the  document
                expect(changeTrack.updateSideBar()).to.equal(true);
            });
        });

        // accepting or rejecting the change track
        describe('method "changeTrack.resolveChangeTracking"', function () {

            it('should accept the existing change track', function (done) {
                firstParagraph = Position.getParagraphElement(pageNode, [0]);
                expect($(firstParagraph).text()).to.equal(fullText1);

                var resolvePromise = changeTrack.resolveChangeTracking();

                resolvePromise.always(function () {
                    expect(resolvePromise.state()).to.equal('resolved');
                    // the full text should still be there
                    expect($(firstParagraph).text()).to.equal(fullText1);
                    // no change tracked nodes in the document yet
                    expect(changeTrack.getNextChangeTrack()).to.equal(false);
                    done();
                });
            });

            it('should undo the acceptance of the change track', function (done) {

                firstParagraph = Position.getParagraphElement(pageNode, [0]);
                expect($(firstParagraph).text()).to.equal(fullText1);

                var undoPromise = model.getUndoManager().undo();

                undoPromise.always(function () {

                    expect(undoPromise.state()).to.equal('resolved');
                    // the full text should still be there
                    expect($(firstParagraph).text()).to.equal(fullText1);
                    // one change tracked node in the document again
                    expect(changeTrack.getNextChangeTrack()).to.equal(true);
                    // checking the change track selection object
                    ctSelection = changeTrack.getChangeTrackSelection();
                    expect(ctSelection).to.be.an.object;
                    expect(ctSelection.activeNode).to.be.an.object;
                    expect($(ctSelection.activeNode).text()).to.equal(insertText1);
                    expect(ctSelection.activeNodeType).to.equal('inserted');
                    expect(ctSelection.activeNodeTypesOrdered.length).to.equal(1);
                    expect(ctSelection.activeNodeTypesOrdered[0]).to.equal('inserted');
                    expect(ctSelection.selectedNodes.length).to.equal(1);
                    expect(ctSelection.allChangeTrackNodes.length).to.equal(1);
                    expect(ctSelection.selectedNodes[0]).to.equal(ctSelection.activeNode);

                    done();
                });
            });

            it('should reject the existing change track', function (done) {
                firstParagraph = Position.getParagraphElement(pageNode, [0]);
                expect($(firstParagraph).text()).to.equal(fullText1);

                var resolvePromise = changeTrack.resolveChangeTracking(null, { accepted: false });

                resolvePromise.always(function () {
                    expect(resolvePromise.state()).to.equal('resolved');
                    // only the start text should still be there
                    expect($(firstParagraph).text()).to.equal(startText);
                    // no change tracked nodes in the document yet
                    expect(changeTrack.getNextChangeTrack()).to.equal(false);
                    done();
                });
            });

            it('should undo the rejecting of the change track', function (done) {

                firstParagraph = Position.getParagraphElement(pageNode, [0]);
                expect($(firstParagraph).text()).to.equal(startText);

                var undoPromise = model.getUndoManager().undo();

                undoPromise.always(function () {

                    expect(undoPromise.state()).to.equal('resolved');
                    // the full text should again be there
                    expect($(firstParagraph).text()).to.equal(fullText1);
                    // one change tracked node in the document again
                    expect(changeTrack.getNextChangeTrack()).to.equal(true);
                    // checking the change track selection object
                    ctSelection = changeTrack.getChangeTrackSelection();
                    expect(ctSelection).to.be.an.object;
                    expect(ctSelection.activeNode).to.be.an.object;
                    expect($(ctSelection.activeNode).text()).to.equal(insertText1);
                    expect(ctSelection.activeNodeType).to.equal('inserted');
                    expect(ctSelection.activeNodeTypesOrdered.length).to.equal(1);
                    expect(ctSelection.activeNodeTypesOrdered[0]).to.equal('inserted');
                    expect(ctSelection.selectedNodes.length).to.equal(1);
                    expect(ctSelection.allChangeTrackNodes.length).to.equal(1);
                    expect(ctSelection.selectedNodes[0]).to.equal(ctSelection.activeNode);

                    done();
                });
            });

        });

        // investigating the change tracked node
        describe('method "changeTrack.getChangeTrackInfoFromNode"', function () {
            it('should exist', function () {
                expect(changeTrack).to.respondTo('getChangeTrackInfoFromNode');
            });
            it('should return the change tracking information from the change tracked node', function () {
                // parameter: node, info, type
                // allowed infos: CHANGE_TRACK_INFOS = ['author', 'date', 'uid'],
                // allowed types: CHANGE_TRACK_TYPES = ['inserted', 'removed', 'modified']
                expect(changeTrack.getChangeTrackInfoFromNode(ctSelection.activeNode, 'author', 'inserted')).to.equal(testAuthor);
                expect(changeTrack.getChangeTrackInfoFromNode(ctSelection.activeNode, 'author', 'removed')).to.equal(null);
                expect(changeTrack.getChangeTrackInfoFromNode(ctSelection.activeNode, 'author', 'modified')).to.equal(null);
                expect(changeTrack.getChangeTrackInfoFromNode(ctSelection.activeNode, 'author', 'test')).to.equal(null); // invalid type

                expect(changeTrack.getChangeTrackInfoFromNode(ctSelection.activeNode, 'name', 'inserted')).to.equal(null); // 'name' is not valid

                // the uid is modified using function TextUtils.resolveUserId(testUid)
                var transferUid = TextUtils.resolveUserId(parseInt(testUid, 10));
                expect(changeTrack.getChangeTrackInfoFromNode(ctSelection.activeNode, 'uid', 'inserted')).to.equal(transferUid);
                expect(changeTrack.getChangeTrackInfoFromNode(ctSelection.activeNode, 'uid', 'removed')).to.equal(null);
                expect(changeTrack.getChangeTrackInfoFromNode(ctSelection.activeNode, 'uid', 'modified')).to.equal(null);
                expect(changeTrack.getChangeTrackInfoFromNode(ctSelection.activeNode, 'uid', 'test')).to.equal(null); // invalid type
            });
        });

    });

    // ========================================================================
});
