/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/clipboard'
], function (Clipboard) {

    'use strict';

    // static class Clipboard =================================================

    describe('Spreadsheet module Clipboard', function () {

        it('should exist', function () {
            expect(Clipboard).to.be.an('object');
        });

        // polyfill String.fromCodePoint --------------------------------------

        /*! http://mths.be/fromcodepoint v0.1.0 by @mathias */
        if (!String.fromCodePoint) {
            (function () {
                var defineProperty = (function () {
                    // IE 8 only supports `Object.defineProperty` on DOM elements
                    var result;
                    try {
                        var object = {};
                        var $defineProperty = Object.defineProperty;
                        result = $defineProperty(object, object, object) && $defineProperty;
                    } catch (error) { /* nothing */ }
                    return result;
                }());
                var stringFromCharCode = String.fromCharCode;
                var floor = Math.floor;
                var fromCodePoint = function () {
                    var MAX_SIZE = 0x4000;
                    var codeUnits = [];
                    var highSurrogate;
                    var lowSurrogate;
                    var index = -1;
                    var length = arguments.length;
                    if (!length) {
                        return '';
                    }
                    var result = '';
                    while (++index < length) {
                        var codePoint = Number(arguments[index]);
                        if (
                            !isFinite(codePoint) ||       // `NaN`, `+Infinity`, or `-Infinity`
                            codePoint < 0 ||              // not a valid Unicode code point
                            codePoint > 0x10FFFF ||       // not a valid Unicode code point
                            floor(codePoint) !== codePoint // not an integer
                        ) {
                            throw new Error('Invalid code point: ' + codePoint);
                        }
                        if (codePoint <= 0xFFFF) { // BMP code point
                            codeUnits.push(codePoint);
                        } else { // Astral code point; split in surrogate halves
                            // http://mathiasbynens.be/notes/javascript-encoding#surrogate-formulae
                            codePoint -= 0x10000;
                            highSurrogate = (codePoint >> 10) + 0xD800;
                            lowSurrogate = (codePoint % 0x400) + 0xDC00;
                            codeUnits.push(highSurrogate, lowSurrogate);
                        }
                        if (index + 1 === length || codeUnits.length > MAX_SIZE) {
                            result += stringFromCharCode.apply(null, codeUnits);
                            codeUnits.length = 0;
                        }
                    }
                    return result;
                };
                if (defineProperty) {
                    defineProperty(String, 'fromCodePoint', {
                        value: fromCodePoint,
                        configurable: true,
                        writable: true
                    });
                } else {
                    String.fromCodePoint = fromCodePoint;
                }
            }());
        }

        // helper classes -----------------------------------------------------

        function getCharRange(start, end) {
            var chars = '';
            for (var i = start; i <= end; i++) {
                chars += String.fromCodePoint(i);
            }
            return chars;
        }

        function createInvalidXmlChars() {
            var chars = '';
            // UTF-8
            chars += getCharRange(0x0, 0x8);
            chars += getCharRange(0xB, 0xC);
            chars += getCharRange(0xE, 0x1F);
            chars += getCharRange(0x7F, 0x84);
            chars += getCharRange(0x86, 0x9F);
            chars += getCharRange(0xFDD0, 0xFDDF);
            // UTF-16
            // [#x1FFFE-#x1FFFF] ... [#x10FFFE-#x10FFFF]
            for (var start = 0x1FFFE; start <= 0x10FFFE; start += 0x10000) {
                chars += getCharRange(start, start + 1);
            }

            return chars;
        }

        // static methods -----------------------------------------------------

        describe('method "removeInvalidXmlChars"', function () {
            it('should exist', function () {
                expect(Clipboard).itself.to.respondTo('removeInvalidXmlChars');
            });
            it('should remove all invalid chars from the String', function () {
                var inputChars = 'foobar' + createInvalidXmlChars() + 'bazqux';
                var resultChars = Clipboard.removeInvalidXmlChars(inputChars);
                expect(resultChars.length).to.equal(12);
                expect(resultChars).to.equal('foobarbazqux');
            });
        });
    });

    // ========================================================================
});
