/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/model/formula/utils/cellref'
], function (Address, CellRef) {

    'use strict';

    // class CellRef ==========================================================

    describe('Spreadsheet class CellRef', function () {

        it('should exist', function () {
            expect(CellRef).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // convenience shortcuts
        var a = Address.parse;

        // create a CellRef instance on-the-fly from $1.$1 notation
        function cell(text) {
            var m = /^(\$)?(\d+)\.(\$)?(\d+)$/i.exec(text);
            return new CellRef(parseInt(m[2], 10), parseInt(m[4], 10), !!m[1], !!m[3]);
        }

        // dummy spreadsheet document model
        var docModel = {
            isValidAddress: function (address) {
                return (address[0] >= 0) && (address[0] <= this.getMaxCol()) &&
                    (address[1] >= 0) && (address[1] <= this.getMaxRow());
            },
            getMaxCol: _.constant(9),
            getMaxRow: _.constant(19)
        };

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a cell reference', function () {
                var r1 = new CellRef(1, 2, false, true);
                expect(r1).to.have.property('col', 1);
                expect(r1).to.have.property('row', 2);
                expect(r1).to.have.property('absCol', false);
                expect(r1).to.have.property('absRow', true);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('toString');
            });
            it('should return an internal string representation', function () {
                expect(new CellRef(1, 2, false, false).toString()).to.equal('B3');
                expect(new CellRef(1, 2, false, true).toString()).to.equal('B$3');
                expect(new CellRef(1, 2, true, false).toString()).to.equal('$B3');
                expect(new CellRef(1, 2, true, true).toString()).to.equal('$B$3');
                // helper self test
                expect(cell('1.2')).to.stringifyTo('B3');
                expect(cell('1.$2')).to.stringifyTo('B$3');
                expect(cell('$1.2')).to.stringifyTo('$B3');
                expect(cell('$1.$2')).to.stringifyTo('$B$3');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(CellRef).itself.to.respondTo('create');
            });
            it('should create a cell reference', function () {
                var r1 = CellRef.create(a('B3'), false, true);
                expect(r1).to.be.an.instanceof(CellRef);
                expect(r1).to.have.property('col', 1);
                expect(r1).to.have.property('row', 2);
                expect(r1).to.have.property('absCol', false);
                expect(r1).to.have.property('absRow', true);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('clone');
            });
            it('should create a clone', function () {
                var r1 = cell('1.$2'), r2 = r1.clone();
                expect(r2).to.be.an.instanceof(CellRef);
                expect(r2).to.not.equal(r1);
                expect(r2).to.stringifyTo('B$3');
            });
        });

        describe('method "equals"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('equals');
            });
            var r1 = cell('1.$2');
            it('should return true for equal cell references', function () {
                expect(r1.equals(cell('1.$2'))).to.equal(true);
            });
            it('should return false for different cell references', function () {
                expect(r1.equals(cell('2.$2'))).to.equal(false);
                expect(r1.equals(cell('$1.$2'))).to.equal(false);
                expect(r1.equals(cell('1.2'))).to.equal(false);
                expect(r1.equals(cell('1.$1'))).to.equal(false);
            });
        });

        describe('method "toAddress"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('toAddress');
            });
            it('should return the address', function () {
                expect(cell('1.2').toAddress()).to.be.an.instanceof(Address);
                expect(cell('1.2').toAddress()).to.stringifyTo('B3');
                expect(cell('$1.2').toAddress()).to.stringifyTo('B3');
                expect(cell('1.$2').toAddress()).to.stringifyTo('B3');
                expect(cell('$1.$2').toAddress()).to.stringifyTo('B3');
            });
        });

        describe('method "isAbs"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('isAbs');
            });
            it('should return the absolute flag', function () {
                expect(cell('1.2').isAbs(false)).to.equal(false);
                expect(cell('1.2').isAbs(true)).to.equal(false);
                expect(cell('$1.2').isAbs(false)).to.equal(false);
                expect(cell('$1.2').isAbs(true)).to.equal(true);
                expect(cell('1.$2').isAbs(false)).to.equal(true);
                expect(cell('1.$2').isAbs(true)).to.equal(false);
                expect(cell('$1.$2').isAbs(false)).to.equal(true);
                expect(cell('$1.$2').isAbs(true)).to.equal(true);
            });
        });

        describe('method "getIndex"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('getIndex');
            });
            it('should return the specified index', function () {
                expect(cell('1.2').getIndex(false)).to.equal(2);
                expect(cell('1.2').getIndex(true)).to.equal(1);
                expect(cell('$1.2').getIndex(false)).to.equal(2);
                expect(cell('$1.2').getIndex(true)).to.equal(1);
                expect(cell('1.$2').getIndex(false)).to.equal(2);
                expect(cell('1.$2').getIndex(true)).to.equal(1);
                expect(cell('$1.$2').getIndex(false)).to.equal(2);
                expect(cell('$1.$2').getIndex(true)).to.equal(1);
            });
        });

        describe('method "relocate"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('relocate');
            });
            it('should relocate relative components without wrapping', function () {
                var r1 = cell('1.2');
                expect(r1.relocate(docModel, a('A2'), a('C5'), false)).to.equal(true);
                expect(r1).to.stringifyTo('D6');
                expect(r1.relocate(docModel, a('C5'), a('A2'), false)).to.equal(true);
                expect(r1).to.stringifyTo('B3');
                expect(cell('0.0').relocate(docModel, a('B2'), a('A2'), false)).to.equal(false);
                expect(cell('0.0').relocate(docModel, a('B2'), a('B1'), false)).to.equal(false);
                expect(cell('9.19').relocate(docModel, a('A1'), a('B1'), false)).to.equal(false);
                expect(cell('9.19').relocate(docModel, a('A1'), a('A2'), false)).to.equal(false);
            });
            it('should relocate relative components with wrapping', function () {
                var r1 = cell('1.2');
                expect(r1.relocate(docModel, a('A2'), a('C5'), true)).to.equal(true);
                expect(r1).to.stringifyTo('D6');
                expect(r1.relocate(docModel, a('C5'), a('A2'), true)).to.equal(true);
                expect(r1).to.stringifyTo('B3');
                expect(r1.relocate(docModel, a('E6'), a('B2'), true)).to.equal(true);
                expect(r1).to.stringifyTo('I19');
                expect(r1.relocate(docModel, a('B2'), a('E6'), true)).to.equal(true);
                expect(r1).to.stringifyTo('B3');
            });
            it('should not relocate absolute components', function () {
                var r1 = cell('$1.$2');
                expect(r1.relocate(docModel, a('A2'), a('C5'), false)).to.equal(true);
                expect(r1).to.stringifyTo('$B$3');
                var r2 = cell('$1.$2');
                expect(r2.relocate(docModel, a('A2'), a('C5'), true)).to.equal(true);
                expect(r2).to.stringifyTo('$B$3');
            });
        });

        describe('method "colText"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('colText');
            });
            it('should return the column text', function () {
                expect(cell('1.$2').colText()).to.equal('B');
                expect(cell('$1.$2').colText()).to.equal('$B');
            });
        });

        describe('method "rowText"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('rowText');
            });
            it('should return the row text', function () {
                expect(cell('1.$2').rowText()).to.equal('$3');
                expect(cell('1.2').rowText()).to.equal('3');
            });
        });

        describe('method "refText"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('refText');
            });
            it('should return the reference text', function () {
                expect(cell('$1.$2').refText()).to.equal('$B$3');
                expect(cell('1.$2').refText()).to.equal('B$3');
                expect(cell('$1.2').refText()).to.equal('$B3');
                expect(cell('1.2').refText()).to.equal('B3');
            });
        });

        describe('method "colTextRC"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('colTextRC');
            });
            it('should return the column text for R1C1', function () {
                expect(cell('$2.$3').colTextRC('RC', a('A1'))).to.equal('C3');
                expect(cell('$2.$3').colTextRC('ZS', a('E1'))).to.equal('S3');
                expect(cell('2.$3').colTextRC('RC', a('B1'))).to.equal('C[1]');
                expect(cell('2.$3').colTextRC('RC', a('C1'))).to.equal('C');
                expect(cell('2.$3').colTextRC('RC', a('D1'))).to.equal('C[-1]');
            });
        });

        describe('method "rowTextRC"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('rowTextRC');
            });
            it('should return the column text for R1C1', function () {
                expect(cell('$2.$3').rowTextRC('RC', a('A1'))).to.equal('R4');
                expect(cell('$2.$3').rowTextRC('ZS', a('A5'))).to.equal('Z4');
                expect(cell('$2.3').rowTextRC('RC', a('A2'))).to.equal('R[2]');
                expect(cell('$2.3').rowTextRC('RC', a('A4'))).to.equal('R');
                expect(cell('$2.3').rowTextRC('RC', a('A6'))).to.equal('R[-2]');
            });
        });

        describe('method "refTextRC"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('refTextRC');
            });
            it('should return the column text for R1C1', function () {
                expect(cell('$2.$3').refTextRC('RC', a('A1'))).to.equal('R4C3');
                expect(cell('$2.$3').refTextRC('ZS', a('E5'))).to.equal('Z4S3');
                expect(cell('2.$3').refTextRC('RC', a('B2'))).to.equal('R4C[1]');
                expect(cell('2.$3').refTextRC('RC', a('C4'))).to.equal('R4C');
                expect(cell('2.$3').refTextRC('RC', a('D6'))).to.equal('R4C[-1]');
                expect(cell('$2.3').refTextRC('RC', a('B2'))).to.equal('R[2]C3');
                expect(cell('$2.3').refTextRC('RC', a('C4'))).to.equal('RC3');
                expect(cell('$2.3').refTextRC('RC', a('D6'))).to.equal('R[-2]C3');
                expect(cell('2.3').refTextRC('RC', a('B2'))).to.equal('R[2]C[1]');
                expect(cell('2.3').refTextRC('RC', a('C4'))).to.equal('RC');
                expect(cell('2.3').refTextRC('RC', a('D6'))).to.equal('R[-2]C[-1]');
            });
        });

        describe('method "parseCol"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('parseCol');
            });
            it('should parse the column text', function () {
                function testParseCol(colText, absFlag, expRes, expCol, expAbs) {
                    var cellRef = new CellRef(0, 0, true, true);
                    expect(cellRef.parseCol(colText, absFlag, 255)).to.equal(expRes);
                    if (expRes) {
                        expect(cellRef.col).to.equal(expCol);
                        expect(cellRef.absCol).to.equal(expAbs);
                    }
                }
                testParseCol('A', '$', true, 0, true);
                testParseCol('A', '', true, 0, false);
                testParseCol('A', null, true, 0, false);
                testParseCol('IV', '$', true, 255, true);
                testParseCol('IW', '$', false);
                testParseCol('1', '$', false);
            });
        });

        describe('method "parseRow"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('parseRow');
            });
            it('should parse the row text', function () {
                function testParseRow(rowText, absFlag, expRes, expRow, expAbs) {
                    var cellRef = new CellRef(0, 0, true, true);
                    expect(cellRef.parseRow(rowText, absFlag, 255)).to.equal(expRes);
                    if (expRes) {
                        expect(cellRef.row).to.equal(expRow);
                        expect(cellRef.absRow).to.equal(expAbs);
                    }
                }
                testParseRow('1', '$', true, 0, true);
                testParseRow('1', '', true, 0, false);
                testParseRow('1', null, true, 0, false);
                testParseRow('256', '$', true, 255, true);
                testParseRow('257', '$', false);
                testParseRow('A', '$', false);
            });
        });

        describe('method "parseColRC"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('parseColRC');
            });
            it('should parse the column text for R1C1', function () {
                function testParseCol(absText, relText, wrap, expRes, expCol, expAbs) {
                    var cellRef = new CellRef(0, 0, true, true);
                    expect(cellRef.parseColRC(absText, relText, 255, 15, wrap)).to.equal(expRes);
                    if (expRes) {
                        expect(cellRef.col).to.equal(expCol);
                        expect(cellRef.absCol).to.equal(expAbs);
                    }
                }
                testParseCol('1', null, false, true, 0, true);
                testParseCol('1', null, true, true, 0, true);
                testParseCol('256', null, false, true, 255, true);
                testParseCol('256', null, true, true, 255, true);
                testParseCol('257', null, false, false);
                testParseCol('257', null, true, false);
                testParseCol('A', null, false, false);
                testParseCol(null, null, false, true, 15, false);
                testParseCol(null, null, true, true, 15, false);
                testParseCol(null, '', false, true, 15, false);
                testParseCol(null, '0', false, true, 15, false);
                testParseCol(null, '-1', false, true, 14, false);
                testParseCol(null, '-1', true, true, 14, false);
                testParseCol(null, '-15', false, true, 0, false);
                testParseCol(null, '-15', true, true, 0, false);
                testParseCol(null, '-16', false, false);
                testParseCol(null, '-16', true, true, 255, false);
                testParseCol(null, '-255', false, false);
                testParseCol(null, '-255', true, true, 16, false);
                testParseCol(null, '-256', false, false);
                testParseCol(null, '-256', true, false);
                testParseCol(null, '1', false, true, 16, false);
                testParseCol(null, '1', true, true, 16, false);
                testParseCol(null, '240', false, true, 255, false);
                testParseCol(null, '240', true, true, 255, false);
                testParseCol(null, '241', false, false);
                testParseCol(null, '241', true, true, 0, false);
                testParseCol(null, '255', false, false);
                testParseCol(null, '255', true, true, 14, false);
                testParseCol(null, '256', false, false);
                testParseCol(null, '256', true, false);
                testParseCol(null, 'A', false, false);
            });
        });

        describe('method "parseRowRC"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('parseRowRC');
            });
            it('should parse the row text for R1C1', function () {
                function testParseRow(absText, relText, wrap, expRes, expRow, expAbs) {
                    var cellRef = new CellRef(0, 0, true, true);
                    expect(cellRef.parseRowRC(absText, relText, 255, 15, wrap)).to.equal(expRes);
                    if (expRes) {
                        expect(cellRef.row).to.equal(expRow);
                        expect(cellRef.absRow).to.equal(expAbs);
                    }
                }
                testParseRow('1', null, false, true, 0, true);
                testParseRow('1', null, true, true, 0, true);
                testParseRow('256', null, false, true, 255, true);
                testParseRow('256', null, true, true, 255, true);
                testParseRow('257', null, false, false);
                testParseRow('257', null, true, false);
                testParseRow('A', null, false, false);
                testParseRow(null, null, false, true, 15, false);
                testParseRow(null, null, true, true, 15, false);
                testParseRow(null, '', false, true, 15, false);
                testParseRow(null, '0', false, true, 15, false);
                testParseRow(null, '-1', false, true, 14, false);
                testParseRow(null, '-1', true, true, 14, false);
                testParseRow(null, '-15', false, true, 0, false);
                testParseRow(null, '-15', true, true, 0, false);
                testParseRow(null, '-16', false, false);
                testParseRow(null, '-16', true, true, 255, false);
                testParseRow(null, '-255', false, false);
                testParseRow(null, '-255', true, true, 16, false);
                testParseRow(null, '-256', false, false);
                testParseRow(null, '-256', true, false);
                testParseRow(null, '1', false, true, 16, false);
                testParseRow(null, '1', true, true, 16, false);
                testParseRow(null, '240', false, true, 255, false);
                testParseRow(null, '240', true, true, 255, false);
                testParseRow(null, '241', false, false);
                testParseRow(null, '241', true, true, 0, false);
                testParseRow(null, '255', false, false);
                testParseRow(null, '255', true, true, 14, false);
                testParseRow(null, '256', false, false);
                testParseRow(null, '256', true, false);
                testParseRow(null, 'A', false, false);
            });
        });
    });

    // ========================================================================
});
