/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/funcs/statisticalfuncs'
], function (AppHelper, SheetHelper, StatisticalFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var r3d = SheetHelper.r3d;
    var r3da = SheetHelper.r3da;
    var mat = SheetHelper.mat;

    // module StatisticalFuncs ================================================

    describe('Spreadsheet module StatisticalFuncs', function () {

        // the operations to be applied at the test document
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'A2', contents: [
                [1, 2, 3, 'abc', '5', true, null, null, '',  6,   ErrorCode.DIV0],
                [1, 2, 4, 8,     16,  32,   64,   128,  256, 512, 1024]
            ] }
        ];

        // initialize the test
        var ooxAppPromise = AppHelper.createSpreadsheetApp('ooxml', OPERATIONS);
        var odfAppPromise = AppHelper.createSpreadsheetApp('odf', OPERATIONS);
        var moduleTester = SheetHelper.createFunctionModuleTester(StatisticalFuncs, ooxAppPromise, odfAppPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('AVERAGE', function (AVERAGE) {
            it('should return the average of its scalar arguments', function () {
                expect(AVERAGE(1)).to.equal(1);
                expect(AVERAGE(1, 2)).to.equal(1.5);
                expect(AVERAGE(1, 2, 6)).to.equal(3);
                expect(AVERAGE(true, 2, 6)).to.equal(3);
                expect(AVERAGE(1, '2', 6)).to.equal(3);
            });
            it('should handle empty arguments as zero', function () {
                expect(AVERAGE(1, null, 5)).to.equal(2);
            });
            it('should fail for non-numeric arguments', function () {
                expect(AVERAGE(1, 'abc', 6)).to.equal(ErrorCode.VALUE);
            });
            it('should return the average of its matrix arguments', function () {
                expect(AVERAGE(mat([[1]]))).to.equal(1);
                expect(AVERAGE(mat([[1, 2]]))).to.equal(1.5);
                expect(AVERAGE(mat([[1, 2], [2, 5]]))).to.equal(2.5);
            });
            it('should skip non-numeric matrix elements', function () {
                expect(AVERAGE(mat([[true, 1, 2]]))).to.equal(1.5);
                expect(AVERAGE(mat([[1, 'a', 2]]))).to.equal(1.5);
                expect(AVERAGE(mat([[1, '2', 2]]))).to.equal(1.5);
            });
            it('should return the average of its reference arguments', function () {
                expect(AVERAGE(r3d('0:0!A2:J2'))).to.equal(3);
            });
        });

        moduleTester.testFunction('BETA.DIST', function (BETADIST, BETADIST_ODF) {
            it('should return the statistical BETA distribution', function () {
                expect(BETADIST(2, 8, 10, true, 1, 3)).to.be.closeTo(0.6854706, 1e-7);
                expect(BETADIST(2, 8, 10, false, 1, 3)).to.be.closeTo(1.4837646, 1e-7);
                expect(BETADIST(0.5, 9, 10, true, 0, 1)).to.be.closeTo(0.592735291, 1e-7);
                expect(BETADIST(0.5, 9, 10, false, 0, 1)).to.be.closeTo(3.338470459, 1e-7);

                expect(BETADIST(2, -8, 10, true, 1, 3)).to.equal(ErrorCode.NUM);
                expect(BETADIST(2, 8, -10, true, 1, 3)).to.equal(ErrorCode.NUM);
                expect(BETADIST(1.4, 8, -10, true, 1.5, 2)).to.equal(ErrorCode.NUM); // x < a
                expect(BETADIST(2.4, 8, -10, true, 1.5, 2)).to.equal(ErrorCode.NUM); // x > b
                expect(BETADIST(2, 8, -10, true, 2, 2)).to.equal(ErrorCode.NUM); // a === b
            });
            it('should return the statistical BETA distribution for ODF', function () {
                expect(BETADIST_ODF(1, 8, 10, true, 2, 3)).to.equal(0); // x < a , cumulative
                expect(BETADIST_ODF(4, 8, 10, true, 2, 3)).to.equal(1); // x > b , cumulative
                expect(BETADIST_ODF(1, 8, 10, false, 2, 3)).to.equal(0); // x < a , not cumulative
                expect(BETADIST_ODF(4, 8, 10, false, 2, 3)).to.equal(0); // x > b , not cumulative
                expect(BETADIST_ODF(1, -8, 10, true, 1, 1)).to.equal(ErrorCode.NUM); // b - a <= 0
            });
        });

        moduleTester.testFunction('BINOM.DIST', function (BINOMDIST) {
            it('should return the statistical binomial distribution', function () {
                expect(BINOMDIST(6, 10, 0.5, false)).to.be.closeTo(0.2050781, 1e-7);
                expect(BINOMDIST(6, 10, 0.5, true)).to.be.closeTo(0.828125, 1e-7);
                expect(BINOMDIST(6, 10, 0, true)).to.equal(1); // p === 0, cumulative === true
                expect(BINOMDIST(6, 10, 0, false)).to.equal(0); // p === 0, cumulative === false
            });
            it('should return error code for invalid data', function () {
                expect(BINOMDIST(6, -0.1, 0.5, true)).to.equal(ErrorCode.NUM); // n < 0
                expect(BINOMDIST(-1, 0.1, 0.5, true)).to.equal(ErrorCode.NUM); // x < 0
                expect(BINOMDIST(6, 5, 0.5, true)).to.equal(ErrorCode.NUM); // x > n
                expect(BINOMDIST(6, 10, 1.1, true)).to.equal(ErrorCode.NUM); // p > 1
            });
        });

        moduleTester.testFunction('BINOM.INV', function (BINOMINV) {
            it('should return the inverse statistical binomial distribution', function () {
                expect(BINOMINV(6, 0.5, 0.75)).to.be.equal(4);
                expect(BINOMINV(6.5, 0.5, 0.75)).to.be.equal(4);
                expect(BINOMINV(50, 0.333, 0.75)).to.be.equal(19);
            });
            it('should return error code for invalid data', function () {
                expect(BINOMINV(50, 0.5, 1)).to.equal(ErrorCode.NA);
                expect(BINOMINV(50, 1.1, 0.5)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('BINOM.DIST.RANGE', function (BINOMDISTRANGE) {
            it('should return the inverse statistical binomial distribution', function () {
                expect(BINOMDISTRANGE(60, 0.75, 48)).to.be.closeTo(0.083974967, 1e-7);
                expect(BINOMDISTRANGE(60, 0.75, 45, 50)).to.be.closeTo(0.523629793, 1e-7);
                expect(BINOMDISTRANGE(60, 1, 45, 50)).to.be.equal(0);
                expect(BINOMDISTRANGE(60, 0, 45, 50)).to.be.equal(0);
            });
            it('should return error code for invalid data', function () {
                expect(BINOMDISTRANGE(0.9, 0.75, 48)).to.equal(ErrorCode.NA);
                expect(BINOMDISTRANGE(50, 0.1, 55)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('CHISQ.DIST', function (CHISQDIST) {
            it('should return the correct result', function () {
                expect(CHISQDIST(0.5, 1, true)).to.be.closeTo(0.52049988, 1e-7);
                expect(CHISQDIST(2, 3, false)).to.be.closeTo(0.20755375, 1e-7);
                expect(CHISQDIST(1, 1, true)).to.be.closeTo(0.682689492, 1e-7);
                expect(CHISQDIST(1, 1, false)).to.be.closeTo(0.241970725, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(CHISQDIST(1, 0.5, true)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('CHISQ.DIST.RT', function (CHISQDISTRT, CHISQDISTRT_ODF) {
            it('should return the ChiSq right tail distribution', function () {
                expect(CHISQDISTRT(18.307, 10)).to.be.closeTo(0.0500006, 1e-6);
                expect(CHISQDISTRT(18, 11)).to.be.closeTo(0.081580614, 1e-6);
                expect(CHISQDISTRT(18, 101)).to.equal(1);
            });
            it('should return the ChiSq right tail distribution for ODF', function () {
                expect(CHISQDISTRT_ODF(-1, 8)).to.equal(1);
            });
        });

        moduleTester.testFunction('CHISQ.INV', function (CHISQINV) {
            it('should return the correct result', function () {
                expect(CHISQINV(0.93, 1)).to.be.closeTo(3.283020287, 1e-7);
                expect(CHISQINV(0.6, 2)).to.be.closeTo(1.832581464, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(CHISQINV(1, 5)).to.equal(ErrorCode.NA);
                expect(CHISQINV(0.2, 0.5)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('CHISQ.INV.RT', function (CHISQINVRT) {
            it('should return the correct result', function () {
                expect(CHISQINVRT(0.050001, 10)).to.be.closeTo(18.306973, 1e-6);
                expect(CHISQINVRT(0.15, 20)).to.be.closeTo(26.49758019, 1e-6);
                expect(CHISQINVRT(0.75, 100)).to.be.closeTo(90.13321975, 1e-6);
            });
            it('should return error code for invalid data', function () {
                expect(CHISQINVRT(0, 10)).to.equal(ErrorCode.NA);
                expect(CHISQINVRT(0.6, 0.9)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('CHISQ.TEST', function (CHISQTEST) {
            var mat1 = mat([[58, 11, 10], [35, 25, 23]]);
            var mat2 = mat([[45.35, 17.56, 16.09], [47.65, 18.44, 16.91]]);
            var mat3 = mat([[1, 2], [1, 2]]);
            it('should return the correct result', function () {
                expect(CHISQTEST(mat1, mat2)).to.be.closeTo(0.0003082, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(CHISQTEST(mat1, mat3)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('CONFIDENCE.NORM', function (CONFIDENCENORM) {
            it('should return the correct result', function () {
                expect(CONFIDENCENORM(0.05, 2.5, 50)).to.be.closeTo(0.692951912, 1e-8);
                expect(CONFIDENCENORM(0.05, 2.5, 100)).to.be.closeTo(0.489990996, 1e-8);
            });
            it('should return error code for invalid data', function () {
                expect(CONFIDENCENORM(-0.05, 2.5, 100)).to.equal(ErrorCode.NA); // alpha <= 0
                expect(CONFIDENCENORM(0.05, 2.5, 0.5)).to.equal(ErrorCode.NA); // n < 1
            });
        });

        moduleTester.testFunction('CONFIDENCE.T', function (CONFIDENCET) {
            it('should return the correct result', function () {
                expect(CONFIDENCET(0.05, 1, 50)).to.be.closeTo(0.284196855, 1e-8);
                expect(CONFIDENCET(0.1, 1, 50)).to.be.closeTo(0.237100101, 1e-8);
            });
            it('should return error code for invalid data', function () {
                expect(CONFIDENCET(-0.05, 2.5, 100)).to.equal(ErrorCode.NA); // alpha <= 0
                expect(CONFIDENCET(0.05, 2.5, 0.5)).to.equal(ErrorCode.NA); // n < 1
            });
        });

        moduleTester.testAliasFunction('CORREL', 'PEARSON');

        moduleTester.testFunction('COUNT', { toOperand: true }, function (COUNT) {
            it('should return the count of scalar numbers', function () {
                expect(COUNT(1)).to.equal(1);
                expect(COUNT(1, 2, 3, 'abc', '4', true, null, null, null, 5, ErrorCode.DIV0)).to.equal(9);
            });
            it('should return the count of numbers in a matrix', function () {
                expect(COUNT(mat([[1]]))).to.equal(1);
                expect(COUNT(mat([[1, 2, 3], ['abc', '4', true], [5, 6, ErrorCode.DIV0]]))).to.equal(5);
            });
            it('should return the count of numbers in a cell range', function () {
                expect(COUNT(r3d('0:0!A2:K2'))).to.equal(4);
            });
        });

        moduleTester.testFunction('COUNTA', { toOperand: true }, function (COUNTA) {
            it('should return the count of scalar values', function () {
                expect(COUNTA(1)).to.equal(1);
                expect(COUNTA(1, 2, 3, 'abc', '4', true, null, null, null, 5, ErrorCode.DIV0)).to.equal(11);
            });
            it('should return the count of values in a matrix', function () {
                expect(COUNTA(mat([[1]]))).to.equal(1);
                expect(COUNTA(mat([[1, 2, 3], ['abc', '4', true], [5, 6, ErrorCode.DIV0]]))).to.equal(9);
            });
            it('should return the count of values in a cell range', function () {
                expect(COUNTA(r3d('0:0!A2:K2'))).to.equal(9);
            });
        });

        moduleTester.testFunction('COUNTBLANK', function (COUNTBLANK) {
            it('should return the number of blank cells', function () {
                expect(COUNTBLANK(r3da('0:0!A2:K2'))).to.equal(3);
                expect(COUNTBLANK(r3da('0:0!A2:K10000'))).to.equal(109970);
            });
        });

        moduleTester.testFunction('COUNTIF', function (COUNTIF) {
            it('should return the number of matching cells', function () {
                expect(COUNTIF(r3d('0:0!A2:K2'), '>2')).to.equal(2);
            });
            it('should count empty cells and empty string results', function () {
                expect(COUNTIF(r3d('0:0!A2:K2'), '')).to.equal(3);
                expect(COUNTIF(r3d('0:0!A2:K2'), '=')).to.equal(2);
                expect(COUNTIF(r3d('0:0!ZZ1:ZZ1000000'), '')).to.equal(1000000);
                expect(COUNTIF(r3d('0:0!ZZ1:ZZ1000000'), '=')).to.equal(1000000);
            });
        });

        moduleTester.testFunction('COUNTIFS', function (COUNTIFS) {
            it('should return the number of matching cells', function () {
                expect(COUNTIFS(r3d('0:0!A2:K2'), '<4', r3d('0:0!A3:K3'), '>1')).to.equal(2);
            });
            it('should count empty cells and empty string results', function () {
                expect(COUNTIFS(r3d('0:0!A2:K2'), '')).to.equal(3);
                expect(COUNTIFS(r3d('0:0!ZZ1:ZZ1000000'), '', r3d('0:0!ZZ1:ZZ1000000'), '=')).to.equal(1000000);
            });
        });

        moduleTester.testAliasFunction('COVAR', 'COVARIANCE.P');

        moduleTester.testFunction('COVARIANCE.P', function (COVARIANCE_P) {
            var mat1 = mat([[3, 2, 4, 5, 6]]);
            var mat2 = mat([[9, 7, 12, 15, 17]]);
            var mat5 = mat([[10]]);
            it('should return covariance, the average of the products of paired deviations', function () {
                expect(COVARIANCE_P(mat1, mat2)).to.equal(5.2);
            });
            it('should return error code for invalid data', function () {
                expect(COVARIANCE_P(mat1, mat5)).to.equal(ErrorCode.NA); // not same dimensions of matrices
            });
        });

        moduleTester.testFunction('COVARIANCE.S', function (COVARIANCE_S) {
            var mat1 = mat([[2, 4, 8]]);
            var mat2 = mat([[5, 11, 12]]);
            var mat5 = mat([[10]]);
            it('should return covariance, the average of the products of paired deviations', function () {
                expect(COVARIANCE_S(mat1, mat2)).to.be.closeTo(9.666666667, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(COVARIANCE_S(mat1, mat5)).to.equal(ErrorCode.NA); // not same dimensions of matrices
            });
        });

        moduleTester.testFunction('EXPON.DIST', function (EXPONDIST) {
            it('should return the correct result', function () {
                expect(EXPONDIST(0.2, 10, true)).to.be.closeTo(0.864664717, 1e-8);
                expect(EXPONDIST(0.2, 10, false)).to.be.closeTo(1.353352832, 1e-8);
            });
            it('should return error code for invalid data', function () {
                expect(EXPONDIST(0.5, 0, true)).to.equal(ErrorCode.NA); // lambda <= 0
            });
        });

        moduleTester.testFunction('F.DIST', function (FDIST) {
            it('should return the correct result', function () {
                expect(FDIST(15.2069, 6, 4, true)).to.be.closeTo(0.99, 1e-7);
                expect(FDIST(15.2069, 6, 4, false)).to.be.closeTo(0.001223792, 1e-8);
            });
            it('should return error code for invalid data', function () {
                expect(FDIST(15.2069, 0.6, 4, true)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('F.DIST.RT', function (FDISTRT) {
            it('should return the correct result', function () {
                expect(FDISTRT(15.2069, 6, 4)).to.be.closeTo(0.01, 1e-2);
                expect(FDISTRT(10, 8, 6)).to.be.closeTo(0.005784045, 1e-8);
            });
            it('should return error code for invalid data', function () {
                expect(FDISTRT(15.2069, 0.6, 4)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('F.INV', function (FINV) {
            it('should return the correct result', function () {
                expect(FINV(0.01, 6, 4)).to.be.closeTo(0.10930991, 1e-7);
                expect(FINV(0.6, 5, 7)).to.be.closeTo(1.194251702, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(FINV(0.01, 0.6, 4)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('F.INV.RT', function (FINVRT) {
            it('should return the correct result', function () {
                expect(FINVRT(0.01, 6, 4)).to.be.closeTo(15.20686486, 1e-7);
                expect(FINVRT(0.6, 5, 7)).to.be.closeTo(0.769927426, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(FINVRT(0.01, 0.6, 4)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('F.TEST', function (FTEST) {
            var mat1 = mat([[6, 7, 9, 15, 21]]);
            var mat2 = mat([[20, 28, 31, 38, 40]]);
            var mat3 = mat([[10, 31, 31, 15, 20]]);
            it('should return the correct result', function () {
                expect(FTEST(mat1, mat2)).to.be.closeTo(0.64831785, 1e-7);
                expect(FTEST(mat1, mat3)).to.be.closeTo(0.453048722, 1e-7);
            });
        });

        moduleTester.testFunction('FISHER', function (FISHER) {
            it('should return the Fisher statistical transformation of x', function () {
                expect(FISHER(0.75)).to.be.closeTo(0.972955075, 1e-9);
                expect(FISHER(0.55)).to.be.closeTo(0.618381314, 1e-9);
                expect(FISHER(-0.75)).to.be.closeTo(-0.972955075, 1e-9);
                expect(FISHER(-0.55)).to.be.closeTo(-0.618381314, 1e-9);
            });
            it('should throw #NUM! error for invalid arguments', function () {
                expect(FISHER(-1)).to.equal(ErrorCode.NUM);
                expect(FISHER(1)).to.equal(ErrorCode.NUM);
                expect(FISHER(5.0)).to.equal(ErrorCode.NUM);
                expect(FISHER(-5.0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testAliasFunction('FISHERINV', 'TANH');

        moduleTester.testAliasFunction('FORECAST', 'FORECAST.LINEAR');

        moduleTester.testFunction('FORECAST.LINEAR', function (FORECAST_LINEAR) {
            var mat1 = mat([[6, 7, 9, 15, 21]]);
            var mat2 = mat([[20, 28, 31, 38, 40]]);
            var mat3 = mat([[20, 28]]);
            it('should return a value along a linear trend', function () {
                expect(FORECAST_LINEAR(30, mat1, mat2)).to.be.closeTo(10.607253, 1e-6);
                expect(FORECAST_LINEAR(20, mat1, mat2)).to.be.closeTo(3.5162037, 1e-6);
            });
            it('should return error code for invalid data', function () {
                expect(FORECAST_LINEAR(30, mat1, mat3)).to.equal(ErrorCode.NA); // not same dimensions of matrices
            });
        });

        moduleTester.testFunction('FREQUENCY', function (FREQUENCY) {
            var data = mat([[79, 85, 78, 85, 50, 81, 95, 88, 97]]);
            it('should return correct frequencies', function () {
                expect(FREQUENCY(data, mat([[70, 79, 89]]))).to.deep.equal(mat([[1], [2], [4], [2]]));
                expect(FREQUENCY(data, mat([[70], [79], [89]]))).to.deep.equal(mat([[1], [2], [4], [2]]));
                expect(FREQUENCY(data, mat([[70, 75, 79, 89]]))).to.deep.equal(mat([[1], [0], [2], [4], [2]]));
                expect(FREQUENCY(data, mat([[79, 75, 70, 89]]))).to.deep.equal(mat([[2], [0], [1], [4], [2]]));
                expect(FREQUENCY(data, mat([[1, 75, 79, 500]]))).to.deep.equal(mat([[0], [1], [2], [6], [0]]));
                expect(FREQUENCY(data, mat([[500, 75, 79, 1]]))).to.deep.equal(mat([[6], [1], [2], [0], [0]]));
                expect(FREQUENCY(data, mat([[500, 75, '79', 1]]))).to.deep.equal(mat([[8], [1], [0], [0]]));
                // bug 54466: duplicate entries in "bins" causes invalid matrix
                expect(FREQUENCY(data, mat([[70, 79, 79, 89]]))).to.deep.equal(mat([[1], [2], [0], [4], [2]]));
            });
        });

        moduleTester.testFunction('GAMMA', function (GAMMA) {
            it('should return the correct result', function () {
                expect(GAMMA(2.5)).to.be.closeTo(1.329340388, 1e-7);
                expect(GAMMA(-3.75)).to.be.closeTo(0.267866129, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(GAMMA(0)).to.equal(ErrorCode.NUM);
                expect(GAMMA(-2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('GAMMA.DIST', function (GAMMADIST) {
            it('should return the correct result', function () {
                expect(GAMMADIST(10.00001131, 9, 2, false)).to.be.closeTo(0.03263913, 1e-7);
                expect(GAMMADIST(10.00001131, 9, 2, true)).to.be.closeTo(0.068094004, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(GAMMADIST(10, -9, 2, false)).to.equal(ErrorCode.NUM);
                expect(GAMMADIST(10, 9, 0, false)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('GAMMA.INV', function (GAMMAINV) {
            it('should return the correct result', function () {
                expect(GAMMAINV(0.068094, 9, 2)).to.be.closeTo(10.00001119, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(GAMMAINV(0.5, -9, 2)).to.equal(ErrorCode.NUM);
                expect(GAMMAINV(0.5, 9, 0)).to.equal(ErrorCode.NUM);
                expect(GAMMAINV(1, 9, 1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testAliasFunction('GAMMALN', 'GAMMALN.PRECISE');

        moduleTester.testFunction('GAMMALN.PRECISE', function (GAMMALN_PRECISE) {
            it('should return the correct result', function () {
                expect(GAMMALN_PRECISE(4)).to.be.closeTo(1.791759469, 1e-8);
                expect(GAMMALN_PRECISE(2.2)).to.be.closeTo(0.096947467, 1e-8);
            });
            it('should return error code for invalid data', function () {
                expect(GAMMALN_PRECISE(0)).to.equal(ErrorCode.NUM);
                expect(GAMMALN_PRECISE(-1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('GAUSS', function (GAUSS) {
            it('should return the Gauss statistical probability of x', function () {
                expect(GAUSS(2)).to.be.closeTo(0.477249868, 1e-9);
                expect(GAUSS(2.22)).to.be.closeTo(0.486790616, 1e-9);
                expect(GAUSS(-0.5)).to.be.closeTo(-0.191462461, 1e-9);
            });
        });

        moduleTester.testFunction('GROWTH', { toOperand: [0, 1, 2] }, function (GROWTH) {
            it('should return the GROWTH statistical result', function () {
                var knownY = mat([[33100, 47300, 69000, 102000, 150000, 220000]]);
                var knownX = mat([[11, 12, 13, 14, 15, 16]]);
                var newX =  mat([[17, 18]]);

                var expectedResult = mat([[32618.2037735397], [47729.4226147478], [69841.3008562175], [102197.0733788320], [149542.4867400460], [218821.8762145950]]);

                SheetHelper.assertMatrixCloseTo(GROWTH(knownY), expectedResult, 1e-7);
                SheetHelper.assertMatrixCloseTo(GROWTH(knownY, knownX), expectedResult, 1e-7);
                SheetHelper.assertMatrixCloseTo(GROWTH(knownY, knownX, newX), mat([[320196.718363472], [468536.054184048]]), 1e-7);
                SheetHelper.assertMatrixCloseTo(GROWTH(knownY, knownX, knownX), expectedResult, 1e-7);

                expect(GROWTH(knownY, newX)).to.equal(ErrorCode.REF);
                expect(GROWTH(mat([[17, 0]]))).to.equal(ErrorCode.NUM);
                expect(GROWTH(mat([[17, '18']]))).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('HYPGEOM.DIST', function (HYPGEOMDIST) {
            it('should return the correct result', function () {
                expect(HYPGEOMDIST(1, 4, 8, 20, true)).to.be.closeTo(0.465428277, 1e-7);
                expect(HYPGEOMDIST(1, 4, 8, 20, false)).to.be.closeTo(0.363261094, 1e-7);
                expect(HYPGEOMDIST(12, 45, 80000, 200000, true)).to.be.closeTo(0.044612892, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(HYPGEOMDIST(-1, 4, 8, 20, true)).to.equal(ErrorCode.NUM); // x < 0
                expect(HYPGEOMDIST(4, 1, 8, 20, true)).to.equal(ErrorCode.NUM); // n < x
                expect(HYPGEOMDIST(1, 40, 8, 20, true)).to.equal(ErrorCode.NUM); // x < n - N + M
            });
        });

        moduleTester.testFunction('INTERCEPT', function (INTERCEPT) {
            var mat1 = mat([[2, 3, 9, 1, 8]]);
            var mat2 = mat([[6, 5, 11, 7, 5]]);
            var mat3 = mat([['a']]);
            it('should return the Intercept of the linear regression line', function () {
                expect(INTERCEPT(mat1, mat2)).to.be.closeTo(0.048387097, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(INTERCEPT(mat3, mat3)).to.equal(ErrorCode.DIV0);
            });
        });

        moduleTester.testFunction('KURT', function (KURT) {
            var mat1 = mat([[3, 4, 5, 2, 3, 4, 5, 6, 4, 7]]);
            var mat2 = mat([[6, 5, 11, 7, 5, 4, 4]]);
            var mat3 = mat([['a']]);
            it('should return the correct result', function () {
                expect(KURT(mat1)).to.be.closeTo(-0.151799637, 1e-7);
                expect(KURT(mat2)).to.be.closeTo(3.155555556, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(KURT(mat3)).to.equal(ErrorCode.DIV0);
            });
        });

        moduleTester.testFunction('LARGE', { toOperand: 0 }, function (LARGE) {
            var mat1 = mat([[3, 5, 3, 5, 4], [4, 2, 4, 6, 7]]);
            it('should return the 3rd and 7th largest numbers from set', function () {
                expect(LARGE(mat1, 3)).to.equal(5);
                expect(LARGE(mat1, 7)).to.equal(4);
            });
            it('should return error code for invalid data', function () {
                expect(LARGE(mat1, 0)).to.equal(ErrorCode.NUM); // k <= 0
                expect(LARGE(mat1, 11)).to.equal(ErrorCode.NUM); // k > mat1.length
            });
        });

        moduleTester.testFunction('LINEST', function (LINEST) {
            it('should return the correct result', function () {
                expect(LINEST(mat([[3100, 3100]]))).to.deep.equal(mat([[0, 3100]]));
                expect(LINEST(mat([[3100, 4500]]), null, false)).to.deep.equal(mat([[2420, 0]]));
                expect(LINEST(mat([[3100, 4500]]), mat([[1, 2]]), true)).to.deep.equal(mat([[1400, 1700]]));
                expect(LINEST(mat([[3100], [4500], [4400], [5400], [7500], [8100]]))).to.deep.equal(mat([[1000, 2000]]));
                expect(LINEST(mat([[3100], [4500], [4400], [5400], [7500], [8100]]), mat([[1], [2], [3], [4], [5], [6]]))).to.deep.equal(mat([[1000, 2000]]));
            });
            it('should return the correct result with special 1D knownX', function () {
                var knownY = mat([[142000], [144000]]);
                SheetHelper.assertMatrixCloseTo(LINEST(knownY, null, false, false), mat([[86000, 0]]), 1e-7);
                SheetHelper.assertMatrixCloseTo(LINEST(knownY, mat([[2], [3]]), false, false), mat([[55076.92308, 0]]), 1e-5);
                SheetHelper.assertMatrixCloseTo(LINEST(knownY, mat([[2310], [2333]]), false, false), mat([[61.598726930698234, 0]]), 1e-7);
            });
            it('should return the correct result with special 2D knownX', function () {
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[0], [0]]), mat([[1, 2], [3, 4]]), false, false), mat([[0, 0, 0]]), 1e-5);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[0], [0]]), mat([[1, 0], [0, 1]]), false, false), mat([[0, 0, 0]]), 1e-5);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[1], [1]]), mat([[1, 0], [0, 1]]), false, false), mat([[1, 1, 0]]), 1e-5);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[1], [66]]), mat([[1, 0], [0, 1]]), false, false), mat([[66, 1, 0]]), 1e-5);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[66], [1]]), mat([[1, 0], [0, 1]]), false, false), mat([[1, 66, 0]]), 1e-5);

                SheetHelper.assertMatrixCloseTo(LINEST(mat([[1], [1]]), mat([[1, 2], [3, 4]]), false, false), mat([[1, -1, 0]]), 1e-5);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[66], [1]]), mat([[1, 2], [3, 4]]), false, false), mat([[98.5, -131, 0]]), 1e-5);

                SheetHelper.assertMatrixCloseTo(LINEST(mat([[1], [1], [1]]), mat([[1, 2], [3, 4], [5, 0]]), true, false), mat([[0, 0, 1]]), 1e-5);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[66], [1], [1]]), mat([[1, 2], [3, 4], [5, 0]]), true, false), mat([[-10.83333333, -21.66666667, 109.3333333]]), 1e-5);

                SheetHelper.assertMatrixCloseTo(LINEST(mat([[142000], [144000], [151000]]), mat([[2310], [2333], [2356]]), false, false), mat([[62.44612203, 0]]), 1e-5);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[142000], [144000], [151000]]), mat([[2310, 2], [2333, 2], [2356, 3]]), false, false), mat([[6041.800495, 56.39534008, 0]]), 1e-5);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[142000], [144000]]), mat([[1, 2], [3, 4]]), false, false), mat([[141000, -140000, 0]]), 1e-5);

                SheetHelper.assertMatrixCloseTo(LINEST(mat([[10], [20]]), mat([[30, 2], [40, 2]]), false, false), mat([[-10, 1, 0]]), 1e-7);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[142000], [144000]]), mat([[2310, 2], [2333, 2]]), false, false), mat([[-29434.782608695226, 86.95652173913008, 0]]), 1e-7);
            });
            it('should return the correct result with stats', function () {

                var knownY = mat([[142000], [144000], [151000], [150000], [139000], [169000], [126000], [142900], [163000], [169000], [149000]]);

                var knownX = mat([
                    [2310, 2, 2,   20],
                    [2333, 2, 2,   12],
                    [2356, 3, 1.5, 33],
                    [2379, 3, 2,   43],
                    [2402, 2, 3,   53],
                    [2425, 4, 2,   23],
                    [2448, 2, 1.5, 99],
                    [2471, 2, 2,   34],
                    [2494, 3, 3,   23],
                    [2517, 4, 4,   55],
                    [2540, 2, 3,   22]
                ]);

                var resultWithStats = mat([
                    [-234.2371645,           2553.210660391537,     12529.768167086753,    27.64138737,    52317.830507291335],
                    [13.26801148,            530.6691519,           400.0668382,           5.429374042,    12237.361602862342],
                    [0.996747993,            970.5784629,           ErrorCode.NA,          ErrorCode.NA,   ErrorCode.NA],
                    [459.7536742,            6,                     ErrorCode.NA,          ErrorCode.NA,   ErrorCode.NA],
                    [1732393319.2292507,     5652135.316203966,     ErrorCode.NA,          ErrorCode.NA,   ErrorCode.NA]
                ]);

                var resultWithoutStats = mat([[-234.2371645, 2553.210660391537, 12529.768167086753, 27.64138737, 52317.830507291335]]);
                var resultWithoutStatsNotConstant = mat([[-250.1350324,  1248.795225, 12540.261860152552, 50.71193750640334, 0]]);

                SheetHelper.assertMatrixCloseTo(LINEST(knownY, null, false, false), mat([[19788.932806324112, 0]]), 1e-7);
                SheetHelper.assertMatrixCloseTo(LINEST(knownY, knownX, false, false), resultWithoutStatsNotConstant, 1e-7);
                SheetHelper.assertMatrixCloseTo(LINEST(knownY, knownX, true, false), resultWithoutStats, 1e-7);
                SheetHelper.assertMatrixCloseTo(LINEST(knownY, knownX, true, true), resultWithStats, 1e-7);
            });
            it('should return the correct result original calcengine did not support', function () {
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[3100]])), mat([[0, 3100]]), 1e-7);
                SheetHelper.assertMatrixCloseTo(LINEST(mat([[3100]]), undefined, false, false), mat([[3100, 0]]), 1e-7);
                // still no support
                //SheetHelper.assertMatrixCloseTo(LINEST(mat([[1], [500]]), mat([[0, 0], [0, 1]]), false, false), mat([[500, 0, 0]]), 1e-5);
                //SheetHelper.assertMatrixCloseTo(LINEST(mat([[1], [1]]), mat([[0, 0], [0, 0]]), false, false), mat([[0, 0, 0]]), 1e-5);
            });
        });

        moduleTester.testFunction('LOGEST', function (LOGEST) {
            // LOGEST does not need so many tests, because its just a variation of LINEST
            it('should return the correct result with stats', function () {

                var knownY = mat([[142000], [144000], [151000], [150000], [139000], [169000], [126000], [142900], [163000], [169000], [149000]]);
                var knownX = mat([
                    [2310, 2, 2,   20],
                    [2333, 2, 2,   12],
                    [2356, 3, 1.5, 33],
                    [2379, 3, 2,   43],
                    [2402, 2, 3,   53],
                    [2425, 4, 2,   23],
                    [2448, 2, 1.5, 99],
                    [2471, 2, 2,   34],
                    [2494, 3, 3,   23],
                    [2517, 4, 4,   55],
                    [2540, 2, 3,   22]
                ]);

                var resultWithStats = mat([
                    [0.998307311, 1.018106214, 1.085434734,  1.000174374,  80387.1472131758],
                    [8.22132E-05, 0.003288211, 0.002478954,  3.36423E-05,  0.075826957],
                    [0.997224229, 0.006014042, ErrorCode.NA, ErrorCode.NA, ErrorCode.NA],
                    [538.8903983, 6,           ErrorCode.NA, ErrorCode.NA, ErrorCode.NA],
                    [0.077963872, 0.000217012, ErrorCode.NA, ErrorCode.NA, ErrorCode.NA]
                ]);

                SheetHelper.assertMatrixCloseTo(LOGEST(knownY, knownX, true, true), resultWithStats, 1e-7);
            });
        });

        moduleTester.testFunction('LOGNORM.DIST', function (LOGNORMDIST) {
            it('should return the 3rd and 7th largest numbers from set', function () {
                expect(LOGNORMDIST(4, 3.5, 1.2, true)).to.be.closeTo(0.0390836, 1e-7);
                expect(LOGNORMDIST(4, 3.5, 1.2, false)).to.be.closeTo(0.0176176, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(LOGNORMDIST(0, 3.5, 1.2, false)).to.equal(ErrorCode.NUM); // x <= 0
                expect(LOGNORMDIST(4, 3.5, 0, false)).to.equal(ErrorCode.NUM); // sigma <= 0
            });
        });

        moduleTester.testFunction('LOGNORM.INV', function (LOGNORMINV) {
            it('should return the 3rd and 7th largest numbers from set', function () {
                expect(LOGNORMINV(0.039084, 3.5, 1.2)).to.be.closeTo(4.0000252, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(LOGNORMINV(0, 3.5, 1.2, false)).to.equal(ErrorCode.NUM); // p <= 0
                expect(LOGNORMINV(4, 3.5, 0, false)).to.equal(ErrorCode.NUM); // sigma <= 0
            });
        });

        moduleTester.testAliasFunction('MODE', 'MODE.SNGL');

        moduleTester.testFunction('MODE.SNGL', function (MODE_SNGL) {
            it('should return the correct result', function () {
                expect(MODE_SNGL(mat([[4, 2, 1, 3, 4, 3, 2, 1, 2, 3, 5, 6, 1]]))).to.equal(2);
            });
            it('should throw #N/A for distinct numbers', function () {
                expect(MODE_SNGL(mat([[1, 3, 2, 4]]))).to.equal(ErrorCode.NA);
            });
            it('should throw #N/A for invalid data', function () {
                expect(MODE_SNGL(mat([['a']]))).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('MODE.MULT', function (MODE_MULT) {
            it('should return the correct result', function () {
                expect(MODE_MULT(mat([[4, 2, 1, 3, 4, 3, 2, 1, 2, 3, 5, 6, 1]]))).to.deep.equal(mat([[2], [1], [3]]));
            });
            it('should throw #N/A for distinct numbers', function () {
                expect(MODE_MULT(mat([[1, 3, 2, 4]]))).to.equal(ErrorCode.NA);
            });
            it('should throw #N/A for invalid data', function () {
                expect(MODE_MULT(mat([['a']]))).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('NEGBINOM.DIST', function (NEGBINOMDIST) {
            it('should return the negative binomial distribution', function () {
                expect(NEGBINOMDIST(10, 5, 0.25, true)).to.be.closeTo(0.3135141, 1e-7);
                expect(NEGBINOMDIST(10, 5, 0.25, false)).to.be.closeTo(0.0550487, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(NEGBINOMDIST(10, 5, -0.1, false)).to.equal(ErrorCode.NUM);
                expect(NEGBINOMDIST(-1, 5, 0.25, false)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('NORM.DIST', function (NORMDIST) {
            it('should return the statistical normal distribution', function () {
                expect(NORMDIST(42, 40, 1.5, true)).to.be.closeTo(0.9087888, 1e-7);
                expect(NORMDIST(42, 40, 1.5, false)).to.be.closeTo(0.10934, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(NORMDIST(1, 1, -1, false)).to.equal(ErrorCode.NUM);
                expect(NORMDIST(1, 1, -0.1, false)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('NORM.S.DIST', function (NORMSDIST) {
            it('should return the statistical standard normal distribution', function () {
                expect(NORMSDIST(1.333333, true)).to.be.closeTo(0.908788726, 1e-7);
                expect(NORMSDIST(1.333333, false)).to.be.closeTo(0.164010148, 1e-7);
            });
        });

        moduleTester.testFunction('NORMSDIST', function (NORMSDIST) {
            it('should return the statistical standard normal distribution', function () {
                expect(NORMSDIST(1.333333)).to.be.closeTo(0.908788726, 1e-7);
            });
        });

        moduleTester.testFunction('NORM.INV', function (NORMINV) {
            it('should return the statistical normal distribution', function () {
                expect(NORMINV(0.908789, 40, 1.5)).to.be.closeTo(42.000002, 1e-8);
                expect(NORMINV(0.908789, 40, 1.5)).not.to.be.closeTo(41.900002, 1e-8);
                expect(NORMINV(0.555555, 20, 0.5)).to.be.closeTo(20.06985445, 1e-8);
            });
            it('should return error code for invalid data', function () {
                expect(NORMINV(1, 1, 0.5)).to.equal(ErrorCode.NUM);
                expect(NORMINV(-1, 1, 0.5)).to.equal(ErrorCode.NUM);
                expect(NORMINV(2, 1, 0.5)).to.equal(ErrorCode.NUM);
                expect(NORMINV(0, 1, 0.5)).to.equal(ErrorCode.NUM);
                expect(NORMINV(1, 1, 0.5)).to.equal(ErrorCode.NUM);
                expect(NORMINV(0.5, 1, -0.5)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('NORM.S.INV', function (NORMSINV) {
            it('should return the statistical inverse normal distribution', function () {
                expect(NORMSINV(0.908789)).to.be.closeTo(1.3333347, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(NORMSINV(0)).to.equal(ErrorCode.NUM);
                expect(NORMSINV(1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PEARSON', function (PEARSON) {
            var mat1 = mat([[9, 7, 5, 3, 1]]);
            var mat2 = mat([[10, 6, 1, 5, 3]]);
            var mat3 = mat([[1, 3, 5, 7, 9]]);
            var mat5 = mat([[10]]);
            it('should return the Pearson product moment correlation coefficient', function () {
                expect(PEARSON(mat1, mat2)).to.be.closeTo(0.699379, 1e-6);
                expect(PEARSON(mat3, mat2)).to.be.closeTo(-0.699379, 1e-6);
                expect(PEARSON(mat1, mat3)).to.equal(-1);
            });
            it('should return error code for invalid data', function () {
                expect(PEARSON(mat1, mat5)).to.equal(ErrorCode.NA); // not same dimensions of matrices
            });
        });

        moduleTester.testFunction('PERMUT', function (PERMUT) {
            it('should return the permutation of assigned integers', function () {
                expect(PERMUT(100, 3)).to.equal(970200);
                expect(PERMUT(3, 2)).to.almostEqual(6);
            });
            it('should return error code for invalid data', function () {
                expect(PERMUT(0, 1)).to.equal(ErrorCode.NUM);
                expect(PERMUT(2, -1)).to.equal(ErrorCode.NUM);
                expect(PERMUT(1, 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PERMUTATIONA', function (PERMUTATIONA) {
            it('should return the permutation of assigned integers', function () {
                expect(PERMUTATIONA(4, 3)).to.equal(64);
                expect(PERMUTATIONA(0, 0)).to.almostEqual(1);
            });
            it('should return error code for invalid data', function () {
                expect(PERMUTATIONA(-1, 1)).to.equal(ErrorCode.NUM);
                expect(PERMUTATIONA(1, -1)).to.equal(ErrorCode.NUM);
                expect(PERMUTATIONA(-1, -1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PHI', function (PHI) {
            it('should return the value of the density function for a standard normal distribution', function () {
                expect(PHI(0.75)).to.be.closeTo(0.301137432, 1e-9);
                expect(PHI(0.55)).to.be.closeTo(0.342943855, 1e-9);
            });
        });

        moduleTester.testAliasFunction('POISSON', 'POISSON.DIST');

        moduleTester.testFunction('POISSON.DIST', function (POISSON_DIST) {
            it('should return the value of the Poisson distribution', function () {
                expect(POISSON_DIST(2, 5, true)).to.be.closeTo(0.124652019, 1e-9);
                expect(POISSON_DIST(2, 5, false)).to.be.closeTo(0.084224337, 1e-9);
            });
            it('should return error code for invalid data', function () {
                expect(POISSON_DIST(1, -1, true)).to.equal(ErrorCode.NUM);
                expect(POISSON_DIST(-1, 1, true)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PROB', function (PROB) {
            var mat1 = mat([[0, 1, 2, 3]]);
            var mat2 = mat([[0.2, 0.3, 0.1, 0.4]]);
            var mat3 = mat([[0.2, 0.3, 0.1, 0.1]]);
            it('should return the probability that values in a range are between two limits', function () {
                expect(PROB(mat1, mat2, 2)).to.equal(0.1);
                expect(PROB(mat1, mat2, 1, 3)).to.equal(0.8);
            });
            it('should return error code for invalid data', function () {
                expect(PROB(mat1, mat3, 2)).to.equal(ErrorCode.VALUE); // sum p < 1
            });
        });

        moduleTester.testAliasFunction('PERCENTILE', 'PERCENTILE.INC');

        moduleTester.testFunction('PERCENTILE.EXC', function (PERCENTILE_EXC) {
            var set1 = mat([[1, 2, 3, 6, 6, 6, 7, 8, 9]]);
            it('should return the correct result', function () {
                expect(PERCENTILE_EXC(set1, 0.25)).to.equal(2.5);
            });
            it('should return error code for invalid data', function () {
                expect(PERCENTILE_EXC(set1, 0)).to.equal(ErrorCode.NUM);
                expect(PERCENTILE_EXC(set1, 0.01)).to.equal(ErrorCode.NUM);
                expect(PERCENTILE_EXC(set1, 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PERCENTILE.INC', function (PERCENTILE_INC) {
            var set1 = mat([[1, 3, 2, 4]]);
            it('should return the correct result', function () {
                expect(PERCENTILE_INC(set1, 0.3)).to.equal(1.9);
            });
            it('should return error code for invalid data', function () {
                expect(PERCENTILE_INC(set1, 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testAliasFunction('PERCENTRANK', 'PERCENTRANK.INC');

        moduleTester.testFunction('PERCENTRANK.EXC', function (PERCENTRANK_EXC) {
            var mat1 = mat([[1, 2, 3, 6, 6, 6, 7, 8, 9]]);
            it('should return the correct result', function () {
                expect(PERCENTRANK_EXC(mat1, 7)).to.equal(0.7);
                expect(PERCENTRANK_EXC(mat1, 5.43)).to.equal(0.381);
                expect(PERCENTRANK_EXC(mat1, 5.43, 1)).to.equal(0.4);
            });
            it('should return error code for invalid data', function () {
                expect(PERCENTRANK_EXC(mat1, 7, 0.9)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PERCENTRANK.INC', function (PERCENTRANK_INC) {
            var mat1 = mat([[13, 12, 11, 8, 4, 3, 2, 1, 1, 1]]);
            it('should return the correct result', function () {
                expect(PERCENTRANK_INC(mat1, 2)).to.equal(0.3333);
                expect(PERCENTRANK_INC(mat1, 4)).to.equal(0.556);
                expect(PERCENTRANK_INC(mat1, 8)).to.equal(0.667);
                expect(PERCENTRANK_INC(mat1, 5)).to.equal(0.583);
            });
            it('should return error code for invalid data', function () {
                expect(PERCENTRANK_INC(mat1, 2, 0.9)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testAliasFunction('QUARTILE', 'QUARTILE.INC');

        moduleTester.testFunction('QUARTILE.EXC', function (QUARTILE_EXC) {
            var set1 = mat([[6, 7, 15, 36, 39, 40, 41, 42, 43, 47, 49]]);
            it('should return the correct result', function () {
                expect(QUARTILE_EXC(set1, 1)).to.equal(15);
                expect(QUARTILE_EXC(set1, 3)).to.equal(43);
                expect(QUARTILE_EXC(set1, 2)).to.equal(40);
            });
            it('should return error code for invalid data', function () {
                expect(QUARTILE_EXC(set1, 0)).to.equal(ErrorCode.NUM);
                expect(QUARTILE_EXC(set1, 4)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('QUARTILE.INC', function (QUARTILE_INC) {
            var set1 = mat([[1, 2, 4, 7, 8, 9, 10, 12]]);
            it('should return the correct result', function () {
                expect(QUARTILE_INC(set1, 1)).to.equal(3.5);
                expect(QUARTILE_INC(set1, 2)).to.equal(7.5);
            });
            it('should return error code for invalid data', function () {
                expect(QUARTILE_INC(set1, -1)).to.equal(ErrorCode.NUM);
                expect(QUARTILE_INC(set1, 5)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('RANK.AVG', function (RANK_AVG) {
            var set = mat([[89, 88, 92, 101, 94, 97, 95]]);
            it('should return the correct result', function () {
                expect(RANK_AVG(94, set)).to.equal(4);
                expect(RANK_AVG(94, set, 1)).to.equal(4);
                expect(RANK_AVG(94, set, 2)).to.equal(4);
                expect(RANK_AVG(101, set)).to.equal(1);
                expect(RANK_AVG(101, set, 1)).to.equal(7);
            });
            it('should return error code for invalid data', function () {
                expect(RANK_AVG(93, set)).to.equal(ErrorCode.NA); // not in the set
            });
        });

        moduleTester.testFunction('RANK.EQ', function (RANK_EQ) {
            var set = mat([[7, 3.5, 3.5, 1, 2]]);
            it('should return the correct result', function () {
                expect(RANK_EQ(7, set, 1)).to.equal(5);
                expect(RANK_EQ(2, set)).to.equal(4);
                expect(RANK_EQ(3.5, set, 1)).to.equal(3);
            });
            it('should return error code for invalid data', function () {
                expect(RANK_EQ(9, set)).to.equal(ErrorCode.NA); // not in the set
            });
        });

        moduleTester.testFunction('RSQ', function (RSQ) {
            var set1 = mat([[2, 3, 9, 1, 8, 7, 5]]);
            var set2 = mat([[6, 5, 11, 7, 5, 4, 4]]);
            it('should return the probability that values in a range are between two limits', function () {
                expect(RSQ(set1, set2)).to.be.closeTo(0.057950192, 1e-7);
            });
        });

        moduleTester.testFunction('SLOPE', function (SLOPE) {
            var mat1 = mat([[3, 4, 10, 2, 9, 8, 6]]);
            var mat2 = mat([[6, 5, 11, 7, 5, 4, 4]]);
            var mat3 = mat([['a']]);
            it('should return the Slope of the linear regression line', function () {
                expect(SLOPE(mat1, mat2)).to.be.closeTo(0.305555556, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(SLOPE(mat3, mat3)).to.equal(ErrorCode.DIV0);
            });
        });

        moduleTester.testFunction('SKEW', function (SKEW) {
            var set1 = mat([[3, 4, 5, 2, 3, 4, 5, 6, 4, 7]]);
            var set2 = mat([['a']]);
            it('should return the Slope of the linear regression line', function () {
                expect(SKEW(set1)).to.be.closeTo(0.359543071, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(SKEW(set2)).to.equal(ErrorCode.DIV0);
            });
        });

        moduleTester.testFunction('SKEW.P', function (SKEW_P) {
            var set1 = mat([[3, 4, 5, 2, 3, 4, 5, 6, 4, 7]]);
            var set2 = mat([['a']]);
            it('should return the Slope of the linear regression line', function () {
                expect(SKEW_P(set1)).to.be.closeTo(0.303193339, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(SKEW_P(set2)).to.equal(ErrorCode.DIV0);
            });
        });

        moduleTester.testFunction('STANDARDIZE', function (STANDARDIZE) {
            it('should return the Slope of the linear regression line', function () {
                expect(STANDARDIZE(42, 40, 1.5)).to.be.closeTo(1.3333333, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(STANDARDIZE(42, 40, -0.5)).to.equal(ErrorCode.NUM);
                expect(STANDARDIZE(42, 40, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('SMALL', { toOperand: 0 }, function (SMALL) {
            var mat1 = mat([[3, 5, 3, 5, 4], [4, 2, 4, 6, 7]]);
            it('should return the 3rd and 7th smallest numbers from set', function () {
                expect(SMALL(mat1, 1)).to.equal(2);
                expect(SMALL(mat1, 5)).to.equal(4);
            });
            it('should return error code for invalid data', function () {
                expect(SMALL(mat1, 0)).to.equal(ErrorCode.NUM); // k <= 0
                expect(SMALL(mat1, 11)).to.equal(ErrorCode.NUM); // k > mat1.length
            });
        });

        moduleTester.testFunction('STEYX', function (STEYX) {
            var mat1 = mat([[2, 3, 9, 1, 8, 7, 5]]);
            var mat2 = mat([[6, 5, 11, 7, 5, 4, 4]]);
            var mat3 = mat([[5, 10]]);
            it('should return the standard error of the predicted y-value for each x in the regression', function () {
                expect(STEYX(mat1, mat2)).to.be.closeTo(3.30571895, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(STEYX(mat1, mat3)).to.equal(ErrorCode.NA); // not same dimensions of matrices
                expect(STEYX(mat3, mat3)).to.equal(ErrorCode.DIV0); //  elem count < 3
            });
        });

        moduleTester.testFunction('TREND', { toOperand: [0, 1, 2] }, function (TREND) {
            var knownY = mat([[33100, 47300, 69000, 102000, 150000, 220000]]);
            var knownX = mat([[11, 12, 13, 14, 15, 16]]);
            var newX =  mat([[17, 18]]);
            it('should return the GROWTH statistical result', function () {
                var expectedResult =  mat([[12452.3809523809], [48898.0952380952], [85343.8095238095], [121789.5238095240], [158235.2380952380], [194680.9523809520]]);
                SheetHelper.assertMatrixCloseTo(TREND(knownY), expectedResult, 1e-7);
                SheetHelper.assertMatrixCloseTo(TREND(knownY, knownX), expectedResult, 1e-7);
                SheetHelper.assertMatrixCloseTo(TREND(knownY, knownX, newX), mat([[231126.666666667], [267572.380952381]]), 1e-7);
                SheetHelper.assertMatrixCloseTo(TREND(knownY, knownX, knownX), expectedResult, 1e-7);
                expect(TREND(mat([[1, 0]]))).to.deep.equal(mat([[1], [0]]));
            });
            it('should return error code for invalid data', function () {
                expect(TREND(knownY, newX)).to.equal(ErrorCode.REF);
                expect(TREND(mat([[17, '18']]))).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('TRIMMEAN', function (TRIMMEAN) {
            var set1 = mat([[4, 5, 6, 7, 2, 3, 4, 5, 1, 2, 3]]);
            var set2 = mat([[3, 6, 7, 8, 6, 5, 4, 2, 1, 9]]);
            var set3 = mat([['a']]);
            it('should return the correct result', function () {
                expect(TRIMMEAN(set1, 0.2)).to.be.closeTo(3.7777777, 1e-7);
                expect(TRIMMEAN(set2, 0.2)).to.equal(5.125);
            });
            it('should return error code for invalid data', function () {
                expect(TRIMMEAN(set3, 0.1)).to.equal(ErrorCode.VALUE); //  empty
            });
        });

        moduleTester.testFunction('T.DIST', function (TDIST) {
            it('should return the Students left-tailed t-distribution', function () {
                expect(TDIST(60, 1, true)).to.be.closeTo(0.99469533, 1e-7);
                expect(TDIST(8, 3, false)).to.be.closeTo(0.00073691, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(TDIST(60, 0, true)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('T.DIST.2T', function (TDIST2T) {
            it('should return the two-tailed Students t-distribution', function () {
                expect(TDIST2T(1.959999998, 60)).to.be.closeTo(0.054645, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(TDIST2T(60, 0)).to.equal(ErrorCode.NUM);
                expect(TDIST2T(-0.5, 60)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('T.DIST.RT', function (TDISTRT) {
            it('should return the right-tailed Students t-distribution', function () {
                expect(TDISTRT(1.959999998, 60)).to.be.closeTo(0.02732246, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(TDISTRT(60, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('T.INV', function (TINV) {
            it('should return the left-tailed inverse of the Students t-distribution', function () {
                expect(TINV(0.75, 2)).to.be.closeTo(0.8164966, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(TINV(60, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('T.INV.2T', function (TINV2T) {
            it('should return the two-tailed inverse of the Students t-distribution', function () {
                expect(TINV2T(0.546449, 60)).to.be.closeTo(0.606533, 1e-6);
            });
            it('should return error code for invalid data', function () {
                expect(TINV2T(-0.54, 20)).to.equal(ErrorCode.NUM);
                expect(TINV2T(0, 20)).to.equal(ErrorCode.NUM);
                expect(TINV2T(1.1, 20)).to.equal(ErrorCode.NUM);
                expect(TINV2T(0.54, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('T.TEST', function (TTEST) {
            var data1 = mat([[3, 4, 5, 8, 9, 1, 2, 4, 5]]);
            var data2 = mat([[6, 19, 3, 2, 14, 4, 5, 17, 1]]);
            it('should return the probability associated with a Students t-Test', function () {
                expect(TTEST(data1, data2, 2, 1)).to.be.closeTo(0.196015785, 1e-7);
                expect(TTEST(data1, data2, 2, 2)).to.be.closeTo(0.191995887, 1e-7);
                expect(TTEST(data1, data2, 2, 3)).to.be.closeTo(0.202293923, 1e-7);
            });
            it('should return error code for invalid data', function () {
                expect(TTEST(data1, data2, 0, 1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testAliasFunction('WEIBULL', 'WEIBULL.DIST');

        moduleTester.testFunction('WEIBULL.DIST', function (WEIBULLDIST) {
            it('should return the correct result', function () {
                expect(WEIBULLDIST(105, 20, 100, true)).to.be.closeTo(0.929581, 1e-6);
                expect(WEIBULLDIST(105, 20, 100, 1)).to.be.closeTo(0.929581, 1e-6);
                expect(WEIBULLDIST(105, 20, 100, false)).to.be.closeTo(0.035589, 1e-6);
                expect(WEIBULLDIST(105, 20, 100, 0)).to.be.closeTo(0.035589, 1e-6);
            });
            it('should return error code for invalid data', function () {
                expect(WEIBULLDIST(-5, 20, 100, false)).to.equal(ErrorCode.NUM);
                expect(WEIBULLDIST(105, 0, 100, false)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('Z.TEST', function (ZTEST) {
            var set = mat([[3, 6, 7, 8, 6, 5, 4, 2, 1, 9]]);
            it('should return the standard error of the predicted y-value for each x in the regression', function () {
                expect(ZTEST(set, 4)).to.be.closeTo(0.090574, 1e-6);
                expect(ZTEST(set, 6)).to.be.closeTo(0.863043, 1e-6);
            });
        });

        moduleTester.testAliasFunction('ZTEST', 'Z.TEST');
    });

    // ========================================================================
});
