/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/spreadsheet/model/condformatcollection'
], function (AppHelper, SheetHelper, DateUtils, Formatter, CondFormatCollection) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var a = SheetHelper.a;

    // class CondFormatCollection =============================================

    describe('Spreadsheet class CondFormatCollection', function () {

        it('should exist', function () {
            expect(CondFormatCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // a formatter needed for date/time tests
        var formatter = new Formatter('window-0');

        function getDate(days, months) {
            var date = DateUtils.makeUTCToday();
            if (_.isNumber(days)) {
                date.setUTCDate(date.getUTCDate() + days);
            }
            if (months) {
                date.setUTCMonth(date.getUTCMonth() + months);
            }
            return formatter.convertDateToNumber(date);
        }

        function mergeAttributes() {
            var attributeSet = _.reduce(arguments, function (attrSet, result) {
                return docModel.getCellAttributePool().extendAttributeSet(result.attributeSet, attrSet);
            }, {});
            return { attributeSet: attributeSet, renderProps: null };
        }

        function getColorResult(type, value) {
            return { attributeSet: { cell: { fillColor: { type: type, value: value } } }, renderProps: null };
        }

        function getDataBarResult(showValue, dataBarProps) {
            return { attributeSet: null, renderProps: { showValue: showValue, dataBar: dataBarProps } };
        }

        // the operations to be applied at the test document
        var RED = { type: 'rgb', value: 'FF0000' };
        var GREEN = { type: 'rgb', value: '00FF00' };
        var BLUE = { type: 'rgb', value: '0000FF' };
        var ACCENT2_BORDER = { width: 26, style: 'single', color: { type: 'scheme', value: 'accent2' } };
        var ACCENT2_ATTRS = { cell: { fillColor: { type: 'scheme', value: 'accent2' } } };
        var ACCENT3_ATTRS = { cell: { fillColor: { type: 'scheme', value: 'accent3' } } };
        var ACCENT4_ATTRS = { cell: { fillColor: { type: 'scheme', value: 'accent4' } } };
        var ACCENT5_ATTRS = { cell: { fillColor: { type: 'scheme', value: 'accent5' } } };
        var ACCENT6_ATTRS = { cell: { fillColor: { type: 'scheme', value: 'accent6' } } };
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { fileFormat: 'ooxml', cols: 16384, rows: 1048576 } } },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'A1', contents: [
                [55,      null, 0.12,    null, 'Hallo', null, '2hallo',             null, getDate(-1),       null, 5,     null, 84,      null, 'Hallo', null, ErrorCode.DIV0, null, 100, null, 88],
                ['Hallo', null, 500,     null, 'Toll',  null, 'dies ist ein text',  null, getDate(-3),       null, 568,   null, 99,      null, 'Du',    null, null,           null, 6,   null, 90],
                [5,       null, 89,      null, null,    null, null,                 null, getDate(7),        null, -41,   null, 358,     null, 1,       null, 'halo',         null, 8,   null, 0],
                [60,      null, null,    null, 1000,    null, 2,                    null, getDate(null, -1), null, 88,    null, 0,       null, null,    null, '',             null, 7,   null, -1],
                [-10,     null, '"200"', null, -5,      null, ' dies ist ein text', null, 'Da',              null, '"0"', null, '"600"', null, '',      null, ' '],
                [66,      null, -10,     null, 0.005,   null, 'haus',               null, null,              null, null,  null, null,    null, 'Du']
            ] },
            { name: 'insertCFRule', sheet: 0, id: '0000', ranges: 'A1:A6', value1: 'OR(A1=55,A1="Hallo",A1>60)', priority: 37, attrs: ACCENT6_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0001', ranges: 'C1:C6', type: 'notBetween', value1: '400', value2: '900', priority: 38, attrs: ACCENT6_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0002', ranges: 'C1:C6', type: 'between', value1: '0', value2: '300', priority: 39, attrs: { cell: { borderLeft: ACCENT2_BORDER, borderTop: ACCENT2_BORDER, borderRight: ACCENT2_BORDER, borderBottom: ACCENT2_BORDER } } },
            { name: 'insertCFRule', sheet: 0, id: '0003', ranges: 'E1:E6', type: 'equal', value1: '"toll"', priority: 36, attrs: ACCENT2_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0004', ranges: 'E1:E6', type: 'less', value1: '20', priority: 35, attrs: ACCENT6_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0005', ranges: 'G1:G6', type: 'contains', value1: '"2"', priority: 34, attrs: ACCENT6_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0006', ranges: 'G1:G6', type: 'beginsWith', value1: '"dies"', priority: 33, attrs: ACCENT2_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0007', ranges: 'I1:I6', type: 'yesterday', priority: 29, attrs: ACCENT6_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0008', ranges: 'I1:I6', type: 'last7Days', priority: 31, attrs: ACCENT5_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0009', ranges: 'I1:I6', type: 'lastMonth', priority: 32, attrs: ACCENT3_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '000A', ranges: 'I1:I6', type: 'nextWeek', priority: 30, attrs: ACCENT4_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '000B', ranges: 'K1:K5', type: 'atMostAverage', priority: 28, attrs: ACCENT5_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '000C', ranges: 'K1:K5', type: 'aboveAverage', value1: 1, priority: 26, attrs: ACCENT4_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '000D', ranges: 'M1:M5', type: 'topN', value1: 2, priority: 12, attrs: ACCENT6_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '000E', ranges: 'M1:M5', type: 'bottomPercent', value1: 30, priority: 11, attrs: ACCENT2_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '000F', ranges: 'O1:O6', type: 'duplicate', priority: 8, attrs: ACCENT6_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0010', ranges: 'O1:O6', type: 'unique', priority: 7, attrs: ACCENT2_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0011', ranges: 'Q1:Q6', type: 'blank', priority: 4, attrs: ACCENT2_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0012', ranges: 'Q1:Q6', type: 'error', priority: 3, attrs: ACCENT6_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0013', ranges: 'U1:U4', type: 'greater', value1: '80', priority: 4, attrs: _.extend({ character: { bold: true, italic: false } }, ACCENT6_ATTRS) },
            { name: 'insertCFRule', sheet: 0, id: '0014', ranges: 'S1:S4', type: 'equal', value1: '8', priority: 3, attrs: ACCENT5_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0015', ranges: 'S1:S4', type: 'greater', value1: '10', priority: 2, attrs: ACCENT6_ATTRS },
            { name: 'insertCFRule', sheet: 0, id: '0016', ranges: 'S1:S4 U1:U4', type: 'greaterEqual', value1: '90', priority: 1, stop: true, attrs: ACCENT2_ATTRS },

            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'changeCells', sheet: 1, start: 'A1', contents: [
                [-80, -40, 0, 'a', true, null, 30, 70],
                [-80, -40, 0, 40,  80],
                [-80, -40, 0, 40,  80],
                [-80, -40, 0, 40,  80]
            ] },
            { name: 'insertCFRule', sheet: 1, id: '0000', ranges: 'A1:H1', type: 'colorScale', colorScale: [{ t: 'min', c: RED }, { t: 'max', c: BLUE }] },
            { name: 'insertCFRule', sheet: 1, id: '0001', ranges: 'A2:F2', type: 'colorScale', colorScale: [{ t: 'formula', v: '-(2^7)', c: RED }, { t: 'formula', v: '2^7-1', c: BLUE }] },
            { name: 'insertCFRule', sheet: 1, id: '0002', ranges: 'A3:F3', type: 'colorScale', colorScale: [{ t: 'formula', v: '40', c: RED }, { t: 'formula', v: '-40', c: BLUE }] },
            { name: 'insertCFRule', sheet: 1, id: '0003', ranges: 'A4:F4', type: 'colorScale', colorScale: [{ t: 'max', c: RED }, { t: 'min', c: GREEN }, { t: 'formula', v: '0', c: BLUE }] },

            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'changeCells', sheet: 2, start: 'A1', contents: [
                [-80, -40, 0, 'a', true, null, 40, 80],
                [-80, -40, 0, 40,  80],
                [-80, -40, 0, 40,  80],
                [-80, -40, 0, 40,  80],
                [-80, -40, 0, 40,  80]
            ] },
            { name: 'insertCFRule', sheet: 2, id: '0000', ranges: 'A1:H1', type: 'dataBar', dataBar: { r1: { t: 'min' }, r2: { t: 'max' }, c: RED, s: true } },
            { name: 'insertCFRule', sheet: 2, id: '0001', ranges: 'A2:E2', type: 'dataBar', dataBar: { r1: { t: 'formula', v: '-60' }, r2: { t: 'formula', v: '$H$1*0.75' }, c: RED, s: false } },
            { name: 'insertCFRule', sheet: 2, id: '0002', ranges: 'A3:E3', type: 'dataBar', dataBar: { r1: { t: 'percent', v: '25' }, r2: { t: 'percent', v: '$H$1-5' }, c: RED, s: true } },
            { name: 'insertCFRule', sheet: 2, id: '0003', ranges: 'A4:E4', type: 'dataBar', dataBar: { r1: { t: 'percentile', v: '25' }, r2: { t: 'percentile', v: '$H$1-5' }, c: RED, s: true } },
            { name: 'insertCFRule', sheet: 2, id: '0004', ranges: 'A5:E5', type: 'dataBar', dataBar: { r1: { t: 'percentile', v: '25' }, r2: { t: 'percentile', v: '$H$1-5' }, c: RED, s: true, g: true } }

        ];

        // initialize test document
        var docModel = null, condFormatCollection = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            condFormatCollection = docModel.getSheetModel(0).getCondFormatCollection();
        });

        var ORANGE_BORDER = { attributeSet: { cell: {
                borderBottom: { color: { type: 'scheme', value: 'accent2' }, style: 'single', width: 26 },
                borderLeft: { color: { type: 'scheme', value: 'accent2' }, style: 'single', width: 26 },
                borderRight: { color: { type: 'scheme', value: 'accent2' }, style: 'single', width: 26 },
                borderTop: { color: { type: 'scheme', value: 'accent2' }, style: 'single', width: 26 }
            } }, renderProps: null },
            GREEN_BG = getColorResult('scheme', 'accent6'),
            ORANGE_BG = getColorResult('scheme', 'accent2'),
            BLUE_BG = getColorResult('scheme', 'accent5'),
            YELLOW_BG = getColorResult('scheme', 'accent4'),
            GRAY_BG = getColorResult('scheme', 'accent3'),
            BOLD = { attributeSet: { character: { bold: true, italic: false } } };

        it('should render formulas', function () {
            expect(condFormatCollection.resolveRules(a('A1'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('A2'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('A3'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('A4'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('A5'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('A6'))).to.deep.equal(GREEN_BG);
        });
        it('should render intervalls', function () {
            expect(condFormatCollection.resolveRules(a('C1'))).to.deep.equal(mergeAttributes(GREEN_BG, ORANGE_BORDER));
            expect(condFormatCollection.resolveRules(a('C2'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('C3'))).to.deep.equal(mergeAttributes(GREEN_BG, ORANGE_BORDER));
            expect(condFormatCollection.resolveRules(a('C4'))).to.deep.equal(mergeAttributes(GREEN_BG, ORANGE_BORDER));
            expect(condFormatCollection.resolveRules(a('C5'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('C6'))).to.deep.equal(GREEN_BG);
        });
        it('should render compare', function () {
            expect(condFormatCollection.resolveRules(a('E1'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('E2'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('E3'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('E4'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('E5'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('E6'))).to.deep.equal(GREEN_BG);
        });
        it('should render text', function () {
            expect(condFormatCollection.resolveRules(a('G1'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('G2'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('G3'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('G4'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('G5'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('G6'))).to.equal(null);
        });
        it('should render date', function () {
            expect(condFormatCollection.resolveRules(a('I1'))).to.deep.equal(GREEN_BG); // yesterday
            expect(condFormatCollection.resolveRules(a('I2'))).to.deep.equal(BLUE_BG); // last7Days
            expect(condFormatCollection.resolveRules(a('I3'))).to.deep.equal(YELLOW_BG); // next Week
            expect(condFormatCollection.resolveRules(a('I4'))).to.deep.equal(GRAY_BG); // last Month
            expect(condFormatCollection.resolveRules(a('I5'))).to.equal(null);
        });
        it('should render average', function () {
            expect(condFormatCollection.resolveRules(a('K1'))).to.deep.equal(BLUE_BG);
            expect(condFormatCollection.resolveRules(a('K2'))).to.deep.equal(YELLOW_BG);
            expect(condFormatCollection.resolveRules(a('K3'))).to.deep.equal(BLUE_BG);
            expect(condFormatCollection.resolveRules(a('K4'))).to.deep.equal(BLUE_BG);
            expect(condFormatCollection.resolveRules(a('K5'))).to.equal(null);
        });
        it('should render top/bottom', function () {
            expect(condFormatCollection.resolveRules(a('M1'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('M2'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('M3'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('M4'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('M5'))).to.equal(null);
        });
        it('should render quantity', function () {
            expect(condFormatCollection.resolveRules(a('O1'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('O2'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('O3'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('O4'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('O5'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('O6'))).to.deep.equal(GREEN_BG);
        });
        it('should render type', function () {
            expect(condFormatCollection.resolveRules(a('Q1'))).to.deep.equal(GREEN_BG);
            expect(condFormatCollection.resolveRules(a('Q2'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('Q3'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('Q4'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('Q5'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('Q6'))).to.deep.equal(ORANGE_BG);
        });
        it('should render rule over two ranges and stop', function () {
            expect(condFormatCollection.resolveRules(a('S1'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('S2'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('S3'))).to.deep.equal(BLUE_BG);
            expect(condFormatCollection.resolveRules(a('S4'))).to.equal(null);
            expect(JSON.stringify(condFormatCollection.resolveRules(a('U1')))).to.deep.equal(JSON.stringify(mergeAttributes(GREEN_BG, BOLD)));
            expect(condFormatCollection.resolveRules(a('U2'))).to.deep.equal(ORANGE_BG);
            expect(condFormatCollection.resolveRules(a('U3'))).to.equal(null);
            expect(condFormatCollection.resolveRules(a('U4'))).to.equal(null);
        });

        it('should resolve color scales', function () {
            var collection = docModel.getSheetModel(1).getCondFormatCollection();
            // minimum/maximum
            expect(collection.resolveRules(a('A1'))).to.deep.equal(getColorResult('rgb', 'FF0000'));
            expect(collection.resolveRules(a('B1'))).to.deep.equal(getColorResult('rgb', 'BB0044'));
            expect(collection.resolveRules(a('C1'))).to.deep.equal(getColorResult('rgb', '770088'));
            expect(collection.resolveRules(a('D1'))).to.equal(null);
            expect(collection.resolveRules(a('E1'))).to.equal(null);
            expect(collection.resolveRules(a('F1'))).to.equal(null);
            expect(collection.resolveRules(a('G1'))).to.deep.equal(getColorResult('rgb', '4400BB'));
            expect(collection.resolveRules(a('H1'))).to.deep.equal(getColorResult('rgb', '0000FF'));
            // calculated start/end
            expect(collection.resolveRules(a('A2'))).to.deep.equal(getColorResult('rgb', 'CF0030'));
            expect(collection.resolveRules(a('B2'))).to.deep.equal(getColorResult('rgb', 'A70058'));
            expect(collection.resolveRules(a('C2'))).to.deep.equal(getColorResult('rgb', '7F0080'));
            expect(collection.resolveRules(a('D2'))).to.deep.equal(getColorResult('rgb', '5700A8'));
            expect(collection.resolveRules(a('E2'))).to.deep.equal(getColorResult('rgb', '2F00D0'));
            // values outside interval; swapped minimum/maximum
            expect(collection.resolveRules(a('A3'))).to.deep.equal(getColorResult('rgb', 'FF0000'));
            expect(collection.resolveRules(a('B3'))).to.deep.equal(getColorResult('rgb', 'FF0000'));
            expect(collection.resolveRules(a('C3'))).to.deep.equal(getColorResult('rgb', '800080'));
            expect(collection.resolveRules(a('D3'))).to.deep.equal(getColorResult('rgb', '0000FF'));
            expect(collection.resolveRules(a('E3'))).to.deep.equal(getColorResult('rgb', '0000FF'));
            // three-step with unordered color stops
            expect(collection.resolveRules(a('A4'))).to.deep.equal(getColorResult('rgb', 'FF0000'));
            expect(collection.resolveRules(a('B4'))).to.deep.equal(getColorResult('rgb', '808000'));
            expect(collection.resolveRules(a('C4'))).to.deep.equal(getColorResult('rgb', '00FF00'));
            expect(collection.resolveRules(a('D4'))).to.deep.equal(getColorResult('rgb', '008080'));
            expect(collection.resolveRules(a('E4'))).to.deep.equal(getColorResult('rgb', '0000FF'));
        });

        it('should resolve data bar rules', function () {
            var collection = docModel.getSheetModel(2).getCondFormatCollection();
            expect(collection.resolveRules(a('A1'))).to.deep.equal(getDataBarResult(true, { min: -80, max: 80, num: -80, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('B1'))).to.deep.equal(getDataBarResult(true, { min: -80, max: 80, num: -40, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('C1'))).to.deep.equal(getDataBarResult(true, { min: -80, max: 80, num: 0, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('D1'))).to.equal(null);
            expect(collection.resolveRules(a('E1'))).to.equal(null);
            expect(collection.resolveRules(a('F1'))).to.equal(null);
            expect(collection.resolveRules(a('G1'))).to.deep.equal(getDataBarResult(true, { min: -80, max: 80, num: 40, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('H1'))).to.deep.equal(getDataBarResult(true, { min: -80, max: 80, num: 80, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('I1'))).to.equal(null);
            expect(collection.resolveRules(a('A2'))).to.deep.equal(getDataBarResult(false, { min: -60, max: 60, num: -80, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('E2'))).to.deep.equal(getDataBarResult(false, { min: -60, max: 60, num: 80, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('A3'))).to.deep.equal(getDataBarResult(true, { min: -40, max: 40, num: -80, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('E3'))).to.deep.equal(getDataBarResult(true, { min: -40, max: 40, num: 80, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('A4'))).to.deep.equal(getDataBarResult(true, { min: -40, max: 40, num: -80, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('E4'))).to.deep.equal(getDataBarResult(true, { min: -40, max: 40, num: 80, color1: '#ff0000', color2: null }));
            expect(collection.resolveRules(a('E5'))).to.deep.equal(getDataBarResult(true, { min: -40, max: 40, num: 80, color1: '#ff0000', color2: 'hsl(0,100%,85%)' }));
        });
    });

    // ========================================================================
});
