/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/tk/keycodes',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/presentation/model/odflayoutmixin',
    'io.ox/office/presentation/utils/presentationutils'
], function (AppHelper, KeyCodes, AttributeUtils, DOM, Position, OdfLayoutMixin, PresentationUtils) {

    'use strict';

    // mix-in class OdfLayoutMixin =======================================

    describe('Presentation mix-in class OdfLayoutMixin', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            selection = null,
            masterId_1 = 'master1',
            slideDefaultName = 'slide_1',
            secondSlideDefaultName = 'slide_2',
            activeSlideId = null,
            activeSlide = null,
            drawings = null,
            drawing1Attrs = null,
            drawing2Attrs = null,
            drawing3Attrs = null,
            defaultTitleTop = 800,
            defaultBodyTop = 5000,
            loadTitleTop = 900,
            loadBodyTop = 5100,
            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: {
                        page: { marginRight: 0, width: 28000, marginBottom: 0, marginTop: 0, marginLeft: 0, height: 21000 }
                    }
                },
                {
                    name: 'insertMasterSlide', id: masterId_1, attrs: {
                        listStyles: {
                            title: {
                                l1: { paragraph: { bulletFont: { followText: false, name: 'StarSymbol' }, alignment: 'center', indentLeft: 600 }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 44, fontSizeAsian: 44, boldAsian: false, fontSizeComplex: 44, italicAsian: false } }
                            },
                            body: {
                                l1: { paragraph: { indentFirstLine: -600, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 500 }, bullet: { character: '-', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 500 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 32, fontSizeAsian: 32, boldAsian: false, fontSizeComplex: 32, italicAsian: false } },
                                l2: { paragraph: { indentFirstLine: 600, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l3: { paragraph: { indentFirstLine: 1200, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l4: { paragraph: { indentFirstLine: 1800, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l5: { paragraph: { indentFirstLine: 2400, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l6: { paragraph: { indentFirstLine: 3000, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l7: { paragraph: { indentFirstLine: 3600, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l8: { paragraph: { indentFirstLine: 4200, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } },
                                l9: { paragraph: { indentFirstLine: 4800, bulletFont: { followText: false, name: 'Arial' }, bulletSize: { type: 'followText' }, spacingBefore: { type: 'fixed', value: 400 }, bullet: { character: '\u2022', type: 'character' }, indentLeft: 900, spacingAfter: { type: 'fixed', value: 400 } }, character: { color: { type: 'auto' }, italicComplex: false, underline: false, strike: 'none', fontNameComplex: 'Lucida Sans', bold: false, italic: false, caps: 'none', fillColor: { type: 'auto' }, fontName: 'Liberation Sans', boldComplex: false, fontNameAsian: 'Microsoft YaHei', fontSize: 28, fontSizeAsian: 28, boldAsian: false, fontSizeComplex: 28, italicAsian: false } }
                            }
                        },
                        fill: { type: 'solid' }
                    }
                },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { presentation: { phType: 'title' }, drawing: { top: defaultTitleTop, left: 1400, width: 25199, height: 3506 } }, target: masterId_1 },
                { name: 'insertParagraph', start: [0, 0, 0], target: masterId_1, attrs: { paragraph: { bullet: { type: 'none' } } } },
                { name: 'insertText', start: [0, 0, 0, 0], target: masterId_1, text: 'Click to edit Master title style' },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { presentation: { phType: 'body' }, drawing: { top: defaultBodyTop, left: 1400, width: 25199, height: 12179 } }, target: masterId_1 },
                { name: 'insertParagraph', start: [0, 1, 0], target: masterId_1, attrs: { paragraph: { level: 0 } } },
                { name: 'insertText', start: [0, 1, 0, 0], target: masterId_1, text: 'Edit Master text styles' },
                { name: 'insertParagraph', start: [0, 1, 1], target: masterId_1, attrs: { paragraph: { level: 1 } } },
                { name: 'insertText', start: [0, 1, 1, 0], target: masterId_1, text: 'Second level' },
                { name: 'insertSlide', start: [0], target: masterId_1, attrs: { slide: { isSlideNum: false, isDate: false, isFooter: false } } },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { presentation: { userTransformed: true, phType: 'title' }, drawing: { top: loadTitleTop, left: 1400, width: 25199, height: 3506 } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertText', start: [0, 0, 0, 0], text: 'Title' },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { presentation: { phType: 'body' }, drawing: { top: loadBodyTop, left: 1400, width: 25199, height: 12179 } } },
                { name: 'insertParagraph', start: [0, 1, 0], attrs: { paragraph: { level: 0, listStyleId: 'L1' } } },
                { name: 'insertText', start: [0, 1, 0, 0], text: 'Item' }
            ];

        function waitForEvent(source, type) {
            return model.waitForEvent(source, type, 1000);
        }

        AppHelper.createPresentationApp('odf', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(OdfLayoutMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "changeODFLayout"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('changeODFLayout');
            });

            it('should have the first slide activated', function () {
                activeSlideId = model.getActiveSlideId();
                expect(activeSlideId).to.equal(slideDefaultName);
            });

            it('should find two drawings on the active slide', function () {
                activeSlide = model.getSlideById(activeSlideId);
                expect(activeSlide.length).to.equal(1);
                drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(2);
            });

            it('should find the shifted drawings on the active slide after loading', function () {
                expect(PresentationUtils.getPlaceHolderDrawingType(drawings[0])).to.equal('title');
                expect(PresentationUtils.getPlaceHolderDrawingType(drawings[1])).to.equal('body');

                drawing1Attrs = AttributeUtils.getExplicitAttributes(drawings[0]);
                drawing2Attrs = AttributeUtils.getExplicitAttributes(drawings[1]);

                expect(drawing1Attrs.drawing.top).to.equal(loadTitleTop);
                expect(drawing2Attrs.drawing.top).to.equal(loadBodyTop);
            });

            it('should move the drawings to its default positions after loading, if the layout is not changed', function () {

                model.changeODFLayout('layout_3', activeSlideId); // switching to 'Title and Content'

                drawing1Attrs = AttributeUtils.getExplicitAttributes(drawings[0]);
                drawing2Attrs = AttributeUtils.getExplicitAttributes(drawings[1]);

                expect(drawing1Attrs.drawing.top).to.equal(defaultTitleTop);
                expect(drawing2Attrs.drawing.top).to.equal(defaultBodyTop);
            });

            it('should move the drawings to its default positions after moving, if the layout is not changed', function () {

                // selecting all drawings
                selection.selectAllDrawingsOnSlide();
                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(2);

                // ... and pressing key down
                model.handleDrawingOperations({ keyCode: KeyCodes.DOWN_ARROW, shiftKey: false, altKey: false });

                drawing1Attrs = AttributeUtils.getExplicitAttributes(drawings[0]);
                drawing2Attrs = AttributeUtils.getExplicitAttributes(drawings[1]);

                expect(drawing1Attrs.drawing.top).to.be.above(defaultTitleTop);
                expect(drawing2Attrs.drawing.top).to.be.above(defaultBodyTop);

                // userTransformed must be 'true' nach move operation
                expect(drawing1Attrs.presentation.userTransformed).to.equal(true);
                expect(drawing2Attrs.presentation.userTransformed).to.equal(true);

                model.changeODFLayout('layout_3', activeSlideId); // switching to 'Title and Content'

                drawing1Attrs = AttributeUtils.getExplicitAttributes(drawings[0]);
                drawing2Attrs = AttributeUtils.getExplicitAttributes(drawings[1]);

                expect(drawing1Attrs.drawing.top).to.equal(defaultTitleTop);
                expect(drawing2Attrs.drawing.top).to.equal(defaultBodyTop);

                // userTransformed must not be set after reset to default values
                expect(drawing1Attrs.presentation.userTransformed).to.be.undefined;
                expect(drawing2Attrs.presentation.userTransformed).to.be.undefined;
            });

            it('should add additional place holders if specified by the layout', function () {

                model.changeODFLayout('layout_4', activeSlideId); // switching to 'Title and 2 Contents'

                drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(3);
            });

            it('should add empty place holders if specified by the layout', function () {
                expect(PresentationUtils.isEmptyPlaceHolderDrawing(drawings[0])).to.equal(false);
                expect(PresentationUtils.isEmptyPlaceHolderDrawing(drawings[1])).to.equal(false);
                expect(PresentationUtils.isEmptyPlaceHolderDrawing(drawings[2])).to.equal(true); // new place holder
            });

            it('should restore missing place holder drawings after deletion', function () {

                var oxoPos = Position.getOxoPosition(model.getNode(), drawings[2], true);
                expect(oxoPos.length).to.equal(2);
                expect(oxoPos[0]).to.equal(0);
                expect(oxoPos[1]).to.equal(2);

                selection.setTextSelection(oxoPos, Position.increaseLastIndex(oxoPos)); // selecting the empty place holder drawing
                expect(selection.isDrawingSelection()).to.equal(true);

                // deleting all selected drawings with 'delete' or 'backspace'
                return model.deleteSelected({ deleteKey: true }).then(function () {

                    drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                    expect(drawings.length).to.equal(2); // only two remaining drawings

                    model.changeODFLayout('layout_4', activeSlideId); // switching to 'Title and 2 Contents'

                    drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                    expect(drawings.length).to.equal(3);

                    expect(PresentationUtils.isEmptyPlaceHolderDrawing(drawings[0])).to.equal(false);
                    expect(PresentationUtils.isEmptyPlaceHolderDrawing(drawings[1])).to.equal(false);
                    expect(PresentationUtils.isEmptyPlaceHolderDrawing(drawings[2])).to.equal(true); // new empty place holder
                });
            });

            it('should increase the number of place holder drawings if required and possible', function () {

                model.changeODFLayout('layout_12', activeSlideId); // switching to 'Title and 6 Contents'

                drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(7);
            });

            it('should decrease the number of place holder drawings if required and possible', function () {

                model.changeODFLayout('layout_4', activeSlideId); // switching to 'Title and 2 Contents'

                drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(3);

                model.changeODFLayout('layout_5', activeSlideId); // switching to 'Title only'

                drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(2); // still two drawings, because both have content

                // checking userTransformed
                drawing1Attrs = AttributeUtils.getExplicitAttributes(drawings[0]);
                drawing2Attrs = AttributeUtils.getExplicitAttributes(drawings[1]);

                // userTransformed must be set to place holder that is not specified on the layout slide
                expect(drawing1Attrs.presentation.userTransformed).to.be.undefined; // the title
                expect(drawing2Attrs.presentation.userTransformed).to.equal(true); // the body is not specified for this layout

                model.changeODFLayout('layout_4', activeSlideId); // switching to 'Title and 2 Contents'

                drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(3);

                expect(PresentationUtils.isEmptyPlaceHolderDrawing(drawings[0])).to.equal(false);
                expect(PresentationUtils.isEmptyPlaceHolderDrawing(drawings[1])).to.equal(false);
                expect(PresentationUtils.isEmptyPlaceHolderDrawing(drawings[2])).to.equal(true); // new empty place holder

                // userTransformed must not be set after reset to default values
                drawing1Attrs = AttributeUtils.getExplicitAttributes(drawings[0]);
                drawing2Attrs = AttributeUtils.getExplicitAttributes(drawings[1]);
                drawing3Attrs = AttributeUtils.getExplicitAttributes(drawings[2]);

                expect(drawing1Attrs.presentation.userTransformed).to.be.undefined;
                expect(drawing2Attrs.presentation.userTransformed).to.be.undefined;
                expect(drawing3Attrs.presentation.userTransformed).to.be.undefined;
            });

            it('should generate the correct undo operations', function () {

                drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(3);

                model.changeODFLayout('layout_12', activeSlideId); // switching to 'Title and 6 Contents'

                drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawings.length).to.equal(7);

                // restoring the drawings again
                return model.getUndoManager().undo().then(function () {
                    drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                    expect(drawings.length).to.equal(3);
                });
            });

            it('should insert a new slide with the specified layout ID', function () {

                var promise = waitForEvent(model, 'change:activeslide:done');

                expect(model.getStandardSlideCount()).to.equal(1);
                model.insertSlide('layout_11'); // inserting new slide with layout 'Title, 4 Contents'
                expect(model.getStandardSlideCount()).to.equal(2);

                return promise.then(function () {
                    expect(model.getActiveSlideIndex()).to.equal(1); // asynchronous slide formatting has to be finished

                    activeSlideId = model.getActiveSlideId();
                    expect(activeSlideId).to.equal(secondSlideDefaultName);

                    activeSlide = model.getSlideById(activeSlideId);
                    expect(activeSlide.length).to.equal(1);

                    drawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                    expect(drawings.length).to.equal(5);
                });
            });
        });
    });
});
