/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('baseobject/timermixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/scheduler'
], function (Utils, Scheduler) {

    'use strict';

    // mix-in class TimerMixin ================================================

    /**
     * A mix-in class for any class based on BaseObject that adds helper
     * methods for delayed, repeated, and debounced code execution. All timers
     * created by these methods consider the lifetime of the instance they have
     * been invoked from. If the instance has been destroyed, no pending timer
     * code for that instance will be executed anymore. See the descriptions of
     * the methods for more details.
     *
     * Intended as mix-in for the class BaseObject or any of its sub classes.
     * The constructor MUST be called with the calling context bound to the
     * object instance to be extended.
     *
     * @constructor
     */
    function TimerMixin() {

        // public methods -----------------------------------------------------

        /**
         * Invokes the passed callback function once in a browser timeout. If
         * this instance will be destroyed before the callback function has
         * been started, it will not be called anymore.
         *
         * Multiple timer requests (also from different instances of this
         * mix-in class) will be synchronized. Between two callback functions,
         * a minimum interval time of 10 milliseconds will be added, in order
         * to allow the browser to process its event queue, rendering queue,
         * and to process network responses.
         *
         * @param {Function} callback
         *  The callback function that will be executed in a browser timeout
         *  after the delay time. Does not receive any parameters. Will be
         *  called in the context of this instance.
         *
         * @param {String} logMessage
         *  A description for the timer that will be inserted into the debug
         *  log used in automated test environments.
         *
         * @param {Object|Number} [options]
         *  Optional parameters:
         *  @param {Number} [options.delay=0]
         *      The time (in milliseconds) the execution of the passed callback
         *      function will be delayed at least.
         *  @param {Boolean} [options.background=false]
         *      Whether the timer is part of a long-running background task
         *      with low priority. These background tasks will be ignored when
         *      waiting for pending tasks in automated test environments.
         *  This parameter can also be a number, allowing to pass the value of
         *  the option 'delay' directly.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the callback function has
         *  been executed. If the callback function returns a simple value or
         *  object, the promise will be resolved with that value. If the
         *  callback function returns a promise by itself, its state and result
         *  value will be forwarded to the promise returned by this method. The
         *  promise contains an additional method abort() that can be called
         *  before the timeout has been fired to cancel the pending execution
         *  of the callback function. In that case, the promise will be
         *  rejected. If this instance will be destroyed before invoking the
         *  callback function, the timer will be aborted automatically, and the
         *  promise will be rejected.
         */
        this.executeDelayed = function (callback, logMessage, options) {

            // the delay time for the next execution of the callback
            var delay = _.isNumber(options) ? Math.max(0, options) : Utils.getIntegerOption(options, 'delay', 0, 0);
            // a pending abortable result promise returned by the callback function
            var abortablePromise = null;
            // the deferred object representing the timer state
            var deferred = Scheduler.createDeferred(this, logMessage, options);

            // bind callback to this instance
            callback = callback.bind(this);

            // create a new synchronized timer
            var timerId = Scheduler.setTimeout(function () {

                // timer fired successfully: immediately clear the timer identifier
                timerId = null;

                // the result of the callback function
                var result = callback();

                // forward the result to the promise returned by this method
                if (Utils.isPromise(result)) {
                    if ((result.state() === 'pending') && _.isFunction(result.abort)) { abortablePromise = result; }
                    result.done(deferred.resolve.bind(deferred));
                    result.fail(deferred.reject.bind(deferred));
                } else {
                    deferred.resolve(result);
                }
            }, delay, this, logMessage, options);

            // the abortable promise for the deferred object
            return this.createAbortablePromise(deferred, function () {
                if (timerId) { Scheduler.clearTimeout(timerId); timerId = null; }
                if (abortablePromise) { abortablePromise.abort(); }
            });
        };

        /**
         * Invokes the passed callback function repeatedly in a browser timeout
         * loop. If this instance will be destroyed before the callback
         * function has been started, or while the callback function will be
         * repeated, it will not be called anymore.
         *
         * @param {Function} callback
         *  The callback function that will be invoked repeatedly in a browser
         *  timeout loop after the initial delay time. Receives the zero-based
         *  index of the execution cycle as first parameter. Will be called in
         *  the context of this instance. The return value of the function will
         *  be used to decide whether to continue the repeated execution. If
         *  the function returns Utils.BREAK, execution will be stopped. If the
         *  function returns a promise, looping will be deferred until the
         *  promise is resolved or rejected. After resolving the promise, the
         *  delay time will start, and the callback will be invoked again.
         *  Otherwise, after the promise has been rejected, execution will be
         *  stopped, and the promise returned by this method will be rejected
         *  too. All other return values will be ignored, and the callback loop
         *  will continue.
         *
         * @param {String} logMessage
         *  A description for the timer loop that will be inserted into the
         *  debug log used in automated test environments.
         *
         * @param {Object|Number} [options]
         *  Optional parameters:
         *  @param {Number|String} [options.delay=0]
         *      The time (in milliseconds) the initial invocation of the passed
         *      callback function will be delayed. If set to the special value
         *      'immediate', the callback function will be executed immediately
         *      (synchronously) with the invocation of this method.
         *  @param {Number} [options.repeatDelay]
         *      The time (in milliseconds) the repeated invocation of the
         *      passed callback function will be delayed. Must not be less than
         *      10 milliseconds. If omitted, the specified initial delay time
         *      passed with the option 'delay' will be used, unless it is less
         *      than 10 milliseconds, or has been set to 'immediate' (in this
         *      case, the default value of 10 milliseconds will be used).
         *  @param {Number} [options.cycles]
         *      If specified, the maximum number of cycles to be executed.
         *  @param {Boolean} [options.fastAsync=false]
         *      If set to true, and the callback function returns a pending
         *      promise, the delay time for the next invocation of the callback
         *      function will be reduced by the time the promise is pending. By
         *      default, the delay time always starts after the promise has
         *      been resolved.
         *  @param {Boolean} [options.background=false]
         *      Whether the timer loop represents a long-running background
         *      task with low priority. These background tasks will be ignored
         *      when waiting for pending tasks in automated test environments.
         *  This parameter can also be a number, allowing to pass the value of
         *  the option 'delay' (and also 'repeatDelay' which defaults to the
         *  value of the option 'delay') directly.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the callback
         *  function has been invoked the last time. The promise will be
         *  resolved, if the callback function has returned the Utils.BREAK
         *  object in the last iteration, or if the maximum number of
         *  iterations (see option 'cycles') has been reached. It will be
         *  rejected, if the callback function has returned a rejected promise.
         *  The returned promise contains an additional method abort() that can
         *  be called before or while the callback loop is executed to stop the
         *  loop immediately. In that case, the promise will be rejected. If
         *  this instance will be destroyed before or during the callback loop,
         *  the loop timer will be aborted automatically, and the promise will
         *  be rejected.
         */
        this.repeatDelayed = function (callback, logMessage, options) {

            // the delay time for the first invocation of the callback
            var initDelay = _.isNumber(options) ? Math.max(0, options) : Utils.getIntegerOption(options, 'delay', 0, 0);
            // the delay time for the next invocations of the callback
            var repeatDelay = Utils.getIntegerOption(options, 'repeatDelay', initDelay, 10);
            // the number of cycles to be executed
            var cycles = Utils.getIntegerOption(options, 'cycles');
            // whether to reduce delay time after asynchronous callbacks
            var fastAsync = Utils.getBooleanOption(options, 'fastAsync', false);
            // whether this timer loop is a background task
            var background = Utils.getBooleanOption(options, 'background', false);
            // the index of the next cycle
            var index = 0;
            // the current timer promise that defers invoking the callback
            var timer = null;
            // current pending abortable result promise returned by the callback function
            var abortablePromise = null;
            // the resulting deferred object
            var deferred = Scheduler.createDeferred(this, logMessage, options);

            // bind callback to this instance
            callback = callback.bind(this);

            // invokes the callback function
            function invokeCallback() {

                // the result of the callback, immediately break the loop if callback returns Utils.BREAK
                var result = callback(index);
                if (result === Utils.BREAK) {
                    deferred.resolve(Utils.BREAK);
                    return;
                }

                // wait for the result
                if (Utils.isPromise(result)) {
                    if ((result.state() === 'pending') && _.isFunction(result.abort)) { abortablePromise = result; }
                    var t0 = (result.state() === 'pending') ? _.now() : -1;
                    result.done(function () { startNextTimer(t0); });
                    result.fail(deferred.reject.bind(deferred));
                } else {
                    startNextTimer(-1);
                }
            }

            // creates and registers a browser timeout that executes the callback
            var createTimer = function (delay) {
                abortablePromise = null;
                timer = this.executeDelayed(invokeCallback, logMessage, background ? { delay: delay, background: true } : delay);
            }.bind(this);

            function startNextTimer(t0) {
                // do not create a new timer if execution has been aborted while the asynchronous code was running
                if (deferred.state() !== 'pending') { return; }
                // decide whether to start the next iteration
                index += 1;
                if (!_.isNumber(cycles) || (index < cycles)) {
                    createTimer((fastAsync && (t0 > 0)) ? Math.max(0, repeatDelay - (_.now() - t0)) : repeatDelay);
                } else {
                    deferred.resolve(null);
                }
            }

            // immediately invoke callback the first time if specified, otherwise start the initial timer
            if (Utils.getStringOption(options, 'delay') === 'immediate') {
                invokeCallback();
            } else {
                createTimer(initDelay);
            }

            // the abortable promise for the deferred object
            return this.createAbortablePromise(deferred, function () {
                if (timer) { timer.abort(); timer = null; }
                if (abortablePromise) { abortablePromise.abort(); }
            });
        };

        /**
         * Invokes the passed callback function repeatedly until it returns a
         * specific value. To prevent performance problems (frozen user
         * interface of the browser), the callback function may be invoked from
         * several time slices of a browser timeout loop. In difference to the
         * method TimerMixin.repeatDelayed() that uses an own browser timeout
         * for each invocation of the callback function, this method tries to
         * pack multiple invocations into a single time slice until the total
         * execution time of the time slice is exceeded, and continues with a
         * new time slice in another browser timeout afterwards. If this
         * instance will be destroyed while the loop is running, it will be
         * aborted immediately.
         *
         * @param {Function} callback
         *  The callback function that will be invoked repeatedly. Receives the
         *  zero-based index of the execution cycle as first parameter. Will be
         *  called in the context of this instance. The return value of the
         *  function will be used to decide whether to continue the repeated
         *  execution (see parameter 'callback' of the method
         *  TimerMixin.repeatDelayed() for details).
         *
         * @param {String} logMessage
         *  A description for the timer loop that will be inserted into the
         *  debug log used in automated test environments.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number|String} [options.delay=0]
         *      The initial delay time (in milliseconds) before invoking the
         *      callback function the first time. If set to the special value
         *      'immediate', the callback function will be executed immediately
         *      (synchronously) with the invocation of this method.
         *  @param {Number} [options.slice=200]
         *      The time (in milliseconds) for a single synchronous time slice.
         *      The callback function will be invoked repeatedly until the time
         *      has elapsed; or until the callback function returns
         *      Utils.BREAK, a pending promise, or a rejected promise (see
         *      parameter 'callback' above for more details).
         *  @param {Number} [options.interval=10]
         *      The delay time (in milliseconds) between two time slices. If
         *      the callback function returns a pending promise, the
         *      implementation will wait for that promise to resolve, and will
         *      add a shorter delay (shortened by the time the promise has
         *      needed to resolve) before continuing the loop.
         *  @param {Number} [options.cycles]
         *      If specified, the maximum number of cycles to be executed.
         *  @param {Boolean} [options.background=false]
         *      Whether the timer loop represents a long-running background
         *      task with low priority. These background tasks will be ignored
         *      when waiting for pending tasks in automated test environments.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the callback function returns
         *  Utils.BREAK, or if the maximum number of iterations (see option
         *  'cycles') has been reached; or that will be rejected, if the
         *  callback function returns a rejected promise. The promise will be
         *  notified each time a new time slice starts (with the zero-based
         *  index of the next iteration cycle). The promise contains an
         *  additional method abort() that can be called to abort the loop
         *  immediately. In that case, the promise will be rejected. If this
         *  instance will be destroyed before or during the callback loop, the
         *  loop timer will be aborted automatically, and the promise will be
         *  rejected.
         */
        this.repeatSliced = function (callback, logMessage, options) {

            // slice time for synchronous processing
            var slice = Utils.getIntegerOption(options, 'slice', 200, 10);
            // the number of cycles to be executed
            var cycles = Utils.getIntegerOption(options, 'cycles');
            // index of the next cycle
            var index = 0;
            // the resulting deferred object
            var deferred = Scheduler.createDeferred(this, logMessage, options);

            // bind callback to this instance
            callback = callback.bind(this);

            // create the interval timer with the passed delay options
            var timer = this.repeatDelayed(function () {

                // start time of this iteration
                var start = _.now();
                // current return value of the callback function
                var result = null;

                // notify the own deferred object about the progress (once per time slice)
                deferred.notify(index);

                // time slice implementation: invoke callback repeatedly (loop body returns on any exit conditions)
                while (true) {

                    // exit loop, if the specified number of cycles has been reached
                    if (_.isNumber(cycles) && (index >= cycles)) {
                        deferred.resolve();
                        return Utils.BREAK;
                    }

                    // invoke the callback function
                    result = callback(index);
                    index += 1;

                    // callback has returned Utils.BREAK: resolve the own deferred
                    // object immediately, and break the background loop
                    if (result === Utils.BREAK) {
                        deferred.resolve(Utils.BREAK);
                        return Utils.BREAK;
                    }

                    // callback has returned a pending or rejected promise: return it to defer
                    // the loop (ignore resolved promises and continue with next invocation)
                    if (Utils.isPromise(result) && (result.state() !== 'resolved')) {
                        return result;
                    }

                    // check elapsed time at the end of the loop body (this ensures
                    // that the callback will be executed at least once)
                    if (_.now() - start >= slice) {
                        return;
                    }
                }
            }, logMessage, {
                delay: Utils.getOption(options, 'delay'),
                repeatDelay: Utils.getIntegerOption(options, 'interval', 10, 10),
                fastAsync: true,
                background: Utils.getBooleanOption(options, 'background', false)
            });

            // resolve/reject the own deferred object when the timer has finished
            timer.done(deferred.resolve.bind(deferred));
            timer.fail(deferred.reject.bind(deferred));

            // the abortable promise for the resulting deferred object
            return this.createAbortablePromise(deferred, function () {
                if (timer) { timer.abort(); timer = null; }
            });
        };

        /**
         * Invokes the passed callback function for all steps of the passed
         * iterator object. To prevent performance problems (frozen user user
         * interface of the browser), the callback function will be invoked
         * from several time slices of a browser timeout loop. If this instance
         * will be destroyed while the loop is running, it will be aborted
         * immediately.
         *
         * @param {Object} iterator
         *  An object implementing the standard iterator protocol (must provide
         *  a method next() that returns an object with the properties 'done'
         *  and 'value').
         *
         * @param {Function} callback
         *  The callback function that will be invoked for each valid result of
         *  the passed iterator. Receives the following parameters:
         *  (1) {Any} value
         *      The 'value' property from the iterator result.
         *  (2) {Object} result
         *      The complete iterator result, that may contain additional
         *      non-standard value properties.
         *  Will be called in the context of this instance. The return value of
         *  the callback function will be used to decide whether to continue to
         *  process the next iterator result (see parameter 'callback' of the
         *  method TimerMixin.repeatDelayed() for details).
         *
         * @param {String} logMessage
         *  A description for the timer loop that will be inserted into the
         *  debug log used in automated test environments.
         *
         * @param {Object} [options]
         *  Optional parameters. See method TimerMixin.repeatSliced() for
         *  details about all supported options.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all iterator steps have been
         *  processed successfully, or the callback function returns
         *  Utils.BREAK; or that will be rejected, if the callback function
         *  returns a rejected promise. The promise will NOT be notified about
         *  the progress (iterators do not provide their size). The promise
         *  contains an additional method abort() that can be called to cancel
         *  the loop immediately while processing the iterator. In that case,
         *  the promise will be rejected. If this instance will be destroyed
         *  before or during the callback loop, the loop timer will be aborted
         *  automatically, and the promise will be rejected.
         */
        this.iterateSliced = function (iterator, callback, logMessage, options) {
            return this.repeatSliced(function () {
                var result = iterator.next();
                return result.done ? Utils.BREAK : callback.call(this, result.value, result);
            }, logMessage, options);
        };

        /**
         * Invokes the passed callback function for all steps of the passed
         * iterator object to reduce the iterator values to a single result
         * value. To prevent performance problems (frozen user user interface
         * of the browser), the callback function will be invoked from several
         * time slices of a browser timeout loop. If this instance will be
         * destroyed while the loop is running, it will be aborted immediately.
         *
         * @param {Any} initial
         *  The initial result value passed to the first invocation of the
         *  callback function (if the iterator is empty, this value becomes the
         *  final result value immediately).
         *
         * @param {Object} iterator
         *  An object implementing the standard iterator protocol (must provide
         *  a method next() that returns an object with the properties 'done'
         *  and 'value').
         *
         * @param {Function} callback
         *  The callback function that will be invoked for each result of the
         *  passed iterator. Receives the following parameters:
         *  (1) {Any} prevValue
         *      The result value of the previous invocation of this callback
         *      function (or the initial value on first step of the iterator).
         *  (2) {Any} value
         *      The 'value' property from the iterator result.
         *  (3) {Object} result
         *      The complete iterator result, that may contain additional
         *      non-standard value properties.
         *  Will be called in the context of this instance. The return value of
         *  the callback function will become the new result value that will be
         *  passed as 'prevValue' with the next iterator step. If the return
         *  value is a promise, it will be waited for, and its resolved value
         *  will become the next result value.
         *
         * @param {String} logMessage
         *  A description for the timer loop that will be inserted into the
         *  debug log used in automated test environments.
         *
         * @param {Object} [options]
         *  Optional parameters. See method TimerMixin.repeatSliced() for
         *  details about all supported options.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the final reduced result value
         *  after all iterator steps have been processed successfully; or that
         *  will be rejected, if the callback function returns a rejected
         *  promise. The promise will NOT be notified about the progress
         *  (iterators do not provide their size). The promise contains an
         *  additional method abort() that can be called to cancel the loop
         *  immediately while processing the iterator. In that case, the
         *  promise will be rejected. If this instance will be destroyed before
         *  or during the callback loop, the loop timer will be aborted
         *  automatically, and the promise will be rejected.
         */
        this.reduceSliced = function (initial, iterator, callback, logMessage, options) {

            // reduce the passed iterator to the final result
            var promise = this.iterateSliced(iterator, function (value, result) {
                var nextValue = callback.call(this, initial, value, result);
                if (Utils.isPromise(nextValue)) {
                    return nextValue.done(function (next) { initial = next; });
                }
                initial = nextValue;
            }, logMessage, options);

            // return the final result through the promise
            return promise.then(function () { return initial; });
        };

        /**
         * Invokes the passed callback function for all elements contained in
         * the passed data array. To prevent performance problems (frozen user
         * user interface of the browser), the callback function will be
         * invoked from several time slices of a browser timeout loop. If this
         * instance will be destroyed while the loop is running, it will be
         * aborted immediately.
         *
         * @param {Array|jQuery|String} array
         *  A JavaScript array, or another array-like object that provides a
         *  'length' property, and element access via bracket notation.
         *
         * @param {Function} callback
         *  The callback function that will be invoked for each array element.
         *  Receives the following parameters:
         *  (1) {Any} element
         *      The current array element.
         *  (2) {Number} index
         *      The array index of the current element.
         *  (3) {Array|jQuery} array
         *      The entire array as passed in the 'array' parameter.
         *  Will be called in the context of this instance. The return value of
         *  the callback function will be used to decide whether to continue to
         *  process the next array elements (see parameter 'callback' of the
         *  method TimerMixin.repeatDelayed() for details).
         *
         * @param {String} logMessage
         *  A description for the timer loop that will be inserted into the
         *  debug log used in automated test environments.
         *
         * @param {Object} [options]
         *  Optional parameters. See method TimerMixin.repeatSliced() for
         *  details about all supported options. Additionally, the following
         *  options are supported:
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the array elements will be visited in reversed
         *      order. In reverse mode, the callback function may remove the
         *      array element currently visited, or elements that are following
         *      the visited element, from the array in-place.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all array elements have been
         *  processed successfully, or the callback function returns
         *  Utils.BREAK; or that will be rejected, if the callback function
         *  returns a rejected promise. The promise will be notified about the
         *  progress (as a floating-point value between 0.0 and 1.0). The
         *  promise contains an additional method abort() that can be called to
         *  cancel the loop immediately while processing the array. In that
         *  case, the promise will be rejected. If this instance will be
         *  destroyed before or during the callback loop, the loop timer will
         *  be aborted automatically, and the promise will be rejected.
         */
        this.iterateArraySliced = function (array, callback, logMessage, options) {

            // whether to iterate in reversed order
            var reverse = Utils.getBooleanOption(options, 'reverse', false);
            // current array index
            var arrayIndex = reverse ? (array.length - 1) : 0;
            // the resulting deferred object
            var deferred = Scheduler.createDeferred(this, logMessage, options);

            // do nothing if passed array is empty
            if (array.length === 0) {
                deferred.notify(1).resolve();
                return this.createAbortablePromise(deferred);
            }

            // bind callback to this instance
            callback = callback.bind(this);

            // start the sliced loop
            var timer = this.repeatSliced(function () {

                // end of array reached
                if ((arrayIndex < 0) || (arrayIndex >= array.length)) {
                    deferred.notify(1).resolve();
                    return Utils.BREAK;
                }

                // invoke the callback function for a single array element
                var result = callback(array[arrayIndex], arrayIndex, array);

                // next array index
                arrayIndex = reverse ? (arrayIndex - 1) : (arrayIndex + 1);

                // immediately resolve after last element unless the result is a pending or rejected promise
                if (((arrayIndex < 0) || (arrayIndex >= array.length)) && !(Utils.isPromise(result) && (result.state() !== 'resolved'))) {
                    deferred.notify(1).resolve();
                    return Utils.BREAK;
                }

                return result;
            }, logMessage, options);

            // resolve/reject the own deferred object according to the timer result
            timer.done(deferred.resolve.bind(deferred));
            timer.fail(deferred.reject.bind(deferred));

            // notify the own deferred object about the progress (once per time slice)
            timer.progress(function (index) { deferred.notify(index / array.length); });

            // the abortable promise of the resulting deferred object
            return this.createAbortablePromise(deferred, function () {
                if (timer) { timer.abort(); timer = null; }
            });
        };

        /**
         * Invokes the passed callback function for all properties contained in
         * the passed object. To prevent performance problems (frozen user user
         * interface of the browser), the callback function will be invoked
         * from several time slices of a browser timeout loop. If this instance
         * will be destroyed while the loop is running, it will be aborted
         * immediately.
         *
         * @param {Object} object
         *  A JavaScript object whose properties will be iterated.
         *
         * @param {Function} callback
         *  The callback function that will be invoked for each object property
         *  (in no specific order). Receives the following parameters:
         *  (1) {Any} value
         *      The value of the current property.
         *  (2) {String} key
         *      The name of the current property.
         *  (3) {Object} object
         *      The entire object as passed in the 'object' parameter.
         *  Will be called in the context of this instance. The return value of
         *  the callback function will be used to decide whether to continue to
         *  process the next object properties (see parameter 'callback' of the
         *  method TimerMixin.repeatDelayed() for details).
         *
         * @param {String} logMessage
         *  A description for the timer loop that will be inserted into the
         *  debug log used in automated test environments.
         *
         * @param {Object} [options]
         *  Optional parameters. See method TimerMixin.repeatSliced() for
         *  details about all supported options.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all object properties have
         *  been processed successfully, or the callback function returns
         *  Utils.BREAK; or that will be rejected, if the callback function
         *  returns a rejected promise. The promise will be notified about the
         *  progress (as a floating-point value between 0.0 and 1.0). The
         *  promise contains an additional method abort() that can be called to
         *  cancel the loop immediately while processing the properties. In
         *  that case, the promise will be rejected. If this instance will be
         *  destroyed before or during the callback loop, the loop timer will
         *  be aborted automatically, and the promise will be rejected.
         */
        this.iterateObjectSliced = function (object, callback, logMessage, options) {

            return this.iterateArraySliced(_.keys(object), function (key) {
                if (key in object) { return callback.call(this, object[key], key, object); }
            }, logMessage, options);
        };

        /**
         * Creates a synchronized method wrapping a callback function that
         * executes asynchronous code. The synchronized method buffers multiple
         * fast invocations and executes the callback function successively,
         * always waiting for the asynchronous code. In difference to debounced
         * methods, invocations of a synchronized method will never be skipped,
         * and each call of the asynchronous callback function receives its
         * original arguments passed to the synchronized method.
         *
         * @param {String} methodName
         *  A descriptive name for the synchronized method that will be used to
         *  generate debug log messages in automated test environments.
         *
         * @param {Function} callback
         *  A function that will be called every time the synchronized method
         *  has been called. Will be called in the context the synchronized
         *  method has been called with, and receives all parameters that have
         *  been passed to the synchronized method. If this function returns a
         *  pending promise, subsequent invocations of the synchronized method
         *  will be postponed until the promise will be resolved or rejected.
         *  All other return values will be interpreted as synchronous
         *  invocations of the callback function.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.background=false]
         *      Whether the synchronized method represents a long-running
         *      background task with low priority. These background tasks will
         *      be ignored when waiting for pending tasks in automated test
         *      environments.
         *
         * @returns {Function}
         *  The synchronized method that can be called multiple times, and that
         *  executes the asynchronous callback function sequentially. Returns
         *  a promise that will be resolved or rejected after the callback
         *  function has been invoked. If the callback function returns a
         *  promise, the synchronized method will wait for it, and will forward
         *  its state and response to its promise. Otherwise, the promise will
         *  be resolved with the return value of the callback function. If this
         *  instance will be destroyed, processing the chain of pending method
         *  invocations will be aborted.
         */
        this.createSynchronizedMethod = function (methodName, callback, options) {

            // self reference
            var self = this;
            // arguments and returned promise of pending calls of the method
            var pendingInvocations = [];
            // the background loop processing all pending invocations
            var timer = null;
            // the message string inserted into the automated test log
            var logMessage = methodName + ' [synchronized method]';
            // whether this timer loop is a background task
            var background = Utils.getBooleanOption(options, 'background', false);

            // invokes the first pending callback function (used as callback for repeatSliced())
            function invokeCallback() {

                // stop background loop if array is empty (all callbacks invoked)
                if (pendingInvocations.length === 0) { return Utils.BREAK; }

                // the first pending invocation (keep in array until it resolves)
                var invocationData = pendingInvocations[0];
                // create the promise representing the result of the callback function
                var promise = $.when(callback.apply(invocationData.ctxt, invocationData.args));
                // the deferred object bound to the result of the callback
                var deferred = invocationData.def;

                // remove invocation data from array when the promise has been resolved/rejected
                promise.always(function () { pendingInvocations.shift(); });

                // forward result of the promise to the deferred object
                promise.done(deferred.resolve.bind(deferred));
                promise.fail(deferred.reject.bind(deferred));

                // return the promise (unless last callback was synchronous and is already resolved)
                return (pendingInvocations.length === 0) ? Utils.BREAK : promise;
            }

            // rejects the deferred objects of all cached invocations
            function rejectInvocations(cause) {
                pendingInvocations.forEach(function (invocationData) {
                    invocationData.def.reject(cause);
                });
            }

            // create and return the synchronized method
            return function () {

                var deferred = Scheduler.createDeferred(self, logMessage, options);
                // all data about the current invocation (arguments and returned deferred object)
                var invocationData = { ctxt: this, args: arguments, def: deferred };

                // push new invocation to the end of the array
                pendingInvocations.push(invocationData);

                // if missing, start a new timer that processes the array
                if (!timer) {
                    timer = self.repeatSliced(invokeCallback, logMessage, { delay: 'immediate', background: background });
                    // reject all pending promises on abort/destruction
                    timer.fail(rejectInvocations);
                    // forget the timer after the last callback invocation
                    timer.always(function () { timer = null; });
                }

                // return a promise that will be resolved/rejected after invocation
                return deferred.promise();
            };
        };

        /**
         * Creates a debounced method that can be called multiple times during
         * the current script execution. The deferred callback will be executed
         * later once in a browser timeout.
         *
         * @param {String} methodName
         *  A descriptive name for the debounced method that will be used to
         *  generate debug log messages in automated test environments.
         *
         * @param {Function|Null} directCallback
         *  A function that will be called every time the debounced method has
         *  been called. Will be called in the context the debounced method has
         *  been called with, and receives all parameters that have been passed
         *  to the created debounced method. Can be set to null, if nothing has
         *  to be done whne calling the debounced method directly.
         *
         * @param {Function} deferredCallback
         *  A function that will be called in a browser timeout after the
         *  debounced method has been called at least once during the execution
         *  of the current script. Will be called in the context the debounced
         *  method has been called with, and does not receive any parameters.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.delay=0]
         *      The delay time (in milliseconds) for the deferred callback
         *      function. The delay time will restart after each call of the
         *      debounced method, causing the execution of the deferred
         *      callback to be postponed until the debounced method has not
         *      been called again during the delay (this is the behavior of the
         *      method _.debounce()).
         *  @param {Number} [options.maxDelay]
         *      If specified, a delay time used as a hard limit to execute the
         *      deferred callback after the first call of the debounced method,
         *      even if it has been called repeatedly afterwards and the normal
         *      delay time is still running.
         *
         * @returns {Function}
         *  The debounced method that can be called multiple times, and that
         *  executes the deferred callback function once after execution of the
         *  current script ends. Passes all arguments to the direct callback
         *  function, and returns its result (returns the calling context, if
         *  no direct callback has been specified). If this instance will be
         *  destroyed, pending debounced methods will not be invoked anymore.
         */
        this.createDebouncedMethod = function (methodName, directCallback, deferredCallback, options) {

            // self reference
            var self = this;
            // delay time per invocation
            var delay = Utils.getIntegerOption(options, 'delay', 0, 0);
            // maximum accumulated delay time
            var maxDelay = Utils.getIntegerOption(options, 'maxDelay', 0, 0);
            // the current timer used to execute the callback
            var minTimer = null;
            // timer used for the maxDelay option
            var maxTimer = null;
            // the message string inserted into the automated test log
            var logMessage = methodName + ' [debounced method]';

            // aborts and clears all timers
            function clearTimers() {
                if (minTimer) { minTimer.abort(); }
                if (maxTimer) { maxTimer.abort(); }
                minTimer = maxTimer = null;
            }

            // creates the timers that execute the deferred callback
            function createTimers(context) {

                // timer callback invoking the deferred callback, bound to the current context (but ignoring the
                // return value, especially promises returned by the callback will not be forwarded to the timer)
                function timerCallback() {
                    clearTimers();
                    deferredCallback.call(context);
                }

                // abort running timer on first call
                if (minTimer) {
                    minTimer.abort();
                    minTimer = null;
                }

                // create a new timeout executing the callback function
                if (!minTimer) {
                    minTimer = self.executeDelayed(timerCallback, logMessage, delay);
                }

                // on first call, create a timer for the maximum delay
                if (!maxTimer && (maxDelay > 0)) {
                    maxTimer = self.executeDelayed(timerCallback, logMessage, maxDelay);
                }
            }

            // create and return the debounced method
            return function () {

                // create a new timeout executing the callback function
                createTimers(this);

                // call the direct callback with the passed arguments
                return directCallback ? directCallback.apply(this, arguments) : this;
            };
        };

        /**
         * Creates a debounced method that can be called multiple times during
         * the current script execution. The deferred callback will be executed
         * later once in browsers animation frame
         *
         * no optional timeout options, because it uses only the native animation frame
         *
         * @param {Function} directCallback
         *  A function that will be called every time the debounced method has
         *  been called. Will be called in the context the debounced method has
         *  been called with, and receives all parameters that have been passed
         *  to the created debounced method.
         *
         * @param {Function} deferredCallback
         *  A function that will be called in a browser animationstep after the
         *  debounced method has been called at least once during the execution
         *  of the current script. Will be called in the context the debounced
         *  method has been called with, and does not receive any parameters.
         *
         * @returns {Function}
         *  The debounced method that can be called multiple times, and that
         *  executes the deferred callback function once after execution of the
         *  current script ends. Passes all arguments to the direct callback
         *  function, and returns its result. If this instance will be
         *  destroyed, pending debounced methods will not be invoked anymore.
         */
        this.createDebouncedAnimationFrameMethod = function (directCallback, deferredCallback) {
            return function () {

                // call the direct callback with the passed arguments
                // directCallback MUST be called before deferredCallback is set to animation frame
                // because the direct callback itself could trigger an animation frame
                var direct = directCallback.apply(this, arguments);

                // create a new timeout executing the callback function
                window.requestAnimationFrame(deferredCallback.bind(this));

                return direct;
            };
        };

    } // class TimerMixin

    // exports ================================================================

    return TimerMixin;

});
