/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/tk/utils/dateutils', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils'
], function (Config, Utils) {

    'use strict';

    //most short dates we know from spreadsheet
    var NO_YEARS = {
        cs: 'D.MMM',
        de: 'DD.MMM',
        en: 'MMM DD',
        es: 'DD/MMM',
        fr: 'DD/MMM',
        hu: 'DD.MMM',
        it: 'DD/MMM',
        ja: 'M\u6708D\u65E5',
        lv: 'MMM DD',
        nl: 'DD-MMM',
        pl: 'DD MMM',
        sk: 'MMM DD',
        zh: 'D-MMM'
    };

    // static class DateUtils =================================================

    var DateUtils = {};

    // constants --------------------------------------------------------------

    /**
     * The number of seconds per day.
     *
     * @constant
     */
    DateUtils.SEC_PER_DAY = 86400;

    /**
     * The number of milliseconds per day.
     *
     * @constant
     */
    DateUtils.MSEC_PER_DAY = DateUtils.SEC_PER_DAY * 1000;

    /**
     * The number of milliseconds per week.
     *
     * @constant
     */
    DateUtils.MSEC_PER_WEEK = DateUtils.MSEC_PER_DAY * 7;

    // static methods ---------------------------------------------------------

    /**
     * Returns an object containing the date components of the passed UTC date
     * value as single numeric properties.
     *
     * @param {Date} date
     *  The date to be converted, as UTC date value.
     *
     * @returns {Object}
     *  The date components, in the properties 'Y' (full year), 'M' (zero-based
     *  month), and 'D' (one-based day in month).
     */
    DateUtils.getUTCDateComponents = function (date) {
        return {
            Y: date.getUTCFullYear(),
            M: date.getUTCMonth(),
            D: date.getUTCDate()
        };
    };

    /**
     * Returns an object containing the date components of the passed local
     * date value as single numeric properties.
     *
     * @param {Date} date
     *  The date to be converted, as local date value.
     *
     * @returns {Object}
     *  The date components, in the properties 'Y' (full year), 'M' (zero-based
     *  month), and 'D' (one-based day in month).
     */
    DateUtils.getLocalDateComponents = function (date) {
        return {
            Y: date.getFullYear(),
            M: date.getMonth(),
            D: date.getDate()
        };
    };

    /**
     * Returns an object containing the date and time components of the passed
     * UTC date value as single numeric properties.
     *
     * @param {Date} date
     *  The date to be converted, as UTC date value.
     *
     * @returns {Object}
     *  The date and time components, in the properties 'Y' (full year), 'M'
     *  (zero-based month), 'D' (one-based day in month), 'h' (hours), 'm'
     *  (minutes), 's' (seconds), and 'ms' (milliseconds).
     */
    DateUtils.getUTCDateTimeComponents = function (date) {
        var comps = DateUtils.getUTCDateComponents(date);
        comps.h = date.getUTCHours();
        comps.m = date.getUTCMinutes();
        comps.s = date.getUTCSeconds();
        comps.ms = date.getUTCMilliseconds();
        return comps;
    };

    /**
     * Returns an object containing the date and time components of the passed
     * local date value as single numeric properties.
     *
     * @param {Date} date
     *  The date to be converted, as local date value.
     *
     * @returns {Object}
     *  The date and time components, in the properties 'Y' (full year), 'M'
     *  (zero-based month), 'D' (one-based day in month), 'h' (hours), 'm'
     *  (minutes), 's' (seconds), and 'ms' (milliseconds).
     */
    DateUtils.getLocalDateTimeComponents = function (date) {
        var comps = DateUtils.getLocalDateComponents(date);
        comps.h = date.getHours();
        comps.m = date.getMinutes();
        comps.s = date.getSeconds();
        comps.ms = date.getMilliseconds();
        return comps;
    };

    /**
     * Creates a UTC date value from the specified date components.
     *
     * @param {Object} comps
     *  The date components, in the properties 'Y', 'M', and 'D', as returned
     *  for example by the method DateUtils.getUTCDateComponents().
     *
     * @returns {Date}
     *  The UTC date value representing the specified date. The time will be
     *  set to midnight.
     */
    DateUtils.makeUTCDate = function (comps) {
        return new Date(Date.UTC(comps.Y, comps.M, comps.D, 0, 0, 0, 0));
    };

    /**
     * Creates a UTC date value from the specified date and time components.
     *
     * @param {Object} comps
     *  The date/time components, in the properties 'Y', 'M', 'D', 'h', 'm',
     *  's', and 'ms', as returned for example by the method
     *  DateUtils.getUTCDateTimeComponents().
     *
     * @returns {Date}
     *  The UTC date value representing the specified date and time.
     */
    DateUtils.makeUTCDateTime = function (comps) {
        return new Date(Date.UTC(comps.Y, comps.M, comps.D, comps.h, comps.m, comps.s, comps.ms));
    };

    /**
     * Creates a UTC date value from the current date.
     *
     * @returns {Date}
     *  The UTC date value representing the current date. The time will be set
     *  to midnight.
     */
    DateUtils.makeUTCToday = function () {
        return DateUtils.makeUTCDate(DateUtils.getLocalDateComponents(new Date()));
    };

    /**
     * Creates a UTC date value from the current date and time.
     *
     * @returns {Date}
     *  The UTC date value representing the current date and time.
     */
    DateUtils.makeUTCNow = function () {
        return DateUtils.makeUTCDateTime(DateUtils.getLocalDateTimeComponents(new Date()));
    };

    /**
     * Converts a one-digit or two-digit year to a four-digit year, according
     * to the passed century threshold value.
     *
     * @param {Number} year
     *  A year number. MUST be non-negative. Values less than 100 will be
     *  converted to a year in the current, or in the preceding century.
     *
     * @param {Number} threshold
     *  An integer in the range 0 to 100 that specifies how to convert a number
     *  below 100 to a 4-digit year. All numbers less than this threshold value
     *  will be moved to the current century (e.g.: with threshold 30, the
     *  number 10 will be converted to the year 2010); all other numbers will
     *  be moved to the preceding century (e.g.: the number 90 will be
     *  converted to 1990).
     *
     * @returns {Number}
     *  The resulting year number.
     */
    DateUtils.expandYear = function (year, threshold) {
        var century = Utils.roundDown(new Date().getFullYear(), 100);
        return (year < threshold) ? Math.floor(year + century) : (year < 100) ? Math.floor(year + century - 100) : Math.floor(year);
    };

    /**
     * Returns whether the passed number is a leap year.
     *
     * @param {Number} year
     *  The year to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed number is a leap year.
     */
    DateUtils.isLeapYear = function (year) {
        return (((year % 4) === 0) && ((year % 100) !== 0)) || ((year % 400) === 0);
    };

    /**
     * Returns the number of days in the specified year.
     *
     * @param {Number} year
     *  The full year.
     *
     * @returns {Number}
     *  The number of days in the specified year.
     */
    DateUtils.getDaysInYear = function (year) {
        return DateUtils.isLeapYear(year) ? 366 : 365;
    };

    /**
     * Returns the number of days in the month February of the specified year.
     *
     * @param {Number} year
     *  The year.
     *
     * @returns {Number}
     *  The number of days in February of the specified year.
     */
    DateUtils.getDaysInFebruary = function (year) {
        return DateUtils.isLeapYear(year) ? 29 : 28;
    };

    /**
     * Returns the number of days in the specified month.
     *
     * @param {Number} year
     *  The year of the month.
     *
     * @param {Number} month
     *  The month. MUST be a zero-based integer (0 to 11).
     *
     * @returns {Number}
     *  The number of days in the specified month.
     */
    DateUtils.getDaysInMonth = (function () {

        // number of days per month (regular years)
        var DAYS_PER_MONTH = [31, 0, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];

        function getDaysInMonth(year, month) {
            // special handling for February in leap years
            return (month === 1) ? DateUtils.getDaysInFebruary(year) : DAYS_PER_MONTH[month];
        }

        return getDaysInMonth;
    }());

    /**
     * Returns a new date with the first day of the month represented by the
     * passed UTC date.
     *
     * @param {Date} date
     *  The original date, as UTC date value.
     *
     * @returns {Date}
     *  The new date, as copy of the passed date but with the day set to 1, as
     *  UTC date value.
     */
    DateUtils.getMonthStart = function (date) {
        return new Date(Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), 1));
    };

    /**
     * Returns a new date with the first day of the year represented by the
     * passed UTC date.
     *
     * @param {Date} date
     *  The original date, as UTC date value.
     *
     * @returns {Date}
     *  The new date, as copy of the passed date but set to January 1st, as UTC
     *  date value.
     */
    DateUtils.getYearStart = function (date) {
        return new Date(Date.UTC(date.getUTCFullYear(), 0, 1));
    };

    /**
     * Adds the specified number of days to the passed date, and returns the
     * resulting date.
     *
     * @param {Date} date
     *  The original date, as UTC date value.
     *
     * @param {Number} days
     *  The number of days to add. If negative, a date in the past will be
     *  returned accordingly. MUST be an integral number.
     *
     * @returns {Date}
     *  The new date, as UTC date value.
     */
    DateUtils.addDaysToDate = function (date, days) {
        return new Date(date.getTime() + days * DateUtils.MSEC_PER_DAY);
    };

    /**
     * Adds the specified number of months to the passed date, and returns the
     * resulting date.
     *
     * @param {Date} date
     *  The original date, as UTC date value.
     *
     * @param {Number} months
     *  The number of months to add. If negative, a date in the past will be
     *  returned accordingly. MUST be an integral number.
     *
     * @param {String} [dayMode]
     *  Specifies how to handle the day in the resulting date. The following
     *  modes are supported:
     *  - 'first': The day in the resulting date will be set to 1.
     *  - 'last': The day in the resulting date will be set to the last day of
     *      the resulting month.
     *  If omitted, the day of the passed original date will be retained. If
     *  the resulting month is shorter and does not contain that day, the last
     *  valid day of the new month will be used.
     *
     * @returns {Date}
     *  The new date, as UTC date value.
     */
    DateUtils.addMonthsToDate = function (date, months, dayMode) {

        // extract the date components from the passed date
        var comps = DateUtils.getUTCDateTimeComponents(date);

        // calculate the new month and year
        comps.M += months;
        comps.Y += Math.floor(comps.M / 12);
        comps.M = Utils.mod(comps.M, 12);

        // validate the day (new month may have less days than the day passed in the original date)
        switch (dayMode) {
            case 'first': comps.D = 1; break;
            case 'last':  comps.D = DateUtils.getDaysInMonth(comps.Y, comps.M); break;
            default:      comps.D = Math.min(comps.D, DateUtils.getDaysInMonth(comps.Y, comps.M));
        }

        // build and return the new date
        return DateUtils.makeUTCDateTime(comps);
    };

    /**
     * Adds the specified number of years to the passed date, and returns the
     * resulting date.
     *
     * @param {Date} date
     *  The original date, as UTC date value.
     *
     * @param {Number} years
     *  The number of years to add. If negative, a date in the past will be
     *  returned accordingly. MUST be an integral number.
     *
     * @param {String} [dayMode]
     *  Specifies how to handle the day in the resulting date. The following
     *  modes are supported:
     *  - 'first': The resulting date will be set to January 1st.
     *  - 'last': The resulting date will be set to December 31st.
     *  If omitted, the day of the passed original date will be retained. If
     *  the original day was a leap day (February 29th), and the new year is
     *  not a leap year, the date will be set to February 28th.
     *
     * @returns {Date}
     *  The new date, as UTC date value.
     */
    DateUtils.addYearsToDate = function (date, years, dayMode) {

        // extract the date components from the passed date
        var comps = DateUtils.getUTCDateTimeComponents(date);

        // calculate the new year
        comps.Y += years;

        // validate the month and day (correct old leap day in non-leap years)
        switch (dayMode) {
            case 'first': comps.M = 0;  comps.D = 1;  break;
            case 'last':  comps.M = 11; comps.D = 31; break;
            default:      if ((comps.M === 1) && (comps.D === 29) && !DateUtils.isLeapYear(comps.Y)) { comps.D = 28; }
        }

        // build and return the new date
        return DateUtils.makeUTCDate(comps);
    };

    /**
     * @param {Number} timestamp
     *
     * @returns {Date}
     *  UTC-date with passed timestamp
     */
    DateUtils.getUTCDate = function (timestamp) {
        var offset = moment().utcOffset() * 60 * 1000;
        return _.isUndefined(timestamp) ? new Date(new Date().getTime() - offset) : new Date(timestamp - offset);
    };

    /**
     * formats assigned date with assigned format
     *
     * @param {Date} date
     *
     * @param {String} format
     *
     * @returns {String}
     *  formated date
     */
    DateUtils.format = function (date, format) {
        var result = moment(date.getTime()).format(format);
        return result;
    };

    /**
     * @returns {String}
     *  dateformat, day month and year
     */
    DateUtils.getDateFormat = function () {
        return moment.localeData()._longDateFormat.L;
    };

    /**
     * @returns {String}
     *  timeformat, normally hour and minutes
     */
    DateUtils.getTimeFormat = function () {
        return moment.localeData()._longDateFormat.LT;
    };

    /**
     * @returns {String}
     *  dateformat for complete weekdays
     */
    DateUtils.getWeekDayFormat = function () {
        return 'dddd';
    };

    /**
     * @returns {String}
     *  dateformat, day month without year, but not implemented at the moment!
     */
    DateUtils.getNoYearFormat = function () {
        return NO_YEARS[Config.LANGUAGE] || NO_YEARS.en;
    };

    /**
     * Converts the UTC date time string to a date string in local time
     * (keeping the final letter 'Z', that indicates UTC).
     *
     * @param {String} utcString
     *  The UTC date string in ISO 8601 format.
     *
     * @param {Object} [options]
     *  A map of options to control the properties of the returned string.
     *  - {Boolean} [options.useMilliSeconds=false]
     *      Whether the milliseconds shall be part of the returned string.
     *      Default is, that the string is returned without milliseconds.
     *
     * @returns {String}
     *  The current local date in an ISO string representation (including ending
     *  'Z', which is not quite correct).
     */
    DateUtils.convertUTCDateToLocalDate = function (utcString, options) {

        var // whether the seconds can be set to '00' in the returned string
            useMilliSeconds = Utils.getBooleanOption(options, 'useMilliSeconds', false),
            // the local date object
            localDate = new Date(utcString),
            // the local date string
            localDateString = null,
            // the offset in hours between local time and UTC time
            offset = localDate.getTimezoneOffset() / 60,
            // the hours of the local date
            hours = localDate.getHours();

        // hours outside the range 0..23 will affect the date automatically
        localDate.setHours(hours - offset);

        localDateString = localDate.toISOString();

        if (!useMilliSeconds) {
            // removing all digits after the final '.' together with this '.'
            localDateString = localDateString.replace(/\.\d{3}Z$/, 'Z');
        }

        return localDateString;
    };

    /**
     * Returns the current UTC time ISO date format. ISO date format:
     * '2012-04-17T13:38:00Z' with 'Z' at the end for UTC (+00:00)
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.useSeconds=true]
     *      Whether the seconds can be set to '00' in the returned string.
     *      Default is, that the string is returned with seconds and milliseconds.
     *
     * @returns {String}
     *  The current UTC date in an ISO string representation.
     */
    DateUtils.getIsoDateString = function (options) {

        var // the current date in UTC time zone
            d = new Date().toISOString(),
            // whether the seconds can be set to '00' in the returned string
            useSeconds = Utils.getBooleanOption(options, 'useSeconds', true);

        if (!useSeconds) {
            // replacing all digits before the trailing Z and after the final ':' with 0
            d = d.replace(/:[\d.]+?Z$/, ':00Z');
        }

        return d;
    };

    /**
     * Returning the current local time faked as UTC time in ISO date format.
     * This is necessary, because the round trip with MS Office requires this.
     * So local time '2012-04-17' and '13:38:00' results in a string in
     * ISO date format: '2012-04-17T13:38:00Z'
     * 'Z' at the end for UTC (+00:00)
     * So string contains always a shifted time, except in UTC time zone.
     *
     * This functions shifts the offset in the milliseconds number and then
     * converts to date object, so that there are no problems with offsets
     * in hours, where the day, month or even year might be adapted.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.useSeconds=true]
     *      Whether the seconds can be set to '00' in the returned string.
     *      Default is, that the string is returned with seconds and milliseconds.
     *
     * @returns {String}
     *  The current local time as UTC date in an ISO string representation.
     */
    DateUtils.getMSFormatIsoDateString = function (options) {

        var // the local date object
            localDate = new Date(),
            // the local date time in milliseconds
            localTime = localDate.getTime(),
            // the offset in milliseconds between local time and UTC time
            offset = localDate.getTimezoneOffset() * 60 * 1000,
            // the shifted local time in milliseconds
            shiftedTime = localTime - offset,
            // the date string in ISO format
            isoDateString = new Date(shiftedTime).toISOString();

        if (!Utils.getBooleanOption(options, 'useSeconds', true)) {
            // replacing all digits before the trailing Z and after the final ':' with 0
            isoDateString = isoDateString.replace(/:[\d.]+?Z$/, ':00Z');
        }

        return isoDateString;
    };

    // exports ================================================================

    return DateUtils;
});
