/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/controls', [
    'io.ox/office/textframework/view/controls',
    'gettext!io.ox/office/text/main'
], function (Controls, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var RadioGroup = Controls.RadioGroup;
    var CompoundSplitButton = Controls.CompoundSplitButton;

    // static class TextControls ==============================================

    /**
     * Provides different classes for GUI form controls for text documents
     * collected within a single map for convenience.
     */
    var TextControls = _.extend({}, Controls);

    // class ParagraphFillColorPicker =========================================

    var PARA_FILL_COLOR_LABEL = gt('Paragraph fill color');

    /**
     * A color picker control for the paragraph background color in text
     * documents.
     *
     * @constructor
     *
     * @extends FillColorPicker
     */
    TextControls.ParagraphFillColorPicker = Controls.FillColorPicker.extend({ constructor: function (docView) {

        Controls.FillColorPicker.call(this, docView, {
            icon: 'docs-para-fill-color',
            tooltip: PARA_FILL_COLOR_LABEL,
            title: PARA_FILL_COLOR_LABEL,
            dropDownVersion: { label: PARA_FILL_COLOR_LABEL }
        });

    } }); // class ParagraphFillColorPicker

    // class ParagraphBorderModePicker ========================================

    var PARA_BORDERS_LABEL = gt('Paragraph borders');

    /**
     * A border type control for paragraph borders in text documents.
     *
     * @constructor
     *
     * @extends BorderModePicker
     */
    TextControls.ParagraphBorderModePicker = Controls.BorderModePicker.extend({ constructor: function (docView) {

        Controls.BorderModePicker.call(this, docView, {
            tooltip: PARA_BORDERS_LABEL,
            // ODF does not support inner borders at paragraphs
            showInsideHor: !docView.getApp().isODF(),
            dropDownVersion: { label: PARA_BORDERS_LABEL }
        });

    } }); // class ParagraphBorderModePicker

    // class HeaderFooterPicker ===============================================

    /**
     * A button with a drop-down menu providing options to insert and remove
     * headers and footers in a text document.
     *
     * @constructor
     *
     * @extends CompoundSplitButton
     */
    TextControls.HeaderFooterPicker = CompoundSplitButton.extend({ constructor: function (docView) {

        CompoundSplitButton.call(this, docView, {
            icon: 'docs-header-footer',
            label: /*#. insert header and footer in document */ gt('Header & footer'),
            tooltip: gt('Insert header & footer in document'),
            smallerVersion: { hideLabel: true },
            splitValue: 'goto',
            updateCaptionMode: 'none'
        });

        this.addGroup('document/insert/headerfooter', new RadioGroup(docView)
            .createOptionButton('default', { label: /*#. header and footer */ gt('Same across entire document') })
            .createOptionButton('first',   { label: /*#. header and footer */ gt('Different first page') })
            .createOptionButton('evenodd', { label: /*#. header and footer */ gt('Different even & odd pages') })
            .createOptionButton('all',     { label: /*#. header and footer */ gt('Different first, even & odd pages') })
        );
        this.addSeparator();
        this.addGroup('document/insert/headerfooter/remove', new Button(docView, { label: /*#. remove all headers and footers from document */ gt('Remove all headers and footers') }));

    } }); // class HeaderFooterPicker

    // exports ================================================================

    return TextControls;

});
