/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 * @author Marko Benigar <marko.benigar@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/view', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/editframework/model/viewattributesmixin',
    'io.ox/office/editframework/view/popup/attributestooltip',
    'io.ox/office/drawinglayer/view/popup/chartlabelsmenu',
    'io.ox/office/textframework/view/view',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/view/statuspane',
    'io.ox/office/spreadsheet/view/formulapane',
    'io.ox/office/spreadsheet/view/gridpane',
    'io.ox/office/spreadsheet/view/headerpane',
    'io.ox/office/spreadsheet/view/cornerpane',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/controls',
    'io.ox/office/spreadsheet/view/toolbars',
    'io.ox/office/spreadsheet/view/popup/namesmenu',
    'io.ox/office/spreadsheet/view/mixin/selectionmixin',
    'io.ox/office/spreadsheet/view/mixin/highlightmixin',
    'io.ox/office/spreadsheet/view/mixin/viewfuncmixin',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/rendercache',
    'io.ox/office/spreadsheet/view/edit/celltexteditor',
    'io.ox/office/spreadsheet/view/edit/drawingtexteditor',
    'gettext!io.ox/office/spreadsheet/main',
    'less!io.ox/office/spreadsheet/view/style'
], function (Utils, KeyCodes, Forms, Tracking, ValueSet, ValueMap, LocaleData, Rectangle, ViewAttributesMixin, AttributesToolTip, ChartLabelsMenu, TextBaseView, Config, SheetUtils, PaneUtils, FormulaUtils, StatusPane, FormulaPane, GridPane, HeaderPane, CornerPane, Labels, Controls, ToolBars, NamesMenu, SelectionMixin, HighlightMixin, ViewFuncMixin, RenderUtils, RenderCache, CellTextEditor, DrawingTextEditor, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var CheckBox = Controls.CheckBox;
    var CompoundButton = Controls.CompoundButton;
    var Address = SheetUtils.Address;
    var Interval = SheetUtils.Interval;
    var IntervalArray = SheetUtils.IntervalArray;
    var RangeArray = SheetUtils.RangeArray;

    // definitions for global view attributes
    var VIEW_ATTRIBUTE_DEFINITIONS = {

        /**
         * An array of active remote clients with selection settings. See
         * method EditApplication.getActiveClients() for details about the
         * contents of this array. The local client must not be part of the
         * array.
         */
        remoteClients: {
            def: [],
            validate: function (clients) { return _.isArray(clients) ? clients : []; }
        },

        /**
         * Whether formula cells, data validations, and other cell contents
         * will be rendered with special highlighting markers in debug mode.
         */
        highlightCells: {
            def: false
        }
    };

    // codes for various message texts used in alert boxes
    var RESULT_MESSAGES = {

        'sheet:name:empty':
            gt('The sheet name must not be empty. Please enter a name.'),

        'sheet:name:invalid':
            gt('This sheet name contains invalid characters. Please ensure that the name does not contain / \\ * ? : [ or ], and does neither start nor end with an apostrophe.'),

        'sheet:name:used':
            gt('This sheet name is already used. Please enter another name.'),

        'sheet:locked':
            gt('You cannot use this command on a protected sheet.'),

        'cols:overflow':
            gt('It is not possible to modify more than %1$d columns at the same time.', _.noI18n(SheetUtils.MAX_CHANGE_COLS_COUNT)),

        'rows:overflow':
            gt('It is not possible to modify more than %1$d rows at the same time.', _.noI18n(SheetUtils.MAX_CHANGE_ROWS_COUNT)),

        'cells:locked':
            gt('Protected cells cannot be modified.'),

        'cells:overflow':
            gt('It is not possible to modify more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)),

        'cells:pushoff':
            gt('It is not possible to insert new cells because non-empty cells would be pushed off the end of the sheet.'),

        'merge:overlap':
            //#. Warning text: trying to merge multiple cell ranges which overlap each other
            gt('Overlapping ranges cannot be merged.'),

        'merge:overflow':
            //#. Warning text: trying to merge too many cell ranges at the same time
            //#. %1$d is the maximum number of cell ranges that can be merged at a time
            //#, c-format
            gt('It is not possible to merge more than %1$d ranges at the same time.', _.noI18n(SheetUtils.MAX_MERGED_RANGES_COUNT)),

        'autofill:merge:overlap':
            //#. Warning text: trying to trigger autofill over only a part of at least one merged cell
            gt('Autofill over merged cells is not allowed.'),

        'name:invalid':
            // #. Warning text: the user-defined name of a cell range contains invalid characters
            gt('This name contains invalid characters. Please enter another name.'),

        'name:address':
            // #. Warning text: the user-defined name of a cell range cannot look like a cell, e.g. 'A1', or 'R1C1'
            gt('A cell address cannot be used as name. Please enter another name.'),

        'name:used':
            // #. Warning text: the user-defined name of a cell range is already used
            gt('This name is already used. Please enter another name.'),

        'table:multiselect':
            //#. Warning text: tried to create a table range from a cell multi-selection
            gt('Table cannot be created on multiple cell ranges.'),

        'table:overlap':
            //#. Warning text: tried to create a table range above cells that are already filtered
            gt('Table cannot be created on another filtered range.'),

        'table:move':
            //#. Warning text: tried to move a few cells of a table range to the right, or down, etc.
            gt('Parts of a table cannot be moved.'),

        'table:change':
            //#. Warning text: tried to change (overwrite, merge, etc.) parts of a table range
            gt('Parts of a table cannot be changed.'),

        'table:header:change':
            //#. Warning text: tried to change (overwrite, merge, etc.) the header cells of a table range
            gt('The header cells of a table cannot be changed.'),

        'table:headerFooter:change':
            //#. Warning text: tried to change (overwrite, merge, etc.) the header cells of a table range
            gt('The header and footer cells of a table cannot be changed.'),

        'autofilter:multiselect':
            //#. Warning text: tried to create an auto filter from a cell multi-selection
            gt('Filter cannot be applied to multiple cell ranges.'),

        'autofilter:overlap':
            //#. Warning text: tried to create an auto filter above cells that are already filtered
            gt('Filter cannot be applied to another filtered range.'),

        'autofilter:blank':
            //#. Warning text: tried to filter an empty cell range
            gt('Filtering cannot be performed on blank cells.'),

        'autofilter:move':
            //#. Warning text: tried to move a few cells of an auto filter to the right, or down, etc.
            gt('Parts of the filter cannot be moved.'),

        'sort:multiselect':
            //#. Warning text: tried to sort data of a cell multi-selection
            gt('Sorting cannot be performed on multiple cell ranges.'),

        'sort:blank':
            //#. Warning text: tried to sort an empty cell range
            gt('Sorting cannot be performed on blank cells.'),

        'sort:overflow':
            //#. Warning text: tried to sort too many rows or columns
            gt('Sorting cannot be performed on more than %1$d columns or rows.', _.noI18n(SheetUtils.MAX_SORT_LINES_COUNT)),

        'sort:merge:overlap':
            //#. Warning text: trying to sort over merged cells is not allowed
            gt('Sorting over merged cells is not allowed.'),

        'search:nothing':
            //#. Information text: no matching text in document
            gt('No results for your search.'),

        'search:finished':
            //#. Information text: no more matches in document (after something has been found)
            gt('No more results for your search.'),

        'replace:nothing':
            //#. Information text: search&replace did not find anything to replace
            gt('Nothing to replace.'),

        'formula:invalid':
            //#. Warning text: a spreadsheet formula entered in a cell or in a dialog text field contains an error
            gt('The formula contains an error.'),

        'formula:matrix:change':
            gt('Parts of a matrix formula cannot be changed.'),

        'formula:matrix:insert':
            gt('It is not possible to insert cells into a matrix formula.'),

        'formula:matrix:delete':
            gt('Parts of a matrix formula cannot be deleted.'),

        'formula:matrix:size':
            //#. Warning text: new matrix formulas must not exceed specific height and width
            gt('It is not possible to insert a matrix formula with more than %1$d elements.', _.noI18n(FormulaUtils.Matrix.MAX_SIZE)),

        'formula:matrix:merged':
            gt('It is not possible to insert a matrix formula over merged cells.'),

        'validation:source':
            //#. Warning text: the values for a drop-down list attached to a spreadsheet cell could not be found
            gt('No source data available for the drop-down list.'),

        'paste:overflow':
            gt('It is not possible to paste more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)),

        'paste:outside':
            gt('It is not possible to paste outside of the valid sheet area.'),

        'paste:locked':
            gt('Pasting into protected cells is not allowed.'),

        'paste:ranges:unfit':
            gt('You cannot paste this here because the copy area and paste area are not the same size. Select just one cell in the paste area or an area that\'s the same size, and try pasting again.'),

        'drop:unsupported':
            //#. Warning text: file type dropped from outside into the document not supported
            gt('File type not supported.'),

        'drawing:insert:locked':
            gt('Objects cannot be inserted into a protected sheet.'),

        'drawing:delete:locked':
            gt('Objects on a protected sheet cannot be deleted.'),

        'drawing:change:locked':
            gt('Objects on a protected sheet cannot be changed.'),

        'comment:insert:locked':
            gt('Comments cannot be inserted into a protected sheet.'),

        'comment:delete:locked':
            gt('Comments on a protected sheet cannot be deleted.'),

        'comment:change:locked':
            gt('Comments on a protected sheet cannot be changed.'),

        'comment:search:finished':
            gt('No more comments found.')
    };

    // the size of the freeze separator nodes, in pixels
    var FROZEN_SPLIT_SIZE = 1;

    // the size of the split separator nodes, in pixels
    var DYNAMIC_SPLIT_SIZE = 2;

    // the highlighted inner size of the split tracking node, in pixels
    var TRACKING_SPLIT_SIZE = 4;

    // the additional margin of the split tracking nodes, in pixels
    var TRACKING_MARGIN = 3;

    // the position offset of tracking nodes compared to split lines
    var TRACKING_OFFSET = (TRACKING_SPLIT_SIZE - DYNAMIC_SPLIT_SIZE) / 2 + TRACKING_MARGIN;

    // default settings for a pane side
    var DEFAULT_PANE_SIDE_SETTINGS = {
        offset: 0,
        size: 1,
        frozen: false,
        showOppositeScroll: false,
        hiddenSize: 0
    };

    // class SpreadsheetView ==================================================

    /**
     * Represents the entire view of a spreadsheet document. Contains the view
     * panes of the sheet currently shown (the 'active sheet'), which contain
     * the scrollable cell grids (there will be several view panes, if the
     * sheet view is split or frozen); and the selections of all existing
     * sheets.
     *
     * Additionally to the events triggered by the base class EditView, an
     * instance of this class triggers events if global contents of the
     * document, or the contents of the current active sheet have been changed.
     * The following events will be triggered:
     * - 'change:doc:viewattributes'
     *      After the global view attributes of the document model have been
     *      changed.
     * - 'before:activesheet'
     *      Before another sheet will be activated. Event handlers receive the
     *      zero-based index of the old active sheet in the document.
     * - 'change:activesheet'
     *      After another sheet has been activated. Event handlers receive the
     *      zero-based index of the new active sheet in the document, and the
     *      sheet model instance of the new active sheet.
     * - 'update:selection:data'
     *      After the selection data (active cell settings, mixed borders,
     *      subtotal results) have been updated.
     * - 'sheet:triggered'
     *      Any event from the active sheet will be forwarded with this event.
     *      Event handlers receive the type name of the event, and all other
     *      parameters sent with the original sheet event.
     * - 'change:sheet:attributes'
     *      After the formatting attributes of the active sheet have been
     *      changed.
     * - 'change:sheet:viewattributes'
     *      After the view attributes of the active sheet have been changed.
     *      Event handlers receive an incomplete attribute map containing all
     *      changed view attributes of the active sheet with their new values.
     * - 'change:selection:prepare'
     *      Will be triggered directly before the 'change:sheet:viewattributes'
     *      event caused by a changed selection. Event handlers receive the new
     *      selection (an instance of the class SheetSelection).
     * - 'change:selection'
     *      Will be triggered directly after the 'change:sheet:viewattributes'
     *      event caused by a changed selection, for convenience. Event
     *      handlers receive the new selection (an instance of the class
     *      SheetSelection).
     * - 'refresh:ranges'
     *      Intended to initiate updating the visual representation of the
     *      notified cell ranges in the active sheet.
     * - 'insert:columns', 'delete:columns', 'change:columns'
     *      The respective event forwarded from the column collection of the
     *      active sheet. See class ColRowCollection for details.
     * - 'insert:rows', 'delete:rows', 'change:rows'
     *      The respective event forwarded from the row collection of the
     *      active sheet. See class ColRowCollection for details.
     * - 'insert:merged', 'delete:merged'
     *      The respective event forwarded from the merge collection of the
     *      active sheet. See class MergeCollection for details.
     * - 'change:cells', 'move:cells'
     *      The respective event forwarded from the cell collection of the
     *      active sheet. See class CellCollection for details.
     *   'insert:rule', 'delete:rule', 'change:rule'
     *      The respective events forwarded from the collection of conditional
     *      formattings of the active sheet. See class CondFormatCollection for
     *      details.
     * - 'insert:table', 'change:table', 'delete:table'
     *      The respective event forwarded from the table collection of the
     *      active sheet. See class TableCollection for details.
     * - 'insert:drawing', 'delete:drawing', 'move:drawing', 'change:drawing',
     *   'change:drawing:text'
     *      The respective event forwarded from the drawing collection of the
     *      active sheet. See class SheetDrawingCollection for details.
     * - 'insert:comment', 'delete:comment', 'move:comment', 'change:comment',
     *   'change:comment:text'
     *      The respective event forwarded from the comment collection of the
     *      active sheet. See class CommentCollection for details.
     * - 'change:usedrange'
     *      After the used area of the active sheet has been changed, e.g.
     *      after inserting or removing cells or merged ranges.
     * - 'change:highlight:ranges'
     *      After the highlighted ranges in this document view have been
     *      changed. See mix-in class HighlightMixin for more details about
     *      highlighted ranges.
     * - 'textedit:enter', 'textedit:leave'
     *      After entering or leaving the text edit mode (editing the text
     *      contents of a cell or a drawing).
     * - 'textedit:change'
     *      After changing text while text edit mode is active (editing the
     *      text contents of a cell or a drawing).
     * - 'textedit:enter:cell', 'textedit:leave:cell'
     *      After entering or leaving the cell edit mode (editing the text
     *      contents of a cell). The event will be triggered directly after the
     *      generic 'textedit:enter' or 'textedit:leave' event.
     * - 'textedit:change:cell'
     *      After changing text while cell edit mode is active (editing the
     *      text contents of a cell). The event will be triggered directly
     *      after the generic 'textedit:change' event.
     * - 'textedit:enter:drawing', 'textedit:leave:drawing'
     *      After entering or leaving the drawing edit mode (editing the text
     *      contents of a drawing object). The event will be triggered directly
     *      after the generic 'textedit:enter' or 'textedit:leave' event.
     * - 'textedit:change:drawing'
     *      After changing text while drawing edit mode is active (editing the
     *      text contents of a drawing object). The event will be triggered
     *      directly after the generic 'textedit:change' event.
     * - 'render:cellselection'
     *      After the cell selection of the active grid pane has actually been
     *      rendered into the DOM selection layer node (rendering may happen
     *      debounced after several 'change:selection' view events).
     * - 'render:drawingselection'
     *      After the drawing selection of the active grid pane has actually
     *      been rendered into the DOM drawing layer (rendering may happen
     *      debounced after several 'change:selection' view events). Event
     *      handlers receive the DOM drawing frames currently selected in the
     *      active grid pane, as jQuery collection.
     * - 'change:scrollpos'
     *      After the scroll position of any of the header and grid panes has
     *      been changed. Event handlers receive the pane side identifier of
     *      the scrolled header pane, and the new scroll position.
     *
     * @constructor
     *
     * @extends TextBaseView
     * @extends ViewAttributesMixin
     * @extends SelectionMixin
     * @extends HighlightMixin
     * @extends ViewFuncMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this spreadsheet view.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model created by the passed application.
     */
    var SpreadsheetView = TextBaseView.extend({ constructor: function (app, docModel) {

        // self reference
        var self = this;

        // the DOM root node of this view containing the headers and grid panes
        var rootNode = $('<div class="abs pane-root">');

        // size of the pane root node upon last refresh of the pane layout
        var rootNodeSize = { width: 0, height: 0 };

        // the root container for DOM models of all drawing objects in all sheets
        var textRootNode = docModel.getNode();

        // the top-left corner pane
        var cornerPane = null;

        // settings for all pane sides, mapped by pane side identifier
        var paneSideSettings = {};

        // the row/column header panes, mapped by pane side identifiers
        var headerPaneMap = new ValueMap();

        // the grid panes, mapped by pane position identifiers
        var gridPaneMap = new ValueMap();

        // the formula pane with the formula input field
        var formulaPane = null;

        // the status pane with the sheet tabs
        var statusPane = null;

        // false while grid panes are not initialized (influences focus handling)
        var panesInitialized = false;

        // the floating menu for defined names
        var namesMenu = null;

        // the floating menu for chart axis and labels formatting
        var chartLabelsMenu = null;

        // rendering cache for cell formatting and text contents
        var renderCache = null;

        // all registered editors
        var textEditorRegistry = new ValueMap();

        // reference to the editor instance that is currently active
        var activeTextEditor = null;

        // the split line separating the left and right panes
        var colSplitLineNode = $('<div class="split-line columns">');

        // the split line separating the top and bottom panes
        var rowSplitLineNode = $('<div class="split-line rows">');

        // the split tracking node separating left and right panes
        var colSplitTrackingNode = $('<div class="abs split-tracking columns" style="width:' + TRACKING_SPLIT_SIZE + 'px;padding:0 ' + TRACKING_MARGIN + 'px;" tabindex="-1">');

        // the split tracking node separating top and bottom panes
        var rowSplitTrackingNode = $('<div class="abs split-tracking rows" style="height:' + TRACKING_SPLIT_SIZE + 'px;padding:' + TRACKING_MARGIN + 'px 0;" tabindex="-1">');

        // the split tracking node covering the intersection of the other tracking points
        var centerSplitTrackingNode = $('<div class="abs split-tracking columns rows" style="width:' + TRACKING_SPLIT_SIZE + 'px;height:' + TRACKING_SPLIT_SIZE + 'px;padding:' + TRACKING_MARGIN + 'px;" tabindex="-1">');

        // all split tracking nodes, as jQuery collection
        var allSplitTrackingNodes = colSplitTrackingNode.add(rowSplitTrackingNode).add(centerSplitTrackingNode);

        // tracking overlay nodes for column/row resizing
        var resizeOverlayNode = $('<div class="abs resize-tracking"><div class="abs leading"></div><div class="abs trailing"></div></div>');

        // the current status text label
        var statusText = null;

        // last known touch position (to open context menu on the correct position); with properties 'pageX' and 'pageY'
        var lastTouchPos = null;

        // the index and model of the active sheet
        var activeSheet = -1;
        var activeSheetModel = null;

        // collections of the active sheet
        var colCollection = null;
        var rowCollection = null;
        var mergeCollection = null;
        var cellCollection = null;
        var tableCollection = null;
        var validationCollection = null;
        var condFormatCollection = null;
        var drawingCollection = null;
        var commentCollection = null;

        // base constructors --------------------------------------------------

        TextBaseView.call(this, app, docModel, {
            initHandler: initHandler,
            initDebugHandler: initDebugHandler,
            initGuiHandler: initGuiHandler,
            initDebugGuiHandler: initDebugGuiHandler,
            grabFocusHandler: grabFocusHandler
        });

        ViewAttributesMixin.call(this, VIEW_ATTRIBUTE_DEFINITIONS);

        // private methods ----------------------------------------------------

        /**
         * Updates the visibility, position, and size of all header panes and
         * grid panes.
         *
         * @param {Number} maxRowCount
         *  The maximum index of the row to be displayed in row header panes.
         */
        var upateHeaderAndGridPanes = RenderUtils.profileMethod('SpreadsheetView.upateHeaderAndGridPanes()', function (maxRowCount) {

            // whether frozen split mode is active
            var frozenSplit = activeSheetModel.hasFrozenSplit();
            // whether dynamic split mode is really active
            var dynamicSplit = !frozenSplit && activeSheetModel.hasSplit();
            // the size of the split lines
            var splitLineSize = frozenSplit ? FROZEN_SPLIT_SIZE : DYNAMIC_SPLIT_SIZE;
            // start position of the split lines
            var splitLineLeft = activeSheetModel.getSplitWidth();
            var splitLineTop = activeSheetModel.getSplitHeight();

            // expand dynamic splits to minimum split size
            if (dynamicSplit) {
                if (splitLineLeft > 0) { splitLineLeft = Math.max(splitLineLeft, PaneUtils.MIN_PANE_SIZE); }
                if (splitLineTop > 0) { splitLineTop = Math.max(splitLineTop, PaneUtils.MIN_PANE_SIZE); }
            }

            // whether the left and top panes are visible
            var visibleSides = {};
            visibleSides.left = (dynamicSplit || frozenSplit) && (splitLineLeft > 0);
            visibleSides.top = (dynamicSplit || frozenSplit) && (splitLineTop > 0);

            // calculate current size of header nodes
            cornerPane.initializePaneLayout(maxRowCount);
            var headerWidth = cornerPane.getWidth();
            var headerHeight = cornerPane.getHeight();

            // calculate inner width of left panes
            var leftSettings = paneSideSettings.left;
            if (visibleSides.left) {
                leftSettings.offset = headerWidth;
                leftSettings.size = splitLineLeft;
                leftSettings.hiddenSize = frozenSplit ? colCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorLeft')) : 0;
            } else {
                leftSettings.offset = leftSettings.size = leftSettings.hiddenSize = 0;
            }

            // calculate inner height of top panes
            var topSettings = paneSideSettings.top;
            if (visibleSides.top) {
                topSettings.offset = headerHeight;
                topSettings.size = splitLineTop;
                topSettings.hiddenSize = frozenSplit ? rowCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorTop')) : 0;
            } else {
                topSettings.offset = topSettings.size = topSettings.hiddenSize = 0;
            }

            // calculate effective position of split lines
            splitLineLeft = leftSettings.offset + leftSettings.size;
            splitLineTop = topSettings.offset + topSettings.size;

            // determine whether right and bottom panes are visible (must have enough room in split and frozen mode)
            visibleSides.right = !visibleSides.left || (splitLineLeft + splitLineSize + PaneUtils.MIN_PANE_SIZE + Utils.SCROLLBAR_WIDTH <= rootNode.width());
            visibleSides.bottom = !visibleSides.top || (splitLineTop + splitLineSize + PaneUtils.MIN_PANE_SIZE + Utils.SCROLLBAR_HEIGHT <= rootNode.height());

            // visibility of the split lines
            var colSplit = visibleSides.left && visibleSides.right;
            var rowSplit = visibleSides.top && visibleSides.bottom;

            // calculate the resulting grid pane positions and sizes of the right panes
            var rightSettings = paneSideSettings.right;
            if (visibleSides.right) {
                rightSettings.offset = colSplit ? (splitLineLeft + splitLineSize) : headerWidth;
                rightSettings.size = rootNode.width() - rightSettings.offset;
                rightSettings.hiddenSize = frozenSplit ? (leftSettings.hiddenSize + leftSettings.size) : 0;
            } else if (visibleSides.left) {
                leftSettings.size = rootNode.width() - headerWidth;
                rightSettings.offset = rightSettings.size = rightSettings.hiddenSize = 0;
            }

            // calculate the resulting grid pane positions and sizes of the bottom panes
            var bottomSettings = paneSideSettings.bottom;
            if (visibleSides.bottom) {
                bottomSettings.offset = rowSplit ? (splitLineTop + splitLineSize) : headerHeight;
                bottomSettings.size = rootNode.height() - bottomSettings.offset;
                bottomSettings.hiddenSize = frozenSplit ? (topSettings.hiddenSize + topSettings.size) : 0;
            } else if (visibleSides.top) {
                topSettings.size = rootNode.height() - headerHeight;
                bottomSettings.offset = bottomSettings.size = bottomSettings.hiddenSize = 0;
            }

            // set frozen mode (left/top panes are not scrollable in frozen mode in their
            // own direction, e.g. left frozen panes are not scrollable to left/right)
            leftSettings.frozen = topSettings.frozen = frozenSplit;
            rightSettings.frozen = bottomSettings.frozen = false;

            // set up scroll bar visibility in the opposite direction (e.g. left/right
            // scroll bars of top panes are hidden, if bottom panes are visible)
            leftSettings.showOppositeScroll = !visibleSides.right;
            topSettings.showOppositeScroll = !visibleSides.bottom;
            rightSettings.showOppositeScroll = bottomSettings.showOppositeScroll = true;

            // initialize the header panes
            headerPaneMap.forEach(function (headerPane, paneSide) {
                headerPane.initializePaneLayout(paneSideSettings[paneSide]);
            });

            // initialize the grid panes
            gridPaneMap.forEach(function (gridPane, panePos) {
                gridPane.initializePaneLayout(paneSideSettings[PaneUtils.getColPaneSide(panePos)], paneSideSettings[PaneUtils.getRowPaneSide(panePos)]);
            });

            // visibility and position of the split lines
            colSplitLineNode.toggle(colSplit).css({ left: splitLineLeft, width: splitLineSize });
            rowSplitLineNode.toggle(rowSplit).css({ top: splitLineTop, height: splitLineSize });

            // visibility and position of the split tracking nodes
            colSplitTrackingNode.toggle(dynamicSplit && colSplit).css({ left: splitLineLeft - TRACKING_OFFSET });
            rowSplitTrackingNode.toggle(dynamicSplit && rowSplit).css({ top: splitLineTop - TRACKING_OFFSET });
            centerSplitTrackingNode.toggle(dynamicSplit && colSplit && rowSplit).css({ left: splitLineLeft - TRACKING_OFFSET, top: splitLineTop - TRACKING_OFFSET });
        });

        /**
         * Updates the layer intervals in all header panes, passes the changed
         * layer intervals to the rendering cache, and forwards the new layer
         * ranges received from the rendering cache to all grid panes.
         *
         * @param {IntervalArray|Null} colIntervals
         *  An array with column intervals specifying all dirty columns in the
         *  sheet that must be recalculated and repainted (e.g. after one or
         *  more document operations such as inserting or deleting columns).
         *
         * @param {IntervalArray|Null} rowIntervals
         *  An array with row intervals specifying all dirty rows in the sheet
         *  that must be recalculated and repainted (e.g. after one or more
         *  document operations such as inserting or deleting rows).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.hide=false]
         *      If set to true, all layers will be explicitly hidden.
         *  - {Boolean} [options.full=false]
         *      If set to true, all cells in the entire sheet are dirty and
         *      must be recalculated and repainted (e.g. when zooming). This
         *      option will be ignored, if the option 'hide' has been set.
         *  - {Boolean} [options.force=false]
         *      If set to true, the layer intervals will always be recalculated
         *      regardless of the current scroll position. In difference to the
         *      option 'full', this option does not cause a full repaint.
         *      This option will be ignored, if the option 'hide' has been set.
         */
        var updateRenderingLayers = RenderUtils.profileMethod('SpreadsheetView.updateRenderingLayers()', function (colIntervals, rowIntervals, options) {

            var maxRowCount = 100;
            headerPaneMap.forEach(function (headerPane, paneSide) {
                if (!PaneUtils.isColumnSide(paneSide)) {
                    var interval = headerPane.getRenderInterval();
                    if (interval) {
                        maxRowCount = Math.max(interval.last, maxRowCount);
                    }
                }
            });

            if (Utils.getBooleanOption(options, 'initialize', false) || !Utils.isSameDigitsLength(maxRowCount, cornerPane.getMaxRowCount())) {
                upateHeaderAndGridPanes(maxRowCount);
            }

            // invalidate the entire sheet, or calculate the dirty ranges from the passed intervals
            var dirtyRanges = new RangeArray();
            if (Utils.getBooleanOption(options, 'full', false)) {
                dirtyRanges.push(docModel.getSheetRange());
            } else {
                if (colIntervals) { dirtyRanges.append(docModel.makeColRanges(colIntervals)); }
                if (rowIntervals) { dirtyRanges.append(docModel.makeRowRanges(rowIntervals)); }
                dirtyRanges = dirtyRanges.merge();
            }

            // force updating the layer intervals, if columns or rows in the sheet are dirty
            if (!dirtyRanges.empty()) {
                RenderUtils.log('dirty ranges: ' + dirtyRanges);
                (options || (options = {})).force = true;
            }
            var newRowCount = 100;
            // update the layer intervals of all header panes
            var headerBoundaryMap = new ValueMap();
            headerPaneMap.forEach(function (headerPane, paneSide) {
                var boundary = headerPane.updateLayerInterval(options);
                if (boundary) { headerBoundaryMap.insert(paneSide, boundary); }
                if (!PaneUtils.isColumnSide(paneSide)) {
                    var interval = headerPane.getRenderInterval();
                    if (interval) {
                        newRowCount = Math.max(interval.last, newRowCount);
                    }
                }
            });

            // This can happens if the user jumps with ctrl+(cursor Up or Down)
            if (!Utils.isSameDigitsLength(newRowCount, cornerPane.getMaxRowCount()) && !Utils.isSameDigitsLength(maxRowCount, newRowCount)) {
                upateHeaderAndGridPanes(newRowCount);
            }

            // nothing to do without changed intervals
            if (headerBoundaryMap.empty()) { return; }

            // pass all layer boundaries to the rendering cache, result contains all changed layer ranges
            var layerBoundaryMap = renderCache.updateLayerRanges(headerBoundaryMap, dirtyRanges);

            // distribute all changed layer ranges to the grid panes
            layerBoundaryMap.forEach(function (boundary, panePos) {
                gridPaneMap.get(panePos).setLayerRange(boundary, dirtyRanges);
            });
        });

        /**
         * Refreshes the visibility, position, and size of all header panes,
         * grid panes, and tracking nodes.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.full=false]
         *      If set to true, all cells in the entire sheet are dirty and
         *      must be recalculated and repainted (e.g. when zooming).
         *  - {Boolean} [options.force=false]
         *      If set to true, the layer intervals will always be recalculated
         *      regardless of the current scroll position. In difference to the
         *      option 'full', this option does not cause a full repaint.
         */
        var initializePanes = (function () {

            // collect flags from multiple debounced calls
            var cachedOptions = {};

            // collects the options passed to multiple invocations
            function registerInvocation(options) {
                cachedOptions.full = cachedOptions.full || Utils.getBooleanOption(options, 'full', false);
                cachedOptions.force = cachedOptions.force || Utils.getBooleanOption(options, 'force', false);
                // ignore updates of active pane caused by focus changes during pane initialization
                panesInitialized = false;
            }

            // initializes the header and grid panes
            var initializePanes = RenderUtils.profileMethod('SpreadsheetView.initializePanes()', function () {

                // view layout updates may be requested in case the import has failed, and no sheet is available
                if (!activeSheetModel) { return; }

                // whether the focus is located in the grid panes
                var hasFocus = self.hasAppFocus();

                // update layer intervals and ranges, and the rendering cache
                cachedOptions.initialize = true;
                updateRenderingLayers(null, null, cachedOptions);

                // reset cached settings for next debounced invocations
                cachedOptions = {};
                panesInitialized = true;

                // update focus (active grid pane may have been changed)
                if (hasFocus) { self.grabFocus(); }
            });

            // return a debounced method that waits for document actions/operations before initializing the panes
            return app.createDebouncedActionsMethodFor(self, 'SpreadsheetView.initializePanes', registerInvocation, initializePanes);
        }());

        /**
         * Triggers all events related to changed view attributes of the active
         * sheet.
         */
        function triggerChangeSheetViewAttributes(newAttributes) {
            var selection = newAttributes.selection;
            if (selection) { self.trigger('change:selection:prepare', selection); }
            self.trigger('change:sheet:viewattributes', newAttributes);
            if (selection) { self.trigger('change:selection', selection); }
        }

        /**
         * Event handler for changed view attributes of the document model.
         * Updates internal settings after changing the active sheet, and
         * triggers 'before:activesheet', 'change:activesheet', and
         * 'change:doc:viewattributes' events.
         */
        function changeModelViewAttributesHandler(event, changedAttributes) {

            if ('activeSheet' in changedAttributes) {

                // nothing to do if the sheet index is not valid (e.g. initially after import,
                // or during sheet operations such as inserting or deleting a sheet)
                if (activeSheet >= 0) {
                    // cancel text edit mode before switching sheets
                    self.cancelTextEditMode();
                    // hide contents in all header and grid panes immediately
                    updateRenderingLayers(null, null, { hide: true });
                    // notify listeners with the index of the old active sheet
                    self.trigger('before:activesheet', activeSheet);
                }

                // initialize the view according to the new active sheet, and notify listeners
                // (nothing to do when invalidating the active sheet temporarily, e.g. during
                // sheet operations such as inserting or deleting a sheet)
                if (changedAttributes.activeSheet >= 0) {

                    // get model instance and collections of the new active sheet
                    activeSheet = changedAttributes.activeSheet;
                    activeSheetModel = docModel.getSheetModel(activeSheet);
                    colCollection = activeSheetModel.getColCollection();
                    rowCollection = activeSheetModel.getRowCollection();
                    mergeCollection = activeSheetModel.getMergeCollection();
                    cellCollection = activeSheetModel.getCellCollection();
                    tableCollection = activeSheetModel.getTableCollection();
                    validationCollection = activeSheetModel.getValidationCollection();
                    condFormatCollection = activeSheetModel.getCondFormatCollection();
                    drawingCollection = activeSheetModel.getDrawingCollection();
                    commentCollection = activeSheetModel.getCommentCollection();

                    // notify listeners
                    self.trigger('change:activesheet', activeSheet, activeSheetModel);
                    triggerChangeSheetViewAttributes(activeSheetModel.getViewAttributes());

                    // refresh the layout of the header and grid panes
                    initializePanes();
                }
            }

            self.trigger('change:doc:viewattributes', changedAttributes);
        }

        /**
         * Finalizes insertion of a sheet into the document.
         */
        function afterInsertSheetHandler(event, sheet, sheetModel) {

            // forwards all sheet events to own listeners while the sheet is active
            function forwardActiveSheetEvents(eventTypes, triggerType) {
                self.listenTo(sheetModel, eventTypes, function (event2) {
                    if (sheetModel === activeSheetModel) {
                        var eventType = triggerType || event2.type;
                        self.trigger.apply(self, [eventType].concat(_.toArray(arguments).slice(1)));
                    }
                });
            }

            // registers a callback for a sheet event that will be invoked while the sheet is active
            function handleActiveSheetEvent(eventType, callback) {
                self.listenTo(sheetModel, eventType, function () {
                    if (sheetModel === activeSheetModel) {
                        callback.apply(self, _.toArray(arguments).slice(1));
                    }
                });
            }

            // listen to changes in the sheet, notify listeners of the view
            forwardActiveSheetEvents('change:attributes', 'change:sheet:attributes');
            handleActiveSheetEvent('change:viewattributes', triggerChangeSheetViewAttributes);
            forwardActiveSheetEvents('refresh:ranges');
            forwardActiveSheetEvents('insert:columns delete:columns change:columns');
            forwardActiveSheetEvents('insert:rows delete:rows change:rows');
            forwardActiveSheetEvents('insert:merged delete:merged');
            forwardActiveSheetEvents('change:cells move:cells');
            forwardActiveSheetEvents('insert:rule delete:rule change:rule');
            forwardActiveSheetEvents('insert:table delete:table change:table');
            forwardActiveSheetEvents('insert:drawing delete:drawing change:drawing change:drawing:text move:drawing');
            forwardActiveSheetEvents('insert:comment delete:comment change:comment change:comment:text move:comment');
            forwardActiveSheetEvents('change:usedrange');

            // a generic 'triggered' event for any event of the active sheet
            forwardActiveSheetEvents('triggered', 'sheet:triggered');
        }

        /**
         * Processes column/row operations that have been applied in the active
         * sheet. Collects the dirty column/row indexes in from the operations,
         * and updates the layer intervals of all visible header panes, which
         * will trigger all dependent rendering.
         */
        var changeColRowHandler = (function () {

            // all columns intervals to be refreshed
            var colIntervals = new IntervalArray();
            // all row intervals to be refreshed
            var rowIntervals = new IntervalArray();

            // collects the minimum column/row index changed by an operation
            function registerOperation(event, interval, changeInfo) {

                // collect the minimum column/row index changed by an operation
                switch (event.type) {
                    case 'insert:columns':
                    case 'delete:columns':
                        colIntervals.push(new Interval(interval.first, docModel.getMaxCol()));
                        break;

                    case 'change:columns':
                        if (changeInfo.sizeChanged) {
                            colIntervals.push(new Interval(interval.first, docModel.getMaxCol()));
                            // update size of frozen panes after changing column size or visibility
                            if (activeSheetModel.hasFrozenSplit()) { initializePanes(); }
                        } else {
                            colIntervals.push(interval);
                        }
                        break;

                    case 'insert:rows':
                    case 'delete:rows':
                        rowIntervals.push(new Interval(interval.first, docModel.getMaxRow()));
                        break;

                    case 'change:rows':
                        if (changeInfo.sizeChanged) {
                            rowIntervals.push(new Interval(interval.first, docModel.getMaxRow()));
                            // update size of frozen panes after changing row size or visibility
                            if (activeSheetModel.hasFrozenSplit()) { initializePanes(); }
                        } else {
                            rowIntervals.push(interval);
                        }
                        break;

                    default:
                        Utils.error('SpreadsheetView.changeColRowHandler(): unknown event "' + event.type + '"');
                        return;
                }
            }

            // bug 36014: debounce if model currently applies more operations (e.g. Undo)
            return app.createDebouncedActionsMethodFor(self, 'SpreadsheetView.changeColRowHandler', registerOperation, function () {
                updateRenderingLayers(colIntervals, rowIntervals);
                colIntervals.clear();
                rowIntervals.clear();
            });
        }());

        /**
         * Handles all tracking events for the split separators.
         */
        function splitTrackingHandler(event) {

            // the event source node
            var sourceNode = $(this);
            // whether column split tracking and/or row split tracking is active
            var colSplit = sourceNode.hasClass('columns');
            var rowSplit = sourceNode.hasClass('rows');
            // minimum and maximum position of split lines
            var minLeft = self.getHeaderWidth();
            var maxLeft = rootNode.width() - Utils.SCROLLBAR_WIDTH - DYNAMIC_SPLIT_SIZE;
            var minTop = self.getHeaderHeight();
            var maxTop = rootNode.height() - Utils.SCROLLBAR_HEIGHT - DYNAMIC_SPLIT_SIZE;

            // returns the X position of a split line according to the current event
            function getTrackingSplitLeft() {
                var offset = paneSideSettings.left.offset + paneSideSettings.left.size + (colSplit ? event.offsetX : 0);
                return (offset - PaneUtils.MIN_PANE_SIZE < minLeft) ? minLeft : (offset + PaneUtils.MIN_PANE_SIZE > maxLeft) ? maxLeft : offset;
            }

            // returns the Y position of a split line according to the current event
            function getTrackingSplitTop() {
                var offset = paneSideSettings.top.offset + paneSideSettings.top.size + (rowSplit ? event.offsetY : 0);
                return (offset - PaneUtils.MIN_PANE_SIZE < minTop) ? minTop : (offset + PaneUtils.MIN_PANE_SIZE > maxTop) ? maxTop : offset;
            }

            function updateNodeOffset(node, propName, offset, min, max) {
                node.toggleClass('collapsed', (offset <= min) || (offset >= max))
                    .add(centerSplitTrackingNode)
                    .css(propName, (offset - TRACKING_OFFSET) + 'px');
            }

            function finalizeTracking() {
                allSplitTrackingNodes.removeClass('tracking-active');
                self.grabFocus();
            }

            switch (event.type) {

                case 'tracking:start':
                    colSplitTrackingNode.toggleClass('tracking-active', colSplit);
                    rowSplitTrackingNode.toggleClass('tracking-active', rowSplit);
                    break;

                case 'tracking:move':
                    if (colSplit) { updateNodeOffset(colSplitTrackingNode, 'left', getTrackingSplitLeft(), minLeft, maxLeft); }
                    if (rowSplit) { updateNodeOffset(rowSplitTrackingNode, 'top', getTrackingSplitTop(), minTop, maxTop); }
                    break;

                case 'tracking:end':
                    var splitWidth = null, splitHeight = null;
                    if (colSplit) {
                        splitWidth = getTrackingSplitLeft();
                        splitWidth = ((minLeft < splitWidth) && (splitWidth < maxLeft)) ? activeSheetModel.convertPixelToHmm(splitWidth - self.getHeaderWidth()) : 0;
                    } else {
                        splitWidth = activeSheetModel.getSplitWidthHmm();
                    }
                    if (rowSplit) {
                        splitHeight = getTrackingSplitTop();
                        splitHeight = ((minTop < splitHeight) || (splitHeight < maxTop)) ? activeSheetModel.convertPixelToHmm(splitHeight - self.getHeaderHeight()) : 0;
                    } else {
                        splitHeight = activeSheetModel.getSplitHeightHmm();
                    }
                    activeSheetModel.setDynamicSplit(splitWidth, splitHeight);
                    finalizeTracking();
                    break;

                case 'tracking:cancel':
                    initializePanes();
                    finalizeTracking();
                    break;
            }
        }

        /**
         * Handles 'keydown' events from the view root node.
         */
        function keyDownHandler(event) {

            // do nothing while text edit mode (cell or drawing) is active
            if (self.isTextEditMode()) { return; }

            // special handling for drawing selection
            if (self.hasDrawingSelection()) {

                // enter text edit mode in first selected drawing
                if (KeyCodes.matchKeyCode(event, 'F2')) {
                    self.enterTextEditMode('drawing');
                    return false;
                }

                // delete selected drawings
                if (KeyCodes.matchKeyCode(event, 'DELETE') || KeyCodes.matchKeyCode(event, 'BACKSPACE')) {
                    if (self.requireEditMode()) {
                        // bug 39533: delete drawings via controller for busy screen etc.
                        self.executeControllerItem('drawing/delete');
                    }
                    return false;
                }

                return;
            }

            // enter cell edit mode with current cell contents
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                self.enterTextEditMode('cell');
                return false;
            }

            // open context menu
            if (KeyCodes.matchKeyCode(event, 'F10')) {
                globalContextMenuHandler(event);
                return false;
            }

            // enter cell edit mode with empty text area
            if (!_.browser.MacOS && KeyCodes.matchKeyCode(event, 'BACKSPACE')) {
                self.enterTextEditMode('cell', { text: '' });
                return false;
            }

            // clear contents (not formatting) of all cells in the selection
            if (KeyCodes.matchKeyCode(event, 'DELETE') || (_.browser.MacOS && KeyCodes.matchKeyCode(event, 'BACKSPACE'))) {
                if (self.requireEditMode()) {
                    // bug 39533: clear cell range via controller for busy screen etc.
                    self.executeControllerItem('cell/clear/values');
                }
                return false;
            }
        }

        /**
         * Handles 'keypress' events from the view root node. Starts the cell
         * edit mode on-the-fly for valid Unicode characters.
         */
        function keyPressHandler(event) {

            // the initial text to be inserted into the text area
            var initialText = null;

            // do not handle 'keypress' events bubbled up from active text edit mode
            if (self.isTextEditMode()) { return; }

            // ignore key events where either CTRL/META or ALT is pressed, ignore
            // SPACE keys with any control keys (used as shortcuts for column/row selection)
            // - Add OS-switch to allow "alt + l" for "@"-sign typing on osx-devices
            if (((event.charCode > 32) && ((_.browser.MacOS && Utils.boolEq(event.ctrlKey, event.metaKey)) || (!_.browser.MacOS && Utils.boolEq(event.ctrlKey || event.metaKey, event.altKey)))) || ((event.charCode === 32) && !KeyCodes.hasModifierKeys(event))) {

                // build string from character code
                initialText = String.fromCharCode(event.charCode);

                // decide which text edit mode will be started
                var editMode = self.hasDrawingSelection() ? 'drawing' : 'cell';
                // the options passed to the text editor
                var options = { text: initialText };

                // special handling for cell edit mode
                if (editMode === 'cell') {
                    // start with quick edit mode (cursor keys will move cell cursor instead of text cursor)
                    options.quick = true;
                    // bug 52949: immediately start auto-completion after first character
                    options.autoComplete = true;
                    // percentage number format: add percent sign to number-like first character
                    if ((self.getNumberFormatCategory() === 'percent') && (/^[-+0-9]$/.test(initialText) || (initialText === LocaleData.DEC))) {
                        initialText += '%';
                        options.pos = 1;
                    }
                }

                // start text edit mode, and drop the key event
                self.enterTextEditMode(editMode, options);
                return false;
            }
        }

        function globalContextMenuHandler(event) {

            // quit, if there is no event
            if (_.isUndefined(event)) { return false; }

            // the event target node, as jQuery object
            var targetNode = $(event.target);

            // quit if the contextmenu event was triggered on an input-/textfield
            if (targetNode.is('input, textarea')) { return; }

            var // was the event triggered by keyboard?
                isGlobalTrigger     = _.browser.IE ? targetNode.is('.clipboard') : Utils.isContextEventTriggeredByKeyboard(event),

                selectedDrawings    = self.getSelectedDrawings(),
                activeGridPane      = self.getActiveGridPane(),
                layerRootNode       = activeGridPane.getLayerRootNode(),
                scrollNode          = layerRootNode.parent(),
                sheetOffset         = layerRootNode.offset(),
                sheetPosition       = layerRootNode.position(),
                // scroll positions needed for scrolling hacking in IE & Firefox
                scrollLeft          = scrollNode.scrollLeft(),
                scrollTop           = scrollNode.scrollTop(),
                // the node, on which the event should be triggered later
                triggerNode         = null;

            // contextmenu event on the header-pane
            if (targetNode.is('.header-pane .cell')) {
                triggerNode = targetNode;

            // contextmenu event on the status-pane
            } else if (targetNode.is('.status-pane .button, .status-pane .button *') || (isGlobalTrigger && $('.status-pane .active-sheet-group.focused .selected').length > 0)) {
                if (isGlobalTrigger && $('.status-pane .active-sheet-group.focused .selected').length > 0) {
                    triggerNode = $('.status-pane .active-sheet-group.focused .button.selected');
                    event.pageX = event.pageX - sheetPosition.left + sheetOffset.left;
                    event.pageY = event.pageY + sheetPosition.top + sheetOffset.top;
                } else {
                    if (event.pageX === 0 || event.pageY === 0 || event.pageX === undefined || event.pageY === undefined) {
                        event.pageX = targetNode.offset().left;
                        event.pageY = targetNode.offset().top;
                    }
                    triggerNode = targetNode;
                }

            // contextmenu event on a drawing
            } else if (selectedDrawings.length > 0) {
                triggerNode = activeGridPane.getDrawingRenderer().getDrawingFrame(selectedDrawings[0]);

                // if we have a last known touch position, use it
                if (lastTouchPos) {
                    event.pageX = lastTouchPos.pageX;
                    event.pageY = lastTouchPos.pageY;

                // otherwise, locate a meaningful position
                } else if (isGlobalTrigger) {
                    var rectangle = drawingCollection.getModel(selectedDrawings[0]).getRectangle();

                    event.target = triggerNode;
                    event.pageX = rectangle.left + (rectangle.width / 2) - sheetPosition.left + sheetOffset.left - scrollLeft;
                    event.pageY = rectangle.top + (rectangle.height / 2) - sheetPosition.top + sheetOffset.top - scrollTop;
                }

            // contextmenu event on a cell
            } else if (targetNode.is('.grid-pane *, html')) {
                triggerNode = activeGridPane.getNode();

                // if we have a last known touch position, use it
                if (lastTouchPos) {
                    event.pageX = lastTouchPos.pageX;
                    event.pageY = lastTouchPos.pageY;

                // otherwise, locate a meaningful position
                } else if (isGlobalTrigger) {
                    var activeCellRect = activeSheetModel.getCellRectangle(self.getActiveCell(), { expandMerged: true });
                    event.pageX = Utils.convertHmmToLength(activeCellRect.centerX(), 'px', 1) - sheetPosition.left + sheetOffset.left - scrollLeft;
                    event.pageY = Utils.convertHmmToLength(activeCellRect.centerY(), 'px', 1) - sheetPosition.top + sheetOffset.top - scrollTop;
                }
            }

            if (triggerNode) {
                triggerNode.trigger(new $.Event('documents:contextmenu', { sourceEvent: event }));
            }

            return false;
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            // handle events of the formula dependency manager (show alerts, etc.)
            var dependencyManager = docModel.getDependencyManager();
            // collect the addresses of all formula cells changed locally (for circular reference alert)
            var changedFormulaMaps = new ValueMap();

            // insert the root node into the application pane
            self.insertContentNode(rootNode);
            // keep the text contents of drawing objects in the hidden DOM
            self.insertHiddenNodes(textRootNode);

            // insert resize overlay node first (will increase its z-index while active)
            rootNode.append(resizeOverlayNode);

            // insert the pane nodes into the DOM
            gridPaneMap.forEach(function (gridPane) { rootNode.append(gridPane.getNode()); });
            headerPaneMap.forEach(function (headerPane) { rootNode.append(headerPane.getNode()); });
            rootNode.append(cornerPane.getNode());

            // append the split lines and split tracking nodes
            rootNode.append(colSplitLineNode, rowSplitLineNode, colSplitTrackingNode, rowSplitTrackingNode, centerSplitTrackingNode);

            // to save the position from 'touchstart', we have to register some global event handlers
            // we need this position to open the context menu on the correct location
            if (_.browser.iOS || _.browser.Android) {
                app.registerGlobalEventHandler($(document), 'touchstart', function (event) {
                    if (!event) { return false; }
                    lastTouchPos = event.originalEvent.changedTouches[0];
                });
                app.registerGlobalEventHandler($(document), 'touchend', function (event) {
                    if (!event) { return false; }
                    // delay, because some contextmenus open only on 'touchend' (drawing context menus for example)
                    self.executeDelayed(function () { lastTouchPos = null; }, 'SpreadsheetView.initHandler', 100);
                });
            }

            // update view when document edit mode changes
            self.listenTo(app, 'docs:editmode', function (editMode) {
                rootNode.toggleClass('edit-mode', editMode);
            });

            // application notifies changed data of remote clients
            self.listenTo(app, 'docs:users:selection', function (activeClients) {
                // filter for remote clients with existing selection data
                var remoteClients = activeClients.filter(function (client) {
                    return client.remote && _.isObject(client.userData) && _.isNumber(client.userData.sheet) && _.isString(client.userData.ranges);
                });
                // store selections (this triggers a change event which causes rendering)
                self.setViewAttribute('remoteClients', remoteClients);
            });

            // set a custom application state during background formula calculation, and show a busy label
            self.listenTo(dependencyManager, 'recalc:start', function () { app.setUserState('recalc'); });
            self.listenTo(dependencyManager, 'recalc:end recalc:cancel', function () { app.setUserState(null); });
            self.registerStatusCaption('recalc', Labels.CALCULATING_LABEL, { busy: true, delay: 1000 });

            // collect all formula cells that have been changed locally
            self.listenTo(docModel, 'change:cells', function (event, sheet, changeDesc, external) {
                if (!external) {
                    var sheetUid = docModel.getSheetModel(sheet).getUid();
                    var addressSet = changedFormulaMaps.getOrConstruct(sheetUid, ValueSet, 'key()');
                    changeDesc.formulaCells.forEach(addressSet.insert, addressSet);
                }
            });

            // show a warning alert when reference cycles have been found during calculation
            self.listenTo(dependencyManager, 'recalc:end', function (event, referenceCycles) {

                // check if the reference cycles cover a formula cell changed locally
                var hasCycle = referenceCycles.some(function (addresses) {
                    return addresses.some(function (address) {
                        var addressSet = changedFormulaMaps.get(address.sheetUid, null);
                        return addressSet && addressSet.has(address);
                    });
                });
                changedFormulaMaps.clear();

                // show an alert box if a new cycle has been found
                if (hasCycle) {
                    self.yellNotification({
                        type: 'info',
                        //#. title for a warning message: a formula addresses its own cell, e.g. =A1 in cell A1
                        headline: gt('Circular Reference'),
                        message: gt('A reference in the formula is dependent on the result of the formula cell.')
                    });
                }
            });

            // show an error message (without timeout) if the document contains too many formulas
            self.listenOnceTo(dependencyManager, 'recalc:overflow', function () {
                self.yellNotification({
                    type: 'error',
                    //#. The number of spreadsheet formulas is limited in the configuration. If a document exceeds this limit, the user will see this alert.
                    message: gt('This document exceeds the spreadsheet size and complexity limits. Calculation has been stopped. Formula results are not updated anymore.')

                //  as for task DOCS-733 - "Improve OOM Error Message in Client" - [https://jira.open-xchange.com/browse/DOCS-733] ...
                //
                //    ... text partial 'This document exceeds the spreadsheet size and complexity limits.'
                //    is already part of 'io.ox/office/baseframework/utils/errormessages' but can not be derived
                //    from there ... SERVER_ERROR_MESSAGES['LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR.SPREADSHEET']
                });
            });

            // show warning message if the document contains any formulas with unimplemented functions
            self.listenOnceTo(dependencyManager, 'recalc:end', function () {

                // the formula grammar needed to receive translated function names
                var formulaGrammar = docModel.getFormulaGrammar('ui');
                // the resource keys of all unimplemented functions occurring in the document
                var funcKeys = dependencyManager.getMissingFunctionKeys();
                // the translated function names of all unsupported functions
                var funcNames = funcKeys.map(formulaGrammar.getFunctionName, formulaGrammar).filter(_.identity).sort();

                // show a warning message if the document contains any unsupported functions
                if (funcNames.length > 0) {
                    funcNames = funcNames.map(function (funcName) { return '\n\xa0\u2022\xa0' + funcName; }).join();
                    self.yellNotification({
                        type: 'warning',
                        //#. A list with all unsupported functions will be shown below this message.
                        message: _.noI18n(gt('This spreadsheet contains formulas with unsupported functions:') + funcNames),
                        duration: 30000
                    });
                }
            });

            // handle sheet collection events
            self.listenTo(docModel, {
                'change:viewattributes': changeModelViewAttributesHandler,
                'insert:sheet:after': afterInsertSheetHandler
            });

            // register GUI event handlers after successful import
            self.waitForImportSuccess(function () {

                // register the context menu handler
                app.registerGlobalEventHandler($(document), 'contextmenu', globalContextMenuHandler);

                // enable tracking and register tracking event handlers
                Tracking.enableTracking(allSplitTrackingNodes);
                allSplitTrackingNodes.on('tracking:start tracking:move tracking:end tracking:cancel', splitTrackingHandler);

                // handle generic key events
                rootNode.on({ keydown: keyDownHandler, keypress: keyPressHandler });
            });

            // hide everything, if import fails
            self.waitForImportFailure(function () {
                rootNode.hide();
                statusPane.hide();
            });

            // create the additional view panes
            self.addPane(formulaPane = new FormulaPane(self));
            self.addPane(statusPane = new StatusPane(self));

            // repaint all grid and header panes when the size of the root node has changed
            self.on('refresh:layout', function () {
                var newSize = { width: rootNode[0].clientWidth, height: rootNode[0].clientHeight };
                if ((rootNodeSize.width !== newSize.width) || (rootNodeSize.height !== newSize.height)) {
                    rootNodeSize = newSize;
                    initializePanes();
                }
            });

            // update locked state of active sheet in the DOM
            self.on('change:activesheet change:sheet:attributes', function () {
                rootNode.toggleClass('sheet-locked', activeSheetModel.isLocked());
                rootNode.toggleClass('cell-sheet', activeSheetModel.isCellType());
            });

            // update header panes after column and/or row operations
            self.on('insert:columns delete:columns change:columns insert:rows delete:rows change:rows', changeColRowHandler);

            // update global settings according to view attributes of active sheet
            self.on('change:sheet:viewattributes', function (event, attributes) {
                if ('zoom' in attributes) {
                    // zooming: initialize panes (size of column/row headers may change)
                    initializePanes({ full: true });
                } else if (Utils.hasProperty(attributes, /^split/)) {
                    // split/freeze settings have changed, refresh all panes
                    initializePanes({ force: true });
                } else if (Utils.hasProperty(attributes, /^anchor/)) {
                    // update visible column/row interval immediately while scrolling
                    updateRenderingLayers(null, null);
                }
            });

            // hide cell selection while drawings are selected
            self.on('change:selection', function (event, selection) {
                rootNode.toggleClass('drawing-selection', selection.drawings.length > 0);
            });

            // update attribute marker for text edit modes
            self.on('textedit:enter', function (event, editMode) { rootNode.attr('data-text-edit', editMode); });
            self.on('textedit:leave', function () { rootNode.removeAttr('data-text-edit'); });

            // restore the correct text framework selection after leaving drawing edit mode
            self.on('textedit:leave:drawing', function () {
                self.insertHiddenNodes(textRootNode);
            });

            // process events triggered by header panes
            headerPaneMap.forEach(function (headerPane, paneSide) {

                var columns = PaneUtils.isColumnSide(paneSide);
                var offsetAttr = columns ? 'left' : 'top';
                var sizeAttr = columns ? 'width' : 'height';
                var showResizeTimer = null;

                function showResizerOverlayNode(offset, size) {
                    resizeOverlayNode.attr('data-orientation', columns ? 'columns' : 'rows');
                    updateResizerOverlayNode(offset, size);
                    // Delay displaying the overlay nodes, otherwise they cover
                    // the resizer drag nodes of the header pane which will
                    // interfere with double click detection.
                    showResizeTimer = self.executeDelayed(function () {
                        rootNode.addClass('tracking-active');
                    }, 'SpreadsheetView.showResizerOverlayNode', 200);
                }

                function updateResizerOverlayNode(offset, size) {
                    var relativeOffset = paneSideSettings[paneSide].offset + offset - headerPane.getVisiblePosition().offset;
                    resizeOverlayNode.toggleClass('collapsed', size === 0);
                    resizeOverlayNode.find('>.leading').css(sizeAttr, (Math.max(0, relativeOffset) + 10) + 'px');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, (relativeOffset + size) + 'px');
                }

                function hideResizerOverlayNode() {
                    if (showResizeTimer) {
                        showResizeTimer.abort();
                        showResizeTimer = null;
                    }
                    rootNode.removeClass('tracking-active');
                    resizeOverlayNode.find('>.leading').css(sizeAttr, '');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, '');
                }

                // visualize resize tracking from header panes
                self.listenTo(headerPane, {
                    'resize:start': function (event, offset, size) {
                        showResizerOverlayNode(offset, size);
                    },
                    'resize:move': function (event, offset, size) {
                        updateResizerOverlayNode(offset, size);
                    },
                    'resize:end': function () {
                        hideResizerOverlayNode();
                    },
                    'change:scrollpos': function (event, scrollPos) {
                        self.trigger('change:scrollpos', paneSide, scrollPos);
                    }
                });
            });
        }

        /**
         * Additional debug initialization after construction.
         */
        function initDebugHandler(operationsPane, clipboardPane) {

            // create the output nodes in the operations pane
            operationsPane
                .addDebugInfoHeader('selection', 'Sheet Selection')
                .addDebugInfoNode('selection', 'sheet', 'Active (visible) sheet')
                .addDebugInfoNode('selection', 'ranges', 'Selected cell ranges')
                .addDebugInfoNode('selection', 'border', 'Mixed border attributes')
                .addDebugInfoNode('selection', 'draw', 'Selected drawing objects')
                .addDebugInfoNode('selection', 'pane', 'Active (focused) grid pane')
                .addDebugInfoNode('selection', 'auto', 'Auto-fill range')
                .addDebugInfoHeader('active', 'Active Cell', 'Formatting of the active cell')
                .addDebugInfoNode('active', 'col', 'Explicit attributes of column containing active cell')
                .addDebugInfoNode('active', 'row', 'Explicit attributes of row containing active cell')
                .addDebugInfoNode('active', 'cell', 'Explicit attributes of active cell')
                .addDebugInfoNode('active', 'url', 'Hyperlink URL of active cell')
                .addDebugInfoHeader('text', 'Text Selection')
                .addDebugInfoNode('text', 'edit', 'Text edit mode identifier')
                .addDebugInfoNode('text', 'start', 'Start position of text selection')
                .addDebugInfoNode('text', 'end', 'End position of text selection')
                .addDebugInfoNode('text', 'dir', 'Direction');

            // log information about the active sheet (index, name, used area)
            function logSheetInfo() {
                var sheetName = docModel.getSheetName(activeSheet);
                var usedRange = (activeSheetModel && activeSheetModel.getUsedRange()) || '<empty>';
                operationsPane.setDebugInfoText('selection', 'sheet', 'index=' + activeSheet + ', name="' + sheetName + '", used=' + usedRange);
            }
            self.on('change:activesheet change:usedrange', logSheetInfo);
            self.listenTo(docModel, 'transform:sheet rename:sheet', logSheetInfo);

            // log all selection events
            self.on('change:sheet:viewattributes', function (event, attributes) {
                if ('selection' in attributes) {
                    var selection = attributes.selection;
                    operationsPane
                        .setDebugInfoText('selection', 'ranges', 'count=' + selection.ranges.length + ', ranges=' + selection.ranges + ', active=' + selection.active + ', cell=' + selection.address + (selection.originCell ? (', origin=' + selection.origin) : ''))
                        .setDebugInfoText('selection', 'draw', 'count=' + selection.drawings.length + ', frames=' + ((selection.drawings.length > 0) ? ('[' + selection.drawings.map(JSON.stringify).join() + ']') : '<none>'));
                }
                if ('activePane' in attributes) {
                    operationsPane.setDebugInfoText('selection', 'pane', attributes.activePane);
                }
                if ('autoFillData' in attributes) {
                    var autoFill = attributes.autoFillData;
                    operationsPane.setDebugInfoText('selection', 'auto', autoFill ? ('border=' + autoFill.border + ', count=' + autoFill.count) : '<none>');
                }
            });

            // log formatting of the active cell, and the entire selection
            self.on('update:selection:data', function () {

                function stringifyAutoStyle(styleId) {
                    var attrs = docModel.getCellAutoStyles().getMergedAttributeSet(styleId);
                    return '<span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>style=' + Utils.escapeHTML(styleId || '<default>') + '</span>';
                }

                function stringifyColRowAttrs(desc, family) {
                    return '<span ' + AttributesToolTip.createAttributeMarkup(Utils.makeSimpleObject(family, desc.merged)) + '>attrs=' + Utils.escapeHTML(Utils.stringifyForDebug(desc.explicit)) + '</span>';
                }

                var activeCell = self.getActiveCell();
                var colDesc = colCollection.getEntry(activeCell[0]);
                var rowDesc = rowCollection.getEntry(activeCell[1]);
                var cellModel = cellCollection.getCellModel(activeCell);

                operationsPane
                    .setDebugInfoMarkup('selection', 'border', '<span ' + AttributesToolTip.createAttributeMarkup({ cell: self.getBorderAttributes() }) + '>mixed-borders</span>')
                    .setDebugInfoMarkup('active', 'col', stringifyAutoStyle(colDesc.style) + ', ' + stringifyColRowAttrs(colDesc, 'column'))
                    .setDebugInfoMarkup('active', 'row', stringifyAutoStyle(rowDesc.style) + ', ' + stringifyColRowAttrs(rowDesc, 'row'))
                    .setDebugInfoMarkup('active', 'cell', 'exists=' + !!cellModel + ', ' + stringifyAutoStyle(cellCollection.getStyleId(activeCell)))
                    .setDebugInfoMarkup('active', 'url', cellCollection.getEffectiveURL(activeCell) || '<none>');
            });

            // log settings for text edit mode
            self.on('textedit:enter textedit:leave', function (event, editMode) {
                operationsPane.setDebugInfoText('text', 'edit', (event.type === 'textedit:enter') ? editMode : '');
            });

            // displays the passed text selection state
            function logTextSelection(start, end, dir) {
                operationsPane
                    .setDebugInfoText('text', 'start', start ? start.join(', ') : '')
                    .setDebugInfoText('text', 'end', end ? end.join(', ') : '')
                    .setDebugInfoText('text', 'dir', dir || '');
            }

            // track the text selection during drawing edit mode
            self.on('textedit:enter:cell', function () {

                function resolvePos(lengths, pos) {
                    var para = 0;
                    for (; lengths[para] < pos; para += 1) {
                        pos -= (lengths[para] + 1);
                    }
                    return [para, pos];
                }

                var timer = self.repeatDelayed(function () {
                    var textArea = rootNode.find('textarea:focus');
                    if (textArea.length === 1) {
                        var lengths = _.pluck(textArea.val().split(/\n/), 'length');
                        var selection = Forms.getInputSelection(textArea);
                        var start = selection.start;
                        var end = selection.end;
                        var dir = (start === end) ? 'cursor' : (start < end) ? 'forwards' : 'backwards';
                        logTextSelection(resolvePos(lengths, start), resolvePos(lengths, end), dir);
                    } else {
                        logTextSelection();
                    }
                }, 'SpreadsheetView.logTextAreaSelection', { repeatDelay: 500, background: true });

                self.one('textedit:leave:cell', function () {
                    timer.abort();
                    logTextSelection();
                });
            });

            // the text selection engine (from text framework)
            var textSelection = docModel.getSelection();
            self.listenTo(textSelection, 'change', function () {
                logTextSelection(textSelection.getStartPosition(), textSelection.getEndPosition(), textSelection.getDirection());
            });

            // hide text selection settings when switching to cell selection
            self.on('change:selection', function (event, selection) {
                if (selection.drawings.length === 0) {
                    logTextSelection();
                }
            });

            // handle events triggered by the grid panes
            gridPaneMap.forEach(function (gridPane) {

                // process debug events triggered by grid panes
                gridPane.on('debug:clipboard', function (event, content) {
                    clipboardPane.setClipboardContent(content);
                });
            });

            // register an attribute formatter for number formats
            operationsPane.registerAttributeFormatter({ family: 'cell', name: 'formatId' }, function (formatId, markup) {

                var numberFormatter = docModel.getNumberFormatter();
                var parsedFormat = numberFormatter.getParsedFormat(formatId);
                var SECTION_TITLES = ['Section', 'Format code', 'Category', 'Color', 'Operator', 'Boundary'];
                var SECTION_HEADER_MARKUP = '<tr>' + SECTION_TITLES.map(function (title) { return '<th>' + title + '</th>'; }).join('') + '</tr>';

                function addCellMarkup(value, quote, cols) {
                    markup += '<td' + ((cols > 1) ? ' colspan="' + cols + '"' : '') + '>';
                    quote = quote ? '"' : '';
                    if (value !== null) {
                        switch (typeof value) {
                            case 'number':  markup += value; break;
                            case 'boolean': markup += value; break;
                            case 'string':  markup += Utils.escapeHTML(quote + value + quote); break;
                            case 'object':  markup += Utils.escapeHTML(Utils.stringifyForDebug(value)); break;
                            default: markup += Utils.escapeHTML('<invalid>');
                        }
                    }
                    markup += '</td>';
                }

                function addLineMarkup(label, value, quote) {
                    markup += '<tr><th>' + label + '</th>';
                    addCellMarkup(value, quote, SECTION_TITLES.length - 1);
                    markup += '</tr>';
                }

                function addSectionMarkup(label, section) {
                    markup += '<tr>';
                    addCellMarkup(label);
                    addCellMarkup(section.formatCode, true);
                    addCellMarkup(section.category);
                    addCellMarkup(section.colorName);
                    addCellMarkup(section.operator, true);
                    addCellMarkup(section.operator ? section.boundary : null);
                    markup += '</tr>';
                }

                markup = '<table>';
                addLineMarkup('Format ID', formatId);
                addLineMarkup('Format code', parsedFormat.formatCode, true);
                addLineMarkup('Category', parsedFormat.category);
                addLineMarkup('State', parsedFormat.syntaxError ? 'error' : 'valid');
                markup += SECTION_HEADER_MARKUP;
                parsedFormat.numberSections.forEach(function (section, index) {
                    addSectionMarkup('Num' + (index + 1), section);
                });
                if (parsedFormat.textSection) {
                    addSectionMarkup('Text', parsedFormat.textSection);
                }
                markup += '</table>';
                return markup;
            });
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         *
         * @param {String} tabId
         *  The id of the tool bar tab that is activated. Or the identifier for generating the tool bar
         *  tabs ('toolbartabs') or for the view menu in the top bar ('viewmenu').
         *
         * @param {CompoundButton} viewMenuGroup
         *  The 'View' drop-down menu group.
         */
        function initGuiHandler(tabId, viewMenuGroup) {

            switch (tabId) {

                case 'toolbartabs':
                    // TABS: prepare all tabs (for normal or combined panes)
                    if (!Config.COMBINED_TOOL_PANES) {
                        self.createToolBarTab('format', { label: Labels.FORMAT_HEADER_LABEL, visibleKey: 'view/toolbartab/format/visible', priority: 1 });
                    } else {
                        self.createToolBarTab('font', { label: Labels.FONT_HEADER_LABEL, visibleKey: 'view/toolbartab/format/visible', priority: 1 });
                        self.createToolBarTab('alignment', { label: Labels.ALIGNMENT_HEADER_LABEL, visibleKey: 'view/toolbartab/format/visible', priority: 1 });
                        self.createToolBarTab('cell', { label: Labels.CELL_HEADER_LABEL, visibleKey: 'document/editable/cell', priority: 1 });
                        self.createToolBarTab('numberformat', { label: Labels.NUMBERFORMAT_HEADER_LABEL, visibleKey: 'view/toolbartab/format/visible', priority: 1 });
                    }

                    self.createToolBarTab('data', { label: Labels.DATA_HEADER_LABEL, visibleKey: 'view/toolbartab/data/visible', priority: 1 });
                    if (!Config.COMBINED_TOOL_PANES) { self.createToolBarTab('insert', { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'view/toolbartab/insert/visible', priority: 1 }); }
                    self.createToolBarTab('colrow', { label: Labels.COL_ROW_HEADER_LABEL, visibleKey: 'view/toolbartab/colrow/visible', priority: 1 });
                    if (!Config.COMBINED_TOOL_PANES) { self.createToolBarTab('table', { label: Labels.TABLE_HEADER_LABEL, visibleKey: 'view/toolbartab/table/visible' }); }
                    self.createToolBarTab('drawing', { labelKey: 'drawing/type/label', visibleKey: 'view/toolbartab/drawing/visible' });
                    self.createToolBarTab('comments', { label: Labels.COMMENT_HEADER_LABEL, visibleKey: 'view/toolbartab/comments/visible', priority: 1 });
                    break;

                case 'viewmenu':
                    viewMenuGroup
                        .addSectionLabel(Labels.ZOOM_LABEL)
                        .addGroup('view/zoom/dec', new Button(self, Labels.ZOOMOUT_BUTTON_OPTIONS), { inline: true, sticky: true })
                        .addGroup('view/zoom/inc', new Button(self, Labels.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
                        .addGroup('view/zoom', new Controls.PercentageLabel(self), { inline: true });

                    viewMenuGroup.addSectionLabel(/*#. menu title: settings for splits and frozen columns/rows in a spreadsheet */ gt.pgettext('sheet-split', 'Split and freeze'))
                        .addGroup('view/split/dynamic', new Controls.DynamicSplitCheckBox(self))
                        .addGroup('view/split/frozen', new Controls.FrozenSplitCheckBox(self));

                    viewMenuGroup.addSectionLabel(Labels.OPTIONS_LABEL)
                        // note: we do not set aria role to 'menuitemcheckbox' or 'button' due to Safari just working correctly with 'checkox'. CheckBox constructor defaults aria role to 'checkbox'
                        .addGroup('view/toolbars/show',    new CheckBox(self, Labels.SHOW_TOOLBARS_CHECKBOX_OPTIONS))
                        .addGroup('document/users',        new CheckBox(self, Labels.SHOW_COLLABORATORS_CHECKBOX_OPTIONS))
                        .addGroup('view/formulapane/show', new CheckBox(self, { label: /*#. check box label: show/hide formula-bar */ gt('Show formula bar'), tooltip: gt('Show or hide the formula bar') }))
                        .addGroup('view/grid/show',        new CheckBox(self, { label: /*#. check box label: show/hide cell grid in the sheet */ gt('Show grid lines'), tooltip: gt('Show or hide the cell grid lines in the current sheet') }))
                        .addGroup('view/statuspane/show',  new CheckBox(self, { label: /*#. check box label: show/hide the sheet tabs at the bottom border */ gt('Show sheet tabs'), tooltip: gt('Show or hide the sheet tabs below the sheet area') }));
                    break;

                case 'format':
                    if (!Config.COMBINED_TOOL_PANES) {
                        // tabbed tool pane mode: provide all formatting in a single tool pane
                        self.addToolBar('format', new ToolBars.FontFamilyToolBar(self),    { priority: 1 });
                        self.addToolBar('format', new ToolBars.FontStyleToolBar(self),     { priority: 2 });
                        self.addToolBar('format', new ToolBars.FormatPainterToolBar(self), { priority: 3 });
                        self.addToolBar('format', new ToolBars.CellColorToolBar(self),     { priority: 4 });
                        self.addToolBar('format', new ToolBars.CellAlignmentToolBar(self), { priority: 5 });
                        self.addToolBar('format', new ToolBars.NumberFormatToolBar(self),  { priority: 6 });
                        self.addToolBar('format', new ToolBars.CellBorderToolBar(self),    { priority: 7 });
                        self.addToolBar('format', new ToolBars.CellStyleToolBar(self),     { priority: 8 });
                        self.addToolBar('format', new ToolBars.BoxAlignmentToolBar(self),  { priority: 9,  visibleKey: 'view/selection/drawing' });
                        self.addToolBar('format', new ToolBars.ListStyleToolBar(self),     { priority: 10, visibleKey: 'view/selection/drawing' });
                    }
                    break;

                case 'font':
                    if (Config.COMBINED_TOOL_PANES) {
                        self.addToolBar('font', new ToolBars.FontFamilyToolBar(self), { priority: 1 });
                        self.addToolBar('font', new ToolBars.FontStyleToolBar(self),  { priority: 2 });
                        self.createToolBar('font', { priority: 3 }).addGroup('character/color', new Controls.TextColorPicker(self));
                    }
                    break;

                case 'alignment':
                    if (Config.COMBINED_TOOL_PANES) {
                        // 'Alignment' tab with cell alignment controls, and merge picker
                        self.addToolBar('alignment', new ToolBars.CellAlignmentToolBar(self));
                    }
                    break;

                case 'cell':
                    // 'Cell' tab with cell fill color (but not text color), and cell border controls
                    if (Config.COMBINED_TOOL_PANES) {
                        self.createToolBar('cell').addGroup('cell/fillcolor', new Controls.CellFillColorPicker(self));
                        self.addToolBar('cell', new ToolBars.CellBorderToolBar(self));
                    }
                    break;

                case 'numberformat':
                    if (Config.COMBINED_TOOL_PANES) {
                        // 'Number format' tab with all number formatting controls
                        self.addToolBar('numberformat', new ToolBars.NumberFormatToolBar(self));
                    }
                    break;

                case 'data':
                    // data manipulation (filter/sorting, named ranges)
                    var dataToolBar = self.createToolBar('data');
                    if (Config.COMBINED_TOOL_PANES) {
                        dataToolBar
                            .addGroup('cell/autoformula', new Controls.InsertAutoSumButton(self))
                            .addGroup('function/insert/dialog', new Controls.InsertFunctionButton(self))
                            .addSeparator();
                    }
                    dataToolBar
                        .addGroup('cell/sort', new Controls.SortMenuButton(self))
                        .addSeparator()
                        .addGroup('table/filter', new Controls.FilterButton(self))
                        .addGroup('table/refresh', new Controls.RefreshButton(self));

                    if (!Config.COMBINED_TOOL_PANES) {
                        dataToolBar
                            .addSeparator()
                            .addGroup('view/namesmenu/toggle', new Controls.NamesMenuButton(self));
                    }
                    dataToolBar
                        .addSeparator()
                        .addGroup('sheet/operation/unlocked/cell', new Controls.CellProtectionMenuButton(self));
                    break;

                case 'insert':
                    // 'Insert' tab with controls for cell contents (auto-sum etc.), and drawings
                    if (!Config.COMBINED_TOOL_PANES) {
                        self.addToolBar('insert', new ToolBars.InsertContentToolBar(self));
                        self.addToolBar('insert', new ToolBars.InsertDrawingToolBar(self));
                    }
                    break;

                case 'colrow':
                    // 'Column/row' tab for column and row operations
                    self.addToolBar('colrow', new ToolBars.ColRowToolBar(self, false));
                    self.addToolBar('colrow', new ToolBars.ColRowToolBar(self, true));
                    break;

                case 'table':
                    // 'Table' tab for table ranges (filter/sorting)
                    if (!Config.COMBINED_TOOL_PANES) {
                        self.addToolBar('table', new ToolBars.TableToolBar(self));
                    }
                    break;

                case 'drawing':
                    // 'Drawing' tab for drawing objects
                    self.createToolBar('drawing')
                        .addGroup('shape/insert', new Controls.ShapeTypePicker(self, { label: null }), { visibleKey: 'drawing/type/shape' })
                        .addGroup('drawing/delete', new Button(self, Labels.DELETE_DRAWING_BUTTON_OPTIONS));

                    if (!Config.COMBINED_TOOL_PANES) {

                        self.createToolBar('drawing', { visibleKey: 'view/toolbar/drawing/connector/visible' })
                            .addGroup('drawing/border/style',     new Controls.BorderPresetStylePicker(self, { label: Labels.LINE_STYLE_LABEL }))
                            .addGroup('drawing/border/color',     new Controls.BorderColorPicker(self, { label: gt('Line color'), smallerVersion: { hideLabel: true } }))
                            .addGroup('drawing/connector/arrows', new Controls.ArrowPresetStylePicker(self));

                        self.createToolBar('drawing', { visibleKey: 'view/toolbar/drawing/border/visible' })
                            .addGroup('drawing/border/style', new Controls.BorderPresetStylePicker(self, { label: Labels.BORDER_STYLE_LABEL }))
                            .addGroup('drawing/border/color', new Controls.BorderColorPicker(self, { label: gt('Border color'), smallerVersion: { hideLabel: true } }));

                        self.createToolBar('drawing', { visibleKey: 'view/toolbar/drawing/fill/visible', hideable: true })
                            .addGroup('drawing/fill/color', new Controls.FillColorPicker(self, { label: gt('Background color'),  icon: 'docs-drawing-fill-color', smallerVersion: { hideLabel: true } }));

                        self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 1 })
                            .addGroup('drawing/charttype', new Controls.ChartTypePicker(self))
                            .addGap()
                            .addGroup('drawing/chartlabels', new Button(self, _.extend({ 'aria-owns': chartLabelsMenu.getUid() }, Labels.CHART_LABELS_BUTTON_OPTIONS)))
                            .addSeparator()
                            .addGroup('drawing/chartcolorset', new Controls.ChartColorSetPicker(self))
                            .addGroup('drawing/chartstyleset', new Controls.ChartStyleSetPicker(self));

                        self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 2, hideable: true })
                            .addGroup('drawing/chart/valid', new CompoundButton(self, Labels.CHART_DATA_POINTS_BUTTON_OPTIONS)
                                .addGroup('drawing/chartdatalabel',  new CheckBox(self, Labels.CHART_SHOW_POINT_LABELS_BUTTON_OPTIONS))
                                .addGroup('drawing/chartvarycolor',  new CheckBox(self, Labels.CHART_VARY_POINT_COLORS_BUTTON_OPTIONS))
                            );

                        self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 3, hideable: true })
                            .addGroup('drawing/chartlegend/pos', new Controls.ChartLegendPicker(self));

                        self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 4, hideable: true })
                            .addGroup('drawing/chartdatasource', new CompoundButton(self, { label: /*#. menu title: options to modify the data source of a chart object */ gt.pgettext('chart-source', 'Data source') })
                                .addGroup('drawing/chartsource',   new Button(self, { label: /*#. change source data for a chart object in a spreadsheet */ gt.pgettext('chart-source', 'Edit data references') }))
                                .addGroup('drawing/chartexchange', new Button(self, { label: /*#. switch orientation of data series in rectangular source data of a chart object */ gt.pgettext('chart-source', 'Switch rows and columns') }))
                                .addGroup('drawing/chartfirstrow', new CheckBox(self, { attributes: { class: 'highlight' }, label: /*#. decide if first row of source data is handled as label (headline) or as contentvalues */ gt.pgettext('chart-source', 'First row as label') }))
                                .addGroup('drawing/chartfirstcol', new CheckBox(self, { attributes: { class: 'highlight' }, label: /*#. decide if first column of source data is handled as label or as contentvalues */ gt.pgettext('chart-source', 'First column as label') }))
                            );
                    }
                    self.createToolBar('drawing')
                        .addGroup('drawing/order', new Controls.DrawingArrangementPicker(self));
                    break;

                case 'comments':
                    self.addToolBar('comments', new ToolBars.CommentsToolBar(self));
                    break;

                default:
            }
        }

        /**
         * Additional initialization of debug GUI after importing the document.
         *
         * @param {String} tabId
         *  The id of the tool bar tab that is activated. Or the identifier for generating the tool bar
         *  tabs ('toolbartabs') or for the view menu in the top bar ('viewmenu').
         *
         * @param {CompoundButton} viewMenuGroup
         *  The 'View' drop-down menu group.
         */
        function initDebugGuiHandler(tabId, viewMenuGroup) {

            switch (tabId) {
                case 'viewmenu':
                    viewMenuGroup
                        .addGroup('debug/highlight/cells', new CheckBox(self, { label: _.noI18n('Highlight cell contents') }));
                    break;
                case 'debug':
                    self.createToolBar('debug')
                        .addGroup('document/formula/recalc/all', new Button(self, { icon: 'fa-refresh', label: _.noI18n('Recalculate all'), tooltip: _.noI18n('Recalculate all formulas in the document') }));
                    break;
                default:
            }
        }

        /**
         * Registers a new text editor for a specific text edit mode.
         *
         * @param {TextEditorBase} textEditor
         *  The text editor to be registered.
         */
        function registerTextEditor(textEditor) {

            // insert the editor into the regisrty
            textEditorRegistry.insert(textEditor.getEditMode(), textEditor);

            // forward all editor events to the listeners of this instance
            self.forwardEvents(textEditor);

            // store the active editor in the variable 'activeEditor'
            self.listenTo(textEditor, 'textedit:enter', function () { activeTextEditor = textEditor; });
            self.listenTo(textEditor, 'textedit:leave', function () { activeTextEditor = null; });
        }

        /**
         * Moves the browser focus into the active sheet pane.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Address} [options.target]
         *      The address of a target cell. In a frozen split view, the grid
         *      pane that contains this address will be focused. See the
         *      description of the method SpreadsheetView.getTargetGridPane()
         *      for more details.
         */
        function grabFocusHandler(options) {

            // prevent JS errors when importing the document fails, and the view does not have activated a sheet yet
            if (!activeSheetModel) { return; }

            // text edit mode has own focus handling
            if (activeTextEditor && activeTextEditor.isActive()) {
                activeTextEditor.grabFocus();
                return;
            }

            // focus the target grid pane if specified
            var target = Utils.getOption(options, 'target', null);
            if (target instanceof Address) {
                self.getTargetGridPane(target).grabFocus();
            } else {
                self.getActiveGridPane().grabFocus();
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Inserts generic sheet actions to the passed compound menu.
         *
         * @param {CompoundMenu|CompoundMenuMixin} compoundMenu
         *  The compound menu to be extended.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.insertSheet=false]
         *      If set to true, a button for the action 'Insert Sheet' will be
         *      created.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.addSheetActionButtons = function (compoundMenu, options) {

            if (Utils.getBooleanOption(options, 'insertSheet', false)) {
                compoundMenu.addGroup('document/insertsheet', new Button(this, { label: Labels.INSERT_SHEET_LABEL }));
                compoundMenu.addSeparator();
            }

            compoundMenu.addGroup('sheet/rename/dialog', new Button(this, { label: Labels.RENAME_SHEET_LABEL }));
            compoundMenu.addGroup('document/copysheet/dialog', new Button(this, { label: Labels.COPY_SHEET_LABEL }), { visibleKey: 'document/ooxml' });
            compoundMenu.addGroup('document/deletesheet', new Button(this, { label: Labels.DELETE_SHEET_LABEL }));
            compoundMenu.addGroup('sheet/visible', new Controls.ShowSheetButton(this));
            compoundMenu.addGroup('sheet/locked', new Controls.LockSheetButton(this));
            compoundMenu.addSeparator();
            compoundMenu.addGroup('document/movesheets/dialog', new Button(this, { label: Labels.REORDER_SHEETS_LABEL }));
            compoundMenu.addGroup('document/showsheets/dialog', new Button(this, { label: Labels.UNHIDE_SHEETS_LABEL }));

            return this;
        };

        /**
         * Returns the specified grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getGridPane = function (panePos) {
            return gridPaneMap.get(panePos);
        };

        /**
         * Returns the specified grid pane, if it is visible. Otherwise,
         * returns the nearest visible grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the preferred grid pane. If only one
         *  grid pane is visible, it will be returned regardless of this
         *  parameter. If one of the pane sides contained in the passed pane
         *  position is not visible, returns the grid pane from the other
         *  visible pane side (for example, if requesting the top-left pane
         *  while only bottom panes are visible, returns the bottom-left pane).
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getVisibleGridPane = function (panePos) {

            // jump to other column pane, if the pane side is hidden
            if (paneSideSettings[PaneUtils.getColPaneSide(panePos)].size === 0) {
                panePos = PaneUtils.getNextColPanePos(panePos);
            }

            // jump to other row pane, if the pane side is hidden
            if (paneSideSettings[PaneUtils.getRowPaneSide(panePos)].size === 0) {
                panePos = PaneUtils.getNextRowPanePos(panePos);
            }

            // now, panePos points to a visible grid pane (at least one pane is always visible)
            return gridPaneMap.get(panePos);
        };

        /**
         * Returns the grid pane that is currently focused.
         *
         * @returns {GridPane}
         *  The focused grid pane instance.
         */
        this.getActiveGridPane = function () {
            return this.getVisibleGridPane(activeSheetModel.getViewAttribute('activePane'));
        };

        /**
         * Returns the grid pane that contains the specified target cell. In a
         * frozen split view, only one grid pane can contain the specified
         * cell. In an unfrozen split view, any visible grid pane can contain
         * the cell. In the latter case, the active grid pane will be returned.
         *
         * @param {Address} address
         *  The address of the target cell.
         *
         * @returns {GridPane}
         *  The target grid pane instance.
         */
        this.getTargetGridPane = function (address) {

            // return active grid pane, if the view is not frozen
            if (!activeSheetModel.hasFrozenSplit()) { return this.getActiveGridPane(); }

            // the frozen column and row intervals
            var colInterval = activeSheetModel.getSplitColInterval();
            var rowInterval = activeSheetModel.getSplitRowInterval();
            // preferred pane side identifiers
            var colSide = (_.isObject(colInterval) && (address[0] <= colInterval.last)) ? 'left' : 'right';
            var rowSide = (_.isObject(rowInterval) && (address[1] <= rowInterval.last)) ? 'top' : 'bottom';

            // return the grid pane containing the cell address
            return this.getVisibleGridPane(PaneUtils.getPanePos(colSide, rowSide));
        };

        /**
         * Returns the grid pane covered by the page coordinates in the passed
         * browser event.
         *
         * @param {Event|jQuery.Event} event
         *  A browser event, expected to contain the numeric properties 'pageX'
         *  and 'pageY'.
         *
         * @returns {GridPane|Null}
         *  The grid pane that contains the page coordinates in the passed
         *  browser event; or null, if the page coordinates are outside of all
         *  grid panes.
         */
        this.getGridPaneForEvent = function (event) {
            return gridPaneMap.find(function (gridPane) {
                return gridPane.containsEventOffset(event);
            }) || null;
        };

        /**
         * Activates the specified grid pane, unless the view is currently in
         * an internal initialization phase, or the cell edit mode is currently
         * active.
         *
         * @param {String} panePos
         *  The identifier of the grid pane to be activated.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.updateActiveGridPane = function (panePos) {
            // do not change active pane while initializing panes, or while in cell
            // edit mode (range selection mode in formulas)
            if (panesInitialized && !this.isTextEditMode('cell')) {
                activeSheetModel.setViewAttribute('activePane', panePos);
            }
            return this;
        };

        /**
         * Invokes the passed iterator function for all grid panes contained in
         * this view.
         *
         * @param {Function} callback
         *  The callback function invoked for all grid pane instances. Receives
         *  the following parameters:
         *  (1) {GridPane} gridPane
         *      The current grid pane instance.
         *  (2) {String} panePos
         *      The position identifier of the grid pane.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.iterateGridPanes = function (callback, context) {
            gridPaneMap.forEach(callback, context);
            return this;
        };

        /**
         * Returns the specified header pane.
         *
         * @param {String} paneSide
         *  The identifier of the pane side ('left', 'right', 'top', 'bottom').
         *
         * @returns {HeaderPane}
         *  The header pane instance at the specified pane side.
         */
        this.getHeaderPane = function (paneSide) {
            return headerPaneMap.get(paneSide);
        };

        /**
         * Returns the horizontal header pane (left or right) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The horizontal header pane instance at the corresponding pane side.
         */
        this.getColHeaderPane = function (panePos) {
            return headerPaneMap.get(PaneUtils.getColPaneSide(panePos));
        };

        /**
         * Returns the vertical header pane (top or bottom) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The vertical header pane instance at the corresponding pane side.
         */
        this.getRowHeaderPane = function (panePos) {
            return headerPaneMap.get(PaneUtils.getRowPaneSide(panePos));
        };

        /**
         * Returns the first visible header pane in the specified direction
         * (for example, for columns returns the left header pane, if it is
         * visible, otherwise the right header pane).
         *
         * @param {Boolean} columns
         *  If set to true, returns the first visible column header pane (left
         *  or right); otherwise returns the first visible row header pane (top
         *  or bottom).
         *
         * @returns {HeaderPane}
         *  The first visible header pane instance in the specified direction.
         */
        this.getVisibleHeaderPane = function (columns) {
            var paneSide = columns ?
                ((paneSideSettings.left.size > 0) ? 'left' : 'right') :
                ((paneSideSettings.top.size > 0) ? 'top' : 'bottom');
            return headerPaneMap.get(paneSide);
        };

        /**
         * Returns a header pane associated to the grid pane that is currently
         * focused.
         *
         * @param {Boolean} columns
         *  If set to true, returns the column header pane (left or right);
         *  otherwise returns the row header pane (top or bottom).
         *
         * @returns {HeaderPane}
         *  A header pane instance associated to the grid pane that is
         *  currently focused.
         */
        this.getActiveHeaderPane = function (columns) {
            return this.getActiveGridPane().getHeaderPane(columns);
        };

        /**
         * Invokes the passed iterator function for all header panes contained
         * in this view.
         *
         * @param {Function} callback
         *  The callback function invoked for all header pane instances.
         *  Receives the following parameters:
         *  (1) {HeaderPane} headerPane
         *      The current header pane instance.
         *  (2) {String} paneSide
         *      The identifier of the pane side ('left', 'top', etc.).
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.iterateHeaderPanes = function (callback, context) {
            headerPaneMap.forEach(callback, context);
            return this;
        };

        /**
         * Returns the top-left corner pane.
         *
         * @returns {CornerPane}
         *  The corner pane instance.
         */
        this.getCornerPane = function () {
            return cornerPane;
        };

        /**
         * Returns the current width of the row header panes.
         *
         * @returns {Number}
         *  The current width of the row header panes, in pixels.
         */
        this.getHeaderWidth = function () {
            return cornerPane.getWidth();
        };

        /**
         * Returns the current height of the column header panes.
         *
         * @returns {Number}
         *  The current height of the column header panes, in pixels.
         */
        this.getHeaderHeight = function () {
            return cornerPane.getHeight();
        };

        /**
         * Returns the formula pane containing the formula input field.
         *
         * @returns {FormulaPane}
         *  The formula pane instance.
         */
        this.getFormulaPane = function () {
            return formulaPane;
        };

        /**
         * Returns the status pane containing the sheet tabs, and the subtotal
         * label.
         *
         * @returns {StatusPane}
         *  The status pane instance.
         */
        this.getStatusPane = function () {
            return statusPane;
        };

        /**
         * Returns the floating menu containing all defined names.
         *
         * @returns {NamesMenu}
         *  The floating menu containing all defined names.
         */
        this.getNamesMenu = function () {
            return namesMenu;
        };

        /**
         * Returns the floating menu for axis and label formatting of chart
         * objects.
         *
         * @returns {ChartLabelsMenu}
         *  The floating menu for axis and label formatting of chart objects.
         */
        this.getChartLabelsMenu = function () {
            return chartLabelsMenu;
        };

        /**
         * Returns the rendering cache containing processed rendering data for
         * the visible columns, rows, and cells.
         *
         * @returns {RenderCache}
         *  The rendering cache of this document.
         */
        this.getRenderCache = function () {
            return renderCache;
        };

        /**
         * Returns the intersection of the passed column/row intervals with the
         * visible intervals of the header panes in the specified direction.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Boolean} columns
         *  Whether to return the column intervals of the left and right header
         *  panes (true), or the row intervals of the top and bottom header
         *  panes (false).
         *
         * @returns {IntervalArray}
         *  The parts of the passed column/row intervals which are covered by
         *  the respective visible header panes.
         */
        this.getVisibleIntervals = function (intervals, columns) {

            // the resulting intervals covered by the header panes
            var resultIntervals = new IntervalArray();

            // convert parameter to an array
            intervals = IntervalArray.get(intervals);

            // pick the intervals of all visible header panes in the correct orientation
            headerPaneMap.forEach(function (headerPane, paneSide) {
                if ((PaneUtils.isColumnSide(paneSide) === columns) && headerPane.isVisible()) {
                    resultIntervals.append(intervals.intersect(headerPane.getRenderInterval()));
                }
            });

            // merge intervals in case the header panes cover the same sheet area
            return resultIntervals.merge();
        };

        /**
         * Shows a notification alert box, if a corresponding message text
         * exists for the passed message code.
         *
         * @param {String} msgCode
         *  The message code to show a notification alert box for.
         *
         * @param {Boolean} [success=false]
         *  If set to true, a success alert box will be shown. By default, an
         *  information alert will be shown.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.yellMessage = function (msgCode, success) {

            // missing edit mode shown via special application method 'rejectEditAttempt()'
            if (msgCode === 'readonly') {
                app.rejectEditAttempt();
            } else if (msgCode === 'image:insert') {
                app.rejectEditAttempt('image');
            } else if (msgCode in RESULT_MESSAGES) {
                this.yellNotification({ type: success ? 'success' : 'info', message: RESULT_MESSAGES[msgCode] });
            } else if (success) {
                this.hideYell(); // bug 47936: hide previously shown warning alert
            }

            return this;
        };

        /**
         * Shows a notification message box, dependant on the Viewer mode state.
         *
         * @param {Object} yellOptions
         *  See baseview.yell() options.
         */
        this.yellNotification = function (yellOptions) {
            if (!app.isViewerMode()) {
                this.yell(yellOptions);
            }
        };

        /**
         * Shows a notification message box, if the passed promise rejects with
         * a message code that has a corresponding message text.
         *
         * @param {jQuery.Promise} promise
         *  A promise representing an asynchronous document model operation. If
         *  the promise will be rejected with an object containing the property
         *  'cause' with a supported message code, an alert box will be shown
         *  to the user.
         *
         * @returns {jQuery.Promise}
         *  The promise passed to this method, for convenience.
         */
        this.yellOnFailure = function (promise) {

            // handle success (e.g., hide previously shown warning alert)
            this.waitForSuccess(promise, function () {
                this.yellMessage('', true);
            });

            // show information alert on error
            this.waitForFailure(promise, function (result) {
                this.yellMessage(Utils.getStringOption(result, 'cause', ''));
            });

            // return the promise for convenience
            return promise;
        };

        /**
         * Shows a notification alert to the user, if the document is in
         * read-only mode.
         *
         * @returns {Boolean}
         *  Whether modifying the document is posssible.
         */
        this.requireEditMode = function () {
            var editMode = this.isEditable();
            if (!editMode) { this.yellMessage('readonly'); }
            return editMode;
        };

        /**
         * Returns the current status text label.
         *
         * @returns {String}
         *  The current status text.
         */
        this.getStatusLabel = function () {
            return statusText;
        };

        /**
         * Sets the passed text as current status text label, shown in the
         * status pane at the bottom border of the application pane.
         *
         * @param {String} text
         *  The new status text.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setStatusLabel = function (text) {
            if (statusText !== text) {
                statusText = text;
                app.getController().update();
            }
            return this;
        };

        /**
         * Returns the zoom factor to be used for the active sheet.
         *
         * @attention
         *  Existence of this method is expected by code in base modules, e.g.
         *  the drawing layer.
         *
         * @returns {Number}
         *  The zoom factor for the active sheet.
         */
        this.getZoomFactor = function () {
            return activeSheetModel ? activeSheetModel.getEffectiveZoom() : 1;
        };

        /**
         * Returns the effective font size for the current zoom factor to be
         * used for the column and row labels in all header panes.
         *
         * @returns {Number}
         *  The effective font size for all header panes, in points.
         */
        this.getHeaderFontSize = function () {
            return Utils.minMax(Utils.round(5 * this.getZoomFactor() + 6, 0.1), 8, Utils.COMPACT_DEVICE ? 12 : 30);
        };

        // active sheet -------------------------------------------------------

        /**
         * Returns the zero-based index of the active sheet currently displayed
         * in this spreadsheet view.
         *
         * @returns {Number}
         *  The zero-based index of the active sheet.
         */
        this.getActiveSheet = function () {
            return activeSheet;
        };

        /**
         * Returns the model instance of the active sheet.
         *
         * @returns {SheetModel}
         *  The model instance of the active sheet.
         */
        this.getSheetModel = function () {
            return activeSheetModel;
        };

        /**
         * Returns the cell collection instance of the active sheet.
         *
         * @returns {CellCollection}
         *  The cell collection instance of the active sheet.
         */
        this.getCellCollection = function () {
            return cellCollection;
        };

        /**
         * Returns the column collection instance of the active sheet.
         *
         * @returns {ColRowCollection}
         *  The column collection instance of the active sheet.
         */
        this.getColCollection = function () {
            return colCollection;
        };

        /**
         * Returns the row collection instance of the active sheet.
         *
         * @returns {ColRowCollection}
         *  The row collection instance of the active sheet.
         */
        this.getRowCollection = function () {
            return rowCollection;
        };

        /**
         * Returns the merge collection instance of the active sheet.
         *
         * @returns {MergeCollection}
         *  The merge collection instance of the active sheet.
         */
        this.getMergeCollection = function () {
            return mergeCollection;
        };

        /**
         * Returns the table collection instance of the active sheet.
         *
         * @returns {TableCollection}
         *  The table collection instance of the active sheet.
         */
        this.getTableCollection = function () {
            return tableCollection;
        };

        /**
         * Returns the validation collection instance of the active sheet.
         *
         * @returns {ValidationCollection}
         *  The validation collection instance of the active sheet.
         */
        this.getValidationCollection = function () {
            return validationCollection;
        };

        /**
         * Returns the conditional formatting collection of the active sheet.
         *
         * @returns {CondFormatCollection}
         *  The conditional formatting collection of the active sheet.
         */
        this.getCondFormatCollection = function () {
            return condFormatCollection;
        };

        /**
         * Returns the drawing collection instance of the active sheet.
         *
         * @returns {SheetDrawingCollection}
         *  The drawing collection instance of the active sheet.
         */
        this.getDrawingCollection = function () {
            return drawingCollection;
        };

        /**
         * Returns the cell comment collection instance of the active sheet.
         *
         * @returns {CommentCollection}
         *  The cell comment collection instance of the active sheet.
         */
        this.getCommentCollection = function () {
            return commentCollection;
        };

        /**
         * Converts the passed length in pixels to a length in 1/100 of
         * millimeters, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in pixels.
         *
         * @returns {Number}
         *  The converted length in 1/100 of millimeters.
         */
        this.convertPixelToHmm = function (length) {
            return activeSheetModel.convertPixelToHmm(length);
        };

        /**
         * Converts the passed length in 1/100 of millimeters to a length in
         * pixels, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The converted length in pixels.
         */
        this.convertHmmToPixel = function (length) {
            return activeSheetModel.convertHmmToPixel(length);
        };

        /**
         * Returns the total width of the active sheet, in pixels.
         *
         * @returns {Number}
         *  The total width of the active sheet, in pixels.
         */
        this.getSheetWidth = function () {
            return colCollection.getTotalSize();
        };

        /**
         * Returns the total height of the active sheet, in pixels.
         *
         * @returns {Number}
         *  The total height of the active sheet, in pixels.
         */
        this.getSheetHeight = function () {
            return rowCollection.getTotalSize();
        };

        /**
         * Returns the area of the active sheet, in pixels according to the
         * current sheet zoom factor.
         *
         * @returns {Rectangle}
         *  The area of the active sheet, in pixels.
         */
        this.getSheetRectangle = function () {
            return new Rectangle(0, 0, this.getSheetWidth(), this.getSheetHeight());
        };

        /**
         * Returns the location of the passed cell range in the active sheet,
         * in pixels according to the current sheet zoom factor.
         *
         * @param {Range} range
         *  The address of the cell range in the active sheet.
         *
         * @returns {Rectangle}
         *  The location of the cell range in the active sheet, in pixels.
         */
        this.getRangeRectangle = function (range) {
            return activeSheetModel.getRangeRectangle(range, { pixel: true });
        };

        /**
         * Returns the location of the specified cell in the active sheet, in
         * pixels according to the current sheet zoom factor.
         *
         * @param {Address} address
         *  The address of the cell in the active sheet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.expandMerged=false]
         *      If set to true, and the cell is part of a merged range, the
         *      location of the entire merged range will be returned, and the
         *      address of the merged range will be stored in the property
         *      'mergedRange' of the result rectangle returned by this method.
         *
         * @returns {Object}
         *  The location of the cell in the active sheet, in pixels.
         */
        this.getCellRectangle = function (address, options) {
            return activeSheetModel.getCellRectangle(address, _.extend({}, options, { pixel: true }));
        };

        /**
         * Visits the selection settings of the active sheet for all registered
         * remote users.
         *
         * @param {Function} callback
         *  The callback function invoked for each remote user that currently
         *  has something selected in the active sheet. Receives the following
         *  parameters:
         *  (1) {String} userName
         *      The display name of the remote user.
         *  (2) {Number} colorIndex
         *      The index of a color in the color scheme associated with the
         *      remote user.
         *  (3) {RangeArray|Null}
         *      A non-empty array of cell range addresses selected by the
         *      remote user; or null if no selected ranges are available.
         *  (4) {Array<Array<Number>>} drawings
         *      A non-empty array with the positions of all drawing objects
         *      selected by the remote user; or null, if no selected drawings
         *      are available.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.iterateRemoteSelections = function (callback, context) {

            this.getViewAttribute('remoteClients').forEach(function (client) {

                // skip users that have another sheet activated
                if (client.userData.sheet !== activeSheet) { return; }

                // the array of cell range addresses (may be null)
                var ranges = Utils.getStringOption(client.userData, 'ranges', null);
                if (ranges) { ranges = docModel.getFormulaParser().parseRangeList('op', ranges); }

                // the positions of all selected drawing objects
                var drawings = Utils.getArrayOption(client.userData, 'drawings', null);
                if (drawings && (drawings.length === 0)) { drawings = null; }

                // invoke the callback function if either ranges or drawings are available
                if (ranges || drawings) {
                    callback.call(context, client.userName, client.colorIndex, ranges, drawings);
                }
            });

            return this;
        };

        // scrolling ----------------------------------------------------------

        /**
         * Scrolls the active grid pane to the specified cell. In frozen split
         * view, the best fitting grid pane for the passed cell address will be
         * picked instead (see method SpreadsheetView.getTargetGridPane() for
         * details).
         *
         * @param {Address} address
         *  The address of the cell to be scrolled to.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.updateFocus=true]
         *      If set to true or omitted, and if the view is in frozen split
         *      mode, the active grid pane containing the passed cell will be
         *      focused. Otherwise, the grid pane that is currently active will
         *      be scrolled to the passed cell address.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address, options) {

            // the target grid pane to be scrolled (grid pane may change in frozen split mode)
            var gridPane = this.getTargetGridPane(address);

            // set focus to the grid pane containing the address (this will update the 'activePane' view attribute)
            if (Utils.getBooleanOption(options, 'updateFocus', true)) {
                this.grabFocus({ target: address });
            }

            // scroll the resulting grid pane to the passed cell address
            gridPane.scrollToCell(address);
            return this;
        };

        /**
         * Scrolls the active grid pane to the specified drawing frame. In
         * frozen split view, scrolls the bottom-right grid pane instead.
         *
         * @param {Array<Number>} position
         *  The position of the drawing frame to be scrolled to.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollToDrawingFrame = function (position) {

            // the resulting grid pane that will be scrolled
            var gridPane = activeSheetModel.hasFrozenSplit() ? this.getVisibleGridPane('bottomRight') : this.getActiveGridPane();

            gridPane.scrollToDrawingFrame(position);
            return this;
        };

        // drawing frames -----------------------------------------------------

        /**
         * Temporarily shows the DOM drawing frame associated to the specified
         * drawing model. This method counts its invocations internally. The
         * method SpreadsheetView.tempHideDrawingFrame() will hide the drawing
         * frame after the respective number of invocations.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to show temporarily.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.tempShowDrawingFrame = function (drawingModel) {
            gridPaneMap.forEach(function (gridPane) {
                gridPane.getDrawingRenderer().tempShowDrawingFrame(drawingModel);
            });
            return this;
        };

        /**
         * Hides the DOM drawing frame associated to the specified drawing
         * model, after it has been made visible temporarily. See method
         * SpreadsheetView.tempShowDrawingFrame() for more details.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to hide.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.tempHideDrawingFrame = function (drawingModel) {
            gridPaneMap.forEach(function (gridPane) {
                gridPane.getDrawingRenderer().tempHideDrawingFrame(drawingModel);
            });
            return this;
        };

        // text edit mode -----------------------------------------------------

        /**
         * Returns whether any text edit mode, or the specified edit mode, is
         * currently active.
         *
         * @param {String} [editMode]
         *  If specified, this method checks if this edit mode is currently
         *  active. If omitted, this method checks if any edit mode is active.
         *
         * @returns {Boolean}
         *  Whether the specified text edit mode is currently active.
         */
        this.isTextEditMode = function (editMode) {
            return (activeTextEditor !== null) && (!editMode || (activeTextEditor.getEditMode() === editMode));
        };

        /**
         * Returns the text editor implementation that is currently active.
         *
         * @returns {BaseTextEditor|Null}
         *  The text editor implementation that is currently active; or null,
         *  if text edit mode is currently not active.
         */
        this.getActiveTextEditor = function () {
            return activeTextEditor;
        };

        /**
         * Starts the specified text edit mode.
         *
         * @param {String} editMode
         *  The identifier of the text edit mode to be started. the following
         *  text edit modes are supported:
         *  - 'cell': Starts editing the text contents of the active cell in
         *      the current cell selection of the active sheet.
         *  - 'drawing': Starts editing the text contents of the selected
         *      drawing object in the active sheet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.restart=false]
         *      If set to true, and the specified text edit mode is currently
         *      active, it will be canceled and restarted. By default, the
         *      active text edit mode will simply be continued.
         *  All options will be passed to the text editor implementation. The
         *  supported options depend on the actual implementation.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the current text edit mode has
         *  been started successfully; or rejected, if the implementation has
         *  decided to reject the edit attempt.
         */
        this.enterTextEditMode = function (editMode, options) {

            // find the specified text editor implementation
            var textEditor = textEditorRegistry.get(editMode, null);
            if (!textEditor) { return this.createRejectedPromise(); }

            // leave active edit mode, if it is different from the specified edit mode

            // cancel active text edit mode if restart is requested
            if ((textEditor === activeTextEditor) && Utils.getBooleanOption(options, 'restart', false)) {
                activeTextEditor.cancelEditMode();
            }

            // leave different text edit mode
            var otherActive = activeTextEditor && (textEditor !== activeTextEditor);
            var promise = otherActive ? activeTextEditor.leaveEditMode() : this.createResolvedPromise();

            // start the new text edit mode
            return promise.then(function () {
                return textEditor.enterEditMode(options);
            });
        };

        /**
         * Leaves the current text edit mode, and commits the pending changes
         * if necessary.
         *
         * @param {Object} [options]
         *  Optional parameters. All options will be passed to the text editor
         *  implementation. The supported options depend on the actual
         *  implementation.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the current text edit mode has
         *  been left successfully; or rejected, if the implementation has
         *  decided to keep the current edit mode active.
         */
        this.leaveTextEditMode = function (options) {
            return activeTextEditor ? activeTextEditor.leaveEditMode(options) : this.createResolvedPromise();
        };

        /**
         * Cancels the current text edit mode (cell or drawing) immediately,
         * without committing any pending changes.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.cancelTextEditMode = function () {
            if (activeTextEditor) { activeTextEditor.cancelEditMode(); }
            return this;
        };

        /**
         * Returns the DOM node intended to contain the browser focus while
         * text edit mode is active.
         *
         * @returns {HTMLElement|Null}
         *  The DOM node intended to contain the browser focus while text edit
         *  mode is active; or null, if no text edit mode is active.
         */
        this.getTextEditFocusNode = function () {
            return activeTextEditor ? activeTextEditor.getFocusNode() : null;
        };

        // drawing text formatting --------------------------------------------

        /**
         * Returns the merged formatting attributes provided by the drawing
         * text selection of the text framework.
         *
         * @returns {Object}
         *  The merged formatting attributes of the text selection engine of
         *  the document.
         */
        this.getTextAttributeSet = function () {
            var attrPool = docModel.getAttributePool();
            var attrSet = {};
            attrPool.extendAttributeSet(attrSet, docModel.getAttributes('paragraph'));
            attrPool.extendAttributeSet(attrSet, docModel.getAttributes('character'));
            return attrSet;
        };

        /**
         * Changes the formatting attributes of the drawing text selection of
         * the text framework.
         *
         * @param {Object} attributeSet
         *  An incomplete attribute set that will be applied to the current
         *  text selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the passed attributes have
         *  been applied successfully.
         */
        this.setTextAttributeSet = function (attributeSet) {

            // drawing objects in locked sheets cannot be modified
            var promise = this.ensureUnlockedDrawings();

            // let the text framework implementation do all the work
            promise = promise.then(function () {
                return docModel.setAttributes('character', attributeSet);
            });

            // show warning alert if needed
            return this.yellOnFailure(promise);
        };

        /**
         * Helper function, no implementation for spreadsheet required (52382).
         */
        this.recalculateDocumentMargin = Utils.NOOP;

        // initialization -----------------------------------------------------

        // marker for touch devices and browser types
        Forms.addDeviceMarkers(rootNode);

        // create floating menus
        namesMenu = new NamesMenu(this);
        chartLabelsMenu = new ChartLabelsMenu(this, {
            anchorBox: function () { return self.getActiveGridPane().getScrollNode(); }
        });

        // create the rendering cache
        renderCache = new RenderCache(this);

        // create the header pane instances
        PaneUtils.ALL_PANE_SIDES.forEach(function (paneSide) {
            paneSideSettings[paneSide] = _.clone(DEFAULT_PANE_SIDE_SETTINGS);
            headerPaneMap.insert(paneSide, new HeaderPane(this, paneSide));
        }, this);

        // create the grid pane instances
        PaneUtils.ALL_PANE_POSITIONS.forEach(function (panePos) {
            gridPaneMap.insert(panePos, new GridPane(this, panePos));
        }, this);

        // create the corner pane instance
        cornerPane = new CornerPane(this);

        // create the implementation of the text edit modes
        registerTextEditor(new CellTextEditor(this));
        registerTextEditor(new DrawingTextEditor(this));

        // mix-in classes expecting fully initialized document view
        SelectionMixin.call(this, rootNode);
        HighlightMixin.call(this);
        ViewFuncMixin.call(this);

        // destroy all class members on destruction
        this.registerDestructor(function () {

            renderCache.destroy();
            gridPaneMap.destroyElements();
            headerPaneMap.destroyElements();
            cornerPane.destroy();
            namesMenu.destroy();
            chartLabelsMenu.destroy();
            textEditorRegistry.forEach(function (textEditor) { textEditor.destroy(); });

            self = app = docModel = rootNode = textRootNode = null;
            gridPaneMap = headerPaneMap = cornerPane = formulaPane = statusPane = null;
            namesMenu = chartLabelsMenu = renderCache = null;
            textEditorRegistry = activeTextEditor = null;
            colSplitLineNode = rowSplitLineNode = resizeOverlayNode = null;
            colSplitTrackingNode = rowSplitTrackingNode = centerSplitTrackingNode = allSplitTrackingNodes = null;
            activeSheetModel = colCollection = rowCollection = mergeCollection = cellCollection = null;
            tableCollection = validationCollection = condFormatCollection = drawingCollection = commentCollection = null;
        });

    } }); // class SpreadsheetView

    // exports ================================================================

    return SpreadsheetView;

});
