/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/trackingpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/baseframework/app/appobjectmixin'
], function (Utils, Tracking, TriggerObject, AppObjectMixin) {

    'use strict';

    // default settings for mouse/touch tracking in tracking panes
    var DEFAULT_TRACKING_OPTIONS = {
        trackModifiers: true,
        autoScroll: true,
        borderMargin: -30,
        borderSize: 60,
        minSpeed: 5,
        maxSpeed: 500,
        acceleration: 1.5
    };

    // private global functions ===============================================

    /**
     * Returns whether the passed DOM node list contains the specified DOM node
     * (either directly, or as a descendant of any of its nodes).
     *
     * @param {NodeList} elementList
     *  A DOM node list containing DOM elements.
     *
     * @param {Node} node
     *  The DOM node to be searched in the passed element list.
     *
     * @returns {Boolean}
     *  Whether the passed DOM node list contains the specified DOM node.
     */
    function nodeListContainsNode(elementList, node) {
        return _.some(elementList, function (element) {
            return (element === node) || Utils.containsNode(element, node);
        });
    }

    // class TrackingPane =====================================================

    /**
     * Base class for all split panes shown in the application edit area (grid
     * panes, and header panes). Provides generic methods to be used from all
     * sub classes, especially the support of tracking events.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends AppObjectMixin
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this pane instance.
     *
     * @param {jQuery} trackingNode
     *  The DOM node used as tracking event source.
     */
    var TrackingPane = TriggerObject.extend({ constructor: function (docView, trackingNode) {

        // self reference
        var self = this;

        // the application instance
        var app = docView.getApp();

        // all registered tracking handlers
        var trackingHandlers = [];

        // descriptor for current tracking cycle (element from 'trackingHandlers' array)
        var currTrackingDesc = null;

        // promise that waits to leave the current text edit mode (operation generator may run asynchronously)
        var textEditPromise = null;

        // promise that waits to leave the current tracking cycle
        var trackingEndPromise = null;

        // the mutation observer used to detect DOM node removals during the tracking cycle
        var mutationObserver = null;

        // type of the last finished tracking cycle
        var lastTrackingType = null;

        // the removed node, detected from the mutationObserver
        var removeTargetNode = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this, docView);
        AppObjectMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Invokes the current tracking handler with the passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event received from the tracking node passed to the
         *  constructor of this instance.
         *
         * @param {jQuery.Promise} [promise]
         *  The promise to be passed to tracking end events.
         *
         * @returns {Any}
         *  The return value of the tracking handler.
         */
        function invokeTrackingHandler(event, promise) {
            if (currTrackingDesc)  {
                return currTrackingDesc.handler.call(event.currentTarget, event, promise, self);
            }
        }

        /**
         * Disconnects the DOM mutation observer, and releases the object
         * reference.
         */
        function disconnectTargetNodeObserver() {
            if (mutationObserver) {
                mutationObserver.disconnect();
                mutationObserver = null;
            }
        }

        /**
         * Creates a new DOM mutation observer for the tracking node, that
         * checks if the passed tracking target node has been removed from the
         * DOM.
         *
         * @param {jQuery} targetNode
         *  The target node of a tracking cycle.
         */
        function connectTargetNodeObserver(targetNode) {

            // create a new mutation observer that checks for removal of the target node
            mutationObserver = new window.MutationObserver(function (mutationRecords) {
                if (trackingNode && removeTargetNode === null) {
                    if (mutationRecords.some(function (mutationRecord) {
                        return nodeListContainsNode(mutationRecord.removedNodes, targetNode[0]);
                    })) {
                        trackingNode.append(targetNode.addClass('touch-tracker-hidden'));
                        removeTargetNode = targetNode;
                    }
                }
            });

            // start observing the tracking root node
            mutationObserver.observe(trackingNode[0], { childList: true, subtree: true });
        }

        /**
         * Handles a 'tracking:start' event and starts a new tracking cycle.
         */
        function trackingStartHandler(startEvent) {

            // do not start any other tracking cycle while still waiting to leave the preceding tracking cycle
            if (trackingEndPromise) {
                Utils.warn('TrackingPane.trackingStartHandler(): ignored "tracking:start" event while leaving tracking cycle');
                startEvent.cancelTracking();
                return;
            }

            // Always activate (focus) the tracked view pane. Done in a timeout as workaround for Firefox,
            // it sends 'focusout' events after the old DOM node that has been clicked is removed (the
            // 'focusout' event causes canceling the tracking sometimes).
            self.executeDelayed(self.grabFocus, 'TrackingPane.trackingStartHandler');

            // ignore inline pop-up menus, and other nodes that handle clicks internally
            var targetNode = $(startEvent.target);
            if (targetNode.closest('.inline-popup,.skip-tracking').length > 0) {
                startEvent.cancelTracking();
                return;
            }

            // find first matching registered tracking handler
            currTrackingDesc = _.find(trackingHandlers, function (desc) {
                switch (typeof desc.selector) {
                    case 'string':
                        return targetNode.closest(desc.selector).length > 0;
                    case 'function':
                        return desc.selector.call(self, startEvent, lastTrackingType);
                }
                return false;
            }) || null;

            // exit for unsupported tracking targets, or in read-only mode
            if (!currTrackingDesc || (!currTrackingDesc.readOnly && !docView.isEditable())) {
                startEvent.cancelTracking();
                return;
            }

            // leave text edit mode and wait for the result
            if (!currTrackingDesc.keepTextEdit && docView.isTextEditMode()) {
                // try to leave text edit mode
                textEditPromise = docView.leaveTextEditMode();
                // cancel tracking cycle if leaving text edit mode fails (e.g. syntax error in cell formula)
                textEditPromise.fail(function () {
                    if (currTrackingDesc) { Tracking.cancelTracking(); }
                });
                // clean-up after leaving text edit mode has succeeded or failed
                textEditPromise.always(function () {
                    textEditPromise = null;
                });
                // nothing more to do, if tracking has been canceled immediately (synchronously)
                if (!currTrackingDesc) { return; }
            }

            // Touch tracking: Use a DOM bserver to keep the target node in the DOM during tracking.
            // On touch devices, the tracked DOM node (the target node of the initial 'touchstart'
            // browser event) must remain in the DOM as long as touch tracking is active. If the
            // node will be removed from the DOM, the browser will not trigger any further 'touchmove'
            // or 'touchend' events. Thus the target node will be reinserted into the tracking root
            // node as long as touch tracking is active. Using a DOM observer ensures that all cases
            // are covered that remove the tracked DOM node, for example when re-rendering parts of
            // the DOM while tracking. This works even for code that removes the tracked node deferred
            // in browser timeouts etc.
            if ((startEvent.trackingType === 'touch') && Utils.containsNode(trackingNode, targetNode)) {
                connectTargetNodeObserver(targetNode);
            }

            // prevent native browser scrolling on touch devices
            if (currTrackingDesc.preventScroll && (startEvent.trackingType === 'touch')) {
                startEvent.preventDefault();
            }

            // finally, invoke the tracking handler function
            return invokeTrackingHandler(startEvent);
        }

        /**
         * Handles a 'tracking:move' event and prevents native scrolling of an
         * ancestor node if specified by the tracking handler.
         */
        function trackingMoveHandler(moveEvent) {

            // nothing to do without active tracking cycle (e.g. canceled imeediately during start)
            if (!currTrackingDesc) { return; }

            // prevent native browser scrolling on touch devices
            if (currTrackingDesc.preventScroll && (moveEvent.trackingType === 'touch')) {
                moveEvent.preventDefault();
            }

            return invokeTrackingHandler(moveEvent);
        }

        /**
         * Handles a 'tracking:end' or 'tracking:cancel' event and finishes the
         * current tracking cycle.
         */
        function trackingEndHandler(endEvent) {

            // nothing to do without active tracking cycle (e.g. canceled imeediately during start)
            if (!currTrackingDesc) { return; }

            // the promise that will be resolved on tracking success, or rejected when canceling the cycle
            if (endEvent.type === 'tracking:cancel') {
                trackingEndPromise = self.createRejectedPromise(false);
            } else if (textEditPromise) {
                trackingEndPromise = textEditPromise.then(_.constant(true), _.constant(false));
            } else {
                trackingEndPromise = self.createResolvedPromise(true);
            }

            // invoke the tracking handler function immediately, pass the end promise for
            // deferred processing after the text edit mode has been left
            var result = invokeTrackingHandler(endEvent, trackingEndPromise);

            // final clean-up for the tracking cycle
            trackingEndPromise.always(function () {
                lastTrackingType = currTrackingDesc.type;
                currTrackingDesc = null;
                trackingEndPromise = null;
            });

            if (removeTargetNode) {
                removeTargetNode.remove();
                removeTargetNode = null;
            }

            disconnectTargetNodeObserver();

            return result;
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the document view that contains this grid pane.
         *
         * @returns {SpreadsheetView}
         *  The document view that contains this grid pane.
         */
        this.getDocView = function () {
            return docView;
        };

        /**
         * Registers a callback handler for mouse or touch tracking in this
         * tracking pane. The callback handler will be called for all tracking
         * events of a matching tracking cycle (see 'selector' below) until the
         * related 'tracking:end' or 'tracking:cancel' event has been received.
         *
         * @param {String} type
         *  A unique type identifier for the tracking handler.
         *
         * @param {String|Function} selector
         *  A CSS selector used to decide whether the target node referred by
         *  the 'tracking:start' event will cause to start tracking mode with
         *  the registered tracking handler. The selector must match the target
         *  node, or any of its ancestors in order to activate tracking mode.
         *  Can also be a predicate function that will be invoked with the
         *  'tracking:start' event as first parameter, and the type identifier
         *  of the last finished tracking cycle as second parameter.
         *
         * @param {Function} handler
         *  The callback function that will handle all tracking events for the
         *  target node matched by the specified selector. Receives the
         *  following parameters:
         *  (1) {jQuery.Event} event
         *      The tracking event as triggered by the tracking framework.
         *  (2) {jQuery.Promise|Null} endPromise
         *      A promise that will be pending if leaving the text edit mode is
         *      still running, and that will be resolved with the value true,
         *      if the tracking cycle has been finished successfully (after a
         *      'tracking:end' event, AND after leaving text edit mode), or
         *      that will be rejected with the value false, if the tracking
         *      cycle has been canceled (after a 'tracking:cancel' event, OR if
         *      leaving text edit mode has failed).
         *  (3) {TrackingPane} trackingPane
         *      A reference to this tracking pane instance, for convenience.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.readOnly=false]
         *      If set to true, tracking will be available if the document is
         *      in read-only mode. By default, the tracking cycle will not be
         *      started in read-only mode, and an active tracking cycle will be
         *      canceled immediately if the document loses edit rights.
         *  - {Boolean} [options.keepTextEdit=false]
         *      If set to true, the current text edit mode will remain active
         *      during the tracking cycle. By default, the edit mode will be
         *      left before the tracking cycle will be started, and if this
         *      fails (e.g. when editing a formula with a syntax error) and the
         *      edit mode remains active, the tracking cycle will not be
         *      started.
         *  - {Boolean} [options.preventScroll=false]
         *      If set to true, the 'tracking:move' events will automatically
         *      prevent the default browser action (usually this will be used
         *      to prevent native scrolling of an ancestor of the tracked node
         *      on touch devices).
         *
         * @returns {TrackingPane}
         *  A reference to this instance.
         */
        this.registerTrackingHandler = function (type, selector, handler, options) {
            trackingHandlers.unshift({
                type: type,
                selector: selector,
                handler: handler,
                readOnly: Utils.getBooleanOption(options, 'readOnly', false),
                keepTextEdit: Utils.getBooleanOption(options, 'keepTextEdit', false),
                preventScroll: Utils.getBooleanOption(options, 'preventScroll', false)
            });
            return this;
        };

        /**
         * Enables tracking events for the tracking node that has been passed
         * to the constructor.
         *
         * @param {Object} [options]
         *  Additional tracking options that will be passed to the method
         *  Tracking.enableTracking().
         *
         * @returns {TrackingPane}
         *  A reference to this instance.
         */
        this.enableTracking = function (options) {
            Tracking.enableTracking(trackingNode, _.extend({}, DEFAULT_TRACKING_OPTIONS, options));
            return this;
        };

        /**
         * Disables tracking events for the tracking node that has been passed
         * to the constructor.
         *
         * @returns {TrackingPane}
         *  A reference to this instance.
         */
        this.disableTracking = function () {
            Tracking.disableTracking(trackingNode);
            return this;
        };

        /**
         * Returns the type identifier of the current tracking cycle.
         *
         * @returns {String|Null}
         *  The type identifier of the current tracking cycle; or null, if no
         *  tracking cycle is currently active.
         */
        this.getTrackingType = function () {
            return currTrackingDesc ? currTrackingDesc.type : null;
        };

        /**
         * Returns the type identifier of the last finished tracking cycle.
         *
         * @returns {String|Null}
         *  The type identifier of the last finished tracking cycle; or null,
         *  if this pane did not process any tracking cycle yet.
         */
        this.getLastTrackingType = function () {
            return lastTrackingType;
        };

        /**
         * Sets the browser focus so that this tracking pane becomes active.
         * MUST be overwritten by subclasses.
         *
         * @returns {TrackingPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            Utils.error('TrackingPane.grabFocus(): missing implementation');
            return this;
        };

        // initialization -----------------------------------------------------

        // handle 'start:tracking' events to start a new tracking cycle
        this.listenTo(trackingNode, 'tracking:start', trackingStartHandler);

        // handle 'tracking:move' events (prevent native scrolling if specified
        this.listenTo(trackingNode, 'tracking:move', trackingMoveHandler);

        // pass synthetic tracking events during a tracking cycle to the handler function
        this.listenTo(trackingNode, 'tracking:repeat tracking:scroll', invokeTrackingHandler);

        // handle 'start:end' and 'tracking:cancel' events to leave the tracking cycle
        this.listenTo(trackingNode, 'tracking:end tracking:cancel', trackingEndHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            disconnectTargetNodeObserver();
            self = docView = app = trackingNode = trackingHandlers = removeTargetNode = null;
            currTrackingDesc = textEditPromise = null;
        });

    } }); // class TrackingPane

    // exports ================================================================

    return TrackingPane;

});
