/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/hyperlinkcontextmenu', [
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/view/labels'
], function (Button, ContextMenu, HyperlinkUtils, Labels) {

    'use strict';

    // class HyperlinkContextMenu =============================================

    /**
     * A context menu for header panes. Provides menu actions to manipulate
     * entire columns/rows in the active sheet.
     *
     * @constructor
     *
     * @extends ContextMenu
     *
     * @param {GridPane} gridPane
     *  The grid pane that contains this instance.
     */
    var HyperlinkContextMenu = ContextMenu.extend({ constructor: function (gridPane) {

        // self reference
        var self = this;

        // the document view
        var docView = gridPane.getDocView();

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, docView, gridPane.getNode(), {
            // do not show context menu during cell edit mode
            enableKey: 'document/readonly/cell',
            delay: 200
        });

        // private methods ----------------------------------------------------

        function initMenu() {
            self.destroyAllGroups();

            // the URL of a hyperlink covering the active cell (check for supported protocol)
            var cellUrl = docView.getCellURL();
            if (!HyperlinkUtils.hasSupportedProtocol(cellUrl)) { cellUrl = null; }

            // the URL returned by a formula cell (HYPERLINK function) if available
            var effectiveUrl = docView.getEffectiveURL();
            if (!HyperlinkUtils.hasSupportedProtocol(effectiveUrl)) { effectiveUrl = null; }

            // show the open hyperlink button for any available hyperlink
            if (effectiveUrl) {
                self.addGroup(null, new Button(docView, { label: Labels.OPEN_HYPERLINK_LABEL, href: effectiveUrl }));
            }

            self.setToolTip(effectiveUrl || '', { anchorBorder: 'top bottom' });
            self.updateAllGroups();
        }

        /**
         * Selects the column/row that has been clicked with the right mouse
         * button, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {
            var effectiveUrl = docView.getEffectiveURL();

            // get out, if user clicked on a drawing frame
            if (!effectiveUrl || !HyperlinkUtils.hasSupportedProtocol(effectiveUrl)) {
                event.preventDefault();
                return;
            }
        }

        // public methods -----------------------------------------------------

        this.getXY = function (event) {
            return {
                pageX: event.sourceEvent.pageX,
                pageY: event.sourceEvent.pageY
            };
        };

        // initialization -----------------------------------------------------

        // lazy initialization of the menu items according to the orientation of the header pane
        this.on('popup:beforeshow', initMenu);

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = gridPane = docView = null;
        });

    } }); // class HyperlinkContextMenu

    // exports ================================================================

    return HyperlinkContextMenu;

});
