/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/drawingcontextmenu', [
    'io.ox/office/spreadsheet/view/controls',
    'io.ox/office/editframework/view/popup/editcontextmenu',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/view/labels'
], function (Controls, EditContextMenu, DrawingFrame, Labels) {

    'use strict';

    // class DrawingContextMenu ================================================

    /**
     * A context menu for drawing objects. Provides menu actions to manipulate
     * the selected drawing objects in the active sheet.
     *
     * @constructor
     *
     * @extends EditContextMenu
     */
    var DrawingContextMenu = EditContextMenu.extend({ constructor: function (gridPane, drawingLayerNode) {

        // self reference
        var self = this;

        // the spreadsheet view
        var docView = gridPane.getDocView();

        // base constructor ---------------------------------------------------

        EditContextMenu.call(this, docView, drawingLayerNode, {
            enableKey: 'view/selection/drawing', // show at least 'copy' for read only drawings
            delay: 200,
            lazyInitHandler: initMenu
        });

        // private methods ----------------------------------------------------

        /**
         * Inserts all menu entries into this context menu.
         */
        function initMenu() {

            // add clipboard commands
            self.addClipboardEntries().addSeparator();

            self.addGroup('drawing/delete', new Controls.Button(docView, { label: Labels.DELETE_DRAWING_LABEL, tooltip: Labels.DELETE_DRAWING_TOOLTIP }));
            self.addSeparator();
            self.addGroup('drawing/order', new Controls.DrawingOrderPicker(docView, { caret: false, icon: 'fa-chevron-right', iconPos: 'trailing', anchorBorder: 'right left' }));
        }

        /**
         * Selects the drawing object that has been clicked with the right
         * mouse button, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {

            // do not show the drawing context menu if no drawing frame can be found
            var drawingFrame = gridPane.getDrawingFrameForEvent(event.sourceEvent);
            if (drawingFrame.length === 0) {
                event.preventDefault();
                return;
            }
        }

        // public methods -----------------------------------------------------

        this.getXY = function (event) {

            if (event.pageX && event.pageY) {
                return {
                    pageX: event.pageX,
                    pageY: event.pageY
                };
            }

            var target = $(event.sourceEvent.target);
            var offset = target.offset();
            return {
                pageX: offset.left + target.width() / 2,
                pageY: offset.top + target.height() / 2
            };
        };

        // initialization -----------------------------------------------------

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = drawingLayerNode = null;
        });

    } }); // class DrawingContextMenu

    // exports ================================================================

    return DrawingContextMenu;

});
