/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/cellcontextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/view/popup/editcontextmenu',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/controls',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, EditContextMenu, HyperlinkUtils, SheetUtils, Labels, Controls, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var CompoundButton = Controls.CompoundButton;
    var MergeMode = SheetUtils.MergeMode;

    // class CellContextMenu ==================================================

    /**
     * A context menu for header panes. Provides menu actions to manipulate
     * entire columns/rows in the active sheet.
     *
     * @constructor
     *
     * @extends EditContextMenu
     *
     * @param {GridPane} gridPane
     *  The grid pane that contains this instance.
     */
    var CellContextMenu = EditContextMenu.extend({ constructor: function (gridPane) {

        // self reference
        var self = this;

        // the document view
        var docView = gridPane.getDocView();

        // the address of the current cell this context menu is shown for
        var address = null;

        // special behavior for OOXML documents
        var ooxml = docView.getApp().isOOXML();

        // base constructor ---------------------------------------------------

        EditContextMenu.call(this, docView, gridPane.getNode(), {
            // do not show context menu during cell edit mode
            enableKey: 'view/editmode/off',
            delay: 200
        });

        // private methods ----------------------------------------------------

        function initMenu() {

            var sheetModel = docView.getSheetModel();
            var selectedRanges = docView.getSelectedRanges();
            var multiCols = selectedRanges.colIntervals().merge().size() > 1;
            var multiRows = selectedRanges.rowIntervals().merge().size() > 1;
            var isSingleCell = docView.isSingleCellSelection();
            var isMergedCell = sheetModel.getMergeCollection().isMergedCell(address);
            var containsMerge = docView.hasMergedRangeSelected();

            self.destroyAllGroups();

            // add clipboard commands
            self.addClipboardEntries().addSeparator();

            // the URL of a hyperlink covering the active cell (check for supported protocol)
            var cellUrl = docView.getCellURL();
            if (!HyperlinkUtils.hasSupportedProtocol(cellUrl)) { cellUrl = null; }

            // the URL returned by a formula cell (HYPERLINK function) if available
            var effectiveUrl = docView.getEffectiveURL();
            if (!HyperlinkUtils.hasSupportedProtocol(effectiveUrl)) { effectiveUrl = null; }

            // add the edit and delete hyperlink buttons for 'real' hyperlinks only (not for URLs from HYPERLINK function)
            if (cellUrl) {
                self.addGroup('hyperlink/edit/dialog', new Button(docView, { label: Labels.EDIT_HYPERLINK_LABEL }));
                self.addGroup('hyperlink/delete', new Button(docView, { label: Labels.REMOVE_HYPERLINK_LABEL }));
            }

            // show the open hyperlink button for any available hyperlink
            if (effectiveUrl) {
                self.addGroup(null, new Button(docView, { label: Labels.OPEN_HYPERLINK_LABEL, href: effectiveUrl }));
                self.addSeparator();
            }

            // sub-menu for all row operations
            var rowMenu = new CompoundButton(docView, { attributes: { 'data-custom': 'row_ctm' }, autoCloseParent: false, autoHideGroups: true, label: multiRows ? gt('Rows') : gt('Row'), anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false })
                .addGroup('row/insert', new Controls.ColRowOpLabelButton(docView, true, false, multiRows))
                .addGroup('row/delete', new Controls.ColRowOpLabelButton(docView, false, false, multiRows));

            // sub-menu for all column operations
            var colMenu = new CompoundButton(docView, { attributes: { 'data-custom': 'column_ctm' }, autoCloseParent: false, autoHideGroups: true, label: multiCols ? gt('Columns') : gt('Column'), anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false })
                .addGroup('column/insert', new Controls.ColRowOpLabelButton(docView, true, true, multiCols))
                .addGroup('column/delete', new Controls.ColRowOpLabelButton(docView, false, true, multiCols));

            // sub-menu for all insert operations (except columns/rows)
            var insertMenu = new CompoundButton(docView, { attributes: { 'data-custom': 'insert_ctm' }, autoCloseParent: false, autoHideGroups: true, label: Labels.INSERT_HEADER_LABEL, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false })
                .addGroup('cell/autoformula',       new Controls.InsertAutoSumButton(docView, { icon: null }))
                .addGroup('function/insert/dialog', new Controls.InsertFunctionButton(docView, { icon: null }))
                .addGroup('hyperlink/edit/dialog',  new Button(docView, Labels.INSERT_HYPERLINK_OPTIONS))
                .addGroup('name/insert/dialog',     new Button(docView, { label: gt('Named range') }))
                .addGroup('image/insert/dialog',    new Button(docView, _.extend({ value: 'drive' }, Labels.INSERT_IMAGE_OPTIONS)))
                .addGroup('chart/insert',           new Button(docView, Utils.extendOptions(Labels.INSERT_CHART_OPTIONS, { value: 'column standard' })));

            // sub-menu for cell comments
            var commentMenu = new Controls.CommentMenuButton(docView, { autoCloseParent: false, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false });

            // sub-menu for cell protection (lock cells / hide formula)
            var protectMenu = new Controls.CellProtectionMenuButton(docView, { attributes: { 'data-custom': 'protect_ctm' }, autoCloseParent: false, anchorBorder: 'right left', icon: 'fa-chevron-right', iconPos: 'trailing', caret: false });

            self.addGroup(null, rowMenu, { visibleKey: 'sheet/operation/unlocked/cell' })
                .addGroup(null, colMenu, { visibleKey: 'sheet/operation/unlocked/cell' })
                .addGroup(null, insertMenu, { visibleKey: 'document/editable' })
                .addSeparator()
                .addGroup(null, commentMenu, { visibleKey: 'comment/model' })
                .addSeparator()
                .addGroup('table/filter',   new Controls.FilterButton(docView, { icon: null }))
                .addGroup('cell/sort/dialog', new Button(docView, { label: gt('Sort') }))
                .addSeparator()
                .addGroup(null, protectMenu, { visibleKey: 'sheet/operation/unlocked/cell' })
                .addSeparator()
                .addGroup('cell/clear/values', new Button(docView, Labels.CLEAN_OPTIONS));

            if (ooxml) {
                var frozenLabel = sheetModel.hasFrozenSplit() ? gt('Unfreeze sheet') : gt('Freeze sheet');
                self.addGroup('view/split/frozen', new Button(docView, { label: frozenLabel, value: 'toggle' }));
            }

            if (containsMerge || isMergedCell) {
                self.addGroup('cell/merge', new Button(docView, { label: gt('Unmerge cells'), value: MergeMode.UNMERGE }));
            } else if (!isSingleCell) {
                self.addGroup('cell/merge', new Button(docView, { label: gt('Merge cells'), value: MergeMode.MERGE }));
            }

            self.setToolTip(effectiveUrl || '', { anchorBorder: 'top bottom' });
            self.updateAllGroups();
        }

        /**
         * Selects the column/row that has been clicked with the right mouse
         * button, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {

            // the source event that causes to open the context menu
            var sourceEvent = event.sourceEvent;

            // do not show the cell context menu if a drawing frame has been clicked
            if (gridPane.getDrawingFrameForEvent(sourceEvent).length > 0) {
                event.preventDefault();
                return;
            }

            // store the address of the clicked cell, needed in method initMenu()
            address = gridPane.getCellDataForEvent(sourceEvent).address;
        }

        // public methods -----------------------------------------------------

        this.getXY = function (event) {
            return {
                pageX: event.sourceEvent.pageX,
                pageY: event.sourceEvent.pageY
            };
        };

        // initialization -----------------------------------------------------

        // lazy initialization of the menu items according to the orientation of the header pane
        this.on('popup:beforeshow', initMenu);

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // hide context menu if scrolling starts
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = gridPane = docView = null;
        });

    } }); // class CellContextMenu

    // exports ================================================================

    return CellContextMenu;

});
