/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/viewfuncmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/mixedborder',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/subtotalresult',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/dialogs',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, Iterator, Color, Border, MixedBorder, HyperlinkUtils, SheetUtils, SubtotalResult, FormulaUtils, Labels, Dialogs, RenderUtils) {

    'use strict';

    // convenience shortcuts
    var TransformIterator = Iterator.TransformIterator;
    var FilterIterator = Iterator.FilterIterator;
    var MoveMode = SheetUtils.MoveMode;
    var MergeMode = SheetUtils.MergeMode;
    var ErrorCode = SheetUtils.ErrorCode;
    var Interval = SheetUtils.Interval;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var IntervalArray = SheetUtils.IntervalArray;
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;
    var Scalar = FormulaUtils.Scalar;
    var Matrix = FormulaUtils.Matrix;
    var Dimension = FormulaUtils.Dimension;

    // tolerance for optimal column width, in 1/100 mm
    var SET_COLUMN_WIDTH_TOLERANCE = 50;

    // tolerance for optimal row height (set explicitly), in 1/100 mm
    var SET_ROW_HEIGHT_TOLERANCE = 15;

    // tolerance for optimal row height (updated implicitly), in 1/100 mm
    var UPDATE_ROW_HEIGHT_TOLERANCE = 50;

    // default value for a visible border
    var DEFAULT_SINGLE_BORDER = { style: 'single', width: Utils.convertLengthToHmm(1, 'px'), color: Color.AUTO };

    // mix-in class ViewFuncMixin =============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides implementations
     * of complex document functionality and status requests, called from the
     * application controller.
     *
     * @constructor
     */
    function ViewFuncMixin() {

        // self reference (spreadsheet view instance)
        var self = this;

        // the spreadsheet model, and other model objects
        var docModel = this.getDocModel();
        var undoManager = docModel.getUndoManager();
        var numberFormatter = docModel.getNumberFormatter();
        var formulaGrammar = docModel.getFormulaGrammar('ui');
        var autoStyles = docModel.getCellAutoStyles();

        // the active sheet model and index
        var activeSheetModel = null;

        // collections of the active sheet
        var colCollection = null;
        var rowCollection = null;
        var mergeCollection = null;
        var cellCollection = null;
        var hyperlinkCollection = null;
        var tableCollection = null;

        // the contents and formatting of the active cell
        var activeAddress = Address.A1;
        var activeCellValue = null;
        var activeTokenDesc = null;
        var activeCellDisplay = '';
        var activeAttributeSet = null;
        var activeParsedFormat = null;

        // the mixed borders of the current selection
        var selectedMixedBorders = {};

        // a deferred object used to wait for the mixed borders of the current selection
        var mixedBordersDef = this.createDeferred('ViewfuncMixin.mixedBordersDef');

        // the subtotal data of the current selection
        var selectedSubtotals = new SubtotalResult();

        // private methods ----------------------------------------------------

        /**
         * Updates the cached data of the active cell (contents, formatting),
         * and the entire selection (border settings, subtotals). To improve
         * performance, updating settings of the entire selection (which is a
         * potentially expensive operation) will be done in a sliced background
         * loop.
         *
         * @param {Boolean} forceUpdate
         *  If set to true, current background timers for mixed border settings
         *  and subtotal results will always be aborted. By default, the
         *  background timers will not be aborted if the selected ranges have
         *  not been changed (e.g. while traversing the active cell inside the
         *  selected ranges).
         */
        var updateSelectionSettings = (function () {

            // initial delay before the background loops will be started
            var INITIAL_DELAY = 300;
            // options for the background timers to collect borders and subtotals in the cell collection
            var BACKGROUND_TIMER_OPTIONS = { slice: 200, interval: 50 };

            // timers for the background loops to collect borders and subtotal results
            var initialDelayTimer = null;
            var mixedBorderTimer = null;
            var subtotalResultTimer = null;
            // the selected cell ranges currently used to collect borders and subtotals
            var selectedRanges = new RangeArray();
            // the new mixed borders, mapped by border keys
            var newMixedBorders = null;

            function getMixedBorder(mixedBorder, border) {
                return mixedBorder ? MixedBorder.mixBorders(mixedBorder, border) : _.clone(border);
            }

            function updateMixedBorder(key, border) {
                var mixedBorder = newMixedBorders[key];
                return (newMixedBorders[key] = getMixedBorder(mixedBorder, border));
            }

            // creates an iterator that provides cell ranges with existing border attributes
            function createBorderAttrIterator(range, visitHor, visitVert, visitDiag) {
                var iterator = cellCollection.createStyleIterator(range, { covered: true });
                return new TransformIterator(iterator, function (styleRange, result) {
                    var borderMap = autoStyles.getBorderMap(result.style);
                    var hasHor  = visitHor  && (borderMap.t || borderMap.b);
                    var hasVert = visitVert && (borderMap.l || borderMap.r);
                    var hasDiag = visitDiag && (borderMap.d || borderMap.u);
                    if (!hasHor && !hasVert && !hasDiag) { return null; }
                    styleRange.borderMap = borderMap;
                    return result;
                });
            }

            function processOuterBorder(range, keyH, keyV) {

                var vertical = !keyH;
                var key = vertical ? keyV : keyH;
                var borderLength = 0;

                var iterator = createBorderAttrIterator(range, !vertical, vertical, false);
                var promise = self.iterateSliced(iterator, function (styleRange) {
                    updateMixedBorder(key, styleRange.borderMap[key] || Border.NONE);
                    borderLength += styleRange.size(!vertical);
                }, 'ViewFuncMixin.updateSelectionSettings.processOuterBorder', BACKGROUND_TIMER_OPTIONS);

                return promise.then(function () {
                    if (borderLength < range.size(!vertical)) {
                        updateMixedBorder(key, Border.NONE);
                    }
                });
            }

            // Returns the parts of the passed band interval with visible borders, and the resulting mixed border.
            // The resulting interval array contains intervals with additional property 'mixedBorder' that will be
            // null if the entire interval does not contain a visible border (i.e. if the result array is empty).
            function getBorderIntervals(bandInterval, vertical, leading) {

                var key = SheetUtils.getOuterBorderKey(vertical, leading);
                var mixedBorder = null;

                var borderIntervals = IntervalArray.map(bandInterval.ranges, function (attrRange) {
                    var border = attrRange.borderMap[key];
                    if (Border.isVisibleBorder(border)) {
                        mixedBorder = getMixedBorder(mixedBorder, border);
                        var interval = attrRange.interval(!vertical);
                        interval.border = border;
                        return interval;
                    }
                });

                // variable 'mixedBorder' still null: no visible borders in the band
                borderIntervals.mixedBorder = mixedBorder;
                return borderIntervals;
            }

            function processBorderRanges(range, borderRanges, key, vertical) {

                var bandIntervals = vertical ? borderRanges.getColBands({ ranges: true }) : borderRanges.getRowBands({ ranges: true });
                var bandLength = range.size(!vertical);
                var first = range.getStart(vertical);
                var last = range.getEnd(vertical);

                // calculates the mixed borders of the passed adjoining intervals
                function processBorderIntervals(intervals1, intervals2) {

                    // Add the precalculated visible mixed borders stored in the interval arrays.
                    var mixedBorder = null;
                    if (intervals1.mixedBorder) { mixedBorder = updateMixedBorder(key, intervals1.mixedBorder); }
                    if (intervals2 && intervals2.mixedBorder) { mixedBorder = updateMixedBorder(key, intervals2.mixedBorder); }

                    // Mix in an additional invisible border, if the visible border intervals do not cover the entire
                    // band length (but do not calculate the size of all border intervals, if the mixed border already
                    // has 'mixed' state, i.e. it already contains visible and invisible border states).
                    if (!mixedBorder || !mixedBorder.mixed) {
                        // merge the passed interval arrays to determine the intervals covered by either of the  arrays
                        // (these intervals contain a visible border, and the remaining intervals do not have a border)
                        var borderLength = intervals2 ? intervals1.concat(intervals2).merge().size() : intervals1.size();
                        if (borderLength < bandLength) { updateMixedBorder(key, Border.NONE); }
                    }
                }

                // no band intervals available: range does not contain any visible border lines
                if (bandIntervals.empty()) {
                    updateMixedBorder(key, Border.NONE);
                    return self.createResolvedPromise();
                }

                // add empty border, if there is a gap with inner borders before the first, after the last, or between band intervals
                if ((first + 1 < bandIntervals.first().first) || (bandIntervals.last().last + 1 < last) || bandIntervals.some(function (bandInterval, index) {
                    return (index > 0) && (bandIntervals[index - 1].last + 2 < bandInterval.first);
                })) {
                    updateMixedBorder(key, Border.NONE);
                }

                return self.iterateArraySliced(bandIntervals, function (bandInterval, index) {

                    // whether to process the borders for the leading boundary of the current band (ignore leading boundary of leading range border)
                    var useLeading = first < bandInterval.first;
                    // whether to process the inner borders in the current band (the band must contain at least two columns/rows)
                    var useInner = bandInterval.size() > 1;
                    // whether to process the borders for the trailing boundary of the current band (ignore trailing boundary of trailing range border)
                    var useTrailing = bandInterval.last < last;

                    // the intervals with visible border attributes at the leading side of the band
                    bandInterval.leading = (useLeading || useInner) ? getBorderIntervals(bandInterval, vertical, true) : null;
                    // the intervals with visible border attributes at the trailing side of the band
                    bandInterval.trailing = (useInner || useTrailing) ? getBorderIntervals(bandInterval, vertical, false) : null;

                    // the preceding band interval, and whether it adjoins to the current band
                    var prevBandInterval = bandIntervals[index - 1];
                    var adjoinsPrevBand = prevBandInterval && (prevBandInterval.last + 1 === bandInterval.first);

                    // the next band interval, and whether it adjoins to the current band
                    var nextBandInterval = bandIntervals[index + 1];
                    var adjoinsNextBand = nextBandInterval && (bandInterval.last + 1 === nextBandInterval.first);

                    // combine the trailing borders of the preceding adjoined band with the leading borders of the current band
                    if (adjoinsPrevBand) { processBorderIntervals(prevBandInterval.trailing, bandInterval.leading); }

                    // process the leading borders of the current band, if the previous band does not adjoin
                    if (useLeading && !adjoinsPrevBand) { processBorderIntervals(bandInterval.leading); }

                    // combine the inner borders in the current band
                    if (useInner) { processBorderIntervals(bandInterval.leading, bandInterval.trailing); }

                    // combine the trailing borders of the current band, if the next band does not adjoin
                    // (adjoining bands will be handled in the next iteration of this loop)
                    if (useTrailing && !adjoinsNextBand) { processBorderIntervals(bandInterval.trailing); }

                }, 'ViewFuncMixin.updateSelectionSettings.processBorderRanges', BACKGROUND_TIMER_OPTIONS);
            }

            // process each range in the passed array (exit the loop early, if mixed border becomes ambiguous)
            function getMixedBorders(ranges, keyH, keyV) {
                return self.iterateArraySliced(ranges, function (range) {

                    // early exit, if the collected mixed borders become completely ambiguous
                    var skipH = !keyH || MixedBorder.isFullyAmbiguousBorder(newMixedBorders[keyH]);
                    var skipV = !keyV || MixedBorder.isFullyAmbiguousBorder(newMixedBorders[keyV]);
                    if (skipH && skipV) { return Utils.BREAK; }

                    // property '_outerBorder' used for ranges located at the sheet boundary (do not collect inner borders)
                    if (range._outerBorder) { return processOuterBorder(range, keyH, keyV); }

                    // otherwise, collect the inner border attributes (skip ranges without inner borders)
                    var processH = !skipH && !range.singleRow();
                    var processV = !skipV && !range.singleCol();
                    if (!processH && !processV) { return; }

                    // use a cell style iterator to collect all cell ranges with visible borders into an array
                    var iterator = createBorderAttrIterator(range, processH, processV, false);
                    var promise2 = self.reduceSliced(new RangeArray(), iterator, function (borderRanges, borderRange) {
                        borderRanges.push(borderRange);
                        return borderRanges;
                    }, 'ViewFuncMixin.updateSelectionSettings.getMixedBorders', BACKGROUND_TIMER_OPTIONS);

                    // process the collected ranges with visible border attributes
                    return promise2.then(function (borderRanges) {
                        return Utils.invokeChainedAsync(
                            processH ? function () { return processBorderRanges(range, borderRanges, keyH, false); } : null,
                            processV ? function () { return processBorderRanges(range, borderRanges, keyV, true); } : null
                        );
                    });
                }, 'ViewFuncMixin.updateSelectionSettings.getMixedBorders', BACKGROUND_TIMER_OPTIONS);
            }

            // collects the mixed border states in the current selection in a background loop
            var collectMixedBorders = SheetUtils.profileAsyncMethod('ViewFuncMixin.collectMixedBorders()', function () {

                var maxCol = docModel.getMaxCol();
                var maxRow = docModel.getMaxRow();

                // Build the iterator ranges for the outer borders of the selected ranges. Use the border column/row inside the range,
                // and the adjacent column/row outside the range, to include border attributes coming from outside cells next to the
                // selected ranges. Range borders located directly at the sheet boundary will not be included here (no cells from
                // outside available). These ranges will be handled below.
                // Example: For the selection range B2:E5, the cells B1:E2 will be used to find the top border settings.
                var borderRangesT = RangeArray.map(selectedRanges, function (range) { return (range.start[1] === 0)    ? null : range.rowRange(-1, 2); });
                var borderRangesB = RangeArray.map(selectedRanges, function (range) { return (range.end[1] === maxRow) ? null : range.rowRange(range.rows() - 1, 2); });
                var borderRangesL = RangeArray.map(selectedRanges, function (range) { return (range.start[0] === 0)    ? null : range.colRange(-1, 2); });
                var borderRangesR = RangeArray.map(selectedRanges, function (range) { return (range.end[0] === maxCol) ? null : range.colRange(range.cols() - 1, 2); });

                // Build the iterator ranges for the outer borders of the selected ranges that are located at the sheet boundary.
                // Example: For the selection range B1:E5, the inner cells B1:E1 will be used to find the top border settings.
                borderRangesT.append(Utils.addProperty(RangeArray.map(selectedRanges, function (range) { return (range.start[1] === 0)    ? range.headerRow()   : null; }), '_outerBorder', true));
                borderRangesB.append(Utils.addProperty(RangeArray.map(selectedRanges, function (range) { return (range.end[1] === maxRow) ? range.footerRow()   : null; }), '_outerBorder', true));
                borderRangesL.append(Utils.addProperty(RangeArray.map(selectedRanges, function (range) { return (range.start[0] === 0)    ? range.leadingCol()  : null; }), '_outerBorder', true));
                borderRangesR.append(Utils.addProperty(RangeArray.map(selectedRanges, function (range) { return (range.end[0] === maxCol) ? range.trailingCol() : null; }), '_outerBorder', true));

                // collect all mixed borders of all outer and inner border positions
                newMixedBorders = {};
                mixedBorderTimer = Utils.invokeChainedAsync(
                    function () { return getMixedBorders(selectedRanges, 'h', 'v'); },
                    function () { return getMixedBorders(borderRangesT, 't', null); },
                    function () { return getMixedBorders(borderRangesB, 'b', null); },
                    function () { return getMixedBorders(borderRangesL, null, 'l'); },
                    function () { return getMixedBorders(borderRangesR, null, 'r'); }
                );

                // resolve the deferred object, provide the new mixed borders as value
                return mixedBorderTimer.done(function () {

                    // add missing entries (if the ranges do not contain any visible border)
                    selectedMixedBorders = {};
                    _.each('tblrhv', function (key) {
                        var borderName = SheetUtils.getBorderName(key);
                        selectedMixedBorders[borderName] = newMixedBorders[key] || _.clone(Border.NONE);
                    });

                    mixedBordersDef.resolve(selectedMixedBorders);
                });
            });

            // callback function for the iterator loop
            function reduceSubtotalResult(subtotals, result) {
                return subtotals.add(result);
            }

            // collects the subtotal results in the current selection in a background loop
            var collectSubtotalResults = SheetUtils.profileAsyncMethod('ViewFuncMixin.collectSubtotalResults()', function () {

                // collect the subtotal results of the selected ranges
                var iterator = cellCollection.createSubtotalIterator(selectedRanges);
                subtotalResultTimer = self.reduceSliced(new SubtotalResult(), iterator, reduceSubtotalResult, 'ViewFuncMixin.updateSelectionSettings.collectSubtotalResults', BACKGROUND_TIMER_OPTIONS);

                // copy the resulting subtotals, and notify all listeners
                return subtotalResultTimer.done(function (subtotals) {
                    selectedSubtotals = subtotals;
                    self.trigger('update:selection:data');
                });
            });

            // the actual implementation of the method updateSelectionSettings()
            return function (forceUpdate) {

                // update settings of the active cell synchronously for fast GUI feedback
                activeAddress = self.getActiveCell();
                activeCellValue = cellCollection.getValue(activeAddress);
                activeTokenDesc = cellCollection.getTokenArray(activeAddress, { fullMatrix: true, grammarId: 'ui' });
                activeCellDisplay = cellCollection.getDisplayString(activeAddress);
                activeAttributeSet = cellCollection.getAttributeSet(activeAddress);
                activeParsedFormat = cellCollection.getParsedFormat(activeAddress);

                // abort running timers, and restart the background loops
                var newSelectedRanges = self.getSelectedRanges();
                if (forceUpdate || !selectedRanges.equals(newSelectedRanges, true)) {

                    // a new pending deferred object that can be used to wait for the border settings
                    if (mixedBordersDef.state() !== 'pending') {
                        mixedBordersDef = self.createDeferred('ViewFuncMixin.mixedBordersDef');
                    }

                    // abort running timers
                    if (initialDelayTimer) { initialDelayTimer.abort(); initialDelayTimer = null; }
                    if (mixedBorderTimer) { mixedBorderTimer.abort(); mixedBorderTimer = null; }
                    if (subtotalResultTimer) { subtotalResultTimer.abort(); subtotalResultTimer = null; }

                    // cache the current selection ranges
                    selectedRanges = newSelectedRanges;

                    // restart the background loops after a delay
                    initialDelayTimer = self.executeDelayed(_.noop, 'ViewFuncMixin.initialDelayTimer', INITIAL_DELAY);
                    initialDelayTimer.then(collectMixedBorders).then(collectSubtotalResults);
                }
            };
        }());

        /**
         * Returns the column/row intervals of the selected ranges, optionally
         * depending on a specific target column/row.
         *
         * @param {Boolean} columns
         *  Whether to return column intervals (true) or row intervals (false).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Number} [options.target]
         *      If specified, the zero-based index of a specific target column
         *      or row. If the current selection contains entire column/row
         *      ranges covering this column/row, the respective intervals of
         *      these ranges will be returned instead; otherwise a single
         *      interval containing the target column/row will be returned.
         *  - {Boolean} [options.visible=false]
         *      If set to true, the returned intervals will be reduced to the
         *      column or row intervals covered by the visible panes.
         *
         * @returns {IntervalArray}
         *  An array of column/row intervals for the current selection, already
         *  merged and sorted.
         */
        function getSelectedIntervals(columns, options) {

            // the selected ranges
            var ranges = self.getSelectedRanges();
            // the target column/row
            var target = Utils.getIntegerOption(options, 'target');
            // the resulting intervals
            var intervals = null;

            // use entire column/row ranges in selection, if target index has been passed
            if (_.isNumber(target)) {
                // filter full column/row ranges in selection
                ranges = ranges.filter(function (range) { return docModel.isFullRange(range, columns); });
                // if the target column/row is not contained in the selection, make an interval for it
                if (!ranges.containsIndex(target, columns)) {
                    intervals = new IntervalArray(new Interval(target));
                }
            }

            // convert selected ranges to column/row intervals
            if (!intervals) {
                intervals = ranges.intervals(columns);
            }

            // reduce the intervals to the header pane intervals
            if (Utils.getBooleanOption(options, 'visible', false)) {
                intervals = self.getVisibleIntervals(intervals, columns);
            }

            // merge and sort the resulting intervals
            return intervals.merge();
        }

        /**
         * Returns the column/row intervals from the current selection, if it
         * contains hidden columns/rows. If the selected columns/rows are
         * completely visible, the selection consists of a single column/row
         * interval, and that interval is located next to a hidden column/row
         * interval (at either side, preferring leading), that hidden interval
         * will be returned instead.
         *
         * @returns {IntervalArray}
         *  The column/row intervals calculated from the current selection,
         *  containing hidden columns/rows. May be an empty array.
         */
        function getSelectedHiddenIntervals(columns) {

            // the entire selected intervals
            var intervals = getSelectedIntervals(columns);

            // safety check, should not happen
            if (intervals.empty()) { return intervals; }

            // the column/row collection
            var collection = columns ? colCollection : rowCollection;
            // the mixed column/row attributes
            var attributes = collection.getMixedAttributes(intervals);

            // if attribute 'visible' is false or null, the intervals contain hidden entries
            if (attributes.visible !== true) { return intervals; }

            // the resulting intervals
            var resultIntervals = new IntervalArray();

            // no special behavior for multiple visible intervals
            if (intervals.length > 1) { return resultIntervals; }

            // try to find preceding hidden interval
            var lastIndex = intervals.first().first - 1;
            if ((lastIndex >= 0) && !collection.isEntryVisible(lastIndex)) {
                var prevEntry = collection.getPrevVisibleEntry(lastIndex);
                resultIntervals.push(new Interval(prevEntry ? (prevEntry.index + 1) : 0, lastIndex));
                return resultIntervals;
            }

            // try to find following hidden interval
            var firstIndex = intervals[0].last + 1;
            if ((firstIndex <= collection.getMaxIndex()) && !collection.isEntryVisible(firstIndex)) {
                var nextEntry = collection.getNextVisibleEntry(firstIndex);
                resultIntervals.push(new Interval(firstIndex, nextEntry ? (nextEntry.index - 1) : collection.getMaxIndex()));
                return resultIntervals;
            }

            // no hidden intervals found
            return resultIntervals;
        }

        /**
         * Generates the operations to set the optimal column width in the
         * specified column intervals, based on the content of cells.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {IntervalArray|Interval} colIntervals
         *  An array of column intervals, or a single column interval.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function generateOptimalColumnWidthOperations(generator, colIntervals) {

            // resulting intervals with the new column widths
            var resultIntervals = new IntervalArray();
            // default width for empty columns
            var defWidth = docModel.getAttributePool().getDefaultValue('column', 'width');
            // maximum width for columns
            var maxWidth = docModel.getMaxColWidthHmm();
            // the rendering cache of the active sheet
            var renderCache = self.getRenderCache();
            // the column iterator
            var colIt = colCollection.createIterator(colIntervals, { visible: true });

            // process all visible columns
            var promise = self.iterateSliced(colIt, function (colDesc) {

                // the new column width
                var colWidth = 0;

                renderCache.iterateCellsInCol(colDesc.index, function (element) {
                    if (!element.hasMergedColumns() && (element.contentWidth > 0)) {
                        colWidth = Math.max(colWidth, element.contentWidth + element.paddingTotal);
                    }
                });

                // convert to 1/100 mm; or use default column width, if no cells are available
                colWidth = (colWidth === 0) ? defWidth : activeSheetModel.convertPixelToHmm(colWidth);

                // bug 40737: restrict to maximum column width allowed in the UI
                // TODO: use real maximum according to file format: 255 digits in OOXML, 1000mm in ODF
                colWidth = Math.min(colWidth, maxWidth);

                // do not generate operations, if the custom width was not modified before, and the new width is inside the tolerance
                if (!colDesc.merged.customWidth && (Math.abs(colWidth - colDesc.merged.width) < SET_COLUMN_WIDTH_TOLERANCE)) {
                    return;
                }

                // insert the optimal column width into the cache
                var lastInterval = resultIntervals.last();
                if (lastInterval && (lastInterval.last + 1 === colDesc.index) && (lastInterval.fillData.attrs.width === colWidth)) {
                    lastInterval.last += 1;
                } else {
                    lastInterval = new Interval(colDesc.index);
                    lastInterval.fillData = { attrs: { width: colWidth, customWidth: false } };
                    resultIntervals.push(lastInterval);
                }
            }, 'ViewFuncMixin.generateOptimalColumnWidthOperations');

            // create the operations for all columns
            return promise.then(function () {
                return colCollection.generateIntervalOperations(generator, resultIntervals);
            });
        }

        /**
         * Generates row operations to set the optimal row height in the
         * specified row intervals, based on the content of cells.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {IntervalArray|Interval} rowIntervals
         *  An array of row intervals, or a single row interval.
         *
         * @param {RangeArray|Range} [changedRanges]
         *  If specified, an operation has caused updating the automatic row
         *  heights implicitly. Rows with user-defined height (row attribute
         *  "customHeight" set to true) will not be changed in this case. By
         *  default, the optimal height of all rows in the passed intervals
         *  will be calculated, and all manual row heights will be reset to
         *  automatic mode.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function generateOptimalRowHeightOperations(generator, rowIntervals, changedRanges) {

            // resulting intervals with the new row heights
            var resultIntervals = new IntervalArray();
            // maximum height for rows
            var maxHeightHmm = docModel.getMaxRowHeightHmm();
            // tolerance for automatic row height
            var heightTolerance = changedRanges ? UPDATE_ROW_HEIGHT_TOLERANCE : SET_ROW_HEIGHT_TOLERANCE;
            // the rendering cache of the active sheet
            var renderCache = self.getRenderCache();
            // current zoom factor
            var zoom = activeSheetModel.getZoom();
            // the row iterator
            var rowIt = rowCollection.createIterator(rowIntervals, { visible: true });

            // do not change custom row height in update mode
            if (changedRanges) {
                rowIt = new FilterIterator(rowIt, function (rowDesc) {
                    return !rowDesc.merged.customHeight;
                });
            }

            // process all visible rows
            var promise = self.iterateSliced(rowIt, function (rowDesc) {

                // calculate resulting optimal height of the current row
                var rowHeightHmm = docModel.getRowHeightHmm(rowDesc.attributes.character);

                // TODO: iterate all columns in the row
                renderCache.iterateCellsInRow(rowDesc.index, function (element) {

                    // the address of the current cell
                    var address = element.address;

                    // recalculate cell height if current cell has changed
                    if (changedRanges && changedRanges.containsAddress(address)) {

                        var cellModel = cellCollection.getCellModel(address);
                        if (!cellModel || !cellModel.d) { return; }

                        var attrSet = cellCollection.getAttributeSet(address);
                        var currentRowHeight = docModel.getRowHeightHmm(attrSet.character);
                        var lines = 0;

                        if (SheetUtils.hasWrappingAttributes(attrSet)) {
                            var fontDesc = docModel.getRenderFont(attrSet.character, null, zoom);
                            var innerRect = RenderUtils.getInnerRectForCell(activeSheetModel, address);
                            var textPadding = activeSheetModel.getTextPadding(attrSet.character);
                            var availableWidth = Math.max(2, innerRect.width - 2 * textPadding);
                            cellModel.d.split(/\n/).forEach(function (paragraph) {
                                // replace any control characters with space characters
                                lines += fontDesc.getTextLines(Utils.cleanString(paragraph), availableWidth).length;
                            });
                        } else {
                            lines = 1;
                        }

                        rowHeightHmm = Math.max(rowHeightHmm, lines * currentRowHeight);
                        return;
                    }

                    // use effective cell content height from rendering cache
                    if (!element.hasMergedRows() && (element.contentHeight > 0)) {
                        rowHeightHmm = Math.max(rowHeightHmm, activeSheetModel.convertPixelToHmm(element.contentHeight));
                    }
                });

                // bug 40737: restrict to maximum row height allowed in the UI
                rowHeightHmm = Math.min(rowHeightHmm, maxHeightHmm);

                // do not generate operations, if the row height was optimal before, and the new height is inside the tolerance
                if (!rowDesc.merged.customHeight) {
                    var oldHeightHmm = docModel.convertRowHeightFromUnit(rowDesc.merged.height);
                    if (Math.abs(rowHeightHmm - oldHeightHmm) < heightTolerance) {
                        return;
                    }
                }

                // insert the optimal row height into the cache
                var lastInterval = resultIntervals.last();
                if (lastInterval && (lastInterval.last + 1 === rowDesc.index) && (lastInterval.fillData.attrs.height === rowHeightHmm)) {
                    lastInterval.last += 1;
                } else {
                    lastInterval = new Interval(rowDesc.index);
                    lastInterval.fillData = { attrs: { height: rowHeightHmm, customHeight: false } };
                    resultIntervals.push(lastInterval);
                }
            }, 'ViewFuncMixin.generateOptimalRowHeightOperations');

            // create the operations for all rows
            return promise.then(function () {
                return rowCollection.generateIntervalOperations(generator, resultIntervals);
            });
        }

        /**
         * Inserts new columns or rows into the active sheet, according to the
         * current selection.
         *
         * @param {Boolean} columns
         *  Whether to insert columns (true), or rows (false).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function insertIntervals(columns) {

            // check that the sheet is unlocked
            var promise = self.ensureUnlockedSheet();

            // generate and apply the operations
            promise = promise.then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var ranges = docModel.makeFullRanges(getSelectedIntervals(columns), columns);
                    var moveMode = columns ? MoveMode.RIGHT : MoveMode.DOWN;
                    return cellCollection.generateMoveCellsOperations(generator, ranges, moveMode);
                }, { storeSelection: true });
            });

            // show warning messages if needed
            return self.yellOnFailure(promise);
        }

        /**
         * Deletes existing columns or rows from the active sheet, according to
         * the current selection.
         *
         * @param {Boolean} columns
         *  Whether to delete columns (true), or rows (false).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function deleteIntervals(columns) {

            // check that the sheet is unlocked
            var promise = self.ensureUnlockedSheet();

            // generate and apply the operations
            promise = promise.then(function () {

                // convert selected columns/rows to cell range addresses
                var ranges = docModel.makeFullRanges(getSelectedIntervals(columns), columns);
                // how to generate undo operations
                var undoMode = null;
                // the promise chain (needed to chain query dialog)
                var promise2 = self.createResolvedPromise();

                // ask to proceed if the column/row ranges have contents that cannot be restored
                if (!activeSheetModel.canRestoreDeletedRanges(ranges)) {
                    undoMode = 'clear';
                    promise2 = promise2.then(function () {
                        return self.showQueryDialog(
                            columns ? Labels.DELETE_COLUMNS_TITLE : Labels.DELETE_ROWS_TITLE,
                            columns ? Labels.DELETE_COLUMNS_QUERY : Labels.DELETE_ROWS_QUERY
                        );
                    });
                }

                return promise2.then(function () {
                    return activeSheetModel.createAndApplyOperations(function (generator) {
                        var moveMode = columns ? MoveMode.LEFT : MoveMode.UP;
                        return cellCollection.generateMoveCellsOperations(generator, ranges, moveMode);
                    }, { storeSelection: true, undoMode: undoMode });
                });
            });

            // show warning messages if needed
            return self.yellOnFailure(promise);
        }

        /**
         * Changes the specified column or row attributes in the active sheet,
         * according to the current selection.
         *
         * @param {Object} attributes
         *  The (incomplete) column/row attributes, as simple map object (NOT
         *  an attribute set with a "column" or "row" sub-object).
         *
         * @param {Boolean} columns
         *  Whether to change columns (true), or rows (false).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function changeIntervals(attributes, columns) {

            // check that the sheet is unlocked
            var promise = self.ensureUnlockedSheet();

            // generate and apply the operations
            promise = promise.then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var collection = columns ? colCollection : rowCollection;
                    var intervals = getSelectedIntervals(columns);
                    return collection.generateFillOperations(generator, intervals, { attrs: attributes });
                }, { storeSelection: true });
            });

            // show warning messages if needed
            return self.yellOnFailure(promise);
        }

        /**
         * Shows all hidden columns or rows in the current selection.
         *
         * @param {Boolean} columns
         *  Whether to show hidden columns (true), or rows (false).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function showIntervals(columns) {

            // check that the sheet is unlocked
            var promise = self.ensureUnlockedSheet();

            // generate and apply the operations
            promise = promise.then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    var collection = columns ? colCollection : rowCollection;
                    var intervals = getSelectedHiddenIntervals(columns);
                    return collection.generateFillOperations(generator, intervals, { attrs: { visible: true } });
                }, { storeSelection: true });
            });

            // show warning messages if needed
            return self.yellOnFailure(promise);
        }

        /**
         * Performs additional checks whether the contents of the passed cell
         * ranges can be edited, regardless of their 'unlocked' attribute, and
         * the lock state of the active sheet. Used as helper method by various
         * public methods of this class that all support the same options.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.lockTables='none']
         *      Specifies how to check table ranges covering the passed cell
         *      range addresses. See description of the method
         *      ViewFuncMixin.ensureUnlockedRanges() for more details.
         *  - {String} [options.lockMatrixes='none']
         *      Specifies how to check matrix formulas covering the passed cell
         *      ranges. See method ViewFuncMixin.ensureUnlockedRanges() for
         *      more details.
         *
         * @returns {String|Null}
         *  The value null, if the contents of the passed cell ranges can be
         *  edited; otherwise one of the following error codes:
         *  - 'table:change': If the option 'lockTables' has been set to
         *      'full', and the passed cell ranges overlap with a table range
         *      in the active sheet.
         *  - 'table:header:change': If the option 'lockTables' has been set to
         *      'header', and the passed cell ranges overlap with the header
         *      cells of a table range in the active sheet.
         *  - 'formula:matrix:change': If the option 'lockMatrixes' has been
         *      set, and the passed cell ranges overlap with a matrix formula
         *      in the active sheet.
         */
        function ensureEditableContents(ranges, options) {

            // check the table ranges according to the passed mode
            switch (Utils.getStringOption(options, 'lockTables', 'none')) {

                case 'full':
                    // auto-filter always editable (getAllTables() does not return the auto-filter)
                    if (tableCollection.getAllTables().some(function (tableModel) {
                        return ranges.overlaps(tableModel.getRange());
                    })) {
                        return 'table:change';
                    }
                    break;

                case 'header':
                    // auto-filter always editable (getAllTables() does not return the auto-filter)
                    if (tableCollection.getAllTables().some(function (tableModel) {
                        var headerRange = tableModel.getHeaderRange();
                        return headerRange && ranges.overlaps(headerRange);
                    })) {
                        return 'table:header:change';
                    }
                    break;

                case 'headerFooter':
                    // auto-filter always editable (getAllTables() does not return the auto-filter)
                    if (tableCollection.getAllTables().some(function (tableModel) {
                        var headerRange = tableModel.getHeaderRange();
                        var footerRange = tableModel.getFooterRange();
                        return (headerRange && ranges.overlaps(headerRange)) || (footerRange && ranges.overlaps(footerRange));
                    })) {
                        return 'table:headerFooter:change';
                    }
                    break;
            }

            // DOCS-196: check the matrix formulas in the active sheet
            switch (Utils.getStringOption(options, 'lockMatrixes', 'none')) {

                case 'full':
                    if (cellCollection.rangesOverlapAnyMatrix(ranges)) {
                        return 'formula:matrix:change';
                    }
                    break;

                case 'partial':
                    if (cellCollection.rangesOverlapAnyMatrix(ranges, { partial: true })) {
                        return 'formula:matrix:change';
                    }
                    break;
            }

            // the passed ranges are considered to be editable
            return null;
        }

        /**
         * A little helper function for various public methods that return a
         * promise after checking various conditions in the active sheet.
         *
         * @param {Null|String} errorCode
         *  The value null to indicate a successful check; or an error code as
         *  string, intended to be passed to SpreadsheetView.yellMessage() or
         *  SpreadsheetView.yellOnFailure().
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.sync=false]
         *      If set to true, this method will return the passed error code
         *      directly instead of a promise.
         *
         * @returns {Promise|Null|String}
         *  A resolved promise, if the passed error code is null, or a promise
         *  that will be rejected with an object with 'cause' property set to
         *  the passed error code. If the option 'sync' has been set to true,
         *  the error code will be returned directly instead of a promise.
         */
        function createEnsureResult(errorCode, options) {

            // if 'sync' option is set, return the result directly
            if (Utils.getBooleanOption(options, 'sync', false)) { return errorCode; }

            // create the resulting (resolved or rejected) promise
            return errorCode ? SheetUtils.makeRejected(errorCode) : self.createResolvedPromise(null);
        }

        /**
         * Generates and applies the operations, and undo operations, to insert
         * the specified cell contents, and to update the optimal row height of
         * the affected cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the cell has been changed, or
         *  that will be rejected on any error.
         */
        function setCellContents(address, contents, options) {

            // whether to update the optimal row heights after changing the cells
            var updateRows = Utils.getBooleanOption(options, 'updateRows', false);
            // the cell range to be selected while applying the operations
            var selectRange = Utils.getOption(options, 'selectRange', null);

            // generate and apply the operations
            return activeSheetModel.createAndApplyOperations(function (generator) {

                // try to fill the cell
                var promise = cellCollection.generateCellContentOperations(generator, address, contents);

                // set optimal row height afterwards (for changed rows only)
                if (updateRows) {
                    promise = promise.then(function (changedRanges) {
                        return self.generateUpdateRowHeightOperations(generator, changedRanges);
                    });
                }

                // select the range passed with the options
                if (selectRange) {
                    self.selectRange(selectRange);
                }

                return promise;
            }, { storeSelection: true });
        }

        /**
         * Generates and applies the operations, and undo operations, to set
         * the contents of a cell to the passed edit text (tries to convert to
         * an appropriate data type, or to calculate a formula string).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function parseCellContents(address, parseText, options) {

            // parse the passed text to a cell contents object
            var contents = cellCollection.parseCellValue(parseText, 'val', address);

            // if specified, ignore a recognized URL for a cell hyperlink
            if (Utils.getBooleanOption(options, 'skipHyperlink', false)) {
                delete contents.url;
            }

            // create and apply the cell operation, update optimal row height
            return setCellContents(address, contents, options);
        }

        /**
         * Generates and applies the operations, and undo operations to fill
         * all cell ranges with the same value and formatting, and to update
         * the optimal row height of the affected cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the cell has been changed, or
         *  that will be rejected on any error.
         */
        function fillRangeContents(ranges, contents, options) {

            // whether to update the optimal row heights after changing the cells
            var updateRows = Utils.getBooleanOption(options, 'updateRows', false);

            // generate and apply the operations
            return activeSheetModel.createAndApplyOperations(function (generator) {

                // try to fill the cell ranges
                var promise = cellCollection.generateFillOperations(generator, ranges, contents, options);

                // set optimal row height afterwards (for changed rows only)
                if (updateRows) {
                    promise = promise.then(function (changedRanges) {
                        return self.generateUpdateRowHeightOperations(generator, changedRanges);
                    });
                }

                return promise;
            }, { storeSelection: true });
        }

        /**
         * Initializes all sheet-dependent class members according to the
         * current active sheet.
         */
        function changeActiveSheetHandler(event, sheet, sheetModel) {

            // store model instance and index of the new active sheet
            activeSheetModel = sheetModel;

            // get collections of the new active sheet
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            mergeCollection = sheetModel.getMergeCollection();
            cellCollection = sheetModel.getCellCollection();
            hyperlinkCollection = sheetModel.getHyperlinkCollection();
            tableCollection = sheetModel.getTableCollection();

            // forced update of selection settings (do not try to use cached data from old sheet)
            updateSelectionSettings(true);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the subtotal results in the current selection.
         *
         * @returns {SubtotalResult}
         *  The subtotal results in the current selection.
         */
        this.getSubtotalResult = function () {
            return selectedSubtotals;
        };

        /**
         * Returns whether the active sheet is locked. In locked sheets it is
         * not possible to insert, delete, or modify columns and rows, to edit
         * protected cells, or to manipulate drawing objects.
         *
         * @returns {Boolean}
         *  Whether the active sheet is locked.
         */
        this.isSheetLocked = function () {
            return activeSheetModel.isLocked();
        };

        /**
         * Returns a resolved or rejected promise according to the locked state
         * of the sheet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.errorCode='sheet:locked']
         *      The message code specifying the warning message shown when the
         *      sheet is locked. See method SpreadsheetView.yellMessage() for
         *      details.
         *  - {Boolean} [options.sync=false]
         *      If set to true, this method will be executed synchronously, and
         *      the return value will be null or an error code instead of a
         *      promise.
         *
         * @returns {jQuery.Promise|Null|String}
         *  A promise that will be resolved, if the active sheet is not locked;
         *  or that will be rejected with an object with 'cause' property set
         *  to one of the following error codes:
         *  - 'readonly': The entire document is in read-only state.
         *  - passed error code or 'sheet:locked': The active sheet is locked.
         *  If the option 'sync' has been set to true, the return value will be
         *  null if the sheet is not locked, otherwise one of the error codes
         *  mentioned above as string.
         */
        this.ensureUnlockedSheet = function (options) {

            // prerequisite is global document edit mode
            var errorCode = this.isEditable() ? null : 'readonly';

            // return the specified error code, if the sheet is locked
            if (!errorCode && this.isSheetLocked()) {
                errorCode = Utils.getStringOption(options, 'errorCode', 'sheet:locked');
            }

            // if 'sync' option is set, return the result directly
            return createEnsureResult(errorCode, options);
        };

        /**
         * Returns a resolved or rejected promise according to the locked state
         * of the cells in the passed cell ranges. A cell is in locked state,
         * if its 'unlocked' attribute is not set, AND if the active sheet is
         * locked; or if the cell is part of another locked sheet structure,
         * according to the passed options.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.errorCode='cells:locked']
         *      The message code specifying the warning message shown when the
         *      ranges are locked. See method SpreadsheetView.yellMessage() for
         *      details.
         *  - {String} [options.lockTables='none']
         *      Specifies how to check a table range covering the cell ranges
         *      (the auto-filter, however, is never considered to be locked):
         *      - 'none' (default): Table ranges are not locked per se.
         *      - 'full': The entire table range is locked.
         *      - 'header': Only the header cells of the table are locked.
         *  - {String} [options.lockMatrixes='none']
         *      Specifies how to check the matrix formulas covering the passed
         *      cell ranges:
         *      - 'none' (default): Matrix formulas are not locked per se.
         *      - 'full': All matrix formulas are locked.
         *      - 'partial': Only matrix formulas covered partially are locked.
         *  - {Boolean} [options.sync=false]
         *      If set to true, this method will be executed synchronously, and
         *      the return value will be null or an error code instead of a
         *      promise.
         *
         * @returns {jQuery.Promise|Null|String}
         *  A resolved promise, if the active sheet is not locked, or if the
         *  cell ranges are not locked; otherwise (i.e. active sheet, AND the
         *  cell ranges are locked) a rejected promise with a specific error
         *  code. If the option 'sync' has been set to true, the return value
         *  will be null if the cell ranges are editable, otherwise an error
         *  code as string.
         */
        this.ensureUnlockedRanges = function (ranges, options) {

            // prerequisite is global document edit mode
            var errorCode = this.isEditable() ? null : 'readonly';

            // try to find a locked cell in a locked sheet
            if (!errorCode && this.isSheetLocked()) {

                // shortcut: use cached settings of the active cell, otherwise search in cell collection
                ranges = RangeArray.get(ranges);
                var isActiveCell = (ranges.length === 1) && ranges.first().single() && ranges.first().startsAt(activeAddress);
                var isLocked = isActiveCell ? !activeAttributeSet.cell.unlocked : cellCollection.findCellWithAttributes(ranges, { cell: { unlocked: false } });

                // extract the actual error code from the passed options
                if (isLocked) { errorCode = Utils.getStringOption(options, 'errorCode', 'cells:locked'); }
            }

            // bug 39869: do not allow to change table ranges (according to passed options)
            if (!errorCode) { errorCode = ensureEditableContents(ranges, options); }

            // if 'sync' option is set, return the result directly
            return createEnsureResult(errorCode, options);
        };

        /**
         * Returns a resolved or rejected promise according to the locked state
         * of the cell ranges in the current selection. A cell is in locked
         * state, if its 'unlocked' attribute is not set, AND if the active
         * sheet is locked; or if it is part of another locked sheet structure,
         * according to the passed options.
         *
         * @param {Object} [options]
         *  Optional parameters. See description of the public method
         *  ViewFuncMixin.ensureUnlockedRanges() for details.
         *
         * @returns {jQuery.Promise|Null|String}
         *  A resolved promise, if the active sheet is not locked, or if the
         *  selected cell ranges are not locked; otherwise (i.e. active sheet,
         *  AND selected ranges are locked) a rejected promise with a specific
         *  error code. If the option 'sync' has been set to true, the return
         *  value will be null if the selected ranges are editable, otherwise
         *  an error code as string.
         */
        this.ensureUnlockedSelection = function (options) {
            return this.ensureUnlockedRanges(this.getSelectedRanges(), options);
        };

        /**
         * Returns a resolved or rejected promise according to the locked state
         * of the active cell in the current selection. The active cell is in
         * locked state, if its 'unlocked' attribute is not set, AND if the
         * active sheet is locked; or if the cell is part of another locked
         * sheet structure, according to the passed options.
         *
         * @param {Object} [options]
         *  Optional parameters. See description of the public method
         *  ViewFuncMixin.ensureUnlockedRanges() for details. The default value
         *  of the option 'lockTables' will be changed to 'header' though!
         *
         * @returns {jQuery.Promise|Null|String}
         *  A resolved promise, if the active sheet is not locked, or if the
         *  active cell in a locked sheet is not locked; otherwise (i.e. active
         *  sheet, AND active cell are locked) a rejected promise with a
         *  specific error code. If the option 'sync' has been set to true, the
         *  return value will be null if the active cell is editable, otherwise
         *  an error code as string.
         */
        this.ensureUnlockedActiveCell = function (options) {
            options = _.extend({ lockTables: 'header' }, options);
            return this.ensureUnlockedRanges(new Range(activeAddress), options);
        };

        /**
         * Returns a resolved or rejected promise according to the editable
         * state of drawing objects in the active sheet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.errorCode='drawing:change:locked']
         *      The message code specifying the warning message shown when the
         *      drawings in the active sheet are locked. See method
         *      SpreadsheetView.yellMessage() for details.
         *  - {Boolean} [options.sync=false]
         *      If set to true, this method will be executed synchronously, and
         *      the return value will be null or an error code instead of a
         *      promise.
         *
         * @returns {jQuery.Promise|Null|String}
         *  A promise that will be resolved, if the drawing objects in the
         *  active sheet are not locked; or that will be rejected with an
         *  object with 'cause' property set to one of the following error
         *  codes:
         *  - 'readonly': The entire document is in read-only state.
         *  - (the passed error code): The active sheet is locked.
         *  If the option 'sync' has been set to true, the return value will be
         *  null if the drawing objects in the sheet are not locked, otherwise
         *  one of the error codes mentioned above as string.
         */
        this.ensureUnlockedDrawings = function (options) {
            return this.ensureUnlockedSheet(_.extend({ errorCode: 'drawing:change:locked' }, options));
        };

        /**
         * Returns a resolved or rejected promise according to the editable
         * state of cell comments in the active sheet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.errorCode='comment:change:locked']
         *      The message code specifying the warning message shown when the
         *      comments in the active sheet are locked. See method
         *      SpreadsheetView.yellMessage() for details.
         *  - {Boolean} [options.sync=false]
         *      If set to true, this method will be executed synchronously, and
         *      the return value will be null or an error code instead of a
         *      promise.
         *
         * @returns {jQuery.Promise|Null|String}
         *  A promise that will be resolved, if the cell comments in the active
         *  sheet are not locked; or that will be rejected with an object with
         *  'cause' property set to one of the following error codes:
         *  - 'readonly': The entire document is in read-only state.
         *  - (the passed error code): The active sheet is locked.
         *  If the option 'sync' has been set to true, the return value will be
         *  null if the cell comments in the sheet are not locked, otherwise
         * one of the error codes mentioned above as string.
         */
        this.ensureUnlockedComments = function (options) {
            return this.ensureUnlockedSheet(_.extend({ errorCode: 'comment:change:locked' }, options));
        };

        // column operations --------------------------------------------------

        /**
         * Returns the mixed formatting attributes of all columns in the active
         * sheet covered by the current selection.
         *
         * @returns {Object}
         *  The mixed attributes of all selected columns, as simple object. All
         *  attributes that could not be resolved unambiguously will be set to
         *  the value null.
         */
        this.getColumnAttributes = function () {
            var intervals = getSelectedIntervals(true);
            return colCollection.getMixedAttributes(intervals);
        };

        /**
         * Returns the settings of the column the active cell is currently
         * located in.
         *
         * @returns {ColRowDescriptor}
         *  The column descriptor of the active cell.
         */
        this.getActiveColumnDescriptor = function () {
            return colCollection.getEntry(activeAddress[0]);
        };

        /**
         * Returns whether the current selection contains hidden columns, or is
         * located next to hidden columns, that can be made visible.
         *
         * @returns {Boolean}
         *  Whether the current selection contains hidden columns.
         */
        this.canShowColumns = function () {
            return getSelectedHiddenIntervals(true).length > 0;
        };

        /**
         * Inserts new columns into the active sheet, according to the current
         * selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.insertColumns = function () {
            return insertIntervals(true);
        };

        /**
         * Deletes existing columns from the active sheet, according to the
         * current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.deleteColumns = function () {
            return deleteIntervals(true);
        };

        /**
         * Changes the specified column attributes in the active sheet,
         * according to the current selection.
         *
         * @param {Object} attributes
         *  The (incomplete) column attributes, as simple map object (NOT an
         *  attribute set with a "column" sub-object).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.changeColumns = function (attributes) {
            return changeIntervals(attributes, true);
        };

        /**
         * Shows all hidden columns in the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.showColumns = function () {
            return showIntervals(true);
        };

        /**
         * Changes the width of columns in the active sheet, according to the
         * current selection.
         *
         * @param {Number} width
         *  The column width in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width will not be
         *  changed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Number} [options.target]
         *      The target column to be changed. If specified, the column width
         *      will be set to that column only. If the current selection
         *      contains any ranges covering this column completely, the column
         *      width will be set to all columns contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setColumnWidth = function (width, options) {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {

                    // the target column to be modified
                    var targetCol = Utils.getIntegerOption(options, 'target');
                    // the column intervals to be modified
                    var intervals = getSelectedIntervals(true, { target: targetCol });
                    // hide columns if passed column width is zero; show hidden columns when width is changed
                    var attributes = (width === 0) ? { visible: false } : { visible: true, width: width, customWidth: true };

                    return colCollection.generateFillOperations(generator, intervals, { attrs: attributes });
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Sets optimal column width based on the content of cells.
         *
         * @param {Number} [targetCol]
         *  The target column to be changed. If the current selection contains
         *  any ranges covering this column completely, the optimal column
         *  width will be set to all columns contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setOptimalColumnWidth = function (targetCol) {

            // generate and apply the column operations
            var promise = activeSheetModel.createAndApplyOperations(function (generator) {
                var intervals = getSelectedIntervals(true, { target: targetCol, visible: true });
                return generateOptimalColumnWidthOperations(generator, intervals);
            }, { storeSelection: true });

            // show error alert on failure
            return this.yellOnFailure(promise);
        };

        // row operations -----------------------------------------------------

        /**
         * Returns the mixed formatting attributes of all rows in the active
         * sheet covered by the current selection.
         *
         * @returns {Object}
         *  The mixed attributes of all selected rows, as simple object. All
         *  attributes that could not be resolved unambiguously will be set to
         *  the value null.
         */
        this.getRowAttributes = function () {
            var intervals = getSelectedIntervals(false);
            return rowCollection.getMixedAttributes(intervals);
        };

        /**
         * Returns the settings of the row the active cell is currently located
         * in.
         *
         * @returns {ColRowDescriptor}
         *  The row descriptor of the active cell.
         */
        this.getActiveRowDescriptor = function () {
            return rowCollection.getEntry(activeAddress[1]);
        };

        /**
         * Returns whether the current selection contains hidden rows, or is
         * located next to hidden rows, that can be made visible.
         *
         * @returns {Boolean}
         *  Whether the current selection contains hidden rows.
         */
        this.canShowRows = function () {
            return getSelectedHiddenIntervals(false).length > 0;
        };

        /**
         * Inserts new rows into the active sheet, according to the current
         * selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.insertRows = function () {
            return insertIntervals(false);
        };

        /**
         * Deletes existing rows from the active sheet, according to the
         * current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.deleteRows = function () {
            return deleteIntervals(false);
        };

        /**
         * Changes the specified row attributes in the active sheet, according
         * to the current selection..
         *
         * @param {Object} attributes
         *  The (incomplete) row attributes, as simple map object (NOT an
         *  attribute set with a "row" sub-object).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.changeRows = function (attributes) {
            return changeIntervals(attributes, false);
        };

        /**
         * Shows all hidden rows in the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.showRows = function () {
            return showIntervals(false);
        };

        /**
         * Changes the height of rows in the active sheet, according to the
         * current selection.
         *
         * @param {Number} height
         *  The row height in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height will not be changed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Number} [options.target]
         *      The target row to be changed. If specified, the row height will
         *      be set to that row only. If the current selection contains any
         *      ranges covering this row completely, the row height will be set
         *      to all rows contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setRowHeight = function (height, options) {

            var promise = this.ensureUnlockedSheet().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {

                    // the target row to be modified
                    var targetRow = Utils.getIntegerOption(options, 'target');
                    // the row intervals to be modified
                    var intervals = getSelectedIntervals(false, { target: targetRow });
                    // hide rows if passed row height is zero; show hidden rows when height is changed
                    var attributes = (height === 0) ? { visible: false } : { visible: true, height: height, customHeight: true };

                    return rowCollection.generateFillOperations(generator, intervals, { attrs: attributes });
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Sets optimal row height based on the content of cells.
         *
         * @param {Number} [targetRow]
         *  The target row to be changed. If the current selection contains any
         *  ranges covering this row completely, the optimal row height will be
         *  set to all rows contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setOptimalRowHeight = function (targetRow) {

            // generate and apply the row operations
            var promise = activeSheetModel.createAndApplyOperations(function (generator) {
                var intervals = getSelectedIntervals(false, { target: targetRow, visible: true });
                return generateOptimalRowHeightOperations(generator, intervals);
            }, { storeSelection: true });

            // show error alert on failure
            return this.yellOnFailure(promise);
        };

        /**
         * Generates row operations to update the optimal row height in the
         * specified changed cell ranges, based on the content of cells.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single range address.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.generateUpdateRowHeightOperations = function (generator, ranges) {

            // the merged and sorted row intervals
            var rowIntervals = RangeArray.get(ranges).rowIntervals().merge();

            // restrict to visible area (performance)
            // TODO: iterate all row intervals, not single rows, important when updating entire column
            rowIntervals = this.getVisibleIntervals(rowIntervals, false);

            // set update mode (do not modify rows with custom height)
            return generateOptimalRowHeightOperations(generator, rowIntervals, ranges);
        };

        // cell operations ----------------------------------------------------

        /**
         * Returns the parsed number format of the active cell in the current
         * selection.
         *
         * @returns {ParsedFormat}
         *  The parsed number format of the active cell.
         */
        this.getActiveParsedFormat = function () {
            return activeParsedFormat;
        };

        /**
         * Returns the URL of a hyperlink at the current active cell.
         *
         * @returns {String|Null}
         *  The URL of a hyperlink attached to the active cell; or null, if the
         *  active cell does not contain a hyperlink.
         */
        this.getCellURL = function () {
            return hyperlinkCollection.getCellURL(activeAddress);
        };

        /**
         * Returns the URL of a hyperlink as returned by a formula (via the
         * function HYPERLINK) of the current active cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {String|Null}
         *  The URL of a hyperlink as returned by a cell formula (via the
         *  function HYPERLINK) of the current active cell.
         */
        this.getFormulaURL = function () {
            return cellCollection.getFormulaURL(activeAddress);
        };

        /**
         * Returns the effective URL of a hyperlink at the current active cell.
         * If the cell contains a regular hyperlink, and a cell formula with a
         * HYPERLINK function, the regular hyperlink will be preferred.
         *
         * @returns {String|Null}
         *  The effective URL of a hyperlink at the active cell.
         */
        this.getEffectiveURL = function () {
            return cellCollection.getEffectiveURL(activeAddress);
        };

        /**
         * Changes the contents and/or formatting of the active cell, and
         * updates the view according to the new cell value and formatting.
         *
         * @param {Object} contents
         *  The new properties to be applied at the cell. See description of
         *  the method CellCollection.generateCellContentOperations() for more
         *  details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCellContents = function (contents) {

            // whether the operation will modify the values of cells
            var changeValues = (contents.u === true) || ('v' in contents) || ('f' in contents);

            // apply operations (do nothing if active cell is locked)
            var lockMatrixes = changeValues ? 'partial' : 'none';
            var promise = this.ensureUnlockedActiveCell({ lockMatrixes: lockMatrixes }).then(function () {
                return setCellContents(activeAddress, contents, { updateRows: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Sets the contents of the active cell to the passed edit text (tries
         * to convert to an appropriate data type, or to calculate a formula
         * string), and updates the view according to the new cell value and
         * formatting.
         *
         * @param {String} parseText
         *  The raw text to be parsed. May result in a number, a boolean value,
         *  an error code, or a formula expression whose result will be
         *  calculated. The empty string will remove the current cell value
         *  (will create a blank cell).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.parseCellContents = function (parseText) {

            // apply operations (do nothing if active cell is locked)
            var promise = this.ensureUnlockedActiveCell({ lockMatrixes: 'partial' }).then(function () {
                return parseCellContents(activeAddress, parseText, { updateRows: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Generates the cell operations, and the undo operations, to change
         * the specified contents for a list of single cells in the active
         * sheet.
         *
         * @param {Array<Object>} contentsArray
         *  A list with the descriptors for all cells to be changed. Each array
         *  element MUST be an object with the following properties:
         *  - {Address} address
         *      The address of the cell to be modified.
         *  - {Object} contents
         *      The cell contents descriptor for the cell. See method
         *      CellCollection.generateCellContentOperations() for details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCellContentsArray = function (contentsArray) {

            // all cell addresses, as range array
            var addresses = AddressArray.map(contentsArray, function (entry) { return entry.address; });
            var ranges = RangeArray.mergeAddresses(addresses);

            // apply operations (do nothing if active cell is locked)
            var promise = this.ensureUnlockedRanges(ranges, { lockTables: 'header', lockMatrixes: 'partial' }).then(function () {

                // generate and apply the operations
                return activeSheetModel.createAndApplyOperations(function (generator) {

                    // try to fill the cell
                    var promise2 = cellCollection.generateContentsArrayOperations(generator, contentsArray);

                    // set optimal row height afterwards (for changed rows only)
                    promise2 = promise2.then(function (changedRanges) {
                        return self.generateUpdateRowHeightOperations(generator, changedRanges);
                    });

                    return promise2;
                }, { storeSelection: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Changes the contents and/or formatting of all cells in the current
         * selection, and updates the view according to the new cell values and
         * formatting.
         *
         * @param {Object} contents
         *  The new properties to be applied at all selected cells. See method
         *  CellCollection.generateCellContentOperations() for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.fillRangeContents = function (contents) {

            // whether the operation will modify the values of cells
            var changeValues = (contents.u === true) || ('v' in contents) || ('f' in contents);

            // do nothing if current selection contains locked cells (do not allow to change table header texts)
            var promise = this.ensureUnlockedSelection(changeValues ? { lockTables: 'header', lockMatrixes: 'partial' } : null).then(function () {
                return fillRangeContents(self.getSelectedRanges(), contents, { updateRows: true, refAddress: activeAddress });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Removes the values, formulas, and formatting of all cells in the
         * current selection (deletes the cell definitions from the document
         * model), and updates the view.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.clearRangeContents = function () {

            // do nothing if current selection contains locked cells (do not allow to delete table header texts)
            var promise = this.ensureUnlockedSelection({ lockTables: 'header', lockMatrixes: 'partial' }).then(function () {
                return fillRangeContents(self.getSelectedRanges(), { u: true }, { updateRows: true });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Inserts a matrix formula into the current selection.
         *
         * @param {Object} contents
         *  The new properties to be applied at all selected cells. See method
         *  CellCollection.generateCellContentOperations() for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setMatrixContents = function (contents) {

            // if matrix edit mode has been left without a formula expression, behave like normal fill mode
            if (!contents.f || !contents.result) { return this.fillRangeContents(contents); }

            // invalid formula result (e.g. circular reference, oversized matrix) results in #N/A
            var matrix = (contents.v instanceof ErrorCode) ? new Matrix([[contents.v]]) : contents.v;
            // if the active cell covers an existing matrix formula, use its bounding range regardless of selection
            var matrixRange = cellCollection.getMatrixRange(activeAddress);
            // whether to update an existing matrix (prevent auto-expansion of 1x1 matrixes)
            var updateMatrix = matrixRange !== null;
            // fall-back to the active range of the selection (ignore multi-selection)
            if (!matrixRange) { matrixRange = this.getActiveRange(); }

            // automatically expand single-cell selection to size of result matrix
            if (!updateMatrix && matrixRange.single()) {
                matrixRange.end[0] = matrixRange.start[0] + matrix.cols() - 1;
                matrixRange.end[1] = matrixRange.start[1] + matrix.rows() - 1;
            }

            // check the resulting matrix range:
            // - do not allow to insert oversized matrixes
            // - do not insert matrix formulas over merged ranges
            var matrixDim = Dimension.createFromRange(matrixRange);
            var errorCode = null;
            if (!Matrix.isValidDim(matrixDim)) {
                errorCode = 'formula:matrix:size';
            } else if (mergeCollection.rangesOverlapAnyMergedRange(matrixRange)) {
                errorCode = 'formula:matrix:merged';
            }

            // do not insert matrix formulas into table ranges, do not allow to change another existing matrix partially
            if (!errorCode) {
                var lockMatrixes = updateMatrix ? 'none' : 'partial';
                errorCode = this.ensureUnlockedRanges(matrixRange, { lockTables: 'full', lockMatrixes: lockMatrixes, sync: true });
            }

            // generate and apply the 'changeCells' operation for the matrix formula
            var promise = createEnsureResult(errorCode).then(function () {

                // build the cell contents matrix
                var cellMatrix = _.times(matrixDim.rows, function (row) {
                    return {
                        c: _.times(matrixDim.cols, function (col) {
                            var cellData = { v: Scalar.getCellValue(matrix.get(row, col)), f: null };
                            // add the formula and bounding range of the matrix to the anchor cell
                            if ((row === 0) && (col === 0)) {
                                cellData.f = contents.f;
                                cellData.mr = matrixRange;
                            }
                            return cellData;
                        })
                    };
                });

                return setCellContents(matrixRange.start, cellMatrix, { updateRows: true, selectRange: matrixRange });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Changes the attributes of the attribute family 'cell' for all cells
         * in the current selection.
         *
         * @param {Object} cellAttrs
         *  A simple map with cell attributes to be changed. All attributes set
         *  to the value null will be removed from the cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setCellAttributes = function (cellAttrs) {
            return this.fillRangeContents({ a: { cell: cellAttrs } });
        };

        /**
         * Returns the number format category of the active cell.
         *
         * @returns {String}
         *  The current number format category.
         */
        this.getNumberFormatCategory = function () {
            return activeParsedFormat.category;
        };

        /**
         * Sets selected number format category for the current active cell in
         * the current selection, and implicitly set the first format code of
         * the predefined template of this category to the cell.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setNumberFormatCategory = function (category) {
            var formatCode = numberFormatter.getDefaultCode(category);
            return (formatCode !== null) ? this.fillRangeContents({ format: formatCode }) : $.when();
        };

        /**
         * Returns the number format code of the active cell.
         *
         * @returns {String}
         *  The format code of the active cell, as string.
         */
        this.getNumberFormat = function () {
            return activeParsedFormat.formatCode;
        };

        /**
         * Returns the format code of the active cell.
         *
         * @param {Number|String} format
         *  The identifier of a number format as integer, or a format code as
         *  string, to be set to the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setNumberFormat = function (format) {

            // do nothing if custom fall-back button is clicked
            if (format === Labels.CUSTOM_FORMAT_VALUE) {
                return this.createResolvedPromise();
            }

            return this.fillRangeContents({ format: format });
        };

        /**
         * Returns the border mode of the current selection.
         *
         * @returns {Object}
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         */
        this.getBorderMode = function () {
            return MixedBorder.getBorderMode(selectedMixedBorders);
        };

        /**
         * Changes the visibility of the borders in all ranges of the current
         * selection.
         *
         * @param {Object} borderMode
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.setBorderMode = function (borderMode) {

            // wait for the background loop that collects the mixed borders of the selection
            var promise = mixedBordersDef.then(function () {

                // convert border mode to real border attributes
                var borderAttributes = MixedBorder.getBorderAttributes(borderMode, selectedMixedBorders, DEFAULT_SINGLE_BORDER);

                // do nothing if current selection contains locked cells
                var promise2 = self.ensureUnlockedSelection().then(function () {
                    return activeSheetModel.createAndApplyOperations(function (generator) {
                        return cellCollection.generateBorderOperations(generator, self.getSelectedRanges(), borderAttributes);
                    }, { storeSelection: true });
                });

                // bug 34021: immediately update the cached mixed borders of the selection for fast GUI feedback
                promise2.done(function () {
                    _.extend(selectedMixedBorders, borderAttributes);
                });

                return promise2;
            });

            // show warning alerts if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Returns the mixed border attributes of the current selection.
         *
         * @returns {Object}
         *  An attribute map containing the mixed borders of all available
         *  border attributes ('borderTop', 'borderLeft', etc.), as well as the
         *  pseudo attributes 'borderInsideHor' and 'borderInsideVert' for
         *  inner borders in the current selection. Each border attribute value
         *  contains the properties 'style', 'width', and 'color'. If the
         *  border property is equal in all affected cells, it will be set as
         *  property value in the border object, otherwise the property will be
         *  set to the value null.
         */
        this.getBorderAttributes = function () {
            return _.copy(selectedMixedBorders, true);
        };

        /**
         * Changes the specified border properties (line style, line color, or
         * line width) of all visible cell borders in the current selection.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the current
         *  selection. Omitted border properties will not be changed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        this.changeVisibleBorders = function (border) {

            // do nothing if current selection contains locked cells
            var promise = this.ensureUnlockedSelection().then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    return cellCollection.generateVisibleBorderOperations(generator, self.getSelectedRanges(), border);
                }, { storeSelection: true });
            });

            // bug 34021: immediately update the cached mixed borders of the selection for fast GUI feedback
            promise.done(function () {
                _.each(selectedMixedBorders, function (mixedBorder) {
                    if (MixedBorder.isVisibleBorder(mixedBorder)) {
                        _.extend(mixedBorder, border);
                    }
                });
            });

            // show warning messages
            return this.yellOnFailure(promise);
        };

        /**
         * Fills or clears a cell range according to the position and contents
         * of the cell range currently selected (the 'auto-fill' feature).
         *
         * @param {String} border
         *  Which border of the selected range will be modified by the
         *  operation. Allowed values are 'left', 'right', 'top', or 'bottom'.
         *
         * @param {Number} count
         *  The number of columns/rows to extend the selected range with
         *  (positive values), or to shrink the selected range (negative
         *  values).
         *
         * @param {Boolean} invertType
         *  Swap the default expansion type according to the source data (copy
         *  versus auto-increment).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the auto-fill target range is
         *  editable, and the auto-fill operation has been applied. Otherwise,
         *  the promise will be rejected.
         */
        this.autoFill = function (border, count, options) {

            var // all cell ranges selected in the sheet
                ranges = this.getSelectedRanges(),
                // whether to expand/shrink the leading or trailing border
                leading = /^(left|top)$/.test(border),
                // whether to expand/delete columns or rows
                columns = /^(left|right)$/.test(border),
                // the target range for auto-fill
                targetRange = ranges.first().clone(),
                // the promise of the entire auto-fill operations
                promise = null;

            // for safety: do nothing if called in a multi-selection
            // (no error message, should not happen as the GUI does not offer auto-fill)
            if (ranges.length !== 1) {
                return $.Deferred().reject();
            }

            // fill cells outside the selected range
            if (count > 0) {

                // build the target range (adjacent to the selected range)
                if (leading) {
                    targetRange.setStart(ranges.first().getStart(columns) - count, columns);
                    targetRange.setEnd(ranges.first().getStart(columns) - 1, columns);
                } else {
                    targetRange.setStart(ranges.first().getEnd(columns) + 1, columns);
                    targetRange.setEnd(ranges.first().getEnd(columns) + count, columns);
                }

                // only the target range must be editable (not the source cells)
                promise = this.ensureUnlockedRanges(targetRange, { lockTables: 'header', lockMatrixes: 'partial' });

                // apply auto-fill, and update optimal row heights
                promise = promise.then(function () {
                    return activeSheetModel.createAndApplyOperations(function (generator) {

                        var promise2 = cellCollection.generateAutoFillOperations(generator, ranges.first(), border, count, options);

                        promise2 = promise2.then(function () {
                            return self.generateUpdateRowHeightOperations(generator, targetRange);
                        });

                        // expand the selected range
                        return promise2.always(function () {
                            self.changeActiveRange(ranges.first().boundary(targetRange));
                        });
                    }, { storeSelection: true });
                });

            // delete cells inside the selected range
            } else if (count < 0) {

                // build the target range (part of the selected range, may be the entire selected range)
                if (leading) {
                    targetRange.setEnd(targetRange.getStart(columns) - count - 1, columns);
                } else {
                    targetRange.setStart(targetRange.getEnd(columns) + count + 1, columns);
                }

                // the target range must be editable (not the entire selected range)
                promise = this.ensureUnlockedRanges(targetRange, { lockTables: 'header', lockMatrixes: 'partial' });

                // remove all values and formulas from the cell range (but not the formatting),
                // and update optimal row heights
                promise = promise.then(function () {
                    return fillRangeContents(targetRange, { v: null, f: null }, { updateRows: true });
                });

                // update selection
                promise.always(function () {
                    // do not change selection after deleting the entire range
                    if (ranges.first().equals(targetRange)) { return; }
                    // select the unmodified part of the original range
                    targetRange = ranges.first().clone();
                    if (leading) {
                        targetRange.start.move(-count, columns);
                    } else {
                        targetRange.end.move(count, columns);
                    }
                    self.changeActiveRange(targetRange);
                });

            // do nothing if no cells will be filled or cleared
            } else {
                promise = this.createResolvedPromise();
            }

            return this.yellOnFailure(promise);
        };

        /**
         * Inserts one or more formulas calculating subtotal results into the,
         * or next to the current selection.
         *
         * @param {String} funcKey
         *  The resource key of the subtotal function to be inserted into the
         *  formulas. MUST be a function that takes an argument of reference
         *  type.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the formulas have been
         *  created successfully.
         */
        this.insertAutoFormula = function (funcKey) {

            // start cell edit mode, if a single cell is selected
            if (this.isSingleCellSelection()) {
                var sourceRange = cellCollection.findAutoFormulaRange(activeAddress);
                var formula = '=' + formulaGrammar.generateAutoFormula(docModel, funcKey, sourceRange, activeAddress);
                // cancel current cell edit mode, and restart edit mode
                return this.enterTextEditMode('cell', { text: formula, pos: -1, restart: true });
            }

            // bug 56321: require unlocked sheet (resulting locations of formula cells
            // is unpredictable with the current implementation)
            var promise = this.ensureUnlockedSheet();

            // create and apply the operations
            promise = promise.then(function () {
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    // generate the 'changeCells' operations, and the undo operations
                    var range = self.getActiveRange();
                    return cellCollection.generateAutoFormulaOperations(generator, funcKey, range).done(function (targetRange) {
                        if (targetRange) { self.selectRange(targetRange); }
                    });
                }, { storeSelection: true });
            });

            return this.yellOnFailure(promise);
        };

        /**
         * Shows the 'Insert Function' dialog, and starts the cell edit mode
         * with a formula containing the function selected in the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after a function has been selected
         *  in the dialog, and the cell edit mode has been started; or that
         *  will be rejected after canceling the dialog.
         */
        this.showInsertFunctionDialog = function () {

            // the active cell must be editable
            var promise = this.ensureUnlockedActiveCell().then(function () {

                // pick the formula string if available, otherwise the display text
                var value = activeTokenDesc ? activeTokenDesc.formula : activeCellDisplay;

                // try to extract a function name
                if (_.isString(value)) {
                    var result = (/^=?\s*([^(]+)/).exec(value);
                    value = result ? result[1] : null;
                }

                // show the dialog, wait for the result
                return new Dialogs.FunctionDialog(self, value).show();
            });

            // start cell edit mode, insert the selected function as formula into the active cell
            promise = promise.then(function (funcKey) {
                var formula = '=' + formulaGrammar.getFunctionName(funcKey) + '()';
                // cancel current cell edit mode, and restart edit mode
                return self.enterTextEditMode('cell', { text: formula, pos: -1, restart: true });
            });

            // show warning alerts if necessary
            return this.yellOnFailure(promise);
        };

        /**
         * Merges or unmerges all cells of the current selection.
         *
         * @param {MergeMode} [mergeMode]
         *  The merge mode. If omitted, the merge mode will be MergeMode.MERGE,
         *  if the selection does not cover any merged ranges, otherwise will
         *  be MergeMode.UNMERGE.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the ranges have been merged
         *  or unmerged successfully, or that will be rejected on any error.
         */
        this.mergeRanges = function (mergeMode) {

            // the selected ranges to be merged/unmerged
            var ranges = this.getSelectedRanges();

            // toggle merge state if no valid merge mode has been passed
            if (!(mergeMode instanceof MergeMode)) {
                mergeMode = mergeCollection.rangesOverlapAnyMergedRange(ranges) ? MergeMode.UNMERGE : MergeMode.MERGE;
            }

            // do nothing, if current selection contains locked cells
            // bug 39869: do not allow to merge any cells in a table range (except auto-filter)
            // DOCS-196: do not allow to merge matrix formulas
            var unmerge = mergeMode === MergeMode.UNMERGE;
            var promise = this.ensureUnlockedSelection(unmerge ? null : { lockTables: 'full', lockMatrixes: 'full' }).then(function () {

                // create and apply the operations
                return activeSheetModel.createAndApplyOperations(function (generator) {
                    // generate the 'mergeCells' operations, and the undo operations
                    return mergeCollection.generateMergeCellsOperations(generator, ranges, mergeMode);
                }, { storeSelection: true });
            });

            // show warning alerts if necessary
            return this.yellOnFailure(promise);
        };

        // hyperlink operations -----------------------------------------------

        /**
         * Inserts new hyperlink ranges covering the selected cells, or the
         * active cell.
         *
         * @param {String} url
         *  The URL of the hyperlink to be inserted.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.activeCell=false]
         *      If set to true, the hyperlink will be inserted into the active
         *      cell of the selection only. By default, the entire selection
         *      will be filled.
         *  - {Boolean} [options.createStyles=false]
         *      If set to true, the cells covered by the new hyperlink will be
         *      set to underline style, and to the current hyperlink text
         *      color.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the hyperlink ranges have
         *  been inserted successfully, ot that will be rejected on any error.
         */
        this.insertHyperlink = function (url, options) {

            // whether to create the hyperlink for the active cell only, instead for the entire selection
            var activeCell = Utils.getBooleanOption(options, 'activeCell', false);
            // whether to create the character styles (underline, text color) for the hyperlink
            var createStyles = Utils.getBooleanOption(options, 'createStyles', false);
            // the character attributes for the cells covered by the hyperlink
            var attributes = createStyles ? { character: { underline: true, color: docModel.createHlinkColor() } } : null;
            // the cell contents object
            var contents = { a: attributes, url: url };

            // fill the active cell, or the entire cell selection
            return activeCell ? this.setCellContents(contents) : this.fillRangeContents(contents);
        };

        /**
         * Deletes the hyperlink ranges covering the selected cells, or the
         * active cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.activeCell=false]
         *      If set to true, the hyperlinks will be removed from the active
         *      cell of the selection only. By default, the entire selection
         *      will be cleared.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the hyperlink ranges have
         *  been deleted successfully, ot that will be rejected on any error.
         */
        this.deleteHyperlink = function (options) {

            // whether to create the hyperlink for the active cell only, instead for the entire selection
            var activeCell = Utils.getBooleanOption(options, 'activeCell', false);
            // the cell contents object (remove hyperlinks, and the character formatting)
            var contents = { a: { character: { underline: null, color: null } }, url: '' };

            // fill the active cell, or the entire cell selection
            return activeCell ? this.setCellContents(contents) : this.fillRangeContents(contents);
        };

        /**
         * Opens the 'Edit Hyperlink' dialog, and applies the settings made in
         * the dialog to the current cell selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the dialog has been closed,
         *  and the hyperlink settings have been applied successfully; or that
         *  will be rejected after canceling the dialog, or on any error.
         */
        this.editHyperlink = function () {

            // the literal string value of the cell, used as initial display string
            var display = null;
            // the current hyperlink of the active cell
            var currUrl = null;
            // whether the cell contained a hyperlink before
            var hasHyperlink = false;

            // try to leave text edit mode
            var promise = this.leaveTextEditMode();

            // active cell must be editable (perform additional check for tables)
            promise = promise.then(function () {
                return self.ensureUnlockedActiveCell();
            });

            // create and show the dialog
            promise = promise.then(function () {

                // extract the hyperlink settings of the active cell
                display = (_.isString(activeCellValue) && !activeTokenDesc) ? activeCellValue : null;
                currUrl = self.getCellURL();
                hasHyperlink = typeof currUrl === 'string';

                // if there is no real hyperlink, try the literal cell string
                if (!currUrl && display) {
                    currUrl = HyperlinkUtils.checkForHyperlink(display);
                }

                // do not duplicate equal URL and display string in the dialog
                if (currUrl === display) { display = null; }

                // show the 'Edit Hyperlink' dialog
                return new Dialogs.HyperlinkDialog(self, currUrl, display).show();
            });

            // apply hyperlink settings after closing the dialog
            promise = promise.then(function (result) {

                // missing result.url indicates to delete a hyperlink ('Remove' button)
                if (!result.url) {
                    return self.deleteHyperlink();
                }

                // collect multiple complex sheet operations in a single undo action
                return undoManager.enterUndoGroup(function () {

                    // add character formatting for a new hyperlink, and generate the operation
                    var promise2 = self.insertHyperlink(result.url, { createStyles: !hasHyperlink });

                    // create a text cell, if user has entered a display string in the dialog;
                    // or if the cell was blank, and a valid URL has been entered (do not change
                    // existing cells, e.g. numbers, formulas, etc.)
                    if (result.text) {
                        promise2 = promise2.then(function () {
                            return parseCellContents(activeAddress, result.text, { updateRows: true, skipHyperlink: true });
                        });
                    } else if (activeCellValue === null) {
                        promise2 = promise2.then(function () {
                            return setCellContents(activeAddress, { v: result.url }, { updateRows: true });
                        });
                    }

                    return promise2;
                });
            });

            // show warning alert if necessary
            return this.yellOnFailure(promise);
        };

        // initialization -----------------------------------------------------

        // initialize selection settings, used until first real update
        activeAttributeSet = docModel.getCellStyles().getDefaultStyleAttributeSet();
        activeParsedFormat = numberFormatter.getParsedFormatForAttributes(activeAttributeSet.cell);

        // initialize sheet-dependent class members according to the active sheet
        this.on('change:activesheet', changeActiveSheetHandler);

        // refresh selection settings after changing the selection
        this.on('change:selection:prepare', function () { updateSelectionSettings(false); });

        // update selection settings after any operational changes in the document
        this.one('change:selection', function () {
            self.listenTo(docModel, 'operations:success', function () { updateSelectionSettings(true); });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = undoManager = numberFormatter = formulaGrammar = autoStyles = null;
            activeSheetModel = colCollection = rowCollection = mergeCollection = cellCollection = null;
            hyperlinkCollection = tableCollection = null;
        });

    } // class ViewFuncMixin

    // exports ================================================================

    return ViewFuncMixin;

});
