/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/edit/cellundomanager', [
    'io.ox/office/tk/object/triggerobject'
], function (TriggerObject) {

    'use strict';

    // class CellEditState ====================================================

    /**
     * A single edit state (text contents, selection, and all text formatting
     * attributes) used in cell edit mdoe.
     *
     * @constructor
     *
     * @property {String} text
     *  The current text contents.
     *
     * @property {Object} sel
     *  The current text selection, in the properties 'start' and 'end'.
     *
     * @property {Object} attrs
     *  Attribute set with all text formatting attributes.
     *
     * @property {Boolean} fixed
     *  Whether this edit state is fixed, i.e. cannot be updated with other
     *  text contents.
     */
    function CellEditState(text, selection, attrs, fixed) {

        this.text = text;
        this.sel = selection;
        this.attrs = _.copy(attrs, true);
        this.fixed = !!fixed;

    } // class CellEditState

    // public methods ---------------------------------------------------------

    /**
     * Inserts the passed text contents and selection into this state instance.
     */
    CellEditState.prototype.update = function (text, selection) {
        this.text = text;
        this.sel = selection;
    };

    // class CellUndoManager ==================================================

    /**
     * A special undo manager that will be used to track the changes made in a
     * cell while the cell edit mode is active (without generating any document
     * operations). Instances of this class provide the same API as an instance
     * of the class UndoManager used by the document model. This allows to
     * register this undo manager at the application controller (see method
     * EditController.registerUndoManager() for details).
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {CellTextEditor} cellEditor
     *  The text editor containing this instance.
     */
    var CellUndoManager = TriggerObject.extend({ constructor: function (cellEditor) {

        // array of all known edit states
        var stateStack = [];

        // index of the current state in the action stack
        var stackIndex = -1;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, cellEditor);

        // public methods -----------------------------------------------------

        /**
         * Removes all edit states from this undo manager.
         *
         * @returns {CellUndoManager}
         *  A reference to this instance.
         */
        this.clearStates = function () {
            if (stateStack.length > 0) {
                stateStack = [];
                stackIndex = -1;
                this.trigger('change:stack');
            }
            return this;
        };

        /**
         * Saves the current edit state of the passed text area element on top
         * of the undo stack. If the current edit state is not on top of the
         * stack, all following edit states will be deleted first.
         *
         * @param {String} text
         *  The current text contents.
         *
         * @param {Object} selection
         *  The current text selection.
         *
         * @param {Object} attrs
         *  The attribute set with text formatting attributes currently used in
         *  the text area element.
         *
         * @param {Boolean} [fixed=false]
         *  If omitted or set to false, and if the top-most edit state on the
         *  stack is currently active and represents a simple text change (its
         *  property 'fixed' is false too), it will be updated instead of
         *  creating a new edit state. This can be used to accumulate multiple
         *  changes of single characters into a single edit state. If set to
         *  true, a new edit state will be created unconditionally.
         *
         * @returns {CellUndoManager}
         *  A reference to this instance.
         */
        this.saveState = function (text, selection, attrs, fixed) {

            // the first inserted state is always fixed (initial state when starting edit mode)
            fixed = fixed || (stateStack.length === 0);

            // try to update the current (top-most, not fixed) state with current text,
            // otherwise delete all remaining (redo) states, and push the new state
            if (!fixed && (this.getRedoCount() === 0) && !stateStack[stackIndex].fixed) {
                stateStack[stackIndex].update(text, selection);
            } else {
                stackIndex += 1;
                stateStack.splice(stackIndex);
                stateStack.push(new CellEditState(text, selection, attrs, fixed));
            }

            // notify listeners
            return this.trigger('change:stack');
        };

        /**
         * Returns the number of edit states available on the undo stack.
         *
         * @returns {Number}
         *  The number of edit states available on the undo stack.
         */
        this.getUndoCount = function () {
            return stackIndex;
        };

        /**
         * Restores the top-most edit state of the undo stack.
         *
         * @returns {CellUndoManager}
         *  A reference to this instance.
         */
        this.undo = function () {
            if (stackIndex > 0) {
                stackIndex -= 1;
                this.trigger('restore:state', stateStack[stackIndex]);
            }
            return this;
        };

        /**
         * Returns the number of edit states available on the redo stack.
         *
         * @returns {Number}
         *  The number of edit states available on the redo stack.
         */
        this.getRedoCount = function () {
            return stateStack.length - stackIndex - 1;
        };

        /**
         * Restores the top-most edit state of the redo stack.
         *
         * @returns {CellUndoManager}
         *  A reference to this instance.
         */
        this.redo = function () {
            if (stackIndex + 1 < stateStack.length) {
                stackIndex += 1;
                this.trigger('restore:state', stateStack[stackIndex]);
            }
            return this;
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            cellEditor = stateStack = null;
        });

    } }); // class CellUndoManager

    // exports ================================================================

    return CellUndoManager;

});
