/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/range3darray', [
    'io.ox/office/tk/container/typedarray',
    'io.ox/office/spreadsheet/utils/range3d',
    'io.ox/office/spreadsheet/utils/intervalarray'
], function (TypedArray, Range3D, IntervalArray) {

    'use strict';

    // class Range3DArray =====================================================

    /**
     * Represents an array of cell range addresses with sheet indexes. The
     * array elements are instances of the class Range3D.
     *
     * @constructor
     *
     * @extends Array
     */
    var Range3DArray = TypedArray.create(Range3D);

    // static methods ---------------------------------------------------------

    /**
     * Creates an array of cell range address with sheet indexes from the
     * passed simple cell range addresses, and the passed sheet indexes. The
     * sheet indexes in the resulting cell range address will be ordered
     * automatically. The start and end address of the original cell range
     * addresses will be deeply cloned when creating the new range addresses.
     *
     * @param {RangeArray|Range} ranges
     *  An array of simple cell range addresses (without sheet indexes, or a
     *  simple cell range address (without sheet indexes).
     *
     * @param {Number} sheet1
     *  The zero-based index of the first sheet of the created cell range
     *  addresses.
     *
     * @param {Number} [sheet2]
     *  The zero-based index of the second sheet of the created cell range
     *  addresses. If omitted, the new ranges will refer to a single sheet.
     *
     * @returns {Range3DArray}
     *  The cell range addresses created from the passed parameters.
     */
    Range3DArray.createFromRanges = function (ranges, sheet1, sheet2) {
        return Range3DArray.map(ranges, function (range) { return Range3D.createFromRange(range, sheet1, sheet2); });
    };

    /**
     * Creates an array of cell range addresses with sheet indexes from the
     * passed column and row intervals, by combining each column interval with
     * each row interval. The sheet indexes in the resulting cell range address
     * will be ordered automatically.
     *
     * @param {IntervalArray|Interval} colIntervals
     *  The column intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter.
     *
     * @param {IntervalArray|Interval} rowIntervals
     *  The row intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter.
     *
     * @param {Number} sheet1
     *  The zero-based index of the first sheet of the created cell range
     *  addresses.
     *
     * @param {Number} [sheet2]
     *  The zero-based index of the second sheet of the created cell range
     *  addresses. If omitted, the new ranges will refer to a single sheet.
     *
     * @returns {Range3DArray}
     *  The array of cell range addresses created from the passed intervals.
     */
    Range3DArray.createFromIntervals = function (colIntervals, rowIntervals, sheet1, sheet2) {
        var result = new Range3DArray();
        IntervalArray.forEach(rowIntervals, function (rowInterval) {
            IntervalArray.forEach(colIntervals, function (colInterval) {
                result.push(Range3D.createFromIntervals(colInterval, rowInterval, sheet1, sheet2));
            });
        });
        return result;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a unique string key for this array of cell range addresses that
     * can be used for example as key in an associative map. This method is
     * faster than the method Range3DArray.toString().
     *
     * @returns {String}
     *  A unique string key for this instance.
     */
    Range3DArray.prototype.key = function () {
        return this.map(function (range) { return range.key(); }).join(' ');
    };

    /**
     * Returns the number of cells covered by all cell range address.
     *
     * @returns {Number}
     *  The number of cells covered by all cell range address.
     */
    Range3DArray.prototype.cells = function () {
        return this.reduce(function (count, range) { return count + range.cells(); }, 0);
    };

    /**
     * Returns whether all cell range addresses in this array refer to the same
     * single sheet.
     *
     * @returns {Boolean}
     *  Whether all cell range addresses in this array refer to the same single
     *  sheet.
     */
    Range3DArray.prototype.singleSheet = function () {
        if (this.empty()) { return false; }
        var sheet = this.first().sheet1;
        return this.every(function (range) { return range.isSheet(sheet); });
    };

    /**
     * Returns whether any cell range address in this array overlaps with any
     * cell range address in the passed array.
     *
     * @param {Range3DArray|Range3D} ranges
     *  The other cell range addresses to be checked. This method also accepts
     *  a single cell range address as parameter.
     *
     * @returns {Boolean}
     *  Whether any cell range address in this array overlaps with any cell
     *  range address in the passed array.
     */
    Range3DArray.prototype.overlaps = function (ranges) {
        return this.some(function (range1) {
            return Range3DArray.some(ranges, function (range2) {
                return range1.overlaps(range2);
            });
        });
    };

    /**
     * Returns the address of the bounding cell range of all range addresses in
     * this array (the smallest range that contains all ranges in the array).
     *
     * @returns {Range3D|Null}
     *  The address of the bounding cell range containing all range addresses;
     *  or null, if this range array is empty.
     */
    Range3DArray.prototype.boundary = function () {

        switch (this.length) {
            case 0:
                return null;
            case 1:
                // this method must not return original array elements
                return this[0].clone();
        }

        var result = this[0].clone();
        this.forEach(function (range) {
            result.sheet1 = Math.min(result.sheet1, range.sheet1);
            result.sheet2 = Math.max(result.sheet2, range.sheet2);
            result.start[1] = Math.min(result.start[1], range.start[1]);
            result.start[0] = Math.min(result.start[0], range.start[0]);
            result.start[1] = Math.min(result.start[1], range.start[1]);
            result.end[0] = Math.max(result.end[0], range.end[0]);
            result.end[1] = Math.max(result.end[1], range.end[1]);
        });
        return result;
    };

    /**
     * Returns a shallow copy of this cell range address array that does not
     * contain any duplicate cell range addresses.
     *
     * @returns {Range3DArray}
     *  A copy of this cell range address array without any duplicates. The
     *  array will contain the same ranges as contained in this array (no deep
     *  clones of the ranges).
     */
    Range3DArray.prototype.unify = function () {

        // nothing to do for empty array, or single range; but return a cloned array
        if (this.length <= 1) { return this.clone(); }

        // a map with the unique keys of all result ranges (prevents to search for duplicates in the result)
        var map = {};

        // remove all ranges from the result that are already contained in the map
        return this.reject(function (range) {
            var key = range.key();
            if (map[key]) { return true; }
            map[key] = true;
        });
    };

    /**
     * Returns the cell range addresses covered by this array, and the passed
     * array of cell range addresses. More precisely, returns the existing
     * intersection ranges from all pairs of the cross product of the arrays of
     * cell range addresses.
     *
     * @param {Range3DArray|Range3D} ranges
     *  The other cell range addresses to be intersected with the cell range
     *  addresses in this array. This method also accepts a single cell range
     *  address as parameter.
     *
     * @returns {Range3DArray}
     *  The cell range addresses covered by this array and the passed array.
     */
    Range3DArray.prototype.intersect = function (ranges) {
        var result = new Range3DArray();
        this.forEach(function (range1) {
            Range3DArray.forEach(ranges, function (range2) {
                var range = range1.intersect(range2);
                if (range) { result.push(range); }
            });
        });
        return result;
    };

    // exports ================================================================

    return Range3DArray;

});
