/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/errorcode', function () {

    'use strict';

    // class ErrorCode ========================================================

    /**
     * Instances of this class represent literal error codes, as used in cells,
     * in formula expressions, and as the result of formulas. Implemented as
     * named class to be able to use the 'instanceof' operator to distinguish
     * error codes from other cell values.
     *
     * @constructor
     *
     * @param {String} key
     *  A unique resource key for the error code, needed to find the correct
     *  translation.
     *
     * @param {Number} [num=NaN]
     *  The internal numeric index of built-in error codes.
     *
     * @property {String} key
     *  A unique key for the error code, as passed to the constructor.
     *
     * @property {Number} num
     *  The internal numeric index of built-in error codes, used for example
     *  when comparing error codes in filters and filtering sheet functions.
     *  Set to NaN for all error codes that are not built-in error codes.
     */
    function ErrorCode(key, num) {

        this.key = key;
        this.num = _.isNumber(num) ? num : Number.NaN;

    } // class ErrorCode

    // static methods ---------------------------------------------------------

    // all supported built-in error codes, mapped by key, with integers used e.g. for comparison in filters
    var ERROR_CODES = _.mapObject({
        NULL: 1,
        DIV0: 2,
        VALUE: 3,
        REF: 4,
        NAME: 5,
        NUM: 6,
        NA: 7,
        DATA: 8
    }, function (num, key) {
        return new ErrorCode(key, num);
    });

    /**
     * Creates a new instance of the class ErrorCode; or returns an existing
     * instance of the same error code created in a preceding call of this
     * method.
     *
     * @param {String} key
     *  A unique key for the error code.
     *
     * @returns {ErrorCode}
     *  An instance of the class ErrorCode representing the passed error code.
     */
    ErrorCode.create = function (key) {
        var errorCode = ERROR_CODES[key];
        return errorCode ? errorCode : (ERROR_CODES[key] = new ErrorCode(key));
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the JSON representation of this error code (a string with a
     * leading hash character).
     *
     * @returns {String}
     *  The JSON string representation of this error code.
     */
    ErrorCode.prototype.toJSON = function () { return '#' + this.key; };

    /**
     * Returns the string representation of this error code for debug logging.
     *
     * @attention
     *  DO NOT USE this method for GUI related code where correct locale
     *  dependent representation of the error code is required.
     *
     * @returns {String}
     *  The string representation of this error code for debug logging.
     */
    ErrorCode.prototype.toString = function () { return '#' + this.key; };

    // constants --------------------------------------------------------------

    /**
     * Predefined error codes with native names, as used in operations.
     *
     * @constant
     */
    ErrorCode.NATIVE_CODES = _.clone(ERROR_CODES);

    // add all native error codes directly to the ErrorCode function object (the property
    // NATIVE_CODES can be used to access all available codes at once, e.g. in loops)
    _.extend(ErrorCode, ERROR_CODES);

    // exports ================================================================

    return ErrorCode;

});
