/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/complexfuncs', [
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/complex'
], function (ErrorCode, Complex) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with complex
     * numbers.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, complex numbers are represented by real
     * numbers (complex number without imaginary part), or by strings looking
     * like complex numbers, e.g. '2+3i' or '-4j'. The client-side formula
     * engine provides the function signature type 'val:comp', and the class
     * Complex whose instances represent complex numbers converted from real
     * numbers, or from string representations of complex numbers.
     *
     *************************************************************************/

    // private global functions ===============================================

    /**
     * Returns a resolver function for IMSUM and IMPRODUCT that reduces all
     * complex numbers of all function parameters to their sum or product.
     *
     * @param {Function} callback
     *  The aggregation callback function. Receives two complex numbers, and
     *  must return the resulting complex number.
     *
     * @returns {Function}
     *  The resulting function implementation for IMSUM and IMPRODUCT.
     */
    function implementComplexAggregation(callback) {

        // create the resolver callback function, will be called with FormulaContext context
        return function complexAggregator() {

            // the resulting complex product
            var result = null;
            // whether any parameter was non-empty
            var hasAny = false;

            this.iterateOperands(0, function (operand) {
                this.iterateScalars(operand, function (value) {
                    var complex = this.convertToComplex(value);
                    if (result) {
                        result = callback.call(this, result, complex);
                    } else {
                        result = new Complex(complex.real, complex.imag, complex.unit);
                    }
                }, {
                    emptyParam: false, // empty parameters are skipped: IMSUM("1+i",) = "1+i"
                    complexRef: false // do not accept multi-range and multi-sheet references
                });
                if (!operand.isEmpty()) { hasAny = true; }
            });

            // only empty arguments: return #N/A instead of zero
            if (!hasAny) { throw ErrorCode.NA; }
            // only references to empty cells: return 0
            return result || new Complex(0, 0);
        };
    }

    // exports ================================================================

    return {

        COMPLEX: {
            category: 'complex',
            minParams: 2,
            maxParams: 3,
            type: 'val:comp',
            signature: 'val:num val:num val:str',
            resolve: function (real, imag, unit) {
                // if specified, unit must be lower-case 'i' or 'j'
                if (unit && (unit !== 'i') && (unit !== 'j')) { throw ErrorCode.VALUE; }
                // create a complex number (will be converted to a string)
                return new Complex(real, imag, unit);
            }
        },

        IMABS: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:comp',
            resolve: function (c) { return c.abs(); }
        },

        IMAGINARY: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:comp',
            resolve: function (c) { return c.imag; }
        },

        IMARGUMENT: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:comp',
            resolve: function (c) { return c.arg(); }
        },

        IMCONJUGATE: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.conj(); }
        },

        IMCOS: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.cos(); }
        },

        IMCOSH: {
            category: 'complex',
            name: { ooxml: '_xlfn.IMCOSH' },
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.cosh(); }
        },

        IMCOT: {
            category: 'complex',
            name: { ooxml: '_xlfn.IMCOT' },
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.cot(); }
        },

        IMCSC: {
            category: 'complex',
            name: { ooxml: '_xlfn.IMCSC' },
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.csc(); }
        },

        IMCSCH: {
            category: 'complex',
            name: { ooxml: '_xlfn.IMCSCH' },
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.csch(); }
        },

        IMDIV: {
            category: 'complex',
            minParams: 2,
            maxParams: 2,
            type: 'val:comp',
            signature: 'val:comp val:comp',
            resolve: Complex.div
        },

        IMEXP: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.exp(); }
        },

        IMLN: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.log(1); }
        },

        IMLOG10: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.log(Math.LN10); }
        },

        IMLOG2: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.log(Math.LN2); }
        },

        IMPOWER: {
            category: 'complex',
            minParams: 2,
            maxParams: 2,
            type: 'val:comp',
            signature: 'val:comp val:num',
            resolve: function (c, exp) { return c.pow(exp); }
        },

        IMPRODUCT: {
            category: 'complex',
            minParams: 1,
            type: 'val:comp',
            signature: 'any|mat:pass',
            resolve: implementComplexAggregation(Complex.mul)
        },

        IMREAL: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:comp',
            resolve: function (c) { return c.real; }
        },

        IMSEC: {
            category: 'complex',
            name: { ooxml: '_xlfn.IMSEC' },
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.sec(); }
        },

        IMSECH: {
            category: 'complex',
            name: { ooxml: '_xlfn.IMSECH' },
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.sech(); }
        },

        IMSIN: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.sin(); }
        },

        IMSINH: {
            category: 'complex',
            name: { ooxml: '_xlfn.IMSINH' },
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.sinh(); }
        },

        IMSQRT: {
            category: 'complex',
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.pow(0.5); }
        },

        IMSUB: {
            category: 'complex',
            minParams: 2,
            maxParams: 2,
            type: 'val:comp',
            signature: 'val:comp val:comp',
            resolve: Complex.sub
        },

        IMSUM: {
            category: 'complex',
            minParams: 1,
            type: 'val:comp',
            signature: 'any|mat:pass',
            resolve: implementComplexAggregation(Complex.add)
        },

        IMTAN: {
            category: 'complex',
            name: { ooxml: '_xlfn.IMTAN' },
            minParams: 1,
            maxParams: 1,
            type: 'val:comp',
            signature: 'val:comp',
            resolve: function (c) { return c.tan(); }
        }
    };

});
