/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/presenter/rtconnection', [
    'io.ox/core/event',
    'io.ox/realtime/groups',
    'io.ox/office/tk/utils'
], function (Events, RTGroups, Utils) {

    'use strict';

    var // names of internal events for the real-time connection state
        INTERNAL_EVENTS = 'online offline reset error:notMember timeout error:stanzaProcessingFailed error:joinFailed error:disposed',

        // names of runtime push messages
        PUSH_EVENTS = 'update';

    // private static functions ===============================================

    /**
     * Encodes file properties of the passed file descriptor to a resource
     * identifier that can be used to initialize a real-time connection.
     *
     * @param {Object} file
     *  A file descriptor object which contains file specific properties, like
     *  'id' and 'folder_id'.
     *
     * @returns {String}
     *  A string which references a file and complies to the resource id
     *  encoding scheme; or null, if the file descriptor doesn't describe a
     *  valid file.
     */
    function encodeFileAsResourceId(file) {
        return encodeURIComponent(file['com.openexchange.realtime.resourceID'] || file.id);
    }

    // class RTConnection =====================================================

    /**
     * Represents the connection to the real-time framework, used to send
     * message to and receive message from the server.
     *
     * Triggers the following events:
     * - 'update': Update notifications containing actions created by remote
     *  clients, and state information for the application (read-only, name and
     *  identifier of the current editor, etc.).
     *
     * @extends Events
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Number} [initOptions.realTimeDelay=100]
     *      The duration in milliseconds the realtime framework will collect
     *      messages before sending them to the server.
     */
    function RTConnection(file, initOptions) {

        var // self reference
            self = this,

            // rt channel id
            channelId = 'presenter',

            // rt group id
            groupId = 'synthetic.' + channelId + '://operations/' + encodeFileAsResourceId(file),

            // create the core real-time group instance for the document
            rtGroup = RTGroups.existsGroup(groupId) ? null : RTGroups.getGroup(groupId),

            // private event hub for internal event handling
            eventHub = null,

            // duration for the realtime framework collecting messages before sending
            realTimeDelay = Utils.getIntegerOption(initOptions, 'realTimeDelay', 100, 0),

            traceActions = {},

            // joining the group
            joining = false,

            // rejoining is possible
            rejoinPossible = false;

        // base constructor ---------------------------------------------------

        Events.extend(this);

        // private methods ----------------------------------------------------

        /**
         * Log the data
         */
        function debugLogUpdateNotification(data) {
            var // the operations array (may be missing completely)
                actions = Utils.getArrayOption(data, 'actions', []),
                // editing user id
                editUserId = Utils.getStringOption(data, 'editUserId', ''),
                // server osn
                serverOSN = Utils.getIntegerOption(data, 'serverOSN', -1),
                // has errors
                hasErrors = Utils.getBooleanOption(data, 'hasErrors', false),
                // server osn
                activeClients = Utils.getArrayOption(data, 'activeUsers', []).length,
                // write protected
                writeProtected = Utils.getBooleanOption(data, 'writeProtected', false),
                // locking user
                lockedByUser = Utils.getStringOption(data, 'lockedByUser', ''),
                // locking user id
                lockedByUserId = Utils.getIntegerOption(data, 'lockedByUserId', -1),
                // output string
                output = '';

            // output content of update notification
            output = 'editUserId: ' + editUserId + '; ';
            output += 'server osn: ' + serverOSN + '; ';
            output += 'active users: ' + activeClients + '; ';
            output += 'hasErrors: ' + hasErrors + '; ';
            output += 'writeProtected: ' + writeProtected + '; ';
            output += 'locked by user: ' + lockedByUser + '; ';
            output += 'locked by id: ' + lockedByUserId;
            Utils.log(output);

            // check that all array elements are objects
            if (actions.every(_.isObject)) {
                Utils.log('Last remote action received by RT-Update: ' + JSON.stringify(_.last(actions)));
            }
        }

        /**
         * Register events for specific RT actions.
         * The event handler 'eventName' is triggered, when
         * an RT event with the action name 'eventName' is received.
         * The triggered eventName will be equal to the 'actionName'.
         * The eventHandler will be called with the received payload data
         */
        function registerRTEvents(eventNames) {
            _.each(eventNames.split(/\s+/), function (eventName) {
                rtGroup.on('receive', function (event, stanza) {
                    _.each(stanza.getAll(channelId, eventName), function (payload) {
                        RTConnection.log('RTConnection.eventHub.trigger (' + channelId + '), ' + eventName);
                        eventHub.trigger(eventName, payload.data);
                    });
                });
            });
        }

        /**
         * Determines whether a certain action should be traced on its way
         * through the realtime backend.
         *
         * @returns {Boolean}
         *  Whether to trace the stanzas for the specified action.
         */
        function shouldTrace(action) {
            return traceActions[action.toLowerCase()];
        }

        /**
         * Send a message to the connected RT object,
         * waiting for the acknowledge to this call.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  acknowledge for this request will have arrived, or
         *  rejected after a time out otherwise.
         */
        function send(action, element, data) {

            var payloads = [{ element: 'action', data: action }],
                defer = null;

            if (_.isString(element) && !_.isUndefined(data)) {
                payloads.push({
                    namespace: channelId,
                    element: element,
                    data: data,
                    verbatim: true
                });
            }

            defer = rtGroup.send({
                element: 'message',
                payloads: payloads,
                trace: shouldTrace(action),
                bufferinterval: realTimeDelay
            });

            // Time out detection now uses the deferred to trigger an optional timeout handler
            defer.fail(function () {
                RTConnection.error('Send message with RT failed, payloads: ' + payloads.toString());
                // RT calls us even after connection, rtgroup are destroyed and references are cleared.
                if (!self.destroyed) { self.trigger('timeout'); }
            });

            return defer.promise();
        }

        /**
         * Send a message request to the connected RT object,
         * returning a promise object that will be resolved,
         * if the result data to this request has arrived.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for this request has arrived, or rejected otherwise.
         */
        function sendQuery(action, element, data) {
            var payloads = [{ element: 'action', data: action }],
                defer;

            if (_.isString(element) && !_.isUndefined(data)) {
                payloads.push({
                    namespace: channelId,
                    element: element,
                    data: data,
                    verbatim: true
                });
            }
            // Add necessary payload to provide the real-time id for
            // correct rights management on the backend
            payloads.push({
                namespace: 'office',
                element: 'rtdata',
                data: { rtid: rtGroup.getUuid(), session: ox.session },
                verbatim: true
            });

            defer = rtGroup.query({
                element: 'message',
                payloads: payloads,
                trace: shouldTrace(action),
                bufferinterval: realTimeDelay
            });

            // Time out detection now uses the deferred to trigger an optional timeout handler
            defer.fail(function () {
                RTConnection.error('Send query message with RT failed, payloads: ' + payloads.toString());
                // RT calls us even after connection, rtgroup are destroyed and references are cleared.
                if (!self.destroyed) { self.trigger('timeout'); }
            });

            return defer.promise();
        }

        /**
         * Handles the response from the RT low-level code and extract the
         * data part from, providing it as result via the deferred/promise.
         *
         * @param {jQuery.Deferred|jQuery.Promise} orgDeferred
         *  The original promise or deferred provided by the RT low-level
         *  code.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the passed original
         *  Deferred object is still pending.
         *
         * @returns {jQuery.Deferred}
         *  The Deferred object that will be resolved if the result for the
         *  original request has arrived, or rejected otherwise.
         */
        function handleResponse(orgDeferred, timeout) {

            var // the Deferred object to be returned
                def = $.Deferred();

            orgDeferred.then(function (response) {
                handleResolvedResponse(def, response);
            }, function (response) {
                handleRejectedResponse(def, response);
            });

            // prevent endless waiting
            if (_.isNumber(timeout) && (timeout > 0)) {
                _.delay(function () {
                    if (def.state() === 'pending') {
                        RTConnection.log('Timeout reached for RT request, timeout=' + timeout + 'ms.');
                        def.reject({ cause: 'timeout' });
                    }
                }, timeout);
            }

            return def;
        }

        /**
         * Handles the resolved response of the RT low-level code and
         * tries to extract the data port from it.
         *
         * @param {jQuery.Deferred} def
         *  The deferred to be resolved with the extracted data.
         *
         * @param {Object} response
         *  The original response provided by the RT low-level code.
         */
        function handleResolvedResponse(def, response) {
            var payloads = Utils.getArrayOption(response, 'payloads', []);

            if (self.destroyed) {
                // reject deferred after destroy
                def.reject({ cause: 'closing' });
            }
            if (payloads.length > 0) {
                // Extract the data and provide it
                def.resolve(payloads[0].data);
            } else {
                // Response from server is not in the expected format, therefore
                // we reject the deferred here.
                def.reject(response);
            }
        }

        /**
         * Handles the rejected response of the RT low-level code and
         * tries to extract the data port from it.
         *
         * @param {jQuery.Deferred} def
         *  The deferred to be rejected with the extracted data.
         *
         * @param {Object} response
         *  The original response provided by the RT low-level code.
         */
        function handleRejectedResponse(def, response) {
            var payloads = Utils.getArrayOption(response, 'payloads', []);

            if (payloads.length > 0) {
                // Extract data even for the rejected case
                def.reject(payloads[0].data);
            } else {
                // If we don't find a payload just provide the original response
                def.reject(response);
            }
        }

        /**
         * Handles the resolved response of the RT low-level join request
         * and makes checks for certain aspects (errors).
         *
         * @param {Object} {jQuery.Deferred} def
         *
         * @paran {Object} response
         *  The original response from a join-request provided by the RT
         *  low-level code.
         */
        function handleJoinResponse(def, response) {
            var // the payloads of the join response
                payloads = Utils.getArrayOption(response, 'payloads', []),
                // the data to resolve/reject the deferred
                data = null,
                // do we found an error
                hasError = false;

            if (self.destroyed) {
                // reject deferred after destroy
                def.reject({ cause: 'closing' });
            }

            if (payloads.length > 0) {
                // Extract the answer payload from the backend answer, look
                // for possible errors sent by RT via payloads, too. Should be
                // handled by RT-core (see #39911).
                _.find(payloads, function (payload) {
                    if (_.isObject(payload) && _.isObject(payload.data) && _.isString(payload.element)) {
                        switch (payload.element) {
                            case 'error':
                                hasError = true;
                                data = { error: payload.data };
                                break;
                            default:
                                if (!data) {
                                    data = payload.data;
                                }
                                break;
                        }
                    }
                    return hasError;
                });

                data = data || payloads[0].data; // as fallback
                if (hasError) { def.reject(data); } else { def.resolve(data); }
            } else {
                // we must receive at least one payload - otherwise we
                // assume an error
                def.reject(response);
            }
        }

        /**
         * Handles the rejected response of the RT low-level join request
         * and makes checks for certain aspects (errors). In special cases
         * a re-join is started to re-enroll the client.
         *
         * @param {Object} {jQuery.Deferred} def
         *
         * @paran {Object} response
         *  The original response from a join-request provided by the RT
         *  low-level code.
         */
        function handleRejectedJoinRequest(def, response, retryHandler) {
            var // do we found error data
                hasError = false;

            if (self.destroyed) {
                // reject deferred after destroy
                def.reject({ cause: 'closing' });
            }

            // we assume that we receive the error data as a property in the response
            if (_.isObject(response) && _.isObject(response.error)) {
                hasError = true;
            }

            // check error code to decide, if we can try to re-join
            if (hasError && response.error.code === 1007) {
                // try to re-join in case of code 1007 (STATE_MISSING)
                handleRejoin(def, response, retryHandler);
            } else {
                joining = false;
                def.reject(response);
            }
        }

        /**
         * Handles the re-join of the client, in case a re-enroll
         * is necessary.
         *
         * @param {Function} retryHandler
         *  A handler which is responsible to start a re-join of the client.
         *  Must be valid function.
         */
        function handleRejoin(def, response, retryHandler) {
            // We have to wait for possible notifications from the
            // low-level RT part to determine, if a retry is possible.
            _.delay(function () {
                // now check the states and decide, if we can try to
                // start a reconnect using a possible retry handler
                if (rejoinPossible) {
                    rejoinPossible = false;
                    retryHandler.call(this, def);
                    joining = false;
                } else {
                    handleJoinResponse(def, response);
                    joining = false;
                }
            }, 1000);
        }

        /**
         * Join a group using the provided options and set the asynchronous
         * state. An optional retryHandler is called, if there is a chance
         * to retry the joining.
         *
         * @param {Object} connectOptions
         *  Contains properties to be sent while joining to the
         *  document real-time connection.
         *
         * @param {jQuery.Deferred} def
         *  The deferred to be resolved/rejected, if joining completed/failed.
         *
         * @param {Function} retryHandler
         *  A function that is called, if a retry is possible.
         */
        function joinGroup(connectOptions, def, retryHandler) {

            joining = true;
            rtGroup.join(connectOptions).then(function (response) {
                var payloads = Utils.getArrayOption(response, 'payloads', []);

                if (self.destroyed) {
                    // reject deferred after destroy
                    def.reject({ cause: 'closing' });
                    joining = false;
                } else if (payloads.length > 0) {
                    handleJoinResponse(def, response);
                    joining = false;
                } else if (_.isFunction(retryHandler)) {
                    // We encountered an illegal state for our join request.
                    // try to re-join if handler is available
                    handleRejoin(def, response, retryHandler);
                } else {
                    // without retry handler just process the original
                    // response
                    handleJoinResponse(def, response);
                    joining = false;
                }
            }, function (response) {
                handleRejectedJoinRequest(def, response, retryHandler);
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Connects the real-time group and sends the initial request for the
         * document import operations.
         *
         * @param {Object} connectData
         *  Contains properties to be sent while joining to the
         *  document real-time connection.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  getActions results.
         */
        this.connect = function (connectData) {
            var connectDeferred = $.Deferred(),
                // optional connect data
                connectOptions = { expectWelcomeMessage: true };

            if (connectData) {
                _.extend(connectOptions, { additionalPayloads: [{ element: 'connectData', namespace: channelId, data: connectData, verbatim: true }] });
            }

            if (!rtGroup.isRTWorking()) {
                // the server doesn't support real-time communication, must be
                // incorrect configured.
                return $.Deferred().reject({ cause: 'bad server component' });
            }

            RTConnection.log('RTConnection.connect called');

            // Handler to be called, if we detected a rejoin scenario - possible
            // if a re-enroll is needed by the low-level RT framework.
            function rejoinHandler(def) {
                // call join again, but this time without retry handler
                joinGroup(connectOptions, def, null);
            }

            // joins the group and provides the retry handler
            joinGroup(connectOptions, connectDeferred, rejoinHandler);

            // Remember *connect* deferred to be able to reject them if user
            // closes document while we are still connecting to the realtime group!
            //return this.createAbortablePromise(connectDeferred);

            // TODO: handle connection abort
            return connectDeferred;
        };

        /**
         * Send the given operations to the connected RT object
         * for further distribution to other connected clients
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.sendActions = function (actions) {
            return send('applyactions', 'actions', actions);
        };

        /**
         * Sends slide data update to the server. This slide update data will
         * broadcasted to all collaborating clients.
         *
         * @param {Object} slideInfo
         *  An object containing relevant slide data.
         *
         * @returns {jQuery.Promise}
         */
        this.updateSlide = function (slideInfo) {
            RTConnection.log('RTConnection.updateSlide called', slideInfo);
            return send('updateslide', 'slideInfo', slideInfo);
        };

        /**
         * Start the presentation as presenter.
         *
         * @param {Object} slideInfo
         *  An object containing relevant data for the initial slide.
         *
         * @returns {jQuery.Promise}
         */
        this.startPresentation = function (slideInfo) {
            RTConnection.log('RTConnection.startPresentation called');
            return send('startpresentation', 'slideInfo', slideInfo);
        };

        /**
         * Pause the presentation as presenter.
         *
         * @returns {jQuery.Promise}
         */
        this.pausePresentation = function () {
            RTConnection.log('RTConnection.pausePresentation called');
            return send('pausepresentation');
        };

        /**
         * Continue the previously paused presentation as presenter.
         *
         * @returns {jQuery.Promise}
         */
        this.continuePresentation = function () {
            RTConnection.log('RTConnection.continuePresentation called');
            return send('continuepresentation');
        };

        /**
         * End the presentation as presenter.
         *
         * @returns {jQuery.Promise}
         */
        this.endPresentation = function () {
            RTConnection.log('RTConnection.endPresentation called');
            return send('endpresentation');
        };

        /**
         * Join the presentation as participant / listener.
         *
         * @returns {jQuery.Promise}
         */
        this.joinPresentation = function () {
            RTConnection.log('RTConnection.joinPresentation called');

            // try to join the presentation
            var promise = send('joinpresentation');

            // US #99684978: wait for the update message, reject with error code
            var def = $.Deferred();
            this.one('update', function (event, data) {
                if (_.isObject(data) && _.isObject(data.error) && (data.error.error === 'NO_ERROR')) {
                    def.resolve();
                } else {
                    def.reject(_.isObject(data) ? data.error : null);
                }
            });

            // return a promise that resolves/rejects after the update message
            return promise.then(_.constant(def));
        };

        /**
         * Leave the presentation as participant / listener.
         *
         * @returns {jQuery.Promise}
         */
        this.leavePresentation = function () {
            RTConnection.log('RTConnection.leavePresentation called');
            return send('leavepresentation');
        };

        /**
         * Sends a real-time request to the server and waits for a response.
         *
         * @param {String} action
         *  The identifier of the action sent to the server.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the query is still
         *  pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  answer of the server request, or rejected on error or timeout.
         */
        this.sendQuery = function (action, timeout) {
            RTConnection.log('RTConnection.sendQuery(' + action + ')');
            return handleResponse(sendQuery(action), timeout);
        };

        /**
         * Leave the connected real-time group.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  closing document status of the request.
         */
        this.close = function () {
            var leaveOptions = { expectSignOffMessage: true };

            RTConnection.log('RTConnection.close called');
            return handleResponse(rtGroup.leave(leaveOptions), 20000);
        };

        /**
         * Retrieves the universal unique id from the real-time framework.
         * This id is used to identify this client in the real-time
         * communiation with the server-side.
         * ATTENTION: If it's known that a http-request will need real-time
         * data, it's essential to add this id to the request. Otherwise
         * the server-side is not able to identify the specific client.
         *
         * @returns {String}
         *  The universal unique id of this client in the real-time
         *  environment.
         */
        this.getUuid = function () {
            return rtGroup.getUuid();
        };

        /**
         * Retrieves the universal unique real-time id from the real-time framework,
         * with the following format: ox:// <ox user id> + @ + <context id> + / + <Uuid>
         *
         * @returns {String}
         *  The universal unique real-time id of this client.
         */
        this.getRTUuid = function () {
            return 'ox://' + ox.user_id + '@' + ox.context_id + '/' + this.getUuid();
        };

        this.isInitialized = function () {
            return _.isObject(rtGroup);
        };

        // disconnect from the RT object, no further calls should be made at this object from now on
        this.dispose = function () {

            // destroy realtime group
            try {
                rtGroup.destroy();
                rtGroup = null;
                RTConnection.log('RTConnection.destroy(): RT group destroyed');
            } catch (ex) {
                Utils.exception(ex);
            }

            // destroy private event hub
            eventHub.destroy();
            eventHub = null;

            // remove references
            traceActions = null;
        };

        // initialization -----------------------------------------------------

        RTConnection.log('RTConnection initialization');

        // read traceable actions from page URL
        if (_.url.hash('office:trace')) {
            _.each(_.url.hash('office:trace').split(/\s*,\s*/), function (action) {
                traceActions[action.toLowerCase()] = true;
            });
        }

        function init() {
            // create private event hub for internal event handling
            eventHub = new Events();

            // register the internal event hub for request events and custom runtime events
            registerRTEvents(PUSH_EVENTS);

            // forward runtime realtime events to own listeners
            rtGroup.on(INTERNAL_EVENTS, function (event, data) {
                // special handling for reset event
                if (event.type === 'reset') {
                    if (joining) {
                        // If we are joining, reset is an indicator that a re-enroll
                        // has been done. In that case we have to try to join again.
                        // This must be limited to error code 1007 which means that
                        // we have to re-enroll. A reset after we received a reset with
                        // data.code 1007 is also normal, ignore that, too.
                        // See #40096, the error structure changed
                        if ((!rejoinPossible && (data && data.code === 1007)) || rejoinPossible) {
                            if (!rejoinPossible) {
                                rejoinPossible = true;
                            }
                        } else {
                            // trigger reset otherwise
                            self.trigger(event.type);
                        }
                    } else {
                        // trigger reset outside of join
                        self.trigger(event.type);
                    }
                } else {
                    self.trigger(event.type);
                }
            });

            // forward OX Documents runtime events to own listeners
            eventHub.on(PUSH_EVENTS, function (event, data) {
                if (event.type === 'update') {
                    debugLogUpdateNotification(data);
                }
                self.trigger(event.type, data);
            });
        }

        if (rtGroup) {
            init();
        }

    } // class RTConnection

    // global initialization ==================================================

    // enable logging for real-time core
    require(['io.ox/realtime/rt']).done(function (CoreRT) {
        CoreRT.debug = RTConnection.debug;
    });

    RTConnection.log = function (msg) { Utils.info('Presenter - RTConnection - log', msg); };
    RTConnection.error = function (msg) { Utils.error('Presenter - RTConnection - error', msg); };

    // exports ================================================================

    return RTConnection;

});
