/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/languagepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/editframework/utils/editconfig',
    'gettext!io.ox/office/editframework/main'
], function (Utils, LocaleData, RadioList, Config, gt) {

    'use strict';

    // locale codes of all supported languages
    var LOCALES = [
        'en_US',
        'en_GB',
        'de_DE',
        'fr_FR',
        'es_ES',
        'cs_CZ',
        'da_DK',
        'nl_NL',
        'fi_FI',
        'el_GR',
        'hu_HU',
        'it_IT',
        'pl_PL',
        'pt_PT',
        'ro_RO',
        'ru_RU',
        'sv_SE'
    ];

    // the locale of the user interface, with dash instead of underscore
    var UI_LOCALE = LocaleData.LOCALE.replace('_', '-');

    // special list item value for the entry 'None' (no language selected)
    var NONE_VALUE = 'none';

    // a promise that will resolve when the locales with dictionaries have been received
    var definitionsPromise = (function () {

        // configuration data for all supported locales
        var localeDefinitions = LOCALES.map(function (locale) {
            return {
                lc: locale, // the original locale identifier
                value: locale.replace('_', '-'), // formatting attribute value (dash instead of underscore)
                supported: false, // whether a dictionary is available, will be updated after server request)
                label: LocaleData.getLanguageName(locale) // translated name of the language
            };
        });

        return Config.getLocalesWithDictionary().then(function (supportedLocales) {
            localeDefinitions.forEach(function (definition) {
                definition.supported = _.contains(supportedLocales, definition.lc);
            });
            return localeDefinitions;
        });
    }());

    // class LanguagePicker ===================================================

    /**
     * A drop-down list with all supported languages.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the RadioList base class.
     *  Additionally, the following options are supported:
     *  @param {Boolean} [initOptions.showListIcons=true]
     *      Whether to show the check boxes in the list.
     *  @param {Boolean} [initOptions.showAllLanguages=true]
     *      If true show "Most Recently Languages", None and all supported Languages  see@ LOCALES.
     *      If false show "Most Recently Languages", None and the Language of the current
     *      selected word.
     */
    function LanguagePicker(docView, initOptions) {

        // self reference
        var self = this;

        var showListIcons = Utils.getBooleanOption(initOptions, 'showListIcons', true);
        var showAllLanguages = Utils.getBooleanOption(initOptions, 'showAllLanguages', true);
        var localeDefinitions;

        // base constructor ---------------------------------------------------

        RadioList.call(this, docView, Utils.extendOptions(initOptions, {
            tooltip: gt('Text language'),
            sortItems: true,
            // show most recently used languages in an own section
            mruSize: showAllLanguages ? 4 : 0,
            mruSettingsKey: 'languagepicker'
        }));

        // private methods ----------------------------------------------------

        function createNoneLanguageButton() {
            self.createOptionButton(NONE_VALUE, {
                section: '[None]',
                //#. A special entry used in a language selector control that means "No language selected" (e.g. no spell checking).
                label: gt.pgettext('language', 'None'),
                labelStyle: 'font-style:italic;'
            });
        }

        /**
         * Add the languages from the local definition to the menu list.
         * @param {type} [languageFilter] filter for the languages the 1param is the Local,
         *  if the filter return true the language will be added to the list.
         */
        function addLanguages(languageFilter) {

            localeDefinitions.forEach(function (definition) {

                if (!languageFilter || languageFilter(definition.value)) {
                    var options = { label: definition.label };
                    if (showListIcons) {
                        options.icon = 'docs-dict-available';
                        if (!definition.supported) {
                            options.iconStyle = 'visibility:hidden;';
                        }
                    }
                    self.createOptionButton(definition.value, options);
                }
            });
        }

        /**
         * Add languages to the Language Picker if the initOptions.showAllLanguages is true;
         */
        function addLanguagePickerLanguages() {
            createNoneLanguageButton();
            addLanguages();
        }

        /**
         * Add languages to the Language Picker if the initOptions.showAllLanguages is false;
         */
        function addReduceLanguagePickerLanguages() {

            // prepare a flag set with all languages to be shown
            var languageSet = Utils.makeSet(self.getMenu().getMRUList());
            languageSet[UI_LOCALE] = languageSet[self.getValue()] = true;

            addLanguages(function (language) { return language in languageSet; });
            createNoneLanguageButton();
        }

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('language-picker');

        this.waitForSuccess(definitionsPromise, function (localeDef) {
            localeDefinitions = localeDef;
            if (Utils.getBooleanOption(initOptions, 'showAllLanguages', true)) {
                addLanguagePickerLanguages();
            } else {
                addReduceLanguagePickerLanguages();
            }
        });

        if (!showAllLanguages) {
            // Register the update handler to add the language of the selected word to the minimized language picker
            this.registerUpdateHandler(function (newLanguage) {
                if (localeDefinitions && _.isString(newLanguage) && self.getMenu().findItemNodes(newLanguage).length === 0) {

                    addLanguages(function (language) {
                        return newLanguage === language;
                    });
                }
            });
        }

        // add an activated language to the MRU list
        this.on('group:change', function (event, languageId) {
            if (languageId !== NONE_VALUE) {
                self.getMenu().setMRUValue(languageId);
            }
        });

        // destroy all members on destruction
        this.registerDestructor(function () {
            self = docView = initOptions = null;
        });

    } // class LanguagePicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: LanguagePicker });

});
