/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Jonas Regier <jonas.regier@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/colorpicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/view/control/colorslider',
    'gettext!io.ox/office/editframework/main'
], function (Utils, Forms, RadioList, AppObjectMixin, Color, ColorSlider, gt) {

    'use strict';

    // predefined color definitions
    var BUILTIN_COLOR_DEFINITIONS = [
        { name: 'dark-red',    label: gt('Dark red'),    color: { type: 'rgb', value: 'C00000' } },
        { name: 'red',         label: gt('Red'),         color: { type: 'rgb', value: 'FF0000' } },
        { name: 'orange',      label: gt('Orange'),      color: { type: 'rgb', value: 'FFC000' } },
        { name: 'yellow',      label: gt('Yellow'),      color: { type: 'rgb', value: 'FFFF00' } },
        { name: 'light-green', label: gt('Light green'), color: { type: 'rgb', value: '92D050' } },
        { name: 'green',       label: gt('Green'),       color: { type: 'rgb', value: '00B050' } },
        { name: 'light-blue',  label: gt('Light blue'),  color: { type: 'rgb', value: '00B0F0' } },
        { name: 'blue',        label: gt('Blue'),        color: { type: 'rgb', value: '0070C0' } },
        { name: 'dark-blue',   label: gt('Dark blue'),   color: { type: 'rgb', value: '002060' } },
        { name: 'purple',      label: gt('Purple'),      color: { type: 'rgb', value: '7030A0' } }
    ];

    // definitions for the theme color table (in order as shown in the GUI color picker)
    // transformation table: 0 = pure color; negative values = darken in %; positive values = lighten in %
    var SCHEME_COLOR_DEFINITIONS = [
        { name: 'background1', label: getBackgroundColorName(1), transformations: [0,  -5, -15, -25, -35, -50], base: 0 },
        { name: 'text1',       label: getTextColorName(1),       transformations: [50, 35,  25,  15,   5,   0], base: 5 },
        { name: 'background2', label: getBackgroundColorName(2), transformations: [0, -10, -25, -50, -75, -90], base: 0 },
        { name: 'text2',       label: getTextColorName(2),       transformations: [80, 60,  40,   0, -25, -50], base: 3 },
        { name: 'accent1',     label: getAccentColorName(1),     transformations: [80, 60,  40,   0, -25, -50], base: 3 },
        { name: 'accent2',     label: getAccentColorName(2),     transformations: [80, 60,  40,   0, -25, -50], base: 3 },
        { name: 'accent3',     label: getAccentColorName(3),     transformations: [80, 60,  40,   0, -25, -50], base: 3 },
        { name: 'accent4',     label: getAccentColorName(4),     transformations: [80, 60,  40,   0, -25, -50], base: 3 },
        { name: 'accent5',     label: getAccentColorName(5),     transformations: [80, 60,  40,   0, -25, -50], base: 3 },
        { name: 'accent6',     label: getAccentColorName(6),     transformations: [80, 60,  40,   0, -25, -50], base: 3 }
    ];

    // whether to show a color slider for scheme colors
    var SHOW_COLOR_SLIDER = Utils.SMALL_DEVICE;

    // the default handle position in the color slider
    var COLOR_SLIDER_DEFAULT_INDEX = 2;

    // private global functions ===============================================

    /**
     * Returns the localized name of a text color in a color scheme.
     */
    function getTextColorName(index) {
        //#. The name of a text color in a color scheme (a color scheme consists
        //#. of two text colors, two background colors, and six accent colors).
        //#. Example result: "Text 1", "Text 2"
        //#. %1$d is the index of the text color
        //#, c-format
        return gt('Text %1$d', _.noI18n(index));
    }

    /**
     * Returns the localized name of a background color in a color scheme.
     */
    function getBackgroundColorName(index) {
        //#. The name of a background color in a color scheme (a color scheme consists
        //#. of two text colors, two background colors, and six accent colors).
        //#. Example result: "Background 1", "Background 2"
        //#. %1$d is the index of the background color
        //#, c-format
        return gt('Background %1$d', _.noI18n(index));
    }

    /**
     * Returns the localized name of an accented color in a color scheme.
     */
    function getAccentColorName(index) {
        //#. The name of an accent color in a color scheme (a color scheme consists
        //#. of two text colors, two background colors, and six accent colors).
        //#. Example result: "Accent 1", "Accent 2"
        //#. %1$d is the index of the accent color
        //#, c-format
        return gt('Accent %1$d', _.noI18n(index));
    }

    // class ColorPicker ======================================================

    /**
     * Creates a control with a drop-down menu used to choose a color from a
     * set of color items. Shows a selection of standard colors, and a table of
     * shaded scheme colors from the current document theme. It has two display
     * modes. One used on small devices with a colorslider to browse
     * through the scheme colors and a mode for all other devices, which has a
     * color table to display all scheme colors.
     *
     * @constructor
     *
     * @extends RadioList
     * @extends AppObjectMixin
     *
     * @param {EditView} docView
     *  The document view instance containing this control.
     *
     * @param {String|Null} colorType
     *  The color type used to resolve the automatic color entry. See method
     *  Color.resolve() for a list of supported color types. If set to null,
     *  the button for the automatic color will not be created at all.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the RadioList base class.
     */
    function ColorPicker(docView, colorType, initOptions) {

        // self reference
        var self = this;

        // the document model
        var docModel = docView.getDocModel();

        // the theme model that has been used to create the scheme color table
        var themeModel = docModel.getThemeModel();

        // the color slider control shown in the drop-down menu
        var colorSlider = null;

        // the color box in the drop-down button
        var colorBox = $('<div class="color-preview-box">');

        // the replacement color for the automatic color entry
        var autoColor = new Color('auto');

        // whether the color table needs to be refreshed
        var dirty = true;

        // base constructors --------------------------------------------------

        RadioList.call(this, docView, Utils.extendOptions({
            itemDesign: 'grid',
            gridColumns: SHOW_COLOR_SLIDER ? SCHEME_COLOR_DEFINITIONS[0].transformations.length : SCHEME_COLOR_DEFINITIONS.length,
            updateCaptionMode: 'none',
            itemMatcher: sameColors,
            dropDownVersion: { label: Utils.getStringOption(initOptions, 'tooltip') },
            isRootAnchoredMenu: Utils.SMALL_DEVICE
        }, initOptions));

        AppObjectMixin.call(this, docView.getApp());

        // private methods ----------------------------------------------------

        /**
         * Resolves the passed JSON color according to the current theme target
         * chain.
         *
         * @param {Object} jsonColor
         *  The JSON color to be resolved.
         *
         * @returns {ColorDescriptor}
         *  The descriptor for the resolved color.
         */
        function resolveColor(jsonColor) {
            var auto = (colorType && autoColor.isAuto()) ? colorType : autoColor;
            return Color.parseJSON(jsonColor).resolve(auto, themeModel);
        }

        /**
         * Sets the passed JSON color as background color at the specified DOM
         * color box node.
         *
         * @param {jQuery} colorBox
         *  The DOM node of a color box, as jQuery collection.
         *
         * @param {Object} jsonColor
         *  The JSON representation of the color to be set as background color,
         *  as used in document operations.
         */
        function setBackgroundColor(boxNode, jsonColor) {
            boxNode.css('background-color', resolveColor(jsonColor).css);
        }

        /**
         * Initializes a new color item button.
         */
        function createItemHandler(event, buttonNode, jsonColor) {
            var boxNode = $('<span class="color-button">');
            setBackgroundColor(boxNode, jsonColor);
            buttonNode.prepend(boxNode);
        }

        /**
         * Updates the color box in the drop-down menu button.
         */
        function updateHandler(jsonColor) {
            setBackgroundColor(colorBox, jsonColor);
        }

        /**
         * Returns whether the passed color objects are considered being equal.
         */
        function sameColors(color1, color2) {

            // quick check: same reference
            if (color1 === color2) { return true; }

            // both colors must be objects
            if (!_.isObject(color1) || !_.isObject(color2)) { return false; }

            var color1Value = _.isString(color1.value) ? color1.value.toLowerCase() : null;
            var color2Value = _.isString(color2.value) ? color2.value.toLowerCase() : null;

            // returns whether type and value are considered being equal
            function sameTypeAndValue() {

                // exactly the same type
                if (color1.type !== color2.type) { return false; }

                // exactly the same value (not for auto colors)
                if ((color1.type === 'auto') || (color1Value === color2Value)) { return true; }

                // scheme colors: match 'textN' and 'darkN', as well as 'backgroundN' and 'lightN'
                // TODO: really always, also in dark color schemes where background would be dark?
                return (color1.type === 'scheme') && (color2.type === 'scheme') && (
                    ((color1Value === 'text1') && (color2Value === 'dark1')) ||
                    ((color1Value === 'dark1') && (color2Value === 'text1')) ||
                    ((color1Value === 'text2') && (color2Value === 'dark2')) ||
                    ((color1Value === 'dark2') && (color2Value === 'text2')) ||
                    ((color1Value === 'background1') && (color2Value === 'light1')) ||
                    ((color1Value === 'light1') && (color2Value === 'background1')) ||
                    ((color1Value === 'background2') && (color2Value === 'light2')) ||
                    ((color1Value === 'light2') && (color2Value === 'background2')));
            }

            // returns whether transformations are considered being equal
            function sameTransformations() {

                // transformations may be missing in both colors
                if (_.isEmpty(color1.transformations) && _.isEmpty(color2.transformations)) { return true; }

                // arrays must be equally sized, and transformations must be similar enough (bug 30085)
                return _.isArray(color1.transformations) && _.isArray(color2.transformations) &&
                    (color1.transformations.length === color2.transformations.length) &&
                    _.all(color1.transformations, function (tf1, index) {
                        var tf2 = color2.transformations[index];
                        return _.isString(tf1.type) && (tf1.type === tf2.type) &&
                            _.isNumber(tf1.value) && _.isNumber(tf2.value) &&
                            // may not be completely equal (bug 30085)
                            (Math.abs(tf1.value - tf2.value) <= 256);
                    });
            }

            // same result color but create with different transformations
            function sameHSL() {
                var hsl1 = resolveColor(color1).hsl;
                var hsl2 = resolveColor(color2).hsl;
                var dist = Math.abs(hsl1.h - hsl2.h) + Math.abs(hsl1.s - hsl2.s) + Math.abs(hsl1.l - hsl2.l); // ignoring alpha channel, #48333
                return dist < 0.002;
            }

            // compare colors directly
            if (sameTypeAndValue() && (sameTransformations() || sameHSL())) {
                return true;
            }

            // special treatment for fall-back values in ODF mode
            return docView.getApp().isODF() && (
                ((color1.type === 'scheme') && _.isString(color1.fallbackValue) && (color1.fallbackValue.toLowerCase() === color2Value)) ||
                ((color2.type === 'scheme') && _.isString(color2.fallbackValue) && (color2.fallbackValue.toLowerCase() === color1Value))
            );
        }

        /**
         * Returns all row nodes containing color boxes for scheme colors.
         *
         * @returns {jQuery}
         *  All row nodes containing color boxes for scheme colors.
         */
        function getSchemeColorRowNodes() {
            return self.getMenu().getSectionNode('theme').find('.grid-row');
        }

        function showSchemeColorRow(rowIndex) {
            getSchemeColorRowNodes().css('display', 'none').eq(rowIndex).css('display', 'block');
        }

        /**
         * Creates or recreates the color slider control.
         */
        function initializeColorSlider() {

            if (colorSlider) { colorSlider.destroy(); }

            // find the item nodes with the scheme base colors
            var baseColorItems = self.getMenu().getItemNodes().filter('[data-scheme-base-color]');
            // extract the RGB color values from the nodes
            var hexColors = baseColorItems.get().map(function (itemNode) {
                return Forms.getButtonValue(itemNode).fallbackValue;
            });

            // create and insert the color slider control
            colorSlider = new ColorSlider(docView, hexColors, COLOR_SLIDER_DEFAULT_INDEX);
            self.getMenu().appendContentNodes(colorSlider.getNode());

            // unhide the row for the default position
            showSchemeColorRow(COLOR_SLIDER_DEFAULT_INDEX);

            // show the correct grid row according to the active color in the slider control
            self.listenTo(colorSlider, 'change:color', function (event, index) {
                showSchemeColorRow(index);
            });
        }

        /**
         * Inserts all available colors into the drop-down menu.
         */
        function initializeColorTable() {

            // add automatic color
            function addAutoColor() {
                if (!_.isString(colorType)) { return; }

                var label = (resolveColor(Color.AUTO).a === 0) ?
                    /*#. no fill color, transparent */ gt('No color') :
                    /*#. automatic text color (white on dark backgrounds, otherwise black) */ gt('Automatic color');

                self.createMenuSection('auto', { gridColumns: 1 });
                self.createOptionButton(Color.AUTO, { section: 'auto', label: label, dataValue: 'auto' });
            }

            function createSchemeColor(definition, index) {

                // the signed percentage
                var percent = definition.transformations[index];
                // the theme color object
                var color = new Color('scheme', definition.name);
                // the data value added at the button
                var dataValue = definition.name;
                // description of the color (tool tip)
                var label = definition.label;

                if (percent > 0) {
                    color.transform('lumMod', (100 - percent) * 1000).transform('lumOff', percent * 1000);
                    dataValue += '-lighter' + percent;
                    //#. The full name of a light theme color (a base color lightened by a specific percentage value)
                    //#. Example result: "Green, lighter 20%"
                    //#. %1$s is the name of the base color
                    //#. %2$d is the percentage value, followed by a literal percent sign
                    //#, c-format
                    label = gt('%1$s, lighter %2$d%', label, _.noI18n(percent));
                } else if (percent < 0) {
                    percent = Math.abs(percent);
                    color.transform('lumMod', (100 - percent) * 1000);
                    dataValue += '-darker' + percent;
                    //#. The full name of a dark theme color (a base color darkened by a specific percentage value)
                    //#. Example result: "Green, darker 20%"
                    //#. %1$s is the name of the base color
                    //#. %2$d is the percentage value, followed by a literal percent sign
                    //#, c-format
                    label = gt('%1$s, darker %2$d%', label, _.noI18n(percent));
                }

                // create the JSON color, add the fall-back value for documents without theme support
                var jsonColor = color.toJSON();
                jsonColor.fallbackValue = resolveColor(jsonColor).hex;

                // create the menu option button
                var attributes = (index === definition.base) ? { 'data-scheme-base-color': true } : null;
                self.createOptionButton(jsonColor, { section: 'theme', tooltip: label, dataValue: dataValue, attributes: attributes });
                return jsonColor;
            }

            function addSchemeColors() {

                // create the menu section for the scheme colors
                self.createMenuSection('theme', { label: gt('Theme Colors') });

                // the number of shaded colors per scheme base color
                var shadeCount = SCHEME_COLOR_DEFINITIONS[0].transformations.length;

                if (SHOW_COLOR_SLIDER) {
                    SCHEME_COLOR_DEFINITIONS.forEach(function (definition) {
                        _.times(shadeCount, function (index) {
                            createSchemeColor(definition, index);
                        });
                    });
                    initializeColorSlider();
                } else {
                    _.times(shadeCount, function (index) {
                        SCHEME_COLOR_DEFINITIONS.forEach(function (definition) {
                            createSchemeColor(definition, index);
                        });
                    });
                }
            }

            function addStandardColors() {
                self.createMenuSection('standard', { label: gt('Standard Colors') });
                BUILTIN_COLOR_DEFINITIONS.forEach(function (definition) {
                    self.createOptionButton(definition.color, { section: 'standard', tooltip: definition.label, dataValue: definition.name });
                });
            }

            var newThemeModel = docModel.getThemeModel();
            dirty = dirty || (themeModel !== newThemeModel);
            themeModel = newThemeModel;
            if (!dirty) { return; }

            // clear the menu, but do not remove the title node
            self.clearMenu('>:not(.title)');

            // section order according to the device type
            if (SHOW_COLOR_SLIDER) {
                addAutoColor();
                addStandardColors();
                addSchemeColors();
            } else {
                addAutoColor();
                addSchemeColors();
                addStandardColors();
            }

            dirty = false;
        }

        // public methods -----------------------------------------------------

        /**
         * Sets the specified color to render the color box in the menu list
         * item representing the automatic color, and repaints that list item.
         *
         * @param {Color} color
         *  The new replacement color for the automatic color entry.
         *
         * @returns {ColorPicker}
         *  A reference to this instance.
         */
        this.setAutoColor = function (color) {
            autoColor = color;
            setBackgroundColor(this.getMenu().findItemNodes(Color.AUTO).find('.color-button'), Color.AUTO);
            return this;
        };

        // initialization -----------------------------------------------------

        // add marker class for additional CSS formatting
        this.getMenuNode().addClass('color-picker');

        // add the color preview box to the menu button
        this.getMenuButton().append(colorBox);

        // register an update handler that updates the color box
        this.registerUpdateHandler(updateHandler);

        // register a handler that inserts a color box into each list item
        this.getMenu().on('create:item', createItemHandler);

        // repaint the color table if the theme target chain has changed
        this.getMenu().on('popup:beforeshow', initializeColorTable);

        // refresh color buttons in the drop-down menu after theme has changed
        this.listenTo(docModel.getThemeCollection(), 'triggered', function () { dirty = true; });

        // initialize the color slider control
        if (SHOW_COLOR_SLIDER) {

            // update the handle position in the color slider
            this.getMenu().on('popup:beforeshow', function () {

                var jsonColor = self.getValue();
                var index = Utils.findFirstIndex(getSchemeColorRowNodes().get(), function (rowNode) {
                    var buttonNodes = $(rowNode).find(Forms.BUTTON_SELECTOR);
                    return buttonNodes.get().some(function (buttonNode) {
                        return sameColors(jsonColor, Forms.getButtonValue(buttonNode));
                    });
                });

                colorSlider.moveHandle((index >= 0) ? index : COLOR_SLIDER_DEFAULT_INDEX);
            });
        }

        // destroy all members on destruction
        this.registerDestructor(function () {
            if (colorSlider) { colorSlider.destroy(); }
            self = docView = docModel = themeModel = colorBox = colorSlider = null;
        });

    } // class ColorPicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: ColorPicker });

});
