/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/model/drawingmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/model/embeddedcollection'
], function (Utils, Iterator, Rectangle, AttributedModel, DrawingUtils, EmbeddedCollection) {

    'use strict';

    // mathematical constants
    var PI_180 = Math.PI / 180;

    // private global functions ===============================================

    function getEffectiveAngle(aRad, flipH, revRot) {
        return (revRot ? -aRad : aRad) + (flipH ? Math.PI : 0);
    }

    function resolveAbsoluteLocation(model) {

        // the location of the current drawing object relative to its parent
        var rect = model.getRectangleHmm();
        // flipping flags of the drawing model
        var flipH = model.isFlippedH();
        var flipV = model.isFlippedV();
        // effective rotation of the positive X axis, according to flipping
        var aRad = 0;

        // calculate the location of embedded drawing model according to its parent
        var parentModel = model.getParentModel();
        if (parentModel) {

            // resolve all positioning data of the parent model recursively
            var parentData = resolveAbsoluteLocation(parentModel);

            // the unrotated/unflipped absolute center point of the drawing model
            var cx = parentData.rect.left + rect.centerX();
            var cy = parentData.rect.top + rect.centerY();
            // polar coordinates of the unrotated/unflipped absolute center point
            var polar = parentData.rect.pointToPolar(cx, cy);
            // effective absolute angle of the polar center point, according to parent rotation/flipping
            var revRot = parentData.flipH !== parentData.flipV;
            polar.a = parentData.aRad + (revRot ? -polar.a : polar.a);
            // effective absolute position of the center point
            var center = parentData.rect.polarToPoint(polar);
            // update the effective absolute location of the drawing model
            rect.left = center.x - rect.width / 2;
            rect.top = center.y - rect.height / 2;

            // effective flipping and rotation of the drawing model
            flipH = parentData.flipH !== flipH;
            flipV = parentData.flipV !== flipV;
            aRad = parentData.aRad + getEffectiveAngle(model.getRotationRad(), model.isFlippedH(), flipH !== flipV);

        } else {

            // effective rotation of top-level drawings
            aRad = getEffectiveAngle(model.getRotationRad(), flipH, flipH !== flipV);
        }

        return { rect: rect, flipH: flipH, flipV: flipV, aRad: aRad };
    }

    // class DrawingModel =====================================================

    /**
     * The model of a drawing object.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {DrawingCollection} parentCollection
     *  The parent drawing collection that will contain this drawing object.
     *
     * @param {String} drawingType
     *  The type of this drawing object.
     *
     * @param {Object|Null} initAttributes
     *  An attribute set with initial formatting attributes for the drawing
     *  object.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  - {String} [initOptions.families]
     *      Additional explicit attribute families supported by this drawing
     *      model object (space-separated). The main attribute family 'drawing'
     *      supported by all drawing objects will be registered implicitly, and
     *      does not have to be added here.
     *  - {Boolean} [initOptions.children=false]
     *      If set to true, this drawing model will contain an embedded drawing
     *      model collection, and it will be possible to insert drawing objects
     *      into this model.
     *  - {Boolean} [initOptions.rotatable=false]
     *      If set to true, this drawing model can be rotated and flipped, i.e.
     *      the attributes 'rotation', 'flipH', and 'flipV' will have an effect
     *      when rendering this drawing object.
     */
    var DrawingModel = AttributedModel.extend({ constructor: function (parentCollection, drawingType, initAttributes, initOptions) {

        // additional attribute families supported by this drawing model
        var attrFamilies = Utils.getStringOption(initOptions, 'families', null);

        // the embedded drawing collection, e.g. of group objects
        var childCollection = null;

        // whether this model can be rotated and flipped
        var rotatable = Utils.getBooleanOption(initOptions, 'rotatable', false);

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, parentCollection.getDocModel(), initAttributes, {
            styleFamily: 'drawing',
            families: 'drawing' + (attrFamilies ? (' ' + attrFamilies) : ''),
            addStyleFamilies: false // attribute families need to be specified individually by drawing type
        });

        // protected methods --------------------------------------------------

        /**
         * Handler for the document operation 'setDrawingAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setDrawingAttributes' operation.
         */
        this.applyChangeOperation = function (context) {
            this.setAttributes(context.getObj('attrs'));
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the type of this drawing model, as specified by the document
         * operation that has created the drawing model.
         *
         * @returns {String}
         *  The type of this drawing model.
         */
        this.getType = function () {
            return drawingType;
        };

        /**
         * Returns whether this drawing model can be restored completely by
         * document operations after deletion. This method may be overwritten
         * by subclasses.
         *
         * @returns {Boolean}
         *  Whether this drawing model can be restored completely by document
         *  operations after deletion.
         */
        this.isRestorable = function () {
            return drawingType !== 'undefined';
        };

        /**
         * Returns whether this drawing model is a line/connector shape (i.e.
         * it will be selected in two-point mode instead of the standard
         * rectangle mode).
         *
         * @returns {Boolean}
         *  Whether this drawing model is a line/connector shape.
         */
        this.isTwoPointShape = function () {
            return DrawingUtils.isTwoPointShape(drawingType, this.getExplicitAttributeSet(true).geometry);
        };

        /**
         * Returns whether this drawing model is in visible state (the 'hidden'
         * attribute is not set to true).
         *
         * @returns {Boolean}
         *  Whether this drawing model is in visible state.
         */
        this.isVisible = function () {
            return !this.getMergedAttributeSet(true).drawing.hidden;
        };

        /**
         * Returns whether this drawing object keeps its width/height ratio
         * during resizing (i.e. the value of the formatting attribute
         * 'aspectLocked').
         *
         * @returns {Boolean}
         *  The value of the formatting attribute 'aspectLocked'.
         */
        this.isAspectLocked = function () {
            return this.getMergedAttributeSet(true).drawing.aspectLocked;
        };

        /**
         * Returns whether this drawing object has been flipped horizontally
         * (i.e. the value of the formatting attribute 'flipH').
         *
         * @returns {Boolean}
         *  The value of the formatting attribute 'flipH'.
         */
        this.isFlippedH = function () {
            return rotatable && this.getMergedAttributeSet(true).drawing.flipH;
        };

        /**
         * Returns whether this drawing model has been flipped vertically (i.e.
         * the value of the formatting attribute 'flipV').
         *
         * @returns {Boolean}
         *  The value of the formatting attribute 'flipV'.
         */
        this.isFlippedV = function () {
            return rotatable && this.getMergedAttributeSet(true).drawing.flipV;
        };

        /**
         * Returns the value of the formatting attribute 'rotation' (i.e. the
         * rotation angle in degrees).
         *
         * @returns {Boolean}
         *  The value of the formatting attribute 'rotation'.
         */
        this.getRotationDeg = function () {
            return rotatable ? this.getMergedAttributeSet(true).drawing.rotation : 0;
        };

        /**
         * Returns the rotation angle of this drawing mdoel in radians (i.e.
         * the value of the formatting attribute 'rotation').
         *
         * @returns {Number}
         *  The value of the formatting attribute 'rotation'.
         */
        this.getRotationRad = function () {
            return this.getRotationDeg() * PI_180;
        };

        /**
         * Returns whether this drawing model is embedded in another drawing
         * model (e.g. a group object).
         *
         * @returns {Boolean}
         *  Whether this drawing model is embedded in another drawing model.
         */
        this.isEmbedded = function () {
            return parentCollection.getParentModel() !== null;
        };

        /**
         * Returns the parent drawing model owning this instance as embedded
         * drawing model.
         *
         * @returns {DrawingModel|Null}
         *  The parent drawing model owning this instance as embedded drawing
         *  model; or null, if this instance is contained in a root collection
         *  without parent drawing model.
         */
        this.getParentModel = function () {
            return parentCollection.getParentModel();
        };

        /**
         * Returns the parent drawing collection that contains this drawing
         * object model.
         *
         * @returns {DrawingCollection}
         *  The parent drawing collection that contains this drawing model.
         */
        this.getParentCollection = function () {
            return parentCollection;
        };

        /**
         * Returns the ancestor top-level drawing model containing this
         * instance as embedded drawing model.
         *
         * @returns {DrawingModel}
         *  The ancestor top-level drawing model containing this instance as
         *  embedded drawing model; or a reference to this drawing model, if it
         *  is a top-level drawing model by itself.
         */
        this.getRootModel = function () {
            var parentModel = this.getParentModel();
            return parentModel ? parentModel.getRootModel() : this;
        };

        /**
         * Returns the ancestor top-level drawing collection that contains all
         * top-level drawing models.
         *
         * @returns {DrawingCollection}
         *  The ancestor top-level drawing collection that contains all
         *  top-level drawing models; or a reference to this instance, if it is
         *  a top-level drawing collection by itself.
         */
        this.getRootCollection = function () {
            return parentCollection.getRootCollection();
        };

        /**
         * Returns the collection of child drawing objects contained in this
         * drawing object model.
         *
         * @returns {DrawingCollection|Null}
         *  The child drawing collection contained in this instance; or null,
         *  if the drawing object does not contain a child collection.
         */
        this.getChildCollection = function () {
            return childCollection;
        };

        /**
         * Invokes the passed callback function, if and only if this drawing
         * model contains a child drawing collection.
         *
         * @param {Function} callback
         *  The callback function that will be invoked if this drawing model
         *  contains a child drawing collection. Receives the child drawing
         *  collection as first parameter.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {Any}
         *  The return value of the callback function if it has been invoked;
         *  otherwise null.
         */
        this.withChildCollection = function (callback, context) {
            return childCollection ? callback.call(context, childCollection) : null;
        };

        /**
         * Creates an iterator that visits all drawing models embedded in the
         * child collection of this drawing model.
         *
         * @param {Object} [options]
         *  Optional parameters. See IndexedCollection.createModelIterator()
         *  for details.
         *
         * @returns {Iterator}
         *  The iterator returned from IndexedCollection.createModelIterator(),
         *  or an empty iterator, if this drawing model does not contain a
         *  child collection.
         */
        this.createChildModelIterator = function (options) {
            return childCollection ? childCollection.createModelIterator(options) : Iterator.EMPTY;
        };

        /**
         * Invokes the passed callback function for all direct child models
         * that are contained in the child drawing collection of this drawing
         * model.
         *
         * @param {Function} callback
         *  The callback function that will be invoked for each child model
         *  contained in the child drawing collection of this drawing model.
         *  Receives the following parameters:
         *  (1) {DrawingModel} childModel
         *      The child drawing model currently visited.
         *  (2) {Number} childIndex
         *      The array index of the visited child drawing model in the child
         *      collection of this drawing model.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {DrawingModel}
         *  A reference to this instance.
         */
        this.forEachChildModel = function (callback, context) {
            Iterator.forEach(this.createChildModelIterator(), function (childModel, iterResult) {
                callback.call(context, childModel, iterResult.index);
            });
            return this;
        };

        /**
         * Returns the document position of this drawing model.
         *
         * @returns {Array<Number>|Null}
         *  The document position of the drawing object, if it is contained in
         *  the drawing collection; or null on any error.
         */
        this.getPosition = function () {

            // get the position of this drawing model in its parent collection
            var position = parentCollection.implResolvePosition(this);
            if (!position) { return null; }

            // add the position of all parent drawing models, if this is an embedded model
            var parentModel = parentCollection.getParentModel();
            if (parentModel) {
                var parentPosition = parentModel.getPosition();
                if (!parentPosition) { return null; }
                position = parentPosition.concat(position);
            }

            return position;
        };

        /**
         * Returns the own array index of this drawing model in the list of all
         * its siblings. This index is used to define the rendering order ("Z
         * order").
         *
         * @returns {Number|Null}
         *  The own index in the list of the siblings of this drawing model; or
         *  null on any error.
         */
        this.getIndex = function () {
            return parentCollection.implResolveIndex(this);
        };

        /**
         * Returns the location of this drawing model relative to its parent
         * area, in 1/100 of millimeters, regardless of its current flipping
         * and rotation attributes.
         *
         * @returns {Rectangle}
         *  The location of this drawing model in its parent area.
         */
        this.getRectangleHmm = function () {

            // the raw position from the drawing attributes of this drawing model
            var rectangle = Rectangle.from(this.getMergedAttributeSet(true).drawing);
            // resolve location of an embedded drawing object inside its parent
            var parentModel = this.getParentModel();
            // directly resolve location of top-level drawing objects from the attributes
            if (!parentModel) { return rectangle; }

            // the location of the parent drawing model, in 1/100 mm
            var parentRectHmm = parentModel.getRectangleHmm();
            // the drawing attributes of the parent drawing model
            var parentAttrs = parentModel.getMergedAttributeSet(true).drawing;

            // move relative to parent coordinates
            rectangle.translateSelf(-parentAttrs.childLeft, -parentAttrs.childTop);

            // scale by parent coordinate system
            var widthRatio = (parentAttrs.childWidth > 0) ? (parentRectHmm.width / parentAttrs.childWidth) : 0;
            var heightRatio = (parentAttrs.childHeight > 0) ? (parentRectHmm.height / parentAttrs.childHeight) : 0;
            rectangle.scaleSelf(widthRatio, heightRatio).roundSelf();

            return rectangle;
        };

        /**
         * Returns whether this drawing model and all its embedded descendant
         * drawing models can be completely restored by document operations
         * after deletion.
         *
         * @returns {Boolean}
         *  Whether this drawing model and all its embedded descendants can be
         *  completely restored by document operations after deletion.
         */
        this.isDeepRestorable = function () {

            // first check this drawing model
            if (!this.isRestorable()) { return false; }

            // check whether all embedded drawing models are supported
            return Iterator.every(this.createChildModelIterator(), function (model) {
                return model.isDeepRestorable();
            });
        };

        /**
         * Returns whether this drawing model and all its embedded descendant
         * drawing models are rotatable.
         *
         * @returns {Boolean}
         *  Whether this drawing model and all its embedded descendants are
         *  rotatable.
         */
        this.isDeepRotatable = function () {

            // this model must be rotatable
            if (!rotatable) { return false; }

            // all embedded descendants must be rotatable too
            return Iterator.every(this.createChildModelIterator(), function (model) {
                return model.isDeepRotatable();
            });
        };

        /**
         * Returns whether this drawing model is effectively visible (i.e. the
         * drawing model itself, and all of its parent models, are visible).
         *
         * @returns {Boolean}
         *  Returns whether this drawing model is effectively visible.
         */
        this.isEffectivelyVisible = function () {
            for (var model = this; model; model = model.getParentModel()) {
                if (!model.isVisible()) { return false; }
            }
            return true;
        };

        /**
         * Returns the effective flipping settings of this drawing model,
         * acording to the flipping attributes of this drawing model and all
         * its ancestors.
         *
         * @returns {Object}
         *  A result descriptor with the following properties:
         *  - {Boolean} flipH
         *      Whether this drawing model is effectively flipped horizontally
         *      (i.e. whether the number of 'flipH' attributes of this drawing
         *      model and all its ancestor models is an odd number).
         *  - {Boolean} flipV
         *      Whether this drawing model is effectively flipped vertically
         *      (i.e. whether the number of 'flipV' attributes of this drawing
         *      model and all its ancestor models is an odd number).
         *  - {Boolean} reverseRot
         *      Whether the rotation angle appears reversed, i.e. whether only
         *      exactly one of the effective flipping flags is active.
         */
        this.getEffectiveFlipping = function () {

            // the current flipping states
            var flipH = false;
            var flipV = false;

            // process this drawing model, and the entire chain of ancestors
            for (var model = this; model; model = model.getParentModel()) {
                flipH = flipH !== model.isFlippedH();
                flipV = flipV !== model.isFlippedV();
            }

            // rotation is reversed if exactly one flipping flag is active
            return { flipH: flipH, flipV: flipV, reverseRot: flipH !== flipV };
        };

        /**
         * Returns the effective absolute location of this drawing model in the
         * root area of all top-level drawings, in 1/100 of millimeters. The
         * rotation angle of the drawing object and all of its ancestor group
         * objects will be taken into account (this method will return the
         * bounding rectangle of the rotated drawing at its effective position,
         * according to the rotation of the parent groups). The flipping
         * attributes of group objects will be determined when calculating the
         * absolute position of their embedded children.
         *
         * @returns {Rectangle}
         *  The effective absolute location of this drawing model in the root
         *  area of all top-level drawings (the bounding box, if the rectangle
         *  is effectively rotated).
         */
        this.getEffectiveRectangleHmm = function () {

            // reesolve the effective location recursively through the chain of ancestors
            var locationData = resolveAbsoluteLocation(this);

            // expand to the bounding box of the effectively rotated drawing model
            return locationData.rect.rotatedBoundingBox(locationData.aRad).roundSelf();
        };

        // initialization -----------------------------------------------------

        // create the embedded drawing collection
        if (Utils.getBooleanOption(initOptions, 'children', false)) {
            childCollection = new EmbeddedCollection(this);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            if (childCollection) { childCollection.destroy(); }
            parentCollection = initAttributes = childCollection = null;
        });

    } }); // class DrawingModel

    // exports ================================================================

    return DrawingModel;

});
