/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/scalarset'
], function (SheetHelper, ScalarSet) {

    'use strict';

    // class ScalarSet ========================================================

    describe('Spreadsheet class ScalarSet', function () {

        it('should exist', function () {
            expect(ScalarSet).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var ErrorCode = SheetHelper.ErrorCode;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a scalar set', function () {
                var set = new ScalarSet();
                expect(set).to.be.an.instanceof(ScalarSet);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "key"', function () {
            it('should exist', function () {
                expect(ScalarSet).to.respondTo('key');
            });
            it('should return equal keys for equal values', function () {
                var set = new ScalarSet();
                expect(set.key(null)).to.equal(set.key(null));
                expect(set.key(0)).to.equal(set.key(0));
                expect(set.key(42)).to.equal(set.key(42));
                expect(set.key('')).to.equal(set.key(''));
                expect(set.key('42')).to.equal(set.key('42'));
                expect(set.key('abc')).to.equal(set.key('abc'));
                expect(set.key(false)).to.equal(set.key(false));
                expect(set.key(ErrorCode.DIV0)).to.equal(set.key(ErrorCode.DIV0));
            });
            it('should ignore string case unless explicitly disabled', function () {
                var set1 = new ScalarSet();
                expect(set1.key('abc') === set1.key('ABC')).to.equal(true);
                var set2 = new ScalarSet({ withCase: true });
                expect(set2.key('abc') === set2.key('ABC')).to.equal(false);
            });
            it('should return different keys for different values', function () {
                var set = new ScalarSet();
                expect(set.key(0) === set.key(null)).to.equal(false);
                expect(set.key(0) === set.key(42)).to.equal(false);
                expect(set.key('') === set.key(null)).to.equal(false);
                expect(set.key('') === set.key(0)).to.equal(false);
                expect(set.key('') === set.key('42')).to.equal(false);
                expect(set.key('0') === set.key(0)).to.equal(false);
                expect(set.key('42') === set.key(42)).to.equal(false);
                expect(set.key('abc') === set.key('def')).to.equal(false);
                expect(set.key(false) === set.key(null)).to.equal(false);
                expect(set.key(false) === set.key(0)).to.equal(false);
                expect(set.key(false) === set.key('')).to.equal(false);
                expect(set.key(false) === set.key('false')).to.equal(false);
                expect(set.key(false) === set.key(true)).to.equal(false);
                expect(set.key(true) === set.key(null)).to.equal(false);
                expect(set.key(true) === set.key(1)).to.equal(false);
                expect(set.key(ErrorCode.DIV0) === set.key(null)).to.equal(false);
                expect(set.key(ErrorCode.DIV0) === set.key(ErrorCode.VALUE)).to.equal(false);
                expect(set.key(ErrorCode.DIV0) === set.key('#DIV/0!')).to.equal(false);
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(ScalarSet).to.respondTo('has');
            });
            it('should return false in empty sets', function () {
                var set = new ScalarSet();
                expect(set.has(null)).to.equal(false);
                expect(set.has(0)).to.equal(false);
                expect(set.has('')).to.equal(false);
                expect(set.has(false)).to.equal(false);
                expect(set.has(ErrorCode.NULL)).to.equal(false);
            });
        });

        describe('method "add"', function () {
            it('should exist', function () {
                expect(ScalarSet).to.respondTo('add');
            });
            it('should add values to the set', function () {
                var set = new ScalarSet();
                expect(set.add(null)).to.equal(set);
                expect(set.has(null)).to.equal(true);
                expect(set.has(0)).to.equal(false);
                expect(set.has('')).to.equal(false);
                expect(set.has(false)).to.equal(false);
                expect(set.has(ErrorCode.NULL)).to.equal(false);
                set.add(null).add(0).add('').add(false).add(ErrorCode.NULL);
                expect(set.has(null)).to.equal(true);
                expect(set.has(0)).to.equal(true);
                expect(set.has('')).to.equal(true);
                expect(set.has(false)).to.equal(true);
                expect(set.has(ErrorCode.NULL)).to.equal(true);
            });
            it('should ignore string case', function () {
                var set = new ScalarSet().add('abc');
                expect(set.has('abc')).to.equal(true);
                expect(set.has('ABC')).to.equal(true);
            });
        });

        describe('method "remove"', function () {
            it('should exist', function () {
                expect(ScalarSet).to.respondTo('remove');
            });
            it('should remove values from the set', function () {
                var set = new ScalarSet();
                set.add(null).add(0).add('').add(false).add(ErrorCode.NULL);
                expect(set.remove(null)).to.equal(set);
                expect(set.has(null)).to.equal(false);
                expect(set.has(0)).to.equal(true);
                expect(set.has('')).to.equal(true);
                expect(set.has(false)).to.equal(true);
                expect(set.has(ErrorCode.NULL)).to.equal(true);
                set.remove(null).remove(0).remove('').remove(false).remove(ErrorCode.NULL);
                expect(set.has(null)).to.equal(false);
                expect(set.has(0)).to.equal(false);
                expect(set.has('')).to.equal(false);
                expect(set.has(false)).to.equal(false);
                expect(set.has(ErrorCode.NULL)).to.equal(false);
            });
            it('should ignore string case', function () {
                var set = new ScalarSet().add('abc').remove('ABC');
                expect(set.has('abc')).to.equal(false);
                expect(set.has('ABC')).to.equal(false);
            });
        });

        describe('method "values"', function () {
            it('should exist', function () {
                expect(ScalarSet).to.respondTo('values');
            });
            it('should return the values of the set', function () {
                var set = new ScalarSet();
                set.add(null).add(0).add('').add(false).add(ErrorCode.NULL);
                var values = set.values();
                expect(values).to.be.an('array').with.length(5);
                expect(values).to.contain(null);
                expect(values).to.contain(0);
                expect(values).to.contain('');
                expect(values).to.contain(false);
                expect(values).to.contain(ErrorCode.NULL);
            });
            it('should keep first version of case-insensitive strings', function () {
                var set = new ScalarSet();
                set.add('abc').add('ABC');
                expect(set.values()).to.deep.equal(['abc']);
            });
        });
    });

    // ========================================================================
});
