/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/spreadsheet/utils/addressarray',
    'io.ox/office/spreadsheet/utils/addressset'
], function (SheetHelper, Iterator, AddressArray, AddressSet) {

    'use strict';

    // class AddressSet =======================================================

    describe('Spreadsheet class AddressSet', function () {

        it('should exist', function () {
            expect(AddressSet).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var a = SheetHelper.a;
        var r = SheetHelper.r;
        var unorderedMatcher = SheetHelper.unorderedAddressesMatcher;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a range set', function () {
                var set = new AddressSet();
                expect(set).to.be.an.instanceof(AddressSet);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "empty"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('empty');
            });
            it('should return true for an empty set', function () {
                var set = new AddressSet();
                expect(set.empty()).to.equal(true);
            });
        });

        describe('method "clear"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('clear');
            });
            it('should clear the set', function () {
                var set = new AddressSet();
                set.insert(a('A2'));
                expect(set.empty()).to.equal(false);
                expect(set.clear()).to.equal(set);
                expect(set.empty()).to.equal(true);
            });
        });

        describe('method "values"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('values');
            });
            it('should return all values', function () {
                var set = new AddressSet();
                var a1 = a('A2'), a2 = a('D5'), a3 = a('I10');
                set.insert(a1);
                set.insert(a2);
                set.insert(a3);
                var values = set.values();
                expect(values).to.be.an.instanceof(AddressArray);
                expect(values).to.have.length(3);
                expect(values.indexOf(a1)).to.be.at.least(0);
                expect(values.indexOf(a2)).to.be.at.least(0);
                expect(values.indexOf(a3)).to.be.at.least(0);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('clone');
            });
            var set = new AddressSet();
            var a1 = a('A2'), a2 = a('D5'), a3 = a('I10');
            set.insert(a1);
            set.insert(a2);
            set.insert(a3);
            it('should create a shallow clone', function () {
                var clone = set.clone();
                expect(clone).to.be.an.instanceof(AddressSet);
                var values1 = set.values().sort();
                var values2 = clone.values().sort();
                expect(values2).to.deep.equal(values1);
                expect(values2[0]).to.equal(values1[0]);
            });
            it('should create a deep clone', function () {
                var clone = set.clone(true);
                expect(clone).to.be.an.instanceof(AddressSet);
                var values1 = set.values().sort();
                var values2 = clone.values().sort();
                expect(values2).to.deep.equal(values1);
                expect(values2[0]).to.not.equal(values1[0]);
            });
            it('should create a deep clone with callback', function () {
                var context = {}, spy = sinon.spy(), clone = set.clone(spy, context);
                expect(clone).to.be.an.instanceof(AddressSet);
                var values1 = set.values().sort();
                var values2 = clone.values().sort();
                expect(values2).to.deep.equal(values1);
                expect(values2[0]).to.not.equal(values1[0]);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.callCount(spy, 3);
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('has');
            });
            it('should return false for an empty set', function () {
                var set = new AddressSet();
                expect(set.has(a('A1'))).to.equal(false);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('get');
            });
            it('should return null for an empty set', function () {
                var set = new AddressSet();
                expect(set.get(a('A1'))).to.equal(null);
            });
        });

        describe('methods "first", "last", "firstVert", "lastVert", "boundary"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('first');
                expect(AddressSet).to.respondTo('last');
                expect(AddressSet).to.respondTo('firstVert');
                expect(AddressSet).to.respondTo('lastVert');
                expect(AddressSet).to.respondTo('boundary');
            });
            it('should return null for an empty set', function () {
                var set = new AddressSet();
                expect(set.first()).to.equal(null);
                expect(set.last()).to.equal(null);
                expect(set.firstVert()).to.equal(null);
                expect(set.lastVert()).to.equal(null);
                expect(set.boundary()).to.equal(null);
            });
            it('should return the first address', function () {
                var set = new AddressSet();
                set.insert(a('B4'));
                set.insert(a('C3'));
                set.insert(a('D4'));
                set.insert(a('C5'));
                expect(set.first()).to.stringifyTo('C3');
                expect(set.last()).to.stringifyTo('C5');
                expect(set.firstVert()).to.stringifyTo('B4');
                expect(set.lastVert()).to.stringifyTo('D4');
                expect(set.boundary()).to.stringifyTo('B3:D5');
            });
        });

        describe('method "insert"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('insert');
            });
            var set = new AddressSet();
            it('should insert a new address', function () {
                var a1 = a('A2'), a2 = a1.clone();
                expect(set.insert(a1)).to.equal(a1);
                expect(set.has(a1)).to.equal(true);
                expect(set.has(a2)).to.equal(true);
                expect(set.get(a1)).to.equal(a1);
                expect(set.get(a2)).to.equal(a1);
                expect(set.insert(a2)).to.equal(a1);
                expect(set.has(a1)).to.equal(true);
                expect(set.has(a2)).to.equal(true);
                expect(set.get(a1)).to.equal(a1);
                expect(set.get(a2)).to.equal(a1);
                expect(set.insert(a2, true)).to.equal(a2);
                expect(set.has(a1)).to.equal(true);
                expect(set.has(a2)).to.equal(true);
                expect(set.get(a1)).to.equal(a2);
                expect(set.get(a2)).to.equal(a2);
                expect(set.has(a('A1'))).to.equal(false);
                expect(set.has(a('A3'))).to.equal(false);
                expect(set.has(a('B1'))).to.equal(false);
                expect(set.has(a('B2'))).to.equal(false);
                expect(set.has(a('B3'))).to.equal(false);
            });
            it('should insert and find other addresses', function () {
                set.insert(a('A5'));
                set.insert(a('A4'));
                set.insert(a('A3'));
                set.insert(a('E2'));
                set.insert(a('D2'));
                set.insert(a('C2'));
                expect(set.values()).to.satisfy(unorderedMatcher('A2 A3 A4 A5 C2 D2 E2'));
            });
        });

        describe('method "remove"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('remove');
            });
            it('should remove addresses', function () {
                var set = new AddressSet();
                var a1 = a('A2');
                set.insert(a1);
                set.insert(a('B3'));
                set.insert(a('C4'));
                expect(set.remove(a('A1'))).to.equal(null);
                expect(set.has(a('A2'))).to.equal(true);
                expect(set.remove(a('A2'))).to.equal(a1);
                expect(set.has(a('A2'))).to.equal(false);
            });
        });

        describe('method "merge"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('merge');
            });
            it('should merge with an object', function () {
                var set = new AddressSet();
                set.insert(a('A2'));
                set.insert(a('D5'));
                set.insert(a('I10'));
                var a1 = a('A2');
                var map = { a: a1, b: a('G8') };
                set.merge(map);
                var values = set.values();
                expect(values).to.satisfy(unorderedMatcher('A2 D5 I10 G8'));
                expect(values.indexOf(a1)).to.equal(-1);
                set.merge(map, true);
                values = set.values();
                expect(values).to.satisfy(unorderedMatcher('A2 D5 I10 G8'));
                expect(values.indexOf(a1)).to.be.at.least(0);
            });
            it('should merge two address sets', function () {
                var set1 = new AddressSet();
                set1.insert(a('A2'));
                set1.insert(a('D5'));
                set1.insert(a('I10'));
                var set2 = new AddressSet();
                var a1 = a('A2');
                set2.insert(a1);
                set2.insert(a('G8'));
                set1.merge(set2);
                var values = set1.values();
                expect(values).to.satisfy(unorderedMatcher('A2 D5 I10 G8'));
                expect(values.indexOf(a1)).to.equal(-1);
                set1.merge(set2, true);
                values = set1.values();
                expect(values).to.satisfy(unorderedMatcher('A2 D5 I10 G8'));
                expect(values.indexOf(a1)).to.be.at.least(0);
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('forEach');
            });
            it('should visit all values', function () {
                var set = new AddressSet();
                set.insert(a('A2'));
                set.insert(a('D5'));
                set.insert(a('I10'));
                var spy = sinon.spy(), context = {};
                set.forEach(spy, context);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.callCount(spy, 3);
            });
        });

        describe('method "iterator"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('iterator');
            });
            it('should visit all addresses', function () {
                var set = new AddressSet();
                set.insert(a('A2'));
                set.insert(a('D5'));
                set.insert(a('I10'));
                var it = set.iterator();
                expect(it).to.respondTo('next');
                expect(it.next().done).to.equal(false);
                expect(it.next().done).to.equal(false);
                expect(it.next().done).to.equal(false);
                expect(it.next().done).to.equal(true);
            });
        });

        describe('method "rangeIterator"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('rangeIterator');
            });
            it('should visit all addresses in a range', function () {
                var set = new AddressSet();
                set.insert(a('A2'));
                set.insert(a('A4'));
                set.insert(a('A6'));
                set.insert(a('C2'));
                set.insert(a('C4'));
                set.insert(a('C6'));
                set.insert(a('E2'));
                set.insert(a('E4'));
                set.insert(a('E6'));
                var it1 = set.rangeIterator(r('A1:C10'));
                expect(it1).to.respondTo('next');
                expect(new AddressArray(Iterator.toArray(it1))).to.satisfy(unorderedMatcher('A2 A4 A6 C2 C4 C6'));
                var it2 = set.rangeIterator(r('A3:J5'));
                expect(new AddressArray(Iterator.toArray(it2))).to.satisfy(unorderedMatcher('A4 C4 E4'));
                var it3 = set.rangeIterator(r('D1:D10'));
                expect(it3.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "findAddresses"', function () {
            it('should exist', function () {
                expect(AddressSet).to.respondTo('findAddresses');
            });
            it('should find all addresses in a range', function () {
                var set = new AddressSet();
                set.insert(a('A2'));
                set.insert(a('A4'));
                set.insert(a('A6'));
                set.insert(a('C2'));
                set.insert(a('C4'));
                set.insert(a('C6'));
                set.insert(a('E2'));
                set.insert(a('E4'));
                set.insert(a('E6'));
                expect(set.findAddresses(r('A1:C10'))).to.satisfy(unorderedMatcher('A2 A4 A6 C2 C4 C6'));
                expect(set.findAddresses(r('A3:J5'))).to.satisfy(unorderedMatcher('A4 C4 E4'));
                expect(set.findAddresses(r('D1:D10'))).to.be.empty;
            });
        });
    });

    // ========================================================================
});
