/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/scalar'
], function (Utils, ErrorCode, Scalar) {

    'use strict';

    // static class Scalar ====================================================

    describe('Spreadsheet module Scalar', function () {

        it('should exist', function () {
            expect(Scalar).to.be.an('object');
        });

        // constants ----------------------------------------------------------

        describe('constant "Type"', function () {
            it('should exist', function () {
                expect(Scalar).to.have.a.property('Type').that.is.an('object');
                expect(Scalar.Type).to.have.a.property('NULL', 0);
                expect(Scalar.Type).to.have.a.property('NUMBER', 1);
                expect(Scalar.Type).to.have.a.property('STRING', 2);
                expect(Scalar.Type).to.have.a.property('BOOLEAN', 3);
                expect(Scalar.Type).to.have.a.property('ERROR', 4);
                expect(Scalar.Type).to.have.a.property('NULL_MAX', 5);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "getCellValue"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('getCellValue');
            });
            it('should not convert regular scalar values', function () {
                expect(Scalar.getCellValue(0)).to.equal(0);
                expect(Scalar.getCellValue(1)).to.equal(1);
                expect(Scalar.getCellValue('')).to.equal('');
                expect(Scalar.getCellValue('a')).to.equal('a');
                expect(Scalar.getCellValue(false)).to.equal(false);
                expect(Scalar.getCellValue(true)).to.equal(true);
                expect(Scalar.getCellValue(ErrorCode.NULL)).to.equal(ErrorCode.NULL);
            });
            it('should convert null to zero', function () {
                expect(Scalar.getCellValue(null)).to.equal(0);
            });
        });

        describe('method "getType"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('getType');
            });
            var Type = Scalar.Type;
            it('should recognize numbers', function () {
                expect(Scalar.getType(0)).to.equal(Type.NUMBER);
                expect(Scalar.getType(3)).to.equal(Type.NUMBER);
                expect(Scalar.getType(-3)).to.equal(Type.NUMBER);
                expect(Scalar.getType(1 / 0)).to.equal(Type.NUMBER);
                expect(Scalar.getType(Number.NaN)).to.equal(Type.NUMBER);
            });
            it('should recognize strings', function () {
                expect(Scalar.getType('')).to.equal(Type.STRING);
                expect(Scalar.getType('abc')).to.equal(Type.STRING);
                expect(Scalar.getType('ABC')).to.equal(Type.STRING);
            });
            it('should recognize booleans', function () {
                expect(Scalar.getType(false)).to.equal(Type.BOOLEAN);
                expect(Scalar.getType(true)).to.equal(Type.BOOLEAN);
            });
            it('should recognize error codes', function () {
                expect(Scalar.getType(ErrorCode.DIV0)).to.equal(Type.ERROR);
                expect(Scalar.getType(ErrorCode.NULL)).to.equal(Type.ERROR);
            });
            it('should recognize null', function () {
                expect(Scalar.getType(null)).to.equal(Type.NULL);
                expect(Scalar.getType(null, false)).to.equal(Type.NULL);
                expect(Scalar.getType(null, true)).to.equal(Type.NULL_MAX);
            });
        });

        describe('method "equalNumbers"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('equalNumbers');
            });
            it('should return true, if the numbers are equal', function () {
                expect(Scalar.equalNumbers(-3, -3)).to.equal(true);
                expect(Scalar.equalNumbers(0, 0)).to.equal(true);
                expect(Scalar.equalNumbers(3, 3)).to.equal(true);
                expect(Scalar.equalNumbers(0, Utils.MIN_NUMBER / 2)).to.equal(true);
                expect(Scalar.equalNumbers(1, 1 + 2 * Utils.MIN_NUMBER)).to.equal(true);
                expect(Scalar.equalNumbers(100, 100 * (1 + 2 * Utils.MIN_NUMBER))).to.equal(true);
            });
            it('should return false, if the numbers are not equal', function () {
                expect(Scalar.equalNumbers(-3, 0)).to.equal(false);
                expect(Scalar.equalNumbers(3, 0)).to.equal(false);
                expect(Scalar.equalNumbers(0, -3)).to.equal(false);
                expect(Scalar.equalNumbers(0, 3)).to.equal(false);
                expect(Scalar.equalNumbers(1, 1 + 1e-14)).to.equal(false);
                expect(Scalar.equalNumbers(100, 100 + 1e-12)).to.equal(false);
            });
            it('should return false, if either number is not finite', function () {
                expect(Scalar.equalNumbers(3, 1 / 0)).to.equal(false);
                expect(Scalar.equalNumbers(1 / 0, 3)).to.equal(false);
                expect(Scalar.equalNumbers(1 / 0, 1 / 0)).to.equal(false);
                expect(Scalar.equalNumbers(3, Number.NaN)).to.equal(false);
                expect(Scalar.equalNumbers(Number.NaN, 3)).to.equal(false);
                expect(Scalar.equalNumbers(Number.NaN, Number.NaN)).to.equal(false);
            });
        });

        describe('method "equalStrings"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('equalStrings');
            });
            it('should return false, if first string is less than second string (ignoring case)', function () {
                expect(Scalar.equalStrings('a', 'b')).to.equal(false);
                expect(Scalar.equalStrings('A', 'b')).to.equal(false);
                expect(Scalar.equalStrings('a', 'B')).to.equal(false);
                expect(Scalar.equalStrings('A', 'B')).to.equal(false);
                expect(Scalar.equalStrings('a', 'aa')).to.equal(false);
                expect(Scalar.equalStrings('ab', 'ba')).to.equal(false);
                expect(Scalar.equalStrings('', 'a')).to.equal(false);
            });
            it('should return true, if the strings are equal (ignoring case)', function () {
                expect(Scalar.equalStrings('a', 'a')).to.equal(true);
                expect(Scalar.equalStrings('a', 'A')).to.equal(true);
                expect(Scalar.equalStrings('A', 'a')).to.equal(true);
                expect(Scalar.equalStrings('A', 'A')).to.equal(true);
                expect(Scalar.equalStrings('abc', 'abc')).to.equal(true);
                expect(Scalar.equalStrings('', '')).to.equal(true);
            });
            it('should return false, if first string is greater than second string (ignoring case)', function () {
                expect(Scalar.equalStrings('b', 'a')).to.equal(false);
                expect(Scalar.equalStrings('B', 'a')).to.equal(false);
                expect(Scalar.equalStrings('b', 'A')).to.equal(false);
                expect(Scalar.equalStrings('B', 'A')).to.equal(false);
                expect(Scalar.equalStrings('aa', 'a')).to.equal(false);
                expect(Scalar.equalStrings('ba', 'ab')).to.equal(false);
                expect(Scalar.equalStrings('a', '')).to.equal(false);
            });
            it('should return false, if first string is less than second string (regarding case)', function () {
                expect(Scalar.equalStrings('a', 'A', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('a', 'b', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('a', 'B', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('a', 'c', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('a', 'C', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('a', 'aa', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('ba', 'bA', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('', 'a', { withCase: true })).to.equal(false);
            });
            it('should return true, if the strings are equal (regarding case)', function () {
                expect(Scalar.equalStrings('a', 'a', { withCase: true })).to.equal(true);
                expect(Scalar.equalStrings('A', 'A', { withCase: true })).to.equal(true);
                expect(Scalar.equalStrings('abc', 'abc', { withCase: true })).to.equal(true);
                expect(Scalar.equalStrings('', '', { withCase: true })).to.equal(true);
            });
            it('should return false, if first string is greater than second string (regarding case)', function () {
                expect(Scalar.equalStrings('A', 'a', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('b', 'a', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('B', 'a', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('c', 'a', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('C', 'a', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('aa', 'a', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('bA', 'ba', { withCase: true })).to.equal(false);
                expect(Scalar.equalStrings('a', '', { withCase: true })).to.equal(false);
            });
        });

        describe('method "equalScalars"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('equal');
            });
            it('should compare null to other types', function () {
                expect(Scalar.equal(null, 0, { nullMode: 'less' })).to.equal(false);
                expect(Scalar.equal(null, 0, { nullMode: 'greater' })).to.equal(false);
                expect(Scalar.equal(null, '', { nullMode: 'less' })).to.equal(false);
                expect(Scalar.equal(null, '', { nullMode: 'greater' })).to.equal(false);
                expect(Scalar.equal(null, false, { nullMode: 'less' })).to.equal(false);
                expect(Scalar.equal(null, false, { nullMode: 'greater' })).to.equal(false);
                expect(Scalar.equal(null, ErrorCode.NULL, { nullMode: 'less' })).to.equal(false);
                expect(Scalar.equal(null, ErrorCode.NULL, { nullMode: 'greater' })).to.equal(false);
            });
            it('should compare numbers to other types', function () {
                expect(Scalar.equal(0, null, { nullMode: 'less' })).to.equal(false);
                expect(Scalar.equal(0, null, { nullMode: 'greater' })).to.equal(false);
                expect(Scalar.equal(0, '0')).to.equal(false);
                expect(Scalar.equal(0, false)).to.equal(false);
                expect(Scalar.equal(0, ErrorCode.NULL)).to.equal(false);
            });
            it('should compare strings to other types', function () {
                expect(Scalar.equal('', null, { nullMode: 'less' })).to.equal(false);
                expect(Scalar.equal('', null, { nullMode: 'greater' })).to.equal(false);
                expect(Scalar.equal('0', 0)).to.equal(false);
                expect(Scalar.equal('false', false)).to.equal(false);
                expect(Scalar.equal('#NULL!', ErrorCode.NULL)).to.equal(false);
            });
            it('should compare booleans to other types', function () {
                expect(Scalar.equal(false, null, { nullMode: 'less' })).to.equal(false);
                expect(Scalar.equal(false, null, { nullMode: 'greater' })).to.equal(false);
                expect(Scalar.equal(false, 0)).to.equal(false);
                expect(Scalar.equal(false, 'false')).to.equal(false);
                expect(Scalar.equal(false, ErrorCode.NULL)).to.equal(false);
            });
            it('should compare error codes to other types', function () {
                expect(Scalar.equal(ErrorCode.NULL, null)).to.equal(false);
                expect(Scalar.equal(ErrorCode.NULL, null, { nullMode: 'less' })).to.equal(false);
                expect(Scalar.equal(ErrorCode.NULL, null, { nullMode: 'greater' })).to.equal(false);
                expect(Scalar.equal(ErrorCode.NULL, 0)).to.equal(false);
                expect(Scalar.equal(ErrorCode.NULL, '#NULL!')).to.equal(false);
                expect(Scalar.equal(ErrorCode.NULL, false)).to.equal(false);
            });
            it('should compare numbers', function () {
                expect(Scalar.equal(0, 1)).to.equal(false);
                expect(Scalar.equal(1, 1)).to.equal(true);
                expect(Scalar.equal(1, 0)).to.equal(false);
                expect(Scalar.equal(3, 1 / 0)).to.equal(false);
            });
            it('should compare strings (ignoring case)', function () {
                expect(Scalar.equal('a', 'a')).to.equal(true);
                expect(Scalar.equal('a', 'A')).to.equal(true);
                expect(Scalar.equal('a', 'b')).to.equal(false);
                expect(Scalar.equal('a', 'B')).to.equal(false);
                expect(Scalar.equal('b', 'a')).to.equal(false);
                expect(Scalar.equal('B', 'a')).to.equal(false);
            });
            it('should compare strings (regarding case)', function () {
                expect(Scalar.equal('a', 'a', { withCase: true })).to.equal(true);
                expect(Scalar.equal('a', 'A', { withCase: true })).to.equal(false);
                expect(Scalar.equal('A', 'a', { withCase: true })).to.equal(false);
                expect(Scalar.equal('a', 'B', { withCase: true })).to.equal(false);
                expect(Scalar.equal('A', 'b', { withCase: true })).to.equal(false);
            });
            it('should compare booleans', function () {
                expect(Scalar.equal(false, false)).to.equal(true);
                expect(Scalar.equal(false, true)).to.equal(false);
                expect(Scalar.equal(true, false)).to.equal(false);
                expect(Scalar.equal(true, true)).to.equal(true);
            });
            it('should compare error codes', function () {
                expect(Scalar.equal(ErrorCode.NULL, ErrorCode.NULL)).to.equal(true);
                expect(Scalar.equal(ErrorCode.NULL, ErrorCode.DIV0)).to.equal(false);
                expect(Scalar.equal(ErrorCode.DIV0, ErrorCode.NULL)).to.equal(false);
            });
            it('should convert null value to zero', function () {
                expect(Scalar.equal(null, -1, { nullMode: 'convert' })).to.equal(false);
                expect(Scalar.equal(null, 0, { nullMode: 'convert' })).to.equal(true);
                expect(Scalar.equal(null, 0)).to.equal(true);
                expect(Scalar.equal(null, 1, { nullMode: 'convert' })).to.equal(false);
                expect(Scalar.equal(-1, null, { nullMode: 'convert' })).to.equal(false);
                expect(Scalar.equal(0, null, { nullMode: 'convert' })).to.equal(true);
                expect(Scalar.equal(0, null)).to.equal(true);
                expect(Scalar.equal(1, null, { nullMode: 'convert' })).to.equal(false);
            });
            it('should convert null value to empty string', function () {
                expect(Scalar.equal(null, '', { nullMode: 'convert' })).to.equal(true);
                expect(Scalar.equal(null, '')).to.equal(true);
                expect(Scalar.equal(null, 'a', { nullMode: 'convert' })).to.equal(false);
                expect(Scalar.equal('', null, { nullMode: 'convert' })).to.equal(true);
                expect(Scalar.equal('', null)).to.equal(true);
                expect(Scalar.equal('a', null, { nullMode: 'convert' })).to.equal(false);
            });
            it('should convert null value to FALSE', function () {
                expect(Scalar.equal(null, false, { nullMode: 'convert' })).to.equal(true);
                expect(Scalar.equal(null, false)).to.equal(true);
                expect(Scalar.equal(null, true, { nullMode: 'convert' })).to.equal(false);
                expect(Scalar.equal(null, true)).to.equal(false);
                expect(Scalar.equal(false, null, { nullMode: 'convert' })).to.equal(true);
                expect(Scalar.equal(false, null)).to.equal(true);
                expect(Scalar.equal(true, null, { nullMode: 'convert' })).to.equal(false);
                expect(Scalar.equal(true, null)).to.equal(false);
            });
            it('should not convert null value to error code', function () {
                expect(Scalar.equal(null, ErrorCode.NULL, { nullMode: 'convert' })).to.equal(false);
                expect(Scalar.equal(null, ErrorCode.NULL)).to.equal(false);
                expect(Scalar.equal(ErrorCode.NULL, null, { nullMode: 'convert' })).to.equal(false);
                expect(Scalar.equal(ErrorCode.NULL, null)).to.equal(false);
            });
            it('should compare two null values', function () {
                expect(Scalar.equal(null, null)).to.equal(true);
                expect(Scalar.equal(null, null, { nullMode: 'less' })).to.equal(true);
                expect(Scalar.equal(null, null, { nullMode: 'greater' })).to.equal(true);
                expect(Scalar.equal(null, null, { nullMode: 'convert' })).to.equal(true);
            });
        });

        describe('method "compareNumbers"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('compareNumbers');
            });
            it('should return -1, if first number is less than second number', function () {
                expect(Scalar.compareNumbers(-3, 0)).to.equal(-1);
                expect(Scalar.compareNumbers(0, 3)).to.equal(-1);
                expect(Scalar.compareNumbers(1, 1 + 1e-14)).to.equal(-1);
                expect(Scalar.compareNumbers(100, 100 + 1e-12)).to.equal(-1);
            });
            it('should return 0, if the numbers are equal', function () {
                expect(Scalar.compareNumbers(-3, -3)).to.equal(0);
                expect(Scalar.compareNumbers(0, 0)).to.equal(0);
                expect(Scalar.compareNumbers(3, 3)).to.equal(0);
                expect(Scalar.compareNumbers(0, Utils.MIN_NUMBER / 2)).to.equal(0);
                expect(Scalar.compareNumbers(1, 1 + 2 * Utils.MIN_NUMBER)).to.equal(0);
                expect(Scalar.compareNumbers(100, 100 * (1 + 2 * Utils.MIN_NUMBER))).to.equal(0);
            });
            it('should return 1, if first number is greater than second number', function () {
                expect(Scalar.compareNumbers(3, 0)).to.equal(1);
                expect(Scalar.compareNumbers(0, -3)).to.equal(1);
                expect(Scalar.compareNumbers(1 + 1e-14, 1)).to.equal(1);
                expect(Scalar.compareNumbers(100 + 1e-12, 100)).to.equal(1);
            });
            it('should return NaN, if either number is not finite', function () {
                expect(Scalar.compareNumbers(3, 1 / 0)).to.be.NaN;
                expect(Scalar.compareNumbers(1 / 0, 3)).to.be.NaN;
                expect(Scalar.compareNumbers(1 / 0, 1 / 0)).to.be.NaN;
                expect(Scalar.compareNumbers(3, Number.NaN)).to.be.NaN;
                expect(Scalar.compareNumbers(Number.NaN, 3)).to.be.NaN;
                expect(Scalar.compareNumbers(Number.NaN, Number.NaN)).to.be.NaN;
            });
        });

        describe('method "compareStrings"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('compareStrings');
            });
            it('should return -1, if first string is less than second string (ignoring case)', function () {
                expect(Scalar.compareStrings('a', 'b')).to.equal(-1);
                expect(Scalar.compareStrings('A', 'b')).to.equal(-1);
                expect(Scalar.compareStrings('a', 'B')).to.equal(-1);
                expect(Scalar.compareStrings('A', 'B')).to.equal(-1);
                expect(Scalar.compareStrings('a', 'aa')).to.equal(-1);
                expect(Scalar.compareStrings('ab', 'ba')).to.equal(-1);
                expect(Scalar.compareStrings('', 'a')).to.equal(-1);
            });
            it('should return 0, if the strings are equal (ignoring case)', function () {
                expect(Scalar.compareStrings('a', 'a')).to.equal(0);
                expect(Scalar.compareStrings('a', 'A')).to.equal(0);
                expect(Scalar.compareStrings('A', 'a')).to.equal(0);
                expect(Scalar.compareStrings('A', 'A')).to.equal(0);
                expect(Scalar.compareStrings('abc', 'abc')).to.equal(0);
                expect(Scalar.compareStrings('', '')).to.equal(0);
            });
            it('should return 1, if first string is greater than second string (ignoring case)', function () {
                expect(Scalar.compareStrings('b', 'a')).to.equal(1);
                expect(Scalar.compareStrings('B', 'a')).to.equal(1);
                expect(Scalar.compareStrings('b', 'A')).to.equal(1);
                expect(Scalar.compareStrings('B', 'A')).to.equal(1);
                expect(Scalar.compareStrings('aa', 'a')).to.equal(1);
                expect(Scalar.compareStrings('ba', 'ab')).to.equal(1);
                expect(Scalar.compareStrings('a', '')).to.equal(1);
            });
            it('should return -1, if first string is less than second string (regarding case)', function () {
                expect(Scalar.compareStrings('a', 'A', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('a', 'b', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('a', 'B', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('a', 'c', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('a', 'C', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('a', 'aa', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('a', 'Aa', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('A', 'aa', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('aa', 'ab', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('aa', 'Ab', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('Aa', 'ab', { withCase: true })).to.equal(-1);
                expect(Scalar.compareStrings('', 'a', { withCase: true })).to.equal(-1);
            });
            it('should return 0, if the strings are equal (regarding case)', function () {
                expect(Scalar.compareStrings('a', 'a', { withCase: true })).to.equal(0);
                expect(Scalar.compareStrings('A', 'A', { withCase: true })).to.equal(0);
                expect(Scalar.compareStrings('abc', 'abc', { withCase: true })).to.equal(0);
                expect(Scalar.compareStrings('', '', { withCase: true })).to.equal(0);
            });
            it('should return 1, if first string is greater than second string (regarding case)', function () {
                expect(Scalar.compareStrings('A', 'a', { withCase: true })).to.equal(1);
                expect(Scalar.compareStrings('b', 'a', { withCase: true })).to.equal(1);
                expect(Scalar.compareStrings('B', 'a', { withCase: true })).to.equal(1);
                expect(Scalar.compareStrings('c', 'a', { withCase: true })).to.equal(1);
                expect(Scalar.compareStrings('C', 'a', { withCase: true })).to.equal(1);
                expect(Scalar.compareStrings('aa', 'a', { withCase: true })).to.equal(1);
                expect(Scalar.compareStrings('bA', 'ba', { withCase: true })).to.equal(1);
                expect(Scalar.compareStrings('a', '', { withCase: true })).to.equal(1);
            });
        });

        describe('method "compareBooleans"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('compareBooleans');
            });
            it('should return the result of Boolean comparison', function () {
                expect(Scalar.compareBooleans(false, false)).to.equal(0);
                expect(Scalar.compareBooleans(false, true)).to.equal(-1);
                expect(Scalar.compareBooleans(true, false)).to.equal(1);
                expect(Scalar.compareBooleans(true, true)).to.equal(0);
            });
        });

        describe('method "compareErrorCodes"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('compareErrorCodes');
            });
            it('should return the result of error code comparison', function () {
                expect(Scalar.compareErrorCodes(ErrorCode.NULL, ErrorCode.NULL)).to.equal(0);
                expect(Scalar.compareErrorCodes(ErrorCode.NULL, ErrorCode.DIV0)).to.equal(-1);
                expect(Scalar.compareErrorCodes(ErrorCode.DIV0, ErrorCode.NULL)).to.equal(1);
            });
        });

        describe('method "compareScalars"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('compare');
            });
            it('should compare null to other types', function () {
                expect(Scalar.compare(null, 0, { nullMode: 'less' })).to.equal(-1);
                expect(Scalar.compare(null, 0, { nullMode: 'greater' })).to.equal(1);
                expect(Scalar.compare(null, '', { nullMode: 'less' })).to.equal(-1);
                expect(Scalar.compare(null, '', { nullMode: 'greater' })).to.equal(1);
                expect(Scalar.compare(null, false, { nullMode: 'less' })).to.equal(-1);
                expect(Scalar.compare(null, false, { nullMode: 'greater' })).to.equal(1);
                expect(Scalar.compare(null, ErrorCode.NULL, { nullMode: 'less' })).to.equal(-1);
                expect(Scalar.compare(null, ErrorCode.NULL, { nullMode: 'greater' })).to.equal(1);
            });
            it('should compare numbers to other types', function () {
                expect(Scalar.compare(0, null, { nullMode: 'less' })).to.equal(1);
                expect(Scalar.compare(0, null, { nullMode: 'greater' })).to.equal(-1);
                expect(Scalar.compare(0, '0')).to.equal(-1);
                expect(Scalar.compare(0, false)).to.equal(-1);
                expect(Scalar.compare(0, ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare strings to other types', function () {
                expect(Scalar.compare('', null, { nullMode: 'less' })).to.equal(1);
                expect(Scalar.compare('', null, { nullMode: 'greater' })).to.equal(-1);
                expect(Scalar.compare('0', 0)).to.equal(1);
                expect(Scalar.compare('false', false)).to.equal(-1);
                expect(Scalar.compare('#NULL!', ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare booleans to other types', function () {
                expect(Scalar.compare(false, null, { nullMode: 'less' })).to.equal(1);
                expect(Scalar.compare(false, null, { nullMode: 'greater' })).to.equal(-1);
                expect(Scalar.compare(false, 0)).to.equal(1);
                expect(Scalar.compare(false, 'false')).to.equal(1);
                expect(Scalar.compare(false, ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare error codes to other types', function () {
                expect(Scalar.compare(ErrorCode.NULL, null)).to.equal(1);
                expect(Scalar.compare(ErrorCode.NULL, null, { nullMode: 'less' })).to.equal(1);
                expect(Scalar.compare(ErrorCode.NULL, null, { nullMode: 'greater' })).to.equal(-1);
                expect(Scalar.compare(ErrorCode.NULL, 0)).to.equal(1);
                expect(Scalar.compare(ErrorCode.NULL, '#NULL!')).to.equal(1);
                expect(Scalar.compare(ErrorCode.NULL, false)).to.equal(1);
            });
            it('should compare numbers', function () {
                expect(Scalar.compare(0, 1)).to.equal(-1);
                expect(Scalar.compare(1, 1)).to.equal(0);
                expect(Scalar.compare(1, 0)).to.equal(1);
                expect(Scalar.compare(3, 1 / 0)).to.be.NaN;
            });
            it('should compare strings (ignoring case)', function () {
                expect(Scalar.compare('a', 'a')).to.equal(0);
                expect(Scalar.compare('a', 'A')).to.equal(0);
                expect(Scalar.compare('a', 'b')).to.equal(-1);
                expect(Scalar.compare('a', 'B')).to.equal(-1);
                expect(Scalar.compare('b', 'a')).to.equal(1);
                expect(Scalar.compare('B', 'a')).to.equal(1);
            });
            it('should compare strings (regarding case)', function () {
                expect(Scalar.compare('a', 'a', { withCase: true })).to.equal(0);
                expect(Scalar.compare('a', 'A', { withCase: true })).to.equal(-1);
                expect(Scalar.compare('A', 'a', { withCase: true })).to.equal(1);
                expect(Scalar.compare('a', 'B', { withCase: true })).to.equal(-1);
                expect(Scalar.compare('A', 'b', { withCase: true })).to.equal(-1);
            });
            it('should compare booleans', function () {
                expect(Scalar.compare(false, false)).to.equal(0);
                expect(Scalar.compare(false, true)).to.equal(-1);
                expect(Scalar.compare(true, false)).to.equal(1);
                expect(Scalar.compare(true, true)).to.equal(0);
            });
            it('should compare error codes', function () {
                expect(Scalar.compare(ErrorCode.NULL, ErrorCode.NULL)).to.equal(0);
                expect(Scalar.compare(ErrorCode.NULL, ErrorCode.DIV0)).to.equal(-1);
                expect(Scalar.compare(ErrorCode.DIV0, ErrorCode.NULL)).to.equal(1);
            });
            it('should convert null value to zero', function () {
                expect(Scalar.compare(null, -1, { nullMode: 'convert' })).to.equal(1);
                expect(Scalar.compare(null, 0, { nullMode: 'convert' })).to.equal(0);
                expect(Scalar.compare(null, 1, { nullMode: 'convert' })).to.equal(-1);
                expect(Scalar.compare(null, 0)).to.equal(0);
                expect(Scalar.compare(-1, null, { nullMode: 'convert' })).to.equal(-1);
                expect(Scalar.compare(0, null, { nullMode: 'convert' })).to.equal(0);
                expect(Scalar.compare(0, null)).to.equal(0);
                expect(Scalar.compare(1, null, { nullMode: 'convert' })).to.equal(1);
            });
            it('should convert null value to empty string', function () {
                expect(Scalar.compare(null, '', { nullMode: 'convert' })).to.equal(0);
                expect(Scalar.compare(null, '')).to.equal(0);
                expect(Scalar.compare(null, 'a', { nullMode: 'convert' })).to.equal(-1);
                expect(Scalar.compare('', null, { nullMode: 'convert' })).to.equal(0);
                expect(Scalar.compare('', null)).to.equal(0);
                expect(Scalar.compare('a', null, { nullMode: 'convert' })).to.equal(1);
            });
            it('should convert null value to FALSE', function () {
                expect(Scalar.compare(null, false, { nullMode: 'convert' })).to.equal(0);
                expect(Scalar.compare(null, false)).to.equal(0);
                expect(Scalar.compare(null, true, { nullMode: 'convert' })).to.equal(-1);
                expect(Scalar.compare(null, true)).to.equal(-1);
                expect(Scalar.compare(false, null, { nullMode: 'convert' })).to.equal(0);
                expect(Scalar.compare(false, null)).to.equal(0);
                expect(Scalar.compare(true, null, { nullMode: 'convert' })).to.equal(1);
                expect(Scalar.compare(true, null)).to.equal(1);
            });
            it('should not convert null value to error code', function () {
                expect(Scalar.compare(null, ErrorCode.NULL, { nullMode: 'convert' })).to.equal(-1);
                expect(Scalar.compare(null, ErrorCode.NULL)).to.equal(-1);
                expect(Scalar.compare(ErrorCode.NULL, null, { nullMode: 'convert' })).to.equal(1);
                expect(Scalar.compare(ErrorCode.NULL, null)).to.equal(1);
            });
            it('should compare two null values', function () {
                expect(Scalar.compare(null, null)).to.equal(0);
                expect(Scalar.compare(null, null, { nullMode: 'less' })).to.equal(0);
                expect(Scalar.compare(null, null, { nullMode: 'greater' })).to.equal(0);
                expect(Scalar.compare(null, null, { nullMode: 'convert' })).to.equal(0);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Scalar).itself.to.respondTo('toString');
            });
            it('should convert strings', function () {
                expect(Scalar.toString('abc')).to.equal('"abc"');
                expect(Scalar.toString('a"b"c')).to.equal('"a""b""c"');
                expect(Scalar.toString('')).to.equal('""');
            });
            it('should convert dates', function () {
                var now = new Date();
                expect(Scalar.toString(now)).to.equal(now.toISOString());
            });
            it('should convert null and booleans', function () {
                expect(Scalar.toString(null)).to.equal('null');
                expect(Scalar.toString(true)).to.equal('TRUE');
                expect(Scalar.toString(false)).to.equal('FALSE');
            });
            it('should convert numbers', function () {
                expect(Scalar.toString(42)).to.equal('42');
                expect(Scalar.toString(-12.5)).to.equal('-12.5');
            });
            it('should convert error codes', function () {
                expect(Scalar.toString(ErrorCode.REF)).to.equal('#REF');
            });
        });
    });

    // ========================================================================
});
