/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/funcs/financialfuncs'
], function (AppHelper, SheetHelper, FinancialFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var d = SheetHelper.d;
    var mat = SheetHelper.mat;
    var dmat = SheetHelper.dmat;

    // module FinancialFuncs ==================================================

    describe('Spreadsheet module FinancialFuncs', function () {

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml');
        var moduleTester = SheetHelper.createFunctionModuleTester(FinancialFuncs, appPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('ACCRINT', function (ACCRINT) {
            it('should return correct results', function () {
                expect(ACCRINT(d('2008-03-01'), d('2008-08-31'), d('2008-05-01'), 0.1, 1000, 2, 0)).to.almostEqual(16 + 2 / 3);

                expect(ACCRINT(d('2008-04-01'), d('2010-04-01'), d('2018-04-01'), 1, 1, 1, 0, true)).to.equal(10);
                //expect(ACCRINT(d('2008-04-01'), d('2010-04-01'), d('2018-04-01'), 1, 1, 1, 0, false)).to.equal(9);

                expect(ACCRINT(d('2008-04-05'), d('2008-08-31'), d('2008-05-01'), 0.1, 1000, 2, 0, true)).to.almostEqual(7 + 2 / 9);
                //expect(ACCRINT(d('2008-03-05'), d('2008-08-31'), d('2008-05-01'), 0.1, 1000, 2, 0, false)).to.almostEqual(15 + 5 / 9);

                expect(ACCRINT(d('2008-03-01'), d('2010-03-01'), d('2009-05-01'), 0.1, 1000, 2, 0, true)).to.almostEqual(116 + 2 / 3);
                //expect(ACCRINT(d('2008-03-01'), d('2010-03-01'), d('2009-05-01'), 0.1, 1000, 2, 0, false)).to.almostEqual(-33 + 1 / 3);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(ACCRINT(d('2009-05-02'), d('2010-03-01'), d('2009-05-01'), 0.1, 1000, 2, 0, true)).to.equal(ErrorCode.NUM);
                expect(ACCRINT(d('2008-03-01'), d('2010-03-01'), d('2009-05-01'), 0.1, -1000, 2, 0, true)).to.equal(ErrorCode.NUM);
                expect(ACCRINT(d('2008-03-01'), d('2010-03-01'), d('2009-05-01'), -0.1, 1000, 2, 0, true)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('ACCRINTM', function (ACCRINTM) {
            it('should return correct results', function () {
                expect(ACCRINTM(d('2008-04-01'), d('2008-06-15'), 0.1, 1000, 3)).to.be.closeTo(20.54794521, 1e-6);
                expect(ACCRINTM(d('2008-04-01'), d('2009-04-01'), 0.1, 1000, 0)).to.equal(100);
                expect(ACCRINTM(d('2008-04-01'), d('2009-04-01'), 0.1, 1000, 1)).to.equal(100);
                expect(ACCRINTM(d('2008-04-01'), d('2009-04-01'), 0.1, 1000, 3)).to.equal(100);
                expect(ACCRINTM(d('2008-04-01'), d('2009-04-01'), 0.1, 1000, 4)).to.equal(100);
                expect(ACCRINTM(d('2008-04-06'), d('2009-04-01'), 0.1, 1000, 2)).to.equal(100);
            });
            it('should return correct results within leapyears', function () {
                expect(ACCRINTM(d('2008-02-15'), d('2009-02-15'), 0.1, 1000, 0)).to.equal(100);
                expect(ACCRINTM(d('2008-02-15'), d('2009-02-15'), 0.1, 1000, 1)).to.equal(100);
                expect(ACCRINTM(d('2008-02-15'), d('2009-02-09'), 0.1, 1000, 2)).to.equal(100);
                expect(ACCRINTM(d('2008-02-15'), d('2009-02-14'), 0.1, 1000, 3)).to.equal(100);
                expect(ACCRINTM(d('2008-02-15'), d('2009-02-15'), 0.1, 1000, 4)).to.equal(100);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(ACCRINTM(d('2009-04-01'), d('2008-04-06'), 0.1, 1000, 2)).to.equal(ErrorCode.NUM);
                expect(ACCRINTM(d('2008-04-01'), d('2008-06-15'), 0, 1000, 2)).to.equal(ErrorCode.NUM);
                expect(ACCRINTM(d('2008-04-01'), d('2008-06-15'), 0.1, 0, 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('AMORDEGRC', function (AMORDEGRC) {
            it('should return correct results', function () {
                expect(AMORDEGRC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 0)).to.equal(776);
                expect(AMORDEGRC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 1)).to.equal(776);
                expect(AMORDEGRC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 3)).to.equal(776);
                expect(AMORDEGRC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 4)).to.equal(777);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(AMORDEGRC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 2)).to.equal(ErrorCode.NUM);
                expect(AMORDEGRC(0, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 0)).to.equal(ErrorCode.NUM);
                expect(AMORDEGRC(2400, d('2008-08-19'), d('2008-12-31'), -1, 1, 0.15, 0)).to.equal(ErrorCode.NUM);
                expect(AMORDEGRC(299, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('AMORLINC', function (AMORLINC) {
            it('should return correct results', function () {
                expect(AMORLINC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 0)).to.equal(360);
                expect(AMORLINC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 1)).to.equal(360);
                expect(AMORLINC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 3)).to.equal(360);
                expect(AMORLINC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 4)).to.equal(360);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(AMORLINC(2400, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 2)).to.equal(ErrorCode.NUM);
                expect(AMORLINC(0, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 0)).to.equal(ErrorCode.NUM);
                expect(AMORLINC(2400, d('2008-08-19'), d('2008-12-31'), -1, 1, 0.15, 0)).to.equal(ErrorCode.NUM);
                expect(AMORLINC(299, d('2008-08-19'), d('2008-12-31'), 300, 1, 0.15, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('COUPDAYBS', function (COUPDAYBS) {
            it('should return correct results', function () {
                expect(COUPDAYBS(d('2011-01-25'), d('2011-11-15'), 2)).to.equal(70);
                expect(COUPDAYBS(d('2011-01-25'), d('2011-11-15'), 2, 1)).to.equal(71);
                expect(COUPDAYBS(d('2011-01-25'), d('2011-11-15'), 2, 2)).to.equal(71);
                expect(COUPDAYBS(d('2011-01-25'), d('2011-11-15'), 2, 3)).to.equal(71);
                expect(COUPDAYBS(d('2011-01-25'), d('2011-11-15'), 2, 4)).to.equal(70);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(COUPDAYBS(d('2011-11-15'), d('2011-01-25'), 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('COUPDAYS', function (COUPDAYS) {
            it('should return correct results', function () {
                expect(COUPDAYS(d('2011-01-25'), d('2011-11-15'), 2)).to.equal(180);
                expect(COUPDAYS(d('2011-01-25'), d('2011-11-15'), 2, 1)).to.equal(181);
                expect(COUPDAYS(d('2011-01-25'), d('2011-11-15'), 2, 2)).to.equal(180);
                expect(COUPDAYS(d('2011-01-25'), d('2011-11-15'), 2, 3)).to.equal(182.5);
                expect(COUPDAYS(d('2011-01-25'), d('2011-11-15'), 2, 4)).to.equal(180);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(COUPDAYS(d('2011-11-15'), d('2011-01-25'), 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('COUPDAYSNC', function (COUPDAYSNC) {
            it('should return correct results', function () {
                expect(COUPDAYSNC(d('2011-01-25'), d('2011-11-15'), 2)).to.equal(110);
                expect(COUPDAYSNC(d('2011-01-25'), d('2011-11-15'), 2, 0)).to.equal(110);
                expect(COUPDAYSNC(d('2011-01-25'), d('2011-11-15'), 2, 4)).to.equal(110);

                expect(COUPDAYSNC(d('2011-01-25'), d('2011-11-15'), 2, 1)).to.equal(110);
                expect(COUPDAYSNC(d('2011-01-25'), d('2011-11-15'), 2, 2)).to.equal(110);
                expect(COUPDAYSNC(d('2011-01-25'), d('2011-11-15'), 2, 3)).to.equal(110);

                expect(COUPDAYSNC(d('2011-01-25'), d('2012-01-25'), 2, 1)).to.equal(181);
                expect(COUPDAYSNC(d('2011-01-25'), d('2012-01-25'), 2, 2)).to.equal(181);
                expect(COUPDAYSNC(d('2011-01-25'), d('2012-01-25'), 2, 3)).to.equal(181);
                expect(COUPDAYSNC(d('2011-01-25'), d('2012-01-25'), 2, 4)).to.equal(180);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(COUPDAYSNC(d('2011-11-15'), d('2011-01-25'), 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('COUPNCD', function (COUPNCD) {
            it('should return correct results', function () {
                expect(COUPNCD(d('2011-01-25'), d('2011-11-15'), 2, 1)).to.deep.equal(d('2011-05-15'));
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(COUPNCD(d('2011-11-15'), d('2011-01-25'), 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('COUPNUM', function (COUPNUM) {
            it('should return correct results', function () {
                expect(COUPNUM(d('2007-01-01'), d('2008-01-01'), 1)).to.equal(1);
                expect(COUPNUM(d('2007-01-01'), d('2008-01-01'), 2)).to.equal(2);
                expect(COUPNUM(d('2007-01-01'), d('2008-01-01'), 4)).to.equal(4);
                expect(COUPNUM(d('2007-01-01'), d('2007-04-01'), 4)).to.equal(1);
                expect(COUPNUM(d('2007-01-01'), d('2007-04-02'), 4)).to.equal(2);
                expect(COUPNUM(d('2011-01-01'), d('2012-01-01'), 4, 1)).to.equal(4);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(COUPNUM(d('2011-11-15'), d('2011-01-25'), 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('COUPPCD', function (COUPPCD) {
            it('should return correct results', function () {
                expect(COUPPCD(d('2011-01-25'), d('2011-11-15'), 2, 0)).to.deep.equal(d('2010-11-15'));
                expect(COUPPCD(d('2011-01-25'), d('2011-11-15'), 2, 1)).to.deep.equal(d('2010-11-15'));
                expect(COUPPCD(d('2011-01-25'), d('2011-11-15'), 2, 2)).to.deep.equal(d('2010-11-15'));
                expect(COUPPCD(d('2011-01-25'), d('2011-11-15'), 2, 3)).to.deep.equal(d('2010-11-15'));
                expect(COUPPCD(d('2011-01-25'), d('2011-11-15'), 2, 4)).to.deep.equal(d('2010-11-15'));

                expect(COUPPCD(d('2012-02-29'), d('2050-08-08'), 1)).to.deep.equal(d('2011-08-08'));
                expect(COUPPCD(d('2012-02-29'), d('2050-08-08'), 2)).to.deep.equal(d('2012-02-08'));
                expect(COUPPCD(d('2012-02-29'), d('2050-08-08'), 4)).to.deep.equal(d('2012-02-08'));
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(COUPPCD(d('2011-11-15'), d('2011-01-25'), 2, 1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('CUMIPMT', function (CUMIPMT) {
            it('should return correct results', function () {
                expect(CUMIPMT(0.09 / 12, 30 * 12, 125000, 13, 24, 0)).to.be.closeTo(-11135.23213, 1e-6);
                expect(CUMIPMT(0.09 / 12, 30 * 12, 125000, 1, 1, 0)).to.equal(-937.5);
                expect(CUMIPMT(0.09 / 12, 30 * 12, 125000, 13, 24, 1)).to.be.closeTo(-11052.33958, 1e-5);
                expect(CUMIPMT(0.09 / 12, 30 * 12, 125000, 1, 1, 1)).to.equal(0);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(CUMIPMT(0.09 / 12, 30 * 12, 125000, 13, 24, 2)).to.equal(ErrorCode.NUM);
                expect(CUMIPMT(0.09 / 12, 30 * 12, 125000, 13, 24, -1)).to.equal(ErrorCode.NUM);
                expect(CUMIPMT(0.09 / 12, 30 * 12, 125000, 24, 13, 1)).to.equal(ErrorCode.NUM);
                expect(CUMIPMT(0.09 / 12, 24, 125000, 13, 360, 1)).to.equal(ErrorCode.NUM);
                expect(CUMIPMT(0.09 / 12, 30 * 12, 125000, 0, 24, 1)).to.equal(ErrorCode.NUM);

                expect(CUMIPMT(0, 30 * 12, 125000, 13, 24, 1)).to.equal(ErrorCode.NUM);
                expect(CUMIPMT(0.09 / 12, 0, 125000, 13, 24, 1)).to.equal(ErrorCode.NUM);
                expect(CUMIPMT(0.09 / 12, 30 * 12, 0, 13, 24, 1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('CUMPRINC', function (CUMPRINC) {
            it('should return correct results', function () {
                expect(CUMPRINC(0.09 / 12, 30 * 12, 125000, 13, 24, 0)).to.be.closeTo(-934.1071234, 1e-6);
                expect(CUMPRINC(0.09 / 12, 30 * 12, 125000, 1, 1, 0)).to.be.closeTo(-68.27827118, 1e-6);

                expect(CUMPRINC(0.09 / 12, 30 * 12, 125000, 13, 24, 1)).to.be.closeTo(-927.1534724, 1e-6);
                expect(CUMPRINC(0.09 / 12, 30 * 12, 125000, 1, 1, 1)).to.be.closeTo(-998.291088, 1e-6);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(CUMPRINC(0.09 / 12, 30 * 12, 125000, 13, 24, 2)).to.equal(ErrorCode.NUM);
                expect(CUMPRINC(0.09 / 12, 30 * 12, 125000, 13, 24, -1)).to.equal(ErrorCode.NUM);
                expect(CUMPRINC(0.09 / 12, 30 * 12, 125000, 24, 13, 1)).to.equal(ErrorCode.NUM);
                expect(CUMPRINC(0.09 / 12, 24, 125000, 13, 360, 1)).to.equal(ErrorCode.NUM);
                expect(CUMPRINC(0.09 / 12, 30 * 12, 125000, 0, 24, 1)).to.equal(ErrorCode.NUM);

                expect(CUMPRINC(0, 30 * 12, 125000, 13, 24, 1)).to.equal(ErrorCode.NUM);
                expect(CUMPRINC(0.09 / 12, 0, 125000, 13, 24, 1)).to.equal(ErrorCode.NUM);
                expect(CUMPRINC(0.09 / 12, 30 * 12, 0, 13, 24, 1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DB', function (DB) {
            it('should return correct results', function () {
                expect(DB(1000000, 100000, 6, 1, 7)).to.be.closeTo(186083.333333, 1e-6);
                expect(DB(1000000, 100000, 6, 2, 7)).to.be.closeTo(259639.416666, 1e-6);
                expect(DB(1000000, 100000, 6, 3, 7)).to.be.closeTo(176814.442750, 1e-6);
                expect(DB(1000000, 100000, 6, 4, 7)).to.be.closeTo(120410.635512, 1e-6);
                expect(DB(1000000, 100000, 6, 5, 7)).to.be.closeTo(81999.642784, 1e-6);
                expect(DB(1000000, 100000, 6, 6, 7)).to.be.closeTo(55841.756736, 1e-6);
                expect(DB(1000000, 100000, 6, 7, 7)).to.be.closeTo(15845.098473, 1e-6);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(DB(-1, 100000, 6, 1, 7)).to.equal(ErrorCode.NUM);
                expect(DB(1000000, -1, 6, 1, 7)).to.equal(ErrorCode.NUM);
                expect(DB(1000000, 100000, -1, 1, 7)).to.equal(ErrorCode.NUM);
                expect(DB(1000000, 100000, 6, -1, 7)).to.equal(ErrorCode.NUM);
                expect(DB(1000000, 100000, 6, 1, 0)).to.equal(ErrorCode.NUM);
                expect(DB(1000000, 100000, 6, 1, 13)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DDB', function (DDB) {
            it('should return correct results', function () {
                expect(DDB(2400, 300, 10 * 365, 1)).to.be.closeTo(1.315068493, 1e-6);
                expect(DDB(2400, 300, 10 * 12, 1, 2)).to.equal(40);
                expect(DDB(2400, 300, 10 * 1, 1, 2)).to.equal(480);
                expect(DDB(2400, 300, 10 * 1, 2, 1.5)).to.closeTo(306, 1e-6);
                expect(DDB(2400, 300, 10 * 1, 10)).to.be.closeTo(22.1225472, 1e-6);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(DDB(-1, 300, 10 * 1, 0, 0.875)).to.equal(ErrorCode.NUM);
                expect(DDB(2400, -1, 10 * 1, 0, 0.875)).to.equal(ErrorCode.NUM);
                expect(DDB(2400, 300, -1 * 1, 0, 0.875)).to.equal(ErrorCode.NUM);
                expect(DDB(2400, 300, 10 * 1, -1, 0.875)).to.equal(ErrorCode.NUM);
                expect(DDB(2400, 300, 10 * 1, 0, -1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DISC', function (DISC) {
            it('should return correct results', function () {
                expect(DISC(d('2007-01-25'), d('2007-06-15'), 97.975, 100, 1)).to.be.closeTo(0.052420213, 1e-6);
                expect(DISC(d('2007-01-25'), d('2008-01-25'), 1000, 10, 0)).to.equal(-99);
                expect(DISC(d('2007-01-25'), d('2008-01-25'), 1000, 10, 1)).to.equal(-99);
                expect(DISC(d('2007-01-25'), d('2008-01-25'), 1000, 10, 3)).to.equal(-99);
                expect(DISC(d('2007-01-25'), d('2008-01-25'), 1000, 10, 4)).to.equal(-99);
                expect(DISC(d('2007-01-25'), d('2008-01-23'), 1000, 10, 2)).to.equal(-(98 + 2 / 11));
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(DISC(d('2007-06-15'), d('2007-01-25'), 97.975, 100, 1)).to.equal(ErrorCode.NUM);
                expect(DISC(d('2007-01-25'), d('2007-06-15'), 0, 100, 1)).to.equal(ErrorCode.NUM);
                expect(DISC(d('2007-01-25'), d('2007-06-15'), 97.975, 0, 1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DOLLARDE', function (DOLLARDE) {
            it('should return correct results', function () {
                expect(DOLLARDE(1.02, 16)).to.almostEqual(1.125);
                expect(DOLLARDE(1.1, 32)).to.almostEqual(1.3125);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(DOLLARDE(1.1, -1)).to.equal(ErrorCode.NUM);
                expect(DOLLARDE(1.1, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DOLLARFR', function (DOLLARFR) {
            it('should return correct results', function () {
                expect(DOLLARFR(1.125, 16)).to.almostEqual(1.02);
                expect(DOLLARFR(1.125, 32)).to.almostEqual(1.04);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(DOLLARFR(1.125, -1)).to.equal(ErrorCode.NUM);
                expect(DOLLARFR(1.125, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DURATION', function (DURATION) {
            it('should return correct results', function () {
                expect(DURATION(d('2008-01-01'), d('2016-01-01'), 0.08, 0.09, 2, 1)).to.be.closeTo(5.99195, 1e-5);
                expect(DURATION(d('2008-01-01'), d('2016-01-01'), 0, 0.09, 2, 1)).to.be.closeTo(8, 1e-2);
                expect(DURATION(d('2008-01-01'), d('2016-01-01'), 0.1, 0, 2, 1)).to.be.closeTo(6.3315, 1e-5);

                expect(DURATION(d('2008-01-01'), d('2008-06-01'), 1, 0, 4, 0)).to.be.closeTo(0.375, 1e-5);
                expect(DURATION(d('2008-01-01'), d('2008-06-01'), 1, 0, 4, 1)).to.be.closeTo(0.373168498, 1e-3);
                expect(DURATION(d('2008-01-01'), d('2008-06-01'), 1, 0, 4, 2)).to.be.closeTo(0.38, 1e-3);
                expect(DURATION(d('2008-01-01'), d('2008-06-01'), 1, 0, 4, 3)).to.be.closeTo(0.37477, 1e-3);
                expect(DURATION(d('2008-01-01'), d('2008-06-01'), 1, 0, 4, 4)).to.be.closeTo(0.375, 1e-5);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(DURATION(d('2016-01-01'), d('2008-01-01'), 0.08, 0.09, 2, 1)).to.equal(ErrorCode.NUM);
                expect(DURATION(d('2008-01-01'), d('2016-01-01'), -0.1, 0.09, 2, 1)).to.equal(ErrorCode.NUM);
                expect(DURATION(d('2008-01-01'), d('2016-01-01'), 0.08, -0.1, 2, 1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('EFFECT', function (EFFECT) {
            it('should return correct results', function () {
                expect(EFFECT(0.0525, 4)).to.be.closeTo(0.053542667, 1e-5);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(EFFECT(0.0525, 0)).to.equal(ErrorCode.NUM);
                expect(EFFECT(0, 4)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('FV', function (FV) {
            it('should return correct results', function () {
                expect(FV(0.06 / 12, 12, -100, -1000, 1)).to.be.closeTo(2301.40183, 1e-6);
                expect(FV(0.06 / 12, 10, -200, -500, 1)).to.be.closeTo(2581.403374, 1e-6);
                expect(FV(0.12 / 12, 12, -1000)).to.be.closeTo(12682.503013, 1e-6);
                expect(FV(0.11 / 12, 35, -2000, null, 1)).to.be.closeTo(82846.246371, 1e-6);
            });
        });

        moduleTester.testFunction('FVSCHEDULE', function (FVSCHEDULE) {
            it('should return correct results', function () {
                expect(FVSCHEDULE(1, mat([[0.09, 0.11, 0.1]]))).to.be.closeTo(1.33089, 1e-10);
                expect(FVSCHEDULE(1, mat([[1, 2, 3]]))).to.be.equal(24);
                expect(FVSCHEDULE(1, mat([[0, 1, 2, 3]]))).to.be.equal(24);
                expect(FVSCHEDULE(1, mat([[0, 1, -2, 3]]))).to.be.equal(-8);

                expect(FVSCHEDULE(1, '22')).to.be.equal(23);
            });
            it('should throw #VALUE! error code for wrong input data', function () {
                expect(FVSCHEDULE(1, mat([[1, 'hello', 3]]))).to.equal(ErrorCode.VALUE);
                expect(FVSCHEDULE(1, mat([[1, true, 3]]))).to.equal(ErrorCode.VALUE);
                expect(FVSCHEDULE(1, true)).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('INTRATE', function (INTRATE) {
            it('should return correct results', function () {
                expect(INTRATE(d('2008-02-15'), d('2008-05-15'), 1000000, 1014420, 2)).to.be.closeTo(0.05768, 1e-10);

                expect(INTRATE(d('2008-02-15'), d('2009-02-15'), 1000000, 2000000, 0)).to.be.closeTo(1, 1e-10);
                expect(INTRATE(d('2008-02-15'), d('2009-02-15'), 1000000, 2000000, 1)).to.be.closeTo(1, 1e-4);
                expect(INTRATE(d('2008-02-15'), d('2009-02-15'), 1000000, 2000000, 4)).to.be.closeTo(1, 1e-10);

                expect(INTRATE(d('2008-02-15'), d('2009-02-14'), 1000000, 2000000, 3)).to.be.closeTo(1, 1e-3);

                expect(INTRATE(d('2008-04-06'), d('2009-04-01'), 1000000, 2000000, 2)).to.be.closeTo(1, 1e-10);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(INTRATE(d('2008-02-15'), d('2008-05-15'), 0, 1014420, 2)).to.equal(ErrorCode.NUM);
                expect(INTRATE(d('2008-02-15'), d('2008-05-15'), 1000000, 0, 2)).to.equal(ErrorCode.NUM);
                expect(INTRATE(d('2008-02-15'), d('2008-02-15'), 1000000, 1014420, 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('IPMT', function (IPMT) {
            it('should return correct results', function () {
                expect(IPMT(1 / 120, 1, 36, 8000)).to.be.closeTo(-200 / 3, 1e-6);
                expect(IPMT(1 / 10, 3, 3, 8000)).to.be.closeTo(-292.4471299, 1e-6);
                expect(IPMT(1 / 120, 1, 36, 8000, 500, 0)).to.be.closeTo(-200 / 3, 1e-6);
                expect(IPMT(1 / 120, 1, 36, 8000, 500, 1)).to.be.almostZero;
                expect(IPMT(1 / 120, 1, 36, 8000, 500, 666)).to.be.almostZero;
            });
            it('should throw #NUM! error on wrong input data', function () {
                expect(IPMT(3, 0.1, 3, 8000)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('IRR', function (IRR) {
            it('should return correct results', function () {
                expect(IRR(mat([[-70000, 12000, 15000, 18000, 21000]]))).to.be.closeTo(-0.02124485, 1e-6);
                expect(IRR(mat([[-70000, 12000, 15000, 18000, 21000, 26000]]))).to.be.closeTo(0.08663095, 1e-6);
                expect(IRR(mat([[-70000, 12000, 15000]]), -0.1)).to.be.closeTo(-0.44350694, 1e-6);
            });
            it('should throw #NUM! error on wrong input data', function () {
                expect(IRR(mat([[70000, 12000, 15000]]))).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('ISPMT', function (ISPMT) {
            it('should return correct results', function () {
                expect(ISPMT(1 / 120, 1, 36, 8000000)).to.be.closeTo(-64814.814814, 1e-6);
                expect(ISPMT(0.1, 1, 3, 8000000)).to.be.closeTo(-533333.333333, 1e-6);
            });
            it('should throw #NUM! error on wrong input data', function () {
                expect(ISPMT(1 / 120, 3, 1, 8000000)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('MDURATION', function (MDURATION) {
            it('should return correct results', function () {
                //MDURATION is almost the same as DURATION
                expect(MDURATION(d('2008-01-01'), d('2016-01-01'), 0.08, 0.09, 2, 1)).to.be.closeTo(5.733, 1e-3);
            });
        });

        moduleTester.testFunction('MIRR', function (MIRR) {
            it('should return correct results', function () {
                expect(MIRR(mat([[-120000, 39000, 30000, 21000, 37000, 46000]]), 0.1, 0.12)).to.be.closeTo(0.12609413, 1e-6);
                expect(MIRR(mat([[-120000, 39000, 30000, 21000]]), 0.1, 0.12)).to.be.closeTo(-0.048044655, 1e-6);
                expect(MIRR(mat([[-120000, 39000, 30000, 21000, 37000, 46000]]), 0.1, 0.14)).to.be.closeTo(0.134759111, 1e-6);
            });
            it('should throw #NUM! error on wrong input data', function () {
                expect(MIRR(mat([[-120000, 39000, 30000, 21000, 37000, 46000]]), 0.1, -1)).to.equal(ErrorCode.DIV0);
            });
        });

        moduleTester.testFunction('NOMINAL', function (NOMINAL) {
            it('should return correct results', function () {
                expect(NOMINAL(1, 1)).to.be.closeTo(1, 1e-6);
                expect(NOMINAL(0.053543, 4)).to.be.closeTo(0.05250032, 1e-6);
            });
            it('should throw #NUM! error on wrong input data', function () {
                expect(NOMINAL(1, 0)).to.equal(ErrorCode.NUM);
                expect(NOMINAL(0, 1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('NPER', function (NPER) {
            it('should return correct results', function () {
                expect(NPER(0.01, -100, -1000, 10000, 1)).to.be.closeTo(59.67386567, 1e-6);
                expect(NPER(0.01, -100, -1000, 10000)).to.be.closeTo(60.08212285, 1e-6);
                expect(NPER(0.01, -100, -1000)).to.be.closeTo(-9.57859404, 1e-6);
            });
            it('should throw #NUM! error on wrong input data', function () {
                expect(NPER(-0.01, -100, -1000, 10000, 1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('NPV', function (NPV) {
            it('should return correct results', function () {
                expect(NPV(1, 1, 6)).to.equal(2);
                expect(NPV(0.08, mat([[8000, 9200, 10000, 12000, 14500]]))).to.be.closeTo(41922.061555, 1e-6);
                expect(NPV(0.08, mat([[8000, 9200, 10000, 12000, 14500, -9000]]))).to.be.closeTo(36250.534913, 1e-6);
                expect(NPV(0.1, mat([[-10000, 3000, 4200, 6800]]))).to.be.closeTo(1188.443412, 1e-6);
            });
            it('should throw #DIV/0! error on wrong input data', function () {
                expect(NPV(-1, mat([[-10000, 3000, 4200, 6800]]))).to.equal(ErrorCode.DIV0);
            });
        });

        moduleTester.testFunction('ODDLYIELD', function (ODDLYIELD) {
            it('should return correct results', function () {
                expect(ODDLYIELD(d('2008-04-20'), d('2008-06-15'), d('2007-12-24'), 0.0375, 99.875, 100, 2, 0)).to.be.closeTo(0.0452, 1e-4);
            });
            it('with whole years should return correct results', function () {
                expect(ODDLYIELD(d('2008-01-01'), d('2009-01-01'), d('2007-01-01'), 1, 100, 100, 1)).to.equal(0.5);
                expect(ODDLYIELD(d('2008-01-01'), d('2009-01-01'), d('2007-01-01'), 1, 100, 100, 1, 0)).to.equal(0.5);
                expect(ODDLYIELD(d('2008-01-01'), d('2009-01-01'), d('2007-01-01'), 1, 100, 100, 1, 1)).to.be.closeTo(0.5, 1e-3);
                expect(ODDLYIELD(d('2008-01-01'), d('2009-01-01'), d('2007-01-01'), 1, 100, 100, 1, 2)).to.be.closeTo(0.5, 1e-2);
                expect(ODDLYIELD(d('2008-01-01'), d('2009-01-01'), d('2007-01-01'), 1, 100, 100, 1, 3)).to.equal(0.5);
                expect(ODDLYIELD(d('2008-01-01'), d('2009-01-01'), d('2007-01-01'), 1, 100, 100, 1, 4)).to.equal(0.5);
            });
            it('with whole leapyears should return correct results', function () {
                expect(ODDLYIELD(d('2011-03-01'), d('2012-02-29'), d('2010-03-01'), 1, 100, 100, 1, 1)).to.be.closeTo(0.5, 1e-3);
                expect(ODDLYIELD(d('2011-03-01'), d('2012-02-29'), d('2010-03-01'), 1, 100, 100, 1, 2)).to.be.closeTo(0.5, 1e-2);
                expect(ODDLYIELD(d('2011-03-01'), d('2012-02-29'), d('2010-03-01'), 1, 100, 100, 1, 3)).to.equal(0.5);
                expect(ODDLYIELD(d('2011-03-01'), d('2012-02-29'), d('2010-03-01'), 1, 100, 100, 1, 4)).to.equal(0.5);

                expect(ODDLYIELD(d('2011-03-01'), d('2012-02-29'), d('2010-03-01'), 1, 100, 100, 1)).to.be.closeTo(366 / (365 * 2), 1e-2);
                expect(ODDLYIELD(d('2011-03-01'), d('2012-02-29'), d('2010-03-01'), 1, 100, 100, 1, 0)).to.be.closeTo(366 / (365 * 2), 1e-2);
            });
            it('should throw correct errors', function () {
                expect(ODDLYIELD(d('2008-06-15'), d('2008-04-20'), d('2007-12-24'), 5.62, 100, 100, 2)).to.equal(ErrorCode.NUM);
                expect(ODDLYIELD(d('2008-04-20'), d('2007-12-24'), d('2008-06-15'), 5.62, 100, 100, 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PDURATION', function (PDURATION) {
            it('should return correct results', function () {
                expect(PDURATION(0.025, 2000, 2200)).to.be.closeTo(3.859866163, 1e-6);
                expect(PDURATION(0.025 / 12, 1000, 1200)).to.be.closeTo(87.60547642, 1e-6);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(PDURATION(0, 2000, 2200)).to.equal(ErrorCode.NUM);
                expect(PDURATION(0.025, 0, 2200)).to.equal(ErrorCode.NUM);
                expect(PDURATION(0.025, 2000, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PMT', function (PMT) {
            it('should return correct results', function () {
                expect(PMT(0.08 / 12, 10, 10000)).to.be.closeTo(-1037.032089, 1e-4);
                expect(PMT(0.08 / 12, 10, 10000, 0, 1)).to.be.closeTo(-1030.164327, 1e-4);
                expect(PMT(0.06 / 12, 18 * 12, 0, 50000)).to.be.closeTo(-129.0811609, 1e-4);
                expect(PMT(0.06 / 12, 18 * 12, 0, 50000, 1)).to.be.closeTo(-128.438966, 1e-4);
            });
        });

        moduleTester.testFunction('PPMT', function (PPMT) {
            it('should return correct results', function () {
                expect(PPMT(0.08, 10, 10, 200000)).to.be.closeTo(-27598.05346, 1e-4);
                expect(PPMT(0.1 / 12, 1, 2 * 12, 2000)).to.be.closeTo(-75.62318601, 1e-4);
            });
        });

        moduleTester.testFunction('PRICE', function (PRICE) {
            it('should return correct results', function () {
                expect(PRICE(d('2008-02-15'), d('2017-05-15'), 0.065, 0.065, 100, 2, 0)).to.be.closeTo(100, 1e-1);
                expect(PRICE(d('2008-02-15'), d('2017-05-15'), 0.1, 0, 100, 2, 0)).to.be.closeTo(192.5, 1e-2);
                expect(PRICE(d('2007-02-15'), d('2008-02-15'), 0, 1, 100, 1, 0)).to.be.closeTo(50, 1e-2);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(PRICE(d('2017-05-15'), d('2008-02-15'), 0.0575, 0.065, 100, 2, 0)).to.equal(ErrorCode.NUM);
                expect(PRICE(d('2008-02-15'), d('2017-05-15'), 0.0575, 0.065, 0, 2, 0)).to.equal(ErrorCode.NUM);
                expect(PRICE(d('2008-02-15'), d('2017-05-15'), -1, 0.065, 100, 2, 0)).to.equal(ErrorCode.NUM);
                expect(PRICE(d('2008-02-15'), d('2017-05-15'), 0.0575, -1, 100, 2, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PRICEDISC', function (PRICEDISC) {
            it('should return correct results', function () {
                expect(PRICEDISC(d('2008-03-01'), d('2009-03-01'), 0.1, 99, 0)).to.be.closeTo(89.1, 1e-5);
                expect(PRICEDISC(d('2008-03-01'), d('2009-03-01'), 0.1, 99, 2)).to.be.closeTo(88.9625, 1e-5);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(PRICEDISC(d('2009-03-01'), d('2008-03-01'), 0.1, 99)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('PRICEMAT', function (PRICEMAT) {
            it('should return correct results', function () {
                expect(PRICEMAT(d('2009-02-15'), d('2010-02-15'), d('2007-11-11'), 0, 0, 0)).to.be.closeTo(100, 1e-5);
                expect(PRICEMAT(d('2009-02-15'), d('2010-02-15'), d('2007-11-11'), 0.2, 0, 0)).to.be.closeTo(120, 1e-5);
                expect(PRICEMAT(d('2009-02-15'), d('2010-02-15'), d('2007-11-11'), 0, 1, 0)).to.be.closeTo(50, 1e-5);
            });
        });

        moduleTester.testFunction('PV', function (PV) {
            it('should return correct results', function () {
                expect(PV(0.08 / 12, 12 * 20, 500, undefined, 0)).to.be.closeTo(-59777.14585, 1e-5);
                expect(PV(0.08 / 12, 12 * 20, 500, undefined, 1)).to.be.closeTo(-60175.66016, 1e-5);
            });
        });

        moduleTester.testFunction('RATE', function (RATE) {
            it('should return correct results', function () {
                expect(RATE(4 * 12, -200, 8000)).to.be.closeTo(0.007701472, 1e-6);
                expect(RATE(0, -200, 8000)).to.equal(ErrorCode.NUM);
                expect(RATE(4 * 12, -200, 0)).to.equal(ErrorCode.NUM);
                expect(RATE(4 * 12, -200, 0, -500, 1, -1)).to.equal(ErrorCode.NUM);
            });
            it('should have fixed behavior', function () {
                //important because of a bug in open office!
                expect(RATE(4 * 12, -200, 0, 0.0001, -500)).to.be.closeTo(-1, 1e-6);
                expect(RATE(4 * 12, -200, 0, 0.00002, -500)).to.be.closeTo(-1, 1e-6);
                expect(RATE(4 * 12, -200, 0, 0.00001, -500)).to.equal(ErrorCode.NUM);

                expect(RATE(1, -200, 8000)).to.be.closeTo(-0.975, 1e-6);
                expect(RATE(1.5, -200, 8000)).to.be.closeTo(-0.91062293265641, 1e-6);
                expect(RATE(0.5, -200, 8000)).to.equal(ErrorCode.NUM);
            });
            it('should return correct results with guess is zero', function () {
                expect(RATE(4, -200, 8000, -500, 1, 0)).to.be.closeTo(-0.452187382445176, 1e-10);
            });
            it('should return correct results with guess is null', function () {
                expect(RATE(4, -200, 8000, -500, 1, null)).to.be.closeTo(-0.452187382445176, 1e-10);
            });
            it('should return correct results with guess is assigned with numbers', function () {
                expect(RATE(4, -200, 8000, -500, 1, 50)).to.be.closeTo(-0.452187382445177, 1e-10);
                expect(RATE(4, -200, 8000, -500, 1, 5)).to.be.closeTo(-0.452187382445171, 1e-10);
                expect(RATE(4, -200, 8000, -500, 1, 1)).to.be.closeTo(-0.452187382441825, 1e-10);
            });
            it('should return correct results with guess is not assigned', function () {
                expect(RATE(4, -200, 8000, -500, 1)).to.be.closeTo(-0.452187382443900, 1e-10);
            });
            it('should return correct results with guess is default (0.1)', function () {
                expect(RATE(4, -200, 8000, -500, 1, 0.1)).to.be.closeTo(-0.452187382443900, 1e-10);
            });
        });

        moduleTester.testFunction('RECEIVED', function (RECEIVED) {
            it('should return correct results', function () {
                expect(RECEIVED(d('2008-02-15'), d('2008-05-15'), 1000000, 0.0575)).to.be.closeTo(1014584.6543, 1e-3);

                expect(RECEIVED(d('2009-02-15'), d('2010-02-15'), 900000, 0.1, 0)).to.almostEqual(1000000);
                expect(RECEIVED(d('2009-02-15'), d('2010-02-15'), 900000, 0.1, 1)).to.almostEqual(1000000);
                expect(RECEIVED(d('2009-02-15'), d('2010-02-15'), 900000, 0.1, 4)).to.almostEqual(1000000);

                expect(RECEIVED(d('2009-02-15'), d('2010-02-10'), 900000, 0.1, 2)).to.almostEqual(1000000);
                expect(RECEIVED(d('2009-02-15'), d('2010-02-15'), 900000, 0.1, 3)).to.almostEqual(1000000);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(RECEIVED(d('2010-05-15'), d('2009-02-15'), 900000, 0.1)).to.equal(ErrorCode.NUM);
                expect(RECEIVED(d('2009-02-15'), d('2010-05-15'), 900000, 0)).to.equal(ErrorCode.NUM);
                expect(RECEIVED(d('2009-02-15'), d('2010-05-15'), 0, 0.1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('RRI', function (RRI) {
            it('should return correct results', function () {
                expect(RRI(96, 10000, 11000)).to.be.closeTo(0.00099330737, 1e-10);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(RRI(0, 10000, 11000)).to.equal(ErrorCode.NUM);
                expect(RRI(96, 0, 11000)).to.equal(ErrorCode.NUM);
                expect(RRI(96, 10000, -1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('SLN', function (SLN) {
            it('should return correct results', function () {
                expect(SLN(30000, 7500, 10)).to.be.closeTo(2250, 1e-10);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(SLN(30000, -1, 10)).to.equal(ErrorCode.NUM);
                expect(SLN(30000, 7500, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('SYD', function (SYD) {
            it('should return correct results', function () {
                expect(SYD(30000, 7500, 10, 1)).to.be.closeTo(4090.9091, 1e-3);
                expect(SYD(30000, 7500, 10, 10)).to.be.closeTo(409.09091, 1e-3);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(SYD(30000, 7500, 0, 10)).to.equal(ErrorCode.NUM);
                expect(SYD(30000, 7500, 10, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('TBILLEQ', function (TBILLEQ) {
            it('should return correct results', function () {
                expect(TBILLEQ(d('2008-03-31'), d('2008-06-01'), 0.09)).to.be.closeTo(0.092686643, 1e-8);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(TBILLEQ(d('2009-04-01'), d('2008-03-31'), 0.09)).to.equal(ErrorCode.NUM);
                expect(TBILLEQ(d('2008-03-31'), d('2008-06-01'), 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('TBILLPRICE', function (TBILLPRICE) {
            it('should return correct results', function () {
                expect(TBILLPRICE(d('2008-03-31'), d('2008-06-01'), 0.09)).to.almostEqual(98.45);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(TBILLPRICE(d('2009-04-01'), d('2008-03-31'), 0.09)).to.equal(ErrorCode.NUM);
                expect(TBILLPRICE(d('2008-03-31'), d('2008-06-01'), 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('TBILLYIELD', function (TBILLYIELD) {
            it('should return correct results', function () {
                expect(TBILLYIELD(d('2008-03-31'), d('2008-06-01'), 98.45)).to.almostEqual(0.09141696292534233);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(TBILLYIELD(d('2008-03-31'), d('2009-04-01'), 98.45)).to.equal(ErrorCode.NUM);
                expect(TBILLYIELD(d('2008-03-31'), d('2008-06-01'), 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('VDB', function (VDB) {
            it('should return correct results', function () {
                expect(VDB(2400, 300, 10 * 365, 0, 1)).to.be.closeTo(1.315068493, 1e-6);
                expect(VDB(2400, 300, 10 * 12, 0, 1)).to.equal(40);
                expect(VDB(2400, 300, 10 * 1, 0, 1)).to.equal(480);
                expect(VDB(2400, 300, 10 * 12, 6, 18)).to.be.closeTo(396.3060533, 1e-6);
                expect(VDB(2400, 300, 10 * 12, 6, 18, 1.5)).to.be.closeTo(311.8089367, 1e-6);
                expect(VDB(2400, 300, 10 * 1, 0, 0.875, 1.5)).to.equal(315);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(VDB(-1, 300, 10 * 1, 0, 0.875, 1.5)).to.equal(ErrorCode.NUM);
                expect(VDB(2400, -1, 10 * 1, 0, 0.875, 1.5)).to.equal(ErrorCode.NUM);
                expect(VDB(2400, 300, -1 * 1, 0, 0.875, 1.5)).to.equal(ErrorCode.NUM);
                expect(VDB(2400, 300, 10 * 1, -1, 0.875, 1.5)).to.equal(ErrorCode.NUM);
                expect(VDB(2400, 300, 10 * 1, 0, -1, 1.5)).to.equal(ErrorCode.NUM);
                expect(VDB(2400, 300, 10 * 1, 0, 0.875, -1)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('XIRR', function (XIRR) {
            it('should return correct results', function () {
                expect(XIRR(mat([[-10000, 2750, 4250, 3250, 2750]]), dmat([['2008-01-01', '2008-03-01', '2008-10-30', '2009-02-15', '2009-04-01']]))).to.be.closeTo(0.373362535, 1e-6);
            });
            it('should throw #NUM! error on wrong input data', function () {
                expect(XIRR(mat([[-10000, 2750, 4250, 3250]]), dmat([['2008-01-01', '2008-03-01', '2008-10-30', '2009-02-15', '2009-04-01']]))).to.equal(ErrorCode.NUM);
                expect(XIRR(mat([[-10000, 2750, 4250, 3250, 2750]]), dmat([['2008-01-01', '2008-03-01', '2008-10-30', '2009-02-15', '2009-04-01']]), -2)).to.equal(ErrorCode.NUM);
                expect(XIRR(mat([[-10000]]), dmat([['2008-01-01']]))).to.equal(ErrorCode.NUM);
                //TODO: test floats!!!
            });
        });

        moduleTester.testFunction('XNPV', function (XNPV) {
            it('should return correct results', function () {
                expect(XNPV(0.09, mat([[-10000, 2750, 4250, 3250, 2750]]), dmat([['2008-01-01', '2008-03-01', '2008-10-30', '2009-02-15', '2009-04-01']]))).to.be.closeTo(2086.647602, 1e-6);
            });
            it('should throw #NUM! error on wrong input data', function () {
                expect(XNPV(0.09, mat([[-10000, 2750, 4250, 3250]]), dmat([['2008-01-01', '2008-03-01', '2008-10-30', '2009-02-15', '2009-04-01']]))).to.equal(ErrorCode.NUM);
                expect(XNPV(0.09, mat([[-10000]]), dmat([['2008-01-01']]))).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('YIELD', function (YIELD) {
            it('should return correct results', function () {
                expect(YIELD(d('2008-02-15'), d('2016-11-15'), 0.0575, 95.04287, 100, 2, 0)).to.almostEqual(0.06500000688075479);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(YIELD(d('2008-02-16'), d('2008-03-01'), -1, 100, 2)).to.equal(ErrorCode.NUM);
                expect(YIELD(d('2008-02-16'), d('2008-03-01'), 99.795, -1, 2)).to.equal(ErrorCode.NUM);
                expect(YIELD(d('2008-03-01'), d('2008-02-16'), 99.795, 100, 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('YIELDDISC', function (YIELDDISC) {
            it('should return correct results', function () {
                expect(YIELDDISC(d('2008-02-16'), d('2008-03-01'), 99.795, 100, 2)).to.almostEqual(0.052822571986860085);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(YIELDDISC(d('2008-02-16'), d('2008-03-01'), -1, 100, 2)).to.equal(ErrorCode.NUM);
                expect(YIELDDISC(d('2008-02-16'), d('2008-03-01'), 99.795, -1, 2)).to.equal(ErrorCode.NUM);
                expect(YIELDDISC(d('2008-03-01'), d('2008-02-16'), 99.795, 100, 2)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('YIELDMAT', function (YIELDMAT) {
            it('should return correct results', function () {
                expect(YIELDMAT(d('2008-03-15'), d('2008-11-03'), d('2007-11-08'), 0.0625, 100.0123, 0)).to.almostEqual(0.060954333691538576);
            });
            it('should throw #NUM! error code for wrong input data', function () {
                expect(YIELDMAT(d('2008-03-15'), d('2008-11-03'), d('2007-11-08'), -1, 100.0123, 0)).to.equal(ErrorCode.NUM);
                expect(YIELDMAT(d('2008-03-15'), d('2008-11-03'), d('2007-11-08'), 0.0625, 0, 0)).to.equal(ErrorCode.NUM);
                expect(YIELDMAT(d('2008-11-03'), d('2008-03-15'), d('2007-11-08'), 0.0625, 100.0123, 0)).to.equal(ErrorCode.NUM);
            });
        });
    });

    // ========================================================================
});
