/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/funcs/datetimefuncs'
], function (AppHelper, SheetHelper, DateTimeFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var d = SheetHelper.d;
    var t = SheetHelper.t;
    var dmat = SheetHelper.dmat;

    // module DateTimeFuncs ===================================================

    describe('Spreadsheet module DateTimeFuncs', function () {

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml');
        var moduleTester = SheetHelper.createFunctionModuleTester(DateTimeFuncs, appPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('DATEVALUE', function (DATEVALUE) {
            it('should return correct results', function () {
                expect(DATEVALUE('22.08.2011')).to.equal(40777);
                expect(DATEVALUE('22. Mai. 2011')).to.equal(40685);
                expect(DATEVALUE('11/3/2011')).to.equal(40613);
                expect(DATEVALUE('2:24')).to.equal(0);
                expect(DATEVALUE('22.08.2011 6:35')).to.equal(40777);
            });
            it('should throw error', function () {
                expect(DATEVALUE('29.12.1899')).to.equal(ErrorCode.VALUE);
                expect(DATEVALUE('1.1.10000')).to.equal(ErrorCode.VALUE);
                expect(DATEVALUE('')).to.equal(ErrorCode.VALUE);
                expect(DATEVALUE('hallo')).to.equal(ErrorCode.VALUE);
                expect(DATEVALUE('22.08 2011')).to.equal(ErrorCode.VALUE);
                expect(DATEVALUE('2011')).to.equal(ErrorCode.VALUE);
                expect(DATEVALUE('5. 5.')).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('DAYS360', function (DAYS360) {
            it('should return correct results', function () {
                expect(DAYS360(d('2011-02-28'), d('2011-03-31'), false)).to.equal(30);
                expect(DAYS360(d('2011-02-28'), d('2011-03-31'), true)).to.equal(32);
                expect(DAYS360(d('2011-02-28'), d('2011-03-31'))).to.equal(30);
                // TODO: more tests
            });
        });

        moduleTester.testFunction('MONTHS', function (MONTHS) {
            it('should return correct results months between two assigned dates', function () {
                expect(MONTHS(d('2010-04-03'), d('2011-06-17'), 0)).to.equal(14);
                expect(MONTHS(d('2010-03-31'), d('2010-04-30'), 0)).to.equal(0);
                expect(MONTHS(d('2010-03-31'), d('2010-06-30'), 0)).to.equal(2);
                expect(MONTHS(d('2010-04-03'), d('2011-06-17'), 1)).to.equal(14);
                expect(MONTHS(d('2010-03-31'), d('2010-04-01'), 1)).to.equal(1);
            });
        });

        moduleTester.testFunction('NETWORKDAYS', function (NETWORKDAYS) {
            it('should return correct results netto workdays between two assigned dates', function () {
                expect(NETWORKDAYS(d('2015-10-05'), d('2015-10-06'))).to.equal(2);
                expect(NETWORKDAYS(d('2015-10-06'), d('2015-10-05'))).to.equal(-2);
                expect(NETWORKDAYS(d('2015-10-05'), d('2015-10-09'))).to.equal(5);
                expect(NETWORKDAYS(d('2015-10-05'), d('2015-10-12'))).to.equal(6);
                expect(NETWORKDAYS(d('2015-10-02'), d('2015-10-12'))).to.equal(7);

                expect(NETWORKDAYS(d('1983-06-23'), d('2015-10-02'))).to.equal(8422);
                expect(NETWORKDAYS(d('2015-10-02'), d('2015-10-05'))).to.equal(2);
                expect(NETWORKDAYS(d('2015-10-05'), d('1983-06-23'))).to.equal(-8423);
                expect(NETWORKDAYS(d('2015-10-05'), d('2915-12-07'))).to.equal(234845);
            });
        });

        moduleTester.testFunction('NETWORKDAYS.INTL', function (NETWORKDAYS_INTL) {
            it('should return correct results netto workdays between two assigned dates', function () {
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'))).to.equal(5);
            });
            it('should return correct results netto workdays between two assigned dates and optional weekenddays', function () {
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 1)).to.equal(5);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 2)).to.equal(4);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 3)).to.equal(3);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 4)).to.equal(3);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 5)).to.equal(3);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 5)).to.equal(3);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 7)).to.equal(4);

                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), true)).to.equal(5); //true is interpreted as 1
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 0)).to.equal(ErrorCode.NUM);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 8)).to.equal(ErrorCode.NUM);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), '000')).to.equal(ErrorCode.VALUE);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), false)).to.equal(ErrorCode.NUM);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 'hello')).to.equal(ErrorCode.VALUE);
            });

            it('should return correct results netto workdays between two assigned dates and optional weekenddays & holidays', function () {
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 1, d('2015-10-07'))).to.equal(4);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 2, d('2015-10-07'))).to.equal(3);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 3, d('2015-10-07'))).to.equal(2);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 4, d('2015-10-07'))).to.equal(3);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 5, d('2015-10-07'))).to.equal(3);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 6, d('2015-10-07'))).to.equal(2);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-09'), 7, d('2015-10-07'))).to.equal(3);

                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-11'), '0000000', dmat([['2015-10-05', '2015-10-06']]))).to.equal(5);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-11'), '0000000', dmat([['2015-10-09', '2015-10-10']]))).to.equal(5);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-11'), '0000000', dmat([['2015-10-09', '2015-10-10', '2015-10-11']]))).to.equal(4);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2015-10-11'), '0000000', dmat([['2015-10-11', '2015-10-09']]))).to.equal(5);

                expect(NETWORKDAYS_INTL(d('1983-06-23'), d('2015-10-02'), 4, d('2015-10-02'))).to.equal(8421);
                expect(NETWORKDAYS_INTL(d('2015-10-02'), d('2015-10-05'), 4, d('2015-10-02'))).to.equal(3);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('1983-06-23'), 4, d('2015-10-02'))).to.equal(-8424);
                expect(NETWORKDAYS_INTL(d('2015-10-05'), d('2915-12-07'), 4, d('2015-10-02'))).to.equal(234844);
            });
        });

        moduleTester.testFunction('TIME', function (TIME) {
            it('should return correct result', function () {
                expect(TIME(0, 0, 0)).to.deep.equal(t('00:00:00'));
                expect(TIME(0, 0, 1)).to.deep.equal(t('00:00:01'));
                expect(TIME(0, 0, 60)).to.deep.equal(t('00:01:00'));
                expect(TIME(0, 1, 0)).to.deep.equal(t('00:01:00'));
                expect(TIME(0, 1, 1)).to.deep.equal(t('00:01:01'));
                expect(TIME(0, 1, 60)).to.deep.equal(t('00:02:00'));
                expect(TIME(0, 1, -1)).to.deep.equal(t('00:00:59'));
                expect(TIME(0, 2, 0)).to.deep.equal(t('00:02:00'));
                expect(TIME(0, 60, 0)).to.deep.equal(t('01:00:00'));
                expect(TIME(1, 0, 0)).to.deep.equal(t('01:00:00'));
                expect(TIME(1, -1, 0)).to.deep.equal(t('00:59:00'));
                expect(TIME(1, 0, -1)).to.deep.equal(t('00:59:59'));
                expect(TIME(2, 60, 3600)).to.deep.equal(t('04:00:00'));
                expect(TIME(-1, 60, 0)).to.deep.equal(t('00:00:00'));
            });
            it('should truncate large results', function () {
                expect(TIME(24, 0, 0)).to.deep.equal(t('00:00:00'));
                expect(TIME(36, 0, 0)).to.deep.equal(t('12:00:00'));
                expect(TIME(48, 0, 0)).to.deep.equal(t('00:00:00'));
                expect(TIME(23, 60, 0)).to.deep.equal(t('00:00:00'));
            });
            it('should return #NUM! error for negative results', function () {
                expect(TIME(0, 0, -1)).to.equal(ErrorCode.NUM);
                expect(TIME(0, 1, -61)).to.equal(ErrorCode.NUM);
                expect(TIME(1, -60, -1)).to.equal(ErrorCode.NUM);
                expect(TIME(-1, 0, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('TIMEVALUE', function (TIMEVALUE) {
            it('should return correct results', function () {
                expect(TIMEVALUE('2:24')).to.almostEqual(0.1);
                expect(TIMEVALUE('2 : 24')).to.almostEqual(0.1);
                expect(TIMEVALUE('22.08.2011 6:00')).to.almostEqual(0.25);
                expect(TIMEVALUE('23. Mai. 2011')).to.equal(0);
            });
            it('should throw error', function () {
                expect(TIMEVALUE('29.12.1899')).to.equal(ErrorCode.VALUE);
                expect(TIMEVALUE('1.1.10000')).to.equal(ErrorCode.VALUE);
                expect(TIMEVALUE('15.59')).to.equal(ErrorCode.VALUE);
                expect(TIMEVALUE('')).to.equal(ErrorCode.VALUE);
                expect(TIMEVALUE('hallo')).to.equal(ErrorCode.VALUE);
                expect(TIMEVALUE('20')).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('WEEKS', function (WEEKS) {
            it('should return correct number of weeks between two dates', function () {
                expect(WEEKS(d('2010-05-03'), d('2010-05-17'), 0)).to.equal(2);
                expect(WEEKS(d('2010-05-04'), d('2010-05-27'), 0)).to.equal(3);
                expect(WEEKS(d('2010-05-06'), d('2010-05-17'), 1)).to.equal(2);
                expect(WEEKS(d('2010-05-09'), d('2010-05-10'), 1)).to.equal(1);
                expect(WEEKS(d('2010-05-03'), d('2010-05-09'), 1)).to.equal(0);
            });
        });

        moduleTester.testFunction('WORKDAY', function (WORKDAY) {
            it('should return correct date', function () {
                expect(WORKDAY(d('2015-10-05'), 2)).to.deep.equal(d('2015-10-07'));
                expect(WORKDAY(d('2015-10-06'), -2)).to.deep.equal(d('2015-10-02'));
                expect(WORKDAY(d('2015-10-05'), 5)).to.deep.equal(d('2015-10-12'));
                expect(WORKDAY(d('2015-10-05'), 6)).to.deep.equal(d('2015-10-13'));
                expect(WORKDAY(d('2015-10-02'), 7)).to.deep.equal(d('2015-10-13'));

                expect(WORKDAY(d('1983-06-23'), 8422)).to.deep.equal(d('2015-10-05'));
                expect(WORKDAY(d('2015-10-02'), 2)).to.deep.equal(d('2015-10-06'));
                expect(WORKDAY(d('2015-10-05'), -8423)).to.deep.equal(d('1983-06-22'));
                expect(WORKDAY(d('2015-10-05'), 234845)).to.deep.equal(d('2915-12-09'));
            });
        });

        moduleTester.testFunction('WORKDAY.INTL', function (WORKDAY_INTL) {
            it('should return correct date', function () {
                expect(WORKDAY_INTL(d('2015-10-05'), 5)).to.deep.equal(d('2015-10-12'));
            });
            it('should return correct results next workday between assigned date, day count and optional weekenddays', function () {
                expect(WORKDAY_INTL(d('2015-10-05'), 5, 1)).to.deep.equal(d('2015-10-12'));

                //first day is a weekend day
                expect(WORKDAY_INTL(d('2015-10-04'), 5, 1)).to.deep.equal(d('2015-10-09'));
                expect(WORKDAY_INTL(d('2015-10-05'), 5, '1000000')).to.deep.equal(d('2015-10-10'));
                expect(WORKDAY_INTL(d('2015-10-05'), 5, 2)).to.deep.equal(d('2015-10-10'));

                expect(WORKDAY_INTL(d('2015-10-05'), 5, 3)).to.deep.equal(d('2015-10-11'));
                expect(WORKDAY_INTL(d('2015-10-05'), 5, 4)).to.deep.equal(d('2015-10-12'));
                expect(WORKDAY_INTL(d('2015-10-05'), 5, 5)).to.deep.equal(d('2015-10-12'));
                expect(WORKDAY_INTL(d('2015-10-05'), 5, 6)).to.deep.equal(d('2015-10-12'));
                expect(WORKDAY_INTL(d('2015-10-05'), 5, 7)).to.deep.equal(d('2015-10-12'));

                expect(WORKDAY_INTL(d('2015-10-05'), 5, true)).to.deep.equal(d('2015-10-12')); //true is interpreted as 1
                expect(WORKDAY_INTL(d('2015-10-05'), 5, 0)).to.equal(ErrorCode.NUM);
                expect(WORKDAY_INTL(d('2015-10-05'), 5, 8)).to.equal(ErrorCode.NUM);
                expect(WORKDAY_INTL(d('2015-10-05'), 5, '000')).to.equal(ErrorCode.VALUE);
                expect(WORKDAY_INTL(d('2015-10-05'), 5, false)).to.equal(ErrorCode.NUM);
                expect(WORKDAY_INTL(d('2015-10-05'), 5, 'hello')).to.equal(ErrorCode.VALUE);
                expect(WORKDAY_INTL(d('2015-10-05'), 5, '1111111')).to.equal(ErrorCode.VALUE);
            });

            it('should return correct results next workday between assigned date, day count and optional weekenddays & holidays', function () {
                expect(WORKDAY_INTL(d('2015-10-05'), 4, 1, d('2015-10-07'))).to.deep.equal(d('2015-10-12'));
                expect(WORKDAY_INTL(d('2015-10-05'), 3, 2, d('2015-10-07'))).to.deep.equal(d('2015-10-09'));
                expect(WORKDAY_INTL(d('2015-10-05'), 2, 3, d('2015-10-07'))).to.deep.equal(d('2015-10-09'));
                expect(WORKDAY_INTL(d('2015-10-05'), 3, 4, d('2015-10-07'))).to.deep.equal(d('2015-10-10'));
                expect(WORKDAY_INTL(d('2015-10-05'), 3, 5, d('2015-10-07'))).to.deep.equal(d('2015-10-10'));
                expect(WORKDAY_INTL(d('2015-10-05'), 2, 6, d('2015-10-07'))).to.deep.equal(d('2015-10-10'));
                expect(WORKDAY_INTL(d('2015-10-05'), 3, 7, d('2015-10-07'))).to.deep.equal(d('2015-10-11'));

                expect(WORKDAY_INTL(d('2015-10-05'), 5, '0000000', dmat([['2015-10-05', '2015-10-06']]))).to.deep.equal(d('2015-10-11'));
                expect(WORKDAY_INTL(d('2015-10-05'), 5, '0000000', dmat([['2015-10-09', '2015-10-10']]))).to.deep.equal(d('2015-10-12'));
                expect(WORKDAY_INTL(d('2015-10-05'), 4, '0000000', dmat([['2015-10-09', '2015-10-10', '2015-10-11']]))).to.deep.equal(d('2015-10-12'));
                expect(WORKDAY_INTL(d('2015-10-05'), 5, '0000000', dmat([['2015-10-11', '2015-10-09']]))).to.deep.equal(d('2015-10-12'));

                expect(WORKDAY_INTL(d('1983-06-23'), 8421, 4, d('2015-10-02'))).to.deep.equal(d('2015-10-03'));
                expect(WORKDAY_INTL(d('2015-10-02'), 3, 4, d('2015-10-02'))).to.deep.equal(d('2015-10-05'));
                expect(WORKDAY_INTL(d('2015-10-05'), 234844, 4, d('2015-10-02'))).to.deep.equal(d('2915-12-08'));
            });

            it('should return correct results last workday (negative workdays) between assigned date, day count and optional weekenddays & holidays', function () {
                expect(WORKDAY_INTL(d('2015-10-05'), -8424, 1)).to.deep.equal(d('1983-06-21'));
                expect(WORKDAY_INTL(d('2015-10-05'), -8424, '0000000')).to.deep.equal(d('1992-09-11'));
                expect(WORKDAY_INTL(d('2015-10-05'), -8424, '0000000', d('2015-10-02'))).to.deep.equal(d('1992-09-10'));
                expect(WORKDAY_INTL(d('2015-10-05'), -8424, 1, d('2015-10-02'))).to.deep.equal(d('1983-06-20'));
                expect(WORKDAY_INTL(d('2015-10-05'), -8424, 2, d('2015-10-02'))).to.deep.equal(d('1983-06-21'));
                expect(WORKDAY_INTL(d('2015-10-05'), -8424, 3, d('2015-10-02'))).to.deep.equal(d('1983-06-22'));
                expect(WORKDAY_INTL(d('2015-10-05'), -8424, 4, d('2015-10-02'))).to.deep.equal(d('1983-06-20'));
            });
        });

        moduleTester.testFunction('YEARFRAC', function (YEARFRAC) {
            // TODO: more test cases to cover all execution paths
            it('should return correct results for date mode 0 (30/360 US)', function () {
                expect(YEARFRAC(d('2011-01-01'), d('2011-01-01'), 0)).to.equal(0);
                expect(YEARFRAC(d('2011-02-28'), d('2012-02-29'), 0)).to.equal(1);
                expect(YEARFRAC(d('2011-02-28'), d('2011-03-31'), 0)).to.equal(31 / 360);
            });
            it('should return correct results for date mode 1 (actual/actual)', function () {
                expect(YEARFRAC(d('2011-01-01'), d('2011-01-01'), 1)).to.equal(0);
                expect(YEARFRAC(d('2011-02-28'), d('2012-02-29'), 1)).to.equal(732 / 731);
                expect(YEARFRAC(d('2011-02-28'), d('2011-03-31'), 1)).to.equal(31 / 365);
            });
            it('should return correct results for date mode 2 (actual/360)', function () {
                expect(YEARFRAC(d('2011-01-01'), d('2011-01-01'), 2)).to.equal(0);
                expect(YEARFRAC(d('2011-02-28'), d('2012-02-29'), 2)).to.equal(61 / 60);
                expect(YEARFRAC(d('2011-02-28'), d('2011-03-31'), 2)).to.equal(31 / 360);
            });
            it('should return correct results for date mode 3 (actual/365)', function () {
                expect(YEARFRAC(d('2011-01-01'), d('2011-01-01'), 3)).to.equal(0);
                expect(YEARFRAC(d('2011-02-28'), d('2012-02-29'), 3)).to.equal(366 / 365);
                expect(YEARFRAC(d('2011-02-28'), d('2011-03-31'), 3)).to.equal(31 / 365);
            });
            it('should return correct results for date mode 4 (30E/360)', function () {
                expect(YEARFRAC(d('2011-01-01'), d('2011-01-01'), 4)).to.equal(0);
                expect(YEARFRAC(d('2011-02-28'), d('2012-02-29'), 4)).to.equal(361 / 360);
                expect(YEARFRAC(d('2011-02-28'), d('2011-03-31'), 4)).to.equal(32 / 360);
            });
            it('should default to date mode 0 (30/360 US)', function () {
                expect(YEARFRAC(d('2011-01-01'), d('2011-01-01'))).to.equal(0);
                expect(YEARFRAC(d('2011-02-28'), d('2012-02-29'))).to.equal(1);
                expect(YEARFRAC(d('2011-02-28'), d('2011-03-31'))).to.equal(31 / 360);
            });
        });

        moduleTester.testFunction('YEARS', function (YEARS) {
            it('should return correct number of years between two dates', function () {
                expect(YEARS(d('2009-04-03'), d('2011-11-17'), 0)).to.equal(2);
                expect(YEARS(d('2011-02-28'), d('2012-02-28'), 0)).to.equal(1);
                expect(YEARS(d('2012-02-29'), d('2013-02-28'), 0)).to.equal(0);
                expect(YEARS(d('2009-04-03'), d('2011-11-17'), 1)).to.equal(2);
                expect(YEARS(d('2009-12-31'), d('2010-01-01'), 1)).to.equal(1);
            });
        });
    });

    // ========================================================================
});
