/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/editframework/utils/mixedborder'
], function (MixedBorder) {

    'use strict';

    // static class MixedBorder ===============================================

    describe('EditFramework module MixedBorder', function () {

        it('should exist', function () {
            expect(MixedBorder).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var AUTO = { type: 'auto' },
            SOLID = { style: 'solid', width: 1, color: AUTO },
            DOTTED = { style: 'dotted', width: 1, color: AUTO },
            THICK = { style: 'solid', width: 10, color: AUTO },
            NONE = { style: 'none' },
            MIXED = { style: 'solid', width: 10, color: { type: 'rgb', value: 'C00000' }, mixed: true };

        // static methods -----------------------------------------------------

        describe('method "isVisibleBorder"', function () {
            it('should exist', function () {
                expect(MixedBorder).itself.to.respondTo('isVisibleBorder');
            });
            it('should return true for visible borders', function () {
                expect(MixedBorder.isVisibleBorder(SOLID)).to.equal(true);
                expect(MixedBorder.isVisibleBorder(DOTTED)).to.equal(true);
                expect(MixedBorder.isVisibleBorder(THICK)).to.equal(true);
                expect(MixedBorder.isVisibleBorder(_.extend({ mixed: false }, SOLID))).to.equal(true);
                expect(MixedBorder.isVisibleBorder(_.extend({ mixed: false }, DOTTED))).to.equal(true);
                expect(MixedBorder.isVisibleBorder(_.extend({ mixed: false }, THICK))).to.equal(true);
            });
            it('should return false for invisible borders', function () {
                expect(MixedBorder.isVisibleBorder(NONE)).to.equal(false);
                expect(MixedBorder.isVisibleBorder(_.extend({ mixed: false }, NONE))).to.equal(false);
                expect(MixedBorder.isVisibleBorder({ style: 'solid', width: 0, color: AUTO })).to.equal(false);
                expect(MixedBorder.isVisibleBorder(null)).to.equal(false);
            });
            it('should return true for visible ambiguous borders', function () {
                expect(MixedBorder.isVisibleBorder({ mixed: false, style: null, width: 1, color: AUTO })).to.equal(true);
                expect(MixedBorder.isVisibleBorder({ mixed: false, style: 'solid', width: null, color: AUTO })).to.equal(true);
                expect(MixedBorder.isVisibleBorder({ mixed: false, style: 'solid', width: 1, color: null })).to.equal(true);
                expect(MixedBorder.isVisibleBorder({ mixed: false, style: null, width: null, color: null })).to.equal(true);
            });
            it('should return true for mixed borders', function () {
                expect(MixedBorder.isVisibleBorder(_.extend({ mixed: true }, SOLID))).to.equal(true);
                expect(MixedBorder.isVisibleBorder(_.extend({ mixed: true }, DOTTED))).to.equal(true);
                expect(MixedBorder.isVisibleBorder(_.extend({ mixed: true }, THICK))).to.equal(true);
            });
            it('should return true for mixed ambiguous borders', function () {
                expect(MixedBorder.isVisibleBorder({ mixed: true, style: null, width: 1, color: AUTO })).to.equal(true);
                expect(MixedBorder.isVisibleBorder({ mixed: true, style: 'solid', width: null, color: AUTO })).to.equal(true);
                expect(MixedBorder.isVisibleBorder({ mixed: true, style: 'solid', width: 1, color: null })).to.equal(true);
                expect(MixedBorder.isVisibleBorder({ mixed: true, style: null, width: null, color: null })).to.equal(true);
            });
        });

        describe('method "hasAmbiguousProperties"', function () {
            it('should exist', function () {
                expect(MixedBorder).itself.to.respondTo('hasAmbiguousProperties');
            });
            it('should return true for ambiguous properties', function () {
                expect(MixedBorder.hasAmbiguousProperties({ style: null, width: 1, color: AUTO })).to.equal(true);
                expect(MixedBorder.hasAmbiguousProperties({ style: 'solid', width: null, color: AUTO })).to.equal(true);
                expect(MixedBorder.hasAmbiguousProperties({ style: 'solid', width: 1, color: null })).to.equal(true);
            });
            it('should return false for simple borders', function () {
                expect(MixedBorder.hasAmbiguousProperties(NONE)).to.equal(false);
                expect(MixedBorder.hasAmbiguousProperties(SOLID)).to.equal(false);
                expect(MixedBorder.hasAmbiguousProperties(DOTTED)).to.equal(false);
                expect(MixedBorder.hasAmbiguousProperties(THICK)).to.equal(false);
                expect(MixedBorder.hasAmbiguousProperties(null)).to.equal(false);
            });
            it('should ignore the mixed flag', function () {
                expect(MixedBorder.hasAmbiguousProperties(MIXED)).to.equal(false);
            });
        });

        describe('method "isFullyAmbiguousBorder"', function () {
            it('should exist', function () {
                expect(MixedBorder).itself.to.respondTo('isFullyAmbiguousBorder');
            });
            it('should return true for an ambiguous mixed border', function () {
                expect(MixedBorder.isFullyAmbiguousBorder({ mixed: true, style: null, width: null, color: null })).to.equal(true);
            });
            it('should return false for simple borders', function () {
                expect(MixedBorder.isFullyAmbiguousBorder(NONE)).to.equal(false);
                expect(MixedBorder.isFullyAmbiguousBorder(SOLID)).to.equal(false);
                expect(MixedBorder.isFullyAmbiguousBorder(DOTTED)).to.equal(false);
                expect(MixedBorder.isFullyAmbiguousBorder(THICK)).to.equal(false);
                expect(MixedBorder.isFullyAmbiguousBorder(null)).to.equal(false);
            });
            it('should return false for all other mixed borders', function () {
                expect(MixedBorder.isFullyAmbiguousBorder({ mixed: false, style: null, width: null, color: null })).to.equal(false);
                expect(MixedBorder.isFullyAmbiguousBorder({ style: null, width: null, color: null })).to.equal(false);
                expect(MixedBorder.isFullyAmbiguousBorder({ mixed: true, style: 'solid', width: null, color: null })).to.equal(false);
                expect(MixedBorder.isFullyAmbiguousBorder({ mixed: true, style: null, width: 0, color: null })).to.equal(false);
                expect(MixedBorder.isFullyAmbiguousBorder({ mixed: true, style: null, width: null, color: AUTO })).to.equal(false);
            });
        });

        describe('method "mixBorders"', function () {
            it('should exist', function () {
                expect(MixedBorder).itself.to.respondTo('mixBorders');
            });
            it('should return equal borders unmodified', function () {
                expect(MixedBorder.mixBorders(SOLID)).to.deep.equal(_.extend({ mixed: false }, SOLID));
                expect(MixedBorder.mixBorders(SOLID, SOLID)).to.deep.equal(_.extend({ mixed: false }, SOLID));
                expect(MixedBorder.mixBorders(SOLID, SOLID, SOLID)).to.deep.equal(_.extend({ mixed: false }, SOLID));
                expect(MixedBorder.mixBorders(DOTTED, DOTTED)).to.deep.equal(_.extend({ mixed: false }, DOTTED));
                expect(MixedBorder.mixBorders(THICK, THICK)).to.deep.equal(_.extend({ mixed: false }, THICK));
                expect(MixedBorder.mixBorders(NONE, NONE)).to.deep.equal(_.extend({ mixed: false }, NONE));
            });
            it('should mix visible borders', function () {
                expect(MixedBorder.mixBorders(SOLID, DOTTED)).to.deep.equal({ mixed: false, style: null, width: 1, color: AUTO });
                expect(MixedBorder.mixBorders(SOLID, THICK)).to.deep.equal({ mixed: false, style: 'solid', width: null, color: AUTO });
                expect(MixedBorder.mixBorders(DOTTED, THICK)).to.deep.equal({ mixed: false, style: null, width: null, color: AUTO });
                expect(MixedBorder.mixBorders(SOLID, DOTTED, THICK)).to.deep.equal({ mixed: false, style: null, width: null, color: AUTO });
            });
            it('should mix visible and invisible borders', function () {
                expect(MixedBorder.mixBorders(SOLID, NONE)).to.deep.equal(_.extend({ mixed: true }, SOLID));
                expect(MixedBorder.mixBorders(NONE, SOLID)).to.deep.equal(_.extend({ mixed: true }, SOLID));
                expect(MixedBorder.mixBorders(DOTTED, NONE)).to.deep.equal(_.extend({ mixed: true }, DOTTED));
                expect(MixedBorder.mixBorders(NONE, DOTTED)).to.deep.equal(_.extend({ mixed: true }, DOTTED));
                expect(MixedBorder.mixBorders(THICK, NONE)).to.deep.equal(_.extend({ mixed: true }, THICK));
                expect(MixedBorder.mixBorders(NONE, THICK)).to.deep.equal(_.extend({ mixed: true }, THICK));
                expect(MixedBorder.mixBorders(SOLID, DOTTED, NONE)).to.deep.equal({ mixed: true, style: null, width: 1, color: AUTO });
                expect(MixedBorder.mixBorders(SOLID, NONE, DOTTED)).to.deep.equal({ mixed: true, style: null, width: 1, color: AUTO });
                expect(MixedBorder.mixBorders(NONE, SOLID, DOTTED)).to.deep.equal({ mixed: true, style: null, width: 1, color: AUTO });
                expect(MixedBorder.mixBorders(SOLID, THICK, NONE)).to.deep.equal({ mixed: true, style: 'solid', width: null, color: AUTO });
                expect(MixedBorder.mixBorders(NONE, SOLID, THICK)).to.deep.equal({ mixed: true, style: 'solid', width: null, color: AUTO });
                expect(MixedBorder.mixBorders(DOTTED, THICK, NONE)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                expect(MixedBorder.mixBorders(NONE, DOTTED, THICK)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                expect(MixedBorder.mixBorders(SOLID, DOTTED, THICK, NONE)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                expect(MixedBorder.mixBorders(NONE, SOLID, DOTTED, THICK)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
            });
            it('should mix already mixed borders', function () {
                expect(MixedBorder.mixBorders(NONE, MIXED)).to.deep.equal(MIXED);
                expect(MixedBorder.mixBorders(MIXED, NONE)).to.deep.equal(MIXED);
                expect(MixedBorder.mixBorders(SOLID, _.extend({ mixed: true }, SOLID))).to.deep.equal(_.extend({ mixed: true }, SOLID));
                expect(MixedBorder.mixBorders(_.extend({ mixed: true }, SOLID), SOLID)).to.deep.equal(_.extend({ mixed: true }, SOLID));
                expect(MixedBorder.mixBorders(NONE, SOLID, DOTTED, MIXED)).to.deep.equal({ mixed: true, style: null, width: null, color: null });
                expect(MixedBorder.mixBorders(MIXED, SOLID, DOTTED, NONE)).to.deep.equal({ mixed: true, style: null, width: null, color: null });
            });
            it('should accept an array as parameter', function () {
                expect(MixedBorder.mixBorders([SOLID])).to.deep.equal(_.extend({ mixed: false }, SOLID));
                expect(MixedBorder.mixBorders([SOLID, NONE])).to.deep.equal(_.extend({ mixed: true }, SOLID));
                expect(MixedBorder.mixBorders([SOLID, DOTTED], THICK, NONE)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                expect(MixedBorder.mixBorders(SOLID, [DOTTED, THICK], NONE)).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                expect(MixedBorder.mixBorders(SOLID, DOTTED, [THICK, NONE])).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
                expect(MixedBorder.mixBorders([SOLID, DOTTED], [THICK, NONE])).to.deep.equal({ mixed: true, style: null, width: null, color: AUTO });
            });
        });

        describe('method "getBorderMode"', function () {
            it('should exist', function () {
                expect(MixedBorder).itself.to.respondTo('getBorderMode');
            });
            it('should return the border modes of a single border', function () {
                expect(MixedBorder.getBorderMode({ borderLeft: SOLID })).to.deep.equal({ left: true });
                expect(MixedBorder.getBorderMode({ borderRight: DOTTED })).to.deep.equal({ right: true });
                expect(MixedBorder.getBorderMode({ borderTop: THICK })).to.deep.equal({ top: true });
                expect(MixedBorder.getBorderMode({ borderBottom: NONE })).to.deep.equal({ bottom: false });
                expect(MixedBorder.getBorderMode({ borderInsideHor: MIXED })).to.deep.equal({ insideh: null });
                expect(MixedBorder.getBorderMode({ borderInsideVert: MIXED })).to.deep.equal({ insidev: null });
            });
            it('should ignore other attributes', function () {
                expect(MixedBorder.getBorderMode({ fillColor: AUTO })).to.deep.equal({});
                expect(MixedBorder.getBorderMode({ borderInside: SOLID })).to.deep.equal({});
            });
            it('should return the border modes of all borders', function () {
                expect(MixedBorder.getBorderMode({ borderLeft: SOLID, borderRight: DOTTED, borderTop: THICK, borderBottom: NONE, borderInsideHor: NONE, borderInsideVert: MIXED })).to.deep.equal({ left: true, right: true, top: true, bottom: false, insideh: false, insidev: null });
            });
            it('should return the border modes of a single border (paragraph mode)', function () {
                expect(MixedBorder.getBorderMode({ borderLeft: SOLID }, { paragraph: true })).to.deep.equal({ left: true });
                expect(MixedBorder.getBorderMode({ borderRight: DOTTED }, { paragraph: true })).to.deep.equal({ right: true });
                expect(MixedBorder.getBorderMode({ borderTop: THICK }, { paragraph: true })).to.deep.equal({ top: true });
                expect(MixedBorder.getBorderMode({ borderBottom: NONE }, { paragraph: true })).to.deep.equal({ bottom: false });
                expect(MixedBorder.getBorderMode({ borderInside: MIXED }, { paragraph: true })).to.deep.equal({ insideh: null });
            });
            it('should return the border modes of all borders (paragraph mode)', function () {
                expect(MixedBorder.getBorderMode({ borderLeft: SOLID, borderRight: DOTTED, borderTop: THICK, borderBottom: NONE, borderInside: MIXED }, { paragraph: true })).to.deep.equal({ left: true, right: true, top: true, bottom: false, insideh: null });
            });
            it('should ignore other attributes (paragraph mode)', function () {
                expect(MixedBorder.getBorderMode({ fillColor: AUTO }, { paragraph: true })).to.deep.equal({});
                expect(MixedBorder.getBorderMode({ borderInsideHor: SOLID }, { paragraph: true })).to.deep.equal({});
                expect(MixedBorder.getBorderMode({ borderInsideVert: SOLID }, { paragraph: true })).to.deep.equal({});
            });
        });

        describe('method "getBorderAttributes"', function () {
            it('should exist', function () {
                expect(MixedBorder).itself.to.respondTo('getBorderAttributes');
            });
            it('should return the border attributes of a single border', function () {
                expect(MixedBorder.getBorderAttributes({ left: true }, { borderLeft: DOTTED }, SOLID)).to.deep.equal({ borderLeft: DOTTED });
                expect(MixedBorder.getBorderAttributes({ right: false }, { borderRight: SOLID }, SOLID)).to.deep.equal({ borderRight: NONE });
                expect(MixedBorder.getBorderAttributes({ top: true }, { borderTop: NONE }, SOLID)).to.deep.equal({ borderTop: SOLID });
                expect(MixedBorder.getBorderAttributes({ bottom: false }, { borderBottom: NONE }, SOLID)).to.deep.equal({ borderBottom: NONE });
                expect(MixedBorder.getBorderAttributes({ insideh: true }, { borderInsideHor: SOLID }, DOTTED)).to.deep.equal({ borderInsideHor: SOLID });
                expect(MixedBorder.getBorderAttributes({ insidev: true }, { borderInsideVert: NONE }, DOTTED)).to.deep.equal({ borderInsideVert: DOTTED });
            });
            it('should return the border attributes of all borders', function () {
                expect(MixedBorder.getBorderAttributes({
                    left: true,
                    right: false,
                    top: true,
                    bottom: false,
                    insideh: true,
                    insidev: true
                }, {
                    borderLeft: DOTTED,
                    borderRight: SOLID,
                    borderTop: NONE,
                    borderBottom: NONE,
                    borderInsideHor: THICK,
                    borderInsideVert: NONE
                }, SOLID)).to.deep.equal({
                    borderLeft: DOTTED,
                    borderRight: NONE,
                    borderTop: SOLID,
                    borderBottom: NONE,
                    borderInsideHor: THICK,
                    borderInsideVert: SOLID
                });
            });
            it('should toggle the border attributes of a single border', function () {
                expect(MixedBorder.getBorderAttributes('left', { borderLeft: DOTTED }, SOLID)).to.deep.equal({ borderLeft: NONE });
                expect(MixedBorder.getBorderAttributes('right', { borderRight: NONE }, SOLID)).to.deep.equal({ borderRight: SOLID });
                expect(MixedBorder.getBorderAttributes('top', { borderTop: SOLID }, DOTTED)).to.deep.equal({ borderTop: NONE });
                expect(MixedBorder.getBorderAttributes('bottom', { borderBottom: NONE }, DOTTED)).to.deep.equal({ borderBottom: DOTTED });
                expect(MixedBorder.getBorderAttributes('insideh', { borderInsideHor: _.extend({ mixed: true }, THICK) }, DOTTED)).to.deep.equal({ borderInsideHor: THICK });
                expect(MixedBorder.getBorderAttributes('insidev', { borderInsideVert: NONE }, DOTTED)).to.deep.equal({ borderInsideVert: DOTTED });
            });
            it('should remove other attributes', function () {
                expect(MixedBorder.getBorderAttributes({ left: true }, { borderLeft: SOLID, borderRight: SOLID, borderInsideHor: SOLID, borderInsideVert: SOLID, borderInside: SOLID, fillColor: AUTO }, SOLID)).to.deep.equal({ borderLeft: SOLID });
            });
            it('should return the border attributes of a single border (paragraph mode)', function () {
                expect(MixedBorder.getBorderAttributes({ left: true }, { borderLeft: DOTTED }, SOLID, { paragraph: true })).to.deep.equal({ borderLeft: DOTTED });
                expect(MixedBorder.getBorderAttributes({ right: false }, { borderRight: SOLID }, SOLID, { paragraph: true })).to.deep.equal({ borderRight: NONE });
                expect(MixedBorder.getBorderAttributes({ top: true }, { borderTop: NONE }, SOLID, { paragraph: true })).to.deep.equal({ borderTop: SOLID });
                expect(MixedBorder.getBorderAttributes({ bottom: false }, { borderBottom: NONE }, SOLID, { paragraph: true })).to.deep.equal({ borderBottom: NONE });
                expect(MixedBorder.getBorderAttributes({ insideh: true }, { borderInside: SOLID }, DOTTED, { paragraph: true })).to.deep.equal({ borderInside: SOLID });
            });
            it('should return the border attributes of all borders (paragraph mode)', function () {
                expect(MixedBorder.getBorderAttributes({
                    left: true,
                    right: false,
                    top: true,
                    bottom: false,
                    insideh: true
                }, {
                    borderLeft: DOTTED,
                    borderRight: SOLID,
                    borderTop: NONE,
                    borderBottom: NONE,
                    borderInside: THICK
                }, SOLID, { paragraph: true })).to.deep.equal({
                    borderLeft: DOTTED,
                    borderRight: NONE,
                    borderTop: SOLID,
                    borderBottom: NONE,
                    borderInside: THICK
                });
            });
            it('should toggle the border attributes of a single border (paragraph mode)', function () {
                expect(MixedBorder.getBorderAttributes('left', { borderLeft: DOTTED }, SOLID, { paragraph: true })).to.deep.equal({ borderLeft: NONE });
                expect(MixedBorder.getBorderAttributes('right', { borderRight: NONE }, SOLID, { paragraph: true })).to.deep.equal({ borderRight: SOLID });
                expect(MixedBorder.getBorderAttributes('top', { borderTop: SOLID }, DOTTED, { paragraph: true })).to.deep.equal({ borderTop: NONE });
                expect(MixedBorder.getBorderAttributes('bottom', { borderBottom: NONE }, DOTTED, { paragraph: true })).to.deep.equal({ borderBottom: DOTTED });
                expect(MixedBorder.getBorderAttributes('insideh', { borderInside: _.extend({ mixed: true }, THICK) }, DOTTED, { paragraph: true })).to.deep.equal({ borderInside: THICK });
            });
            it('should remove other attributes (paragraph mode)', function () {
                expect(MixedBorder.getBorderAttributes({ left: true }, { borderLeft: SOLID, borderRight: SOLID, borderInsideHor: SOLID, borderInsideVert: SOLID, borderInside: SOLID, fillColor: AUTO }, SOLID, { paragraph: true })).to.deep.equal({ borderLeft: SOLID });
            });
        });
    });

    // ========================================================================
});
