/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/drawinglayer/utils/drawingutils'
], function (Rectangle, DrawingUtils) {

    'use strict';

    // static class DrawingUtils ==============================================

    describe('DrawingLayer module DrawingUtils', function () {

        it('should exist', function () {
            expect(DrawingUtils).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        function r(l, t, w, h) {
            return new Rectangle(l, t, w, h);
        }

        // static methods -----------------------------------------------------

        describe('method "createDefaultSizeForShapeId"', function () {
            it('should exist', function () {
                expect(DrawingUtils).itself.to.respondTo('createDefaultSizeForShapeId');
            });
            it('should create a width and height for a given presetShapeId that fits in the provided dimensions', function () {
                expect(DrawingUtils.createDefaultSizeForShapeId('ellipse', 100, 100)).to.deep.equal({ width: 100, height: 100 });
                expect(DrawingUtils.createDefaultSizeForShapeId('leftRightArrow', 100, 100)).to.deep.equal({ width: 100, height: 50 });
                expect(DrawingUtils.createDefaultSizeForShapeId('upDownArrow', 100, 100)).to.deep.equal({ width: 50, height: 100 });
                // test shape with a format that is not supported (e.g. scale-vertical-double) - > use square
                expect(DrawingUtils.createDefaultSizeForShapeId('bentConnector3_2', 100, 100)).to.deep.equal({ width: 100, height: 100 });
            });
        });

        describe('method "getTrackingRectangle"', function () {
            it('should exist', function () {
                expect(DrawingUtils).itself.to.respondTo('getTrackingRectangle');
            });
            it('should return a frame rectangle for normal tracking mode', function () {
                var options = { modifierEvent: { shiftKey: false, ctrlKey: false, altKey: false } };
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 200, options).equals(r(200, 200, 5, 5))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 250, options).equals(r(200, 200, 21, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 250, options).equals(r(200, 200, 5, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 250, options).equals(r(180, 200, 21, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 200, options).equals(r(180, 200, 21, 5))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 150, options).equals(r(180, 150, 21, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 150, options).equals(r(200, 150, 5, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 150, options).equals(r(200, 150, 21, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 200, options).equals(r(200, 200, 21, 5))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 150).equals(r(200, 150, 21, 51))).to.equal(true);
                options.boundRect = r(180, 190, 41, 21);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 200, options).equals(r(200, 200, 5, 5))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 210, 210, options).equals(r(200, 200, 11, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 220, options).equals(r(200, 200, 21, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(200, 200, 21, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 190, 190, options).equals(r(190, 190, 11, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 180, options).equals(r(180, 190, 21, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(180, 190, 21, 11))).to.equal(true);
            });
            it('should return a frame rectangle for square tracking mode', function () {
                var options = { modifierEvent: { shiftKey: true, ctrlKey: false, altKey: false } };
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 200, options).equals(r(200, 200, 5, 5))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 250, options).equals(r(200, 200, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 250, 220, options).equals(r(200, 200, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 250, options).equals(r(200, 200, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 250, options).equals(r(150, 200, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 150, 220, options).equals(r(150, 200, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 200, options).equals(r(180, 200, 21, 21))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 150, options).equals(r(150, 150, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 150, 180, options).equals(r(150, 150, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 150, options).equals(r(200, 150, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 150, options).equals(r(200, 150, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 250, 180, options).equals(r(200, 150, 51, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 200, options).equals(r(200, 200, 21, 21))).to.equal(true);
                options.boundRect = r(180, 190, 41, 21);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 200, options).equals(r(200, 200, 5, 5))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(200, 200, 11, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 170, options).equals(r(200, 190, 11, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(190, 190, 11, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 230, options).equals(r(190, 200, 11, 11))).to.equal(true);
            });
            it('should return a frame rectangle for center tracking mode', function () {
                var options = { modifierEvent: { shiftKey: false, ctrlKey: true, altKey: true } };
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 200, options).equals(r(198, 198, 5, 5))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 250, options).equals(r(180, 150, 41, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 250, options).equals(r(198, 150, 5, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 250, options).equals(r(180, 150, 41, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 200, options).equals(r(180, 198, 41, 5))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 150, options).equals(r(180, 150, 41, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 150, options).equals(r(198, 150, 5, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 150, options).equals(r(180, 150, 41, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 200, options).equals(r(180, 198, 41, 5))).to.equal(true);
                options.boundRect = r(180, 190, 51, 31);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(180, 190, 41, 21))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(180, 190, 41, 21))).to.equal(true);
                options.boundRect = r(170, 180, 51, 31);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(180, 190, 41, 21))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(180, 190, 41, 21))).to.equal(true);
                options.boundRect = r(190, 180, 31, 51);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(190, 180, 21, 41))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(190, 180, 21, 41))).to.equal(true);
                options.boundRect = r(180, 170, 31, 51);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(190, 180, 21, 41))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(190, 180, 21, 41))).to.equal(true);
            });
            it('should return a frame rectangle for square/center tracking mode', function () {
                var options = { modifierEvent: { shiftKey: true, ctrlKey: true, altKey: true } };
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 200, options).equals(r(198, 198, 5, 5))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 250, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 250, 220, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 250, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 250, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 150, 220, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 200, options).equals(r(180, 180, 41, 41))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 150, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 150, 180, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 150, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 150, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 250, 180, options).equals(r(150, 150, 101, 101))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 200, options).equals(r(180, 180, 41, 41))).to.equal(true);
                options.boundRect = r(180, 190, 51, 31);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(190, 190, 21, 21))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(190, 190, 21, 21))).to.equal(true);
                options.boundRect = r(170, 180, 51, 31);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(190, 190, 21, 21))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(190, 190, 21, 21))).to.equal(true);
                options.boundRect = r(190, 180, 31, 51);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(190, 190, 21, 21))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(190, 190, 21, 21))).to.equal(true);
                options.boundRect = r(180, 170, 31, 51);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(190, 190, 21, 21))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(190, 190, 21, 21))).to.equal(true);
            });
            it('should return a frame rectangle for two-point mode', function () {
                var options = { modifierEvent: { shiftKey: false, ctrlKey: false, altKey: false }, twoPointMode: true };
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 200, options).equals(r(200, 200, 1, 1))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 250, options).equals(r(200, 200, 21, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 250, options).equals(r(200, 200, 1, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 250, options).equals(r(180, 200, 21, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 200, options).equals(r(180, 200, 21, 1))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 150, options).equals(r(180, 150, 21, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 150, options).equals(r(200, 150, 1, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 150, options).equals(r(200, 150, 21, 51))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 200, options).equals(r(200, 200, 21, 1))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 150).equals(r(200, 150, 21, 51))).to.equal(true);
                options.boundRect = r(180, 190, 41, 21);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 200, 200, options).equals(r(200, 200, 1, 1))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 210, 210, options).equals(r(200, 200, 11, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 220, 220, options).equals(r(200, 200, 21, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 230, 230, options).equals(r(200, 200, 21, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 190, 190, options).equals(r(190, 190, 11, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 180, 180, options).equals(r(180, 190, 21, 11))).to.equal(true);
                expect(DrawingUtils.getTrackingRectangle(200, 200, 170, 170, options).equals(r(180, 190, 21, 11))).to.equal(true);
            });
            it('should add custom properties to the resulting rectangle', function () {
                var rect1 = DrawingUtils.getTrackingRectangle(200, 200, 230, 240);
                expect(rect1).to.have.a.property('origShift', 50);
                expect(rect1).to.have.a.property('reverseX', false);
                expect(rect1).to.have.a.property('reverseY', false);
                var rect2 = DrawingUtils.getTrackingRectangle(200, 200, 170, 240);
                expect(rect2).to.have.a.property('origShift', 50);
                expect(rect2).to.have.a.property('reverseX', true);
                expect(rect2).to.have.a.property('reverseY', false);
                var rect3 = DrawingUtils.getTrackingRectangle(200, 200, 170, 160);
                expect(rect3).to.have.a.property('origShift', 50);
                expect(rect3).to.have.a.property('reverseX', true);
                expect(rect3).to.have.a.property('reverseY', true);
                var rect4 = DrawingUtils.getTrackingRectangle(200, 200, 230, 160);
                expect(rect4).to.have.a.property('origShift', 50);
                expect(rect4).to.have.a.property('reverseX', false);
                expect(rect4).to.have.a.property('reverseY', true);
                var rect5 = DrawingUtils.getTrackingRectangle(200, 200, 200, 200);
                expect(rect5).to.have.a.property('origShift', 0);
                expect(rect5).to.have.a.property('reverseX', false);
                expect(rect5).to.have.a.property('reverseY', false);
            });
        });
    });

    // ========================================================================
});
