/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('globals/sheethelper', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/utils/matrix',
    'io.ox/office/spreadsheet/model/formula/interpret/operand',
    'io.ox/office/spreadsheet/model/formula/interpret/formulacontext'
], function (Utils, SheetUtils, Matrix, Operand, FormulaContext) {

    'use strict';

    // private global functions ===============================================

    function split(str) {
        return str ? str.split(/\s+/).filter(_.identity) : [];
    }

    // private global functions ===============================================

    /**
     * Returns a function that resolves the implementation of an operator or a
     * function used in sheet formulas.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model used to resolve cell references passed to the
     *  operator or function implementation.
     *
     * @param {Object} descriptor
     *  The raw descriptor of the operator or function, as provided by the
     *  implementation modules in spreadsheet/model/formula/impl.
     *
     * @param {Object} [options]
     *  Optional parameters. See method FunctionModuleTester.testFunction() for
     *  details.
     *
     * @returns {Function}
     *  A resolver function wrapping the passed document model, and operator
     *  descriptor. This function can be invoked in the same way as the wrapped
     *  operator or function (according to its parameter signature), and
     *  returns the unconverted result value. If the implementation throws a
     *  formula error code (an instance of the class ErrorCode), this error
     *  code will be caught and returned. Other exceptions will be thrown.
     */
    function createFunctionResolver(docModel, descriptor, options) {

        // create a formula context with the passed reference position
        var refSheet = Utils.getIntegerOption(options, 'refSheet', 0);
        var targetAddress = SheetHelper.a(Utils.getStringOption(options, 'targetAddress', 'A1'));
        var context = new FormulaContext(docModel, { refSheet: refSheet, targetAddress: targetAddress });
        var fileFormat = docModel.getApp().getFileFormat();
        var toOperands = options && options.toOperand;

        // invokes the resolver function for the passed operator descriptor
        function invoke(/*operands...*/) {
            try {

                // create and validate the function parameter values
                var values = _.toArray(arguments);
                values.forEach(function (value, index) {
                    if (_.isArray(value)) {
                        console.warn('do not use plain arrays!');
                        values[index] = new Matrix([value]);
                    } else if (value instanceof Matrix) {
                        values[index] = value.clone();
                    }
                });

                // convert some values to Operand objects if specified
                if (toOperands === true) {
                    values = values.map(function (value) {
                        return new Operand(context, value);
                    });
                } else if (_.isNumber(toOperands)) {
                    if (toOperands < values.length) {
                        values[toOperands] = new Operand(context, values[toOperands]);
                    }
                } else if (_.isArray(toOperands)) {
                    toOperands.forEach(function (index) {
                        if (index < values.length) {
                            values[index] = new Operand(context, values[index]);
                        }
                    });
                }

                // create an array of Operand objects for the operand resolver mock
                var operands = values.map(function (value) {
                    return (value instanceof Operand) ? value : new Operand(context, value);
                });

                // create a mocked operand resolver
                context.pushOperandResolver({
                    size: _.constant(operands.length),
                    get: function (i) { return operands[i]; },
                    resolve: function (i) { return operands[i]; }
                }, 'val', 0, 0);

                // invoke the function implementation, return the result
                var resolveFunc = descriptor.resolve;
                if (_.isObject(resolveFunc) && !_.isFunction(resolveFunc)) {
                    resolveFunc = resolveFunc[fileFormat];
                }
                return resolveFunc.apply(context, values);

            } catch (error) {
                if (error instanceof SheetUtils.ErrorCode) { return error; }
                throw error;
            }
        }

        // default behavior: convert Operand object to the result value
        function resolver() {
            var result = invoke.apply(null, arguments);
            return (result instanceof Operand) ? result.getRawValue() : result;
        }

        // special behavior: return the value unmodified
        resolver.invoke = invoke;

        return resolver;
    }

    /**
     * Generates a test implementation function that provides the additional
     * function properties 'skip' and 'only', that can be forwarded to the
     * methods 'describe.skip' and 'describe.only' respectively.
     *
     * @param {Function} implFunc
     *  The implementation of the test function. The first parameter that will
     *  be passed to this function is the actual describe() implementation
     *  (either 'describe', 'describe.skip', or 'describe.only'). The remaining
     *  parameters will be the ones that have been passed to the generated test
     *  function.
     *
     * @returns {Function}
     *  The generated test implementation function.
     */
    function createImplementation(implFunc) {

        function createImpl(describeFunc) {
            return function () {
                return implFunc.apply(this, [describeFunc].concat(_.toArray(arguments)));
            };
        }

        var defaultImpl = createImpl(describe);
        defaultImpl.skip = createImpl(describe.skip);
        defaultImpl.only = createImpl(describe.only);
        return defaultImpl;
    }

    // class FunctionModuleTester =============================================

    /**
     * A tester for function implementation modules.
     *
     * @constructor
     */
    function FunctionModuleTester(funcModule) {

        // the promises for the test applications to be used to test the function resolvers
        var appPromises = _.toArray(arguments).slice(1);

        // the array of document models to be used for the function resolvers
        var docModels = null;

        // public methods -----------------------------------------------------

        /**
         * Creates a test context for a single spreadsheet function implemented
         * in the implementation module passed to the constructor.
         *
         * @param {String} funcKey
         *  The resource key of the function to be tested.
         *
         * @param {Object} [options]
         *  Optional parameters for the function resolvers:
         *  @param {Number} [options.refSheet=0]
         *      The zero-based index of the reference sheet used by the formula
         *      context (the position of the 'simulated' formula cell
         *      evaluating a function). Defaults to the first sheet in the test
         *      document.
         *  @param {String} [options.targetAddress='A1']
         *      The name of the target reference address used by the formula
         *      context, in A1 notation (the position of the 'simulated'
         *      formula cell evaluating a function).
         *  @param {Number|Array<Number>|Boolean} [options.toOperand=false]
         *      If set to a number, or an array of numbers, the specified
         *      function arguments (zero-based parameter indexes) that will be
         *      passed to the resolver functions will be converted to instances
         *      of the class Operand automatically. If set to true, all
         *      arguments will be converted to instances of Operand.
         *  The options parameter can be omitted completely (the following
         *  callback can be passed as second parameter).
         *
         * @param {Function} callback
         *  The callback function that implemented the tests for the specified
         *  spreadsheet function. Receives one or more function resolvers as
         *  parameters, according to the number of application promises passed
         *  to the constructor. Each resolver is a JavaScript function wrapping
         *  the implementation of the specified spreadsheet function, and can
         *  be invoked in the same way as the wrapped function in a real
         *  spreadsheet formula, according to its parameter signature. It will
         *  return the unconverted result value. If the implementation throws a
         *  formula error code (an instance of the class ErrorCode), this error
         *  code will be caught and returned as value. Other exceptions will be
         *  thrown. Additionally, each resolver contains a method invoke() that
         *  returns the original value of the implementation (no conversion of
         *  instances of class Operand to the wrapped values).
         */
        this.testFunction = createImplementation(function (describeFunc, funcKey, options, callback) {

            // options can be omitted completely
            if (!callback) { callback = options; options = null; }

            // create a text particle for the function, and a context for the before() block
            describeFunc('function "' + funcKey + '"', function () {

                // the function descriptor (add a few standard tests)
                var descriptor = funcModule[funcKey];
                it('should exist', function () {
                    expect(descriptor).to.be.an('object');
                });
                it('should be implemented', function () {
                    expect(descriptor).to.have.a.property('resolve');
                    if (!_.isFunction(descriptor.resolve)) {
                        expect(descriptor.resolve).to.be.an('object');
                        expect(descriptor.resolve).to.respondTo('ooxml');
                        expect(descriptor.resolve).to.respondTo('odf');
                    }
                });

                // create the function resolvers in a before block (this defers their creation
                // until the enclosing test block runs)
                var resolvers = null;
                before(function () {
                    resolvers = docModels.map(function (docModel) {
                        return createFunctionResolver(docModel, descriptor, options);
                    });
                });

                // immediately register the tests implemented in the callback function, but pass
                // placeholder callbacks that call the real function resolvers on their invocation
                // (resolvers will be initialized deferred in the before() block above)
                callback.apply(null, _.times(appPromises.length, function (index) {
                    function getResolver() { return resolvers[index]; }
                    function invokeResolver() { return getResolver().apply(this, arguments); }
                    invokeResolver.invoke = function () { return getResolver().invoke.apply(this, arguments); };
                    return invokeResolver;
                }).concat([descriptor]));
            });
        });

        /**
         * Creates a simple test that the specified spreadsheet function has
         * been implemented by making it an alias of another function.
         *
         * @param {String} aliasFuncKey
         *  The resource key of the function to be tested.
         *
         * @param {String} implFuncKey
         *  The resource key of the function the descriptor of the specified
         *  alias function should refer to.
         */
        this.testAliasFunction = createImplementation(function (describeFunc, aliasFuncKey, implFuncKey) {

            // create a text particle for the function
            describeFunc('function "' + aliasFuncKey + '"', function () {

                var descriptor = funcModule[aliasFuncKey];
                it('should exist', function () {
                    expect(descriptor).to.be.an('object');
                });
                it('should be implemented', function () {
                    expect(descriptor).to.have.a.property('resolve', implFuncKey);
                });
            });
        });

        // initialization -----------------------------------------------------

        // resolve the document models from the application promises (happens before the before() and it() blocks)
        $.when.apply($, appPromises).done(function () { docModels = _.invoke(arguments, 'getModel'); });

        // test the existence of the passed module
        it('should exist', function () { expect(funcModule).to.be.an('object'); });

    } // class FunctionModuleTester

    // static class SheetHelper ===============================================

    var SheetHelper = {};

    // static methods ---------------------------------------------------------

    /**
     * Convenience shortcut to the class ErrorCode.
     */
    SheetHelper.ErrorCode = SheetUtils.ErrorCode;

    /**
     * Parses a column or row interval string in simple A1 notation (e.g. 'B:C'
     * or '2:3') to an instance of class Interval.
     */
    SheetHelper.i = function (str) {
        return /^\d/.test(str) ? SheetUtils.Interval.parseAsRows(str) : SheetUtils.Interval.parseAsCols(str);
    };

    /**
     * Parses a cell address string in simple A1 notation (e.g. 'B3') to an
     * instance of class Address.
     */
    SheetHelper.a = SheetUtils.Address.parse;

    /**
     * Parses a cell range address string in simple A1 notation (e.g. 'B3:C4')
     * to an instance of class Range.
     */
    SheetHelper.r = SheetUtils.Range.parse;

    /**
     * Parses a 3D cell range address string in simple A1 notation with integer
     * sheet indexes (e.g. '1:2!B3:C4') to an instance of class Range3D.
     */
    SheetHelper.r3d = function (str) {
        var matches = /^(\d+):(\d+)!(.*)$/.exec(str);
        return SheetUtils.Range3D.createFromRange(SheetHelper.r(matches[3]), parseInt(matches[1], 10), parseInt(matches[2], 10));
    };

    /**
     * Parses a space-separated list of column intervals or row intervals in
     * simple A1 notation (e.g. 'B:C D:E 7:8') to an instance of class
     * IntervalArray.
     */
    SheetHelper.ia = function (str) {
        return SheetUtils.IntervalArray.map(split(str), SheetHelper.i);
    };

    /**
     * Parses a space-separated list of cell addresses in simple A1 notation
     * (e.g. 'B3 C4') to an instance of class AddressArray.
     */
    SheetHelper.aa = function (str) {
        return SheetUtils.AddressArray.map(split(str), SheetHelper.a);
    };

    /**
     * Parses a space-separated list of cell range addresses in simple A1
     * notation (e.g. 'B3:C4 D5:E6') to an instance of class RangeArray.
     */
    SheetHelper.ra = function (str) {
        return SheetUtils.RangeArray.map(split(str), SheetHelper.r);
    };

    /**
     * Parses a space-separated list of 3D cell range addresses in simple A1
     * notation with integer sheet indexes (e.g. '1:2!B3:C4 2:3!D5:E6') to an
     * instance of class Range3DArray.
     */
    SheetHelper.r3da = function (str) {
        return SheetUtils.Range3DArray.map(split(str), SheetHelper.r3d);
    };

    /**
     * Converts a date string in the exact format 'YYYY-MM-DD' to an instance
     * of the Date class.
     */
    SheetHelper.d = function (date) {
        var matches = /^(\d{4})-(\d{2})-(\d{2})$/.exec(date);
        return new Date(Date.UTC(parseInt(matches[1], 10), parseInt(matches[2], 10) - 1, parseInt(matches[3], 10)));
    };

    /**
     * Converts a time string in the exact format 'hh:mm:ss' to an instance
     * of the Date class.
     */
    SheetHelper.t = function (time) {
        var matches = /^(\d{2}):(\d{2}):(\d{2})$/.exec(time);
        return new Date(Date.UTC(1899, 11, 30, parseInt(matches[1], 10), parseInt(matches[2], 10), parseInt(matches[3], 10), 0));
    };

    /**
     * Creates a new instance of the class Matrix from the passed 2-dimensional
     * array of scalar values.
     */
    SheetHelper.mat = function (arr) {
        return new Matrix(arr);
    };

    /**
     * Creates a new instance of the class Matrix containing Date objects
     * converted from the passed 2-dimensional array of strings. The dates will
     * be created with the method SheetHelper.d().
     */
    SheetHelper.dmat = function (arr) {
        return new Matrix(arr).transform(SheetHelper.d);
    };

    /**
     * Creates a matcher predicate for Chai's satisfy() assertion that matches
     * the passed index intervals but ignores any additional properties of the
     * result intervals.
     *
     * @param {String} expected
     *  The expected index intervals, as string (either as column intervals, or
     *  as row intervals). See description of the method SheetHelper.ia() for
     *  details.
     *
     * @returns {Function}
     *  A matcher predicate that takes an instance of IntervalArray, and
     *  returns whether it matches the specified index intervals.
     */
    SheetHelper.orderedIntervalsMatcher = function (expected) {
        var expectedKey = SheetHelper.ia(expected).toString();
        return function (result) {
            var resultKey = (result instanceof SheetUtils.IntervalArray) ? result.toString() : null;
            if (expectedKey === resultKey) { return true; }
            console.error('SheetHelper.orderedIntervalsMatcher(): wrong result intervals');
            console.error('  exp=' + expected);
            console.error('  got=' + result);
        };
    };

    /**
     * Creates a matcher predicate for Chai's satisfy() assertion that matches
     * the passed index intervals regardless of their order in the result
     * array.
     *
     * @param {String} expected
     *  The expected index intervals, as string (either as column intervals, or
     *  as row intervals). See description of the method SheetHelper.ia() for
     *  details.
     *
     * @returns {Function}
     *  A matcher predicate that takes an instance of IntervalArray, and
     *  returns whether it matches the specified index intervals.
     */
    SheetHelper.unorderedIntervalsMatcher = function (expected) {
        function getKey(intervals) { return _.invoke(intervals, 'toString').sort().join(' '); }
        var expectedKey = getKey(SheetHelper.ia(expected));
        return function (result) {
            var resultKey = (result instanceof SheetUtils.IntervalArray) ? getKey(result) : '';
            if (expectedKey === resultKey) { return true; }
            console.error('SheetHelper.unorderedIntervalsMatcher(): wrong result intervals');
            console.error('  exp=' + expected);
            console.error('  got=' + result);
        };
    };

    /**
     * Creates a matcher predicate for Chai's satisfy() assertion that matches
     * the passed cell addresses regardless of their order in the result array.
     *
     * @param {String} expected
     *  The expected cell addresses, as string. See description of the method
     *  SheetHelper.aa() for details.
     *
     * @returns {Function}
     *  A matcher predicate that takes an instance of AddressArray, and returns
     *  whether it matches the specified cell addresses.
     */
    SheetHelper.unorderedAddressesMatcher = function (expected) {
        function getKey(addresses) { return _.invoke(addresses, 'key').sort().join(' '); }
        var expectedKey = getKey(SheetHelper.aa(expected));
        return function (result) {
            var resultKey = (result instanceof SheetUtils.AddressArray) ? getKey(result) : '';
            if (expectedKey === resultKey) { return true; }
            console.error('SheetHelper.unorderedAddressesMatcher(): wrong result addresses');
            console.error('  exp=' + expected);
            console.error('  got=' + result);
        };
    };

    /**
     * Creates a matcher predicate for Chai's satisfy() assertion that matches
     * the passed cell range addresses regardless of their order in the result
     * array.
     *
     * @param {String} expected
     *  The expected cell range addresses, as string. See description of the
     *  method SheetHelper.ra() for details.
     *
     * @returns {Function}
     *  A matcher predicate that takes an instance of RangeArray, and returns
     *  whether it matches the specified cell range addresses.
     */
    SheetHelper.unorderedRangesMatcher = function (expected) {
        function getKey(ranges) { return _.invoke(ranges, 'key').sort().join(' '); }
        var expectedKey = getKey(SheetHelper.ra(expected));
        return function (result) {
            var resultKey = (result instanceof SheetUtils.RangeArray) ? getKey(result) : '';
            if (expectedKey === resultKey) { return true; }
            console.error('SheetHelper.unorderedRangesMatcher(): wrong result ranges');
            console.error('  exp=' + expected);
            console.error('  got=' + result);
        };
    };

    /**
     * Creates a matcher predicate for Chai's satisfy() assertion that matches
     * the passed cell range addresses by merging them and the result ranges.
     *
     * @param {String} expected
     *  The expected cell range addresses, as string. See description of the
     *  method SheetHelper.r() for details.
     *
     * @returns {Function}
     *  A matcher predicate that takes an instance of RangeArray, and returns
     *  whether the ranges cover the exact same cells as the passed ranges,
     *  regardless if and how the result range overlap each other.
     */
    SheetHelper.mergedRangesMatcher = function (expected) {
        expected = SheetHelper.ra(expected).merge();
        return function (result) {
            var mergedResult = (result instanceof SheetUtils.RangeArray) ? result.merge() : null;
            if (mergedResult && expected.equals(mergedResult, true)) { return true; }
            console.error('SheetHelper.mergedRangesMatcher(): wrong result ranges');
            console.error('  exp=' + expected);
            console.error('  got=' + result);
        };
    };

    /**
     * Calls a BDD test for the two passed matrices.
     *
     * @param {Any} actual
     *
     * @param {Matrix} expected
     *
     * @param {Number} threshold
     */
    SheetHelper.assertMatrixCloseTo = function (actual, expected, threshold) {

        expect(actual).to.be.an.instanceof(Matrix);
        expect(actual.rows()).to.equal(expected.rows());
        expect(actual.cols()).to.equal(expected.cols());

        // no usage of forEach, because it break the error stack!
        for (var row = 0; row < actual.rows(); row += 1) {
            for (var col = 0; col < actual.cols(); col += 1) {
                var actualValue = actual.get(row, col);
                var expectedValue = expected.get(row, col);
                if (_.isNumber(expectedValue)) {
                    expect(actualValue).to.be.closeTo(expectedValue, threshold);
                } else {
                    expect(actualValue).to.equal(expectedValue);
                }
            }
        }
    };

    /**
     * Returns a new tester for a function implementation module.
     *
     * @param {Any} funcModule
     *  The function module. Is expected to be a simple object with function
     *  descriptors, mapped by function resource keys.
     *
     * @param {jQuery.Promise} appPromise...
     *  One or more promises resolving with test application instances, e.g.
     *  for different file formats. The tester will provide individual function
     *  resolvers for each application.
     *
     * @returns {FunctionModuleTester}
     *  A new tester for the specified function implementation module.
     */
    SheetHelper.createFunctionModuleTester = (function () {
        function Helper(args) { return FunctionModuleTester.apply(this, args); }
        Helper.prototype = FunctionModuleTester.prototype;
        return function () { return new Helper(arguments); };
    }());

    // exports ================================================================

    return SheetHelper;

});
