/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/class', function () {

    'use strict';

    // static class Class =====================================================

    // the exported Class class
    var Class = {};

    // static methods ---------------------------------------------------------

    /**
     * Connects the prototype chains of the passed class constructors.
     *
     * @param {Function} SuperClass
     *  The constructor of the class that shall become the super class of the
     *  sub class passed to this method.
     *
     * @param {Function} SubClass
     *  The constructor of the class that shall become the sub class of the
     *  super class passed to this method.
     *
     * @returns {Function}
     *  The extended sub class, for convenience.
     */
    Class.extend = function (SuperClass, SubClass) {

        // initialize prototype chain
        function ClassPrototype() {}
        ClassPrototype.prototype = SuperClass.prototype;
        SubClass.prototype = new ClassPrototype();
        SubClass.prototype.constructor = SubClass;

        // store super reference at class constructor
        SubClass.__super__ = SuperClass.prototype;

        // copy static properties
        var hasOwnProp = Object.prototype.hasOwnProperty.bind(SuperClass);
        for (var key in SuperClass) {
            if (hasOwnProp(key)) {
                SubClass[key] = SuperClass[key];
            }
        }

        // add static extend() method that extends from the sub class
        return Class.extendable(SubClass);
    };

    /**
     * Adds a static method "extend()" to the passed class constructor that
     * allows to create sub classes of that class.
     *
     * @param {Function} SuperClass
     *  The constructor of the class that will be extended with a static method
     *  "extend()". This method expects the constructor of a sub class as only
     *  parameter, connects the prototypes of the super class and sub class,
     *  and adds a new static method "extend()" that is able to extend from the
     *  sub class.
     *
     * @returns {Function}
     *  The extended class, for convenience.
     */
    Class.extendable = function (SuperClass) {
        SuperClass.extend = function (SubClass) {
            return Class.extend(SuperClass, SubClass);
        };
        return SuperClass;
    };

    // exports ================================================================

    return Class;

});
