/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/dialog/tabbeddialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/dialog/basedialog'
], function (Utils, Forms, KeyCodes, BaseDialog) {

    'use strict';

    // class TabbedDialog =====================================================

    /**
     * Creates an empty modal dialog that contains a tab bar and tab panels.
     *
     * Additionally to the events triggered by the base class BaseDialog, an
     * instance of this class triggers the following events:
     * - 'tab:hide'
     *      Before the active tab panel will be deactivated. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the active tab panel.
     *      (3) {jQuery} tabPanel
     *          The root node of the active tab panel.
     *      (4) {jQuery} tabButton
     *          The button element of the active tab button.
     * - 'tab:show'
     *      After a tab panel has been activated. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the active tab panel.
     *      (3) {jQuery} tabPanel
     *          The root node of the active tab panel.
     *      (4) {jQuery} tabButton
     *          The button element of the active tab button.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application that is
     *  creating the dialog, or an object with a method getWindowId() that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  BaseDialog.
     */
    var TabbedDialog = BaseDialog.extend(function (windowId, initOptions) {

        // base constructor
        BaseDialog.call(this, windowId, initOptions);

        // the container for the tab buttons
        this._tabBarNode = $('<ul class="nav nav-tabs" role="tablist">');
        // the container for the tab panel elements
        this._tabPanelNode = $('<div class="tab-content" style="margin-top:13px;">');
        // maps tab panels by user identifiers
        this._tabPanelsMap = {};
        // all tab panel identifiers in insertion order
        this._tabPanelIds = [];
        // the identifier of the visible tab panel
        this._activeId = null;

        // insert the nodes into the dialog body
        this.append(this._tabBarNode, this._tabPanelNode);

        // initially focus the active tab button
        this.on('show', this._focusActiveTabButton.bind(this));

        // trigger 'tab:hide' event before the active tab panel will be hidden
        this._tabBarNode.on('show.bs.tab', function (event) {
            var tabButton = $(event.relatedTarget).closest('a');
            var tabId = tabButton.attr('data-user-id');
            if (tabButton.length > 0) {
                tabButton.attr('aria-selected', false);
                this.trigger('tab:hide', tabId, this._tabPanelsMap[tabId], tabButton);
            }
        }.bind(this));

        // trigger 'tab:show' event after a tab panel has been shown
        this._tabBarNode.on('shown.bs.tab', function (event) {
            var tabButton = $(event.target).closest('a');
            var activeId = this._activeId = tabButton.attr('data-user-id');
            tabButton.attr('aria-selected', true);
            this.trigger('tab:show', activeId, this._tabPanelsMap[activeId], tabButton);
        }.bind(this));

        // handle keyboard events for focus navigation
        this._tabBarNode.on('keydown', function (event) {

            var tabPanelIds = this._tabPanelIds;
            // index of the active tab panel
            var activeIndex = tabPanelIds.indexOf(this._activeId);
            // identifier of the new tab panel to be activated
            var newTabId = null;

            switch (event.keyCode) {
                case KeyCodes.RIGHT_ARROW:
                    newTabId = tabPanelIds[activeIndex + 1];
                    break;
                case KeyCodes.LEFT_ARROW:
                    newTabId = tabPanelIds[activeIndex - 1];
                    break;
                case KeyCodes.SPACE:
                    newTabId = $(event.target).attr('data-user-id');
                    break;
            }

            if (newTabId) {
                // focus active tab button first to allow the active tab panel
                // to finally focus a control inside the panel
                this._focusActiveTabButton();
                this.activateTabPanel(newTabId);
                return false;
            }

            // we need to call preventDefault() because of focus travelling back
            // to the document when this dialog has no input fields
            event.preventDefault();

        }.bind(this));

    }); // class TabbedDialog

    // private methods --------------------------------------------------------

    /**
     * Sets the browser focus into the button element of the active tab panel.
     */
    TabbedDialog.prototype._focusActiveTabButton = function () {
        Utils.setFocus(this._tabBarNode.find('li.active a').first());
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a new tab panel container element associated to the specified
     * identifier.
     *
     * @param {String} tabId
     *  The identifier of the tab panel. The identifiers of all tab panels
     *  created for this dialog must be different.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.label]
     *      The text label to be shown in the tab button activating the new tab
     *      panel. If omitted, the tab button will not contain a text label.
     *  - {String} [options.icon]
     *      The CSS class name of the icon to be shown in the tab button
     *      activating the new tab panel. If omitted, the tab button will not
     *      contain an icon. Must be an icon provided by the Font Awesome icon
     *      set.
     *
     * @returns {jQuery}
     *  The container element of the specified tab panel, as jQuery object.
     */
    TabbedDialog.prototype.addTabPanel = function (tabId, options) {

        // the unique DOM identifier for the tab button
        var buttonId = _.uniqueId('tab');
        // the unique DOM identifier for the tab panel container
        var panelId = _.uniqueId('panel');

        // the button element in the tab bar
        var tabButton = $('<a href="#' + panelId + '" id="' + buttonId + '" tabindex="0" role="tab" data-toggle="tab" data-user-id="' + tabId + '" aria-selected="false" aria-controls="' + panelId + '">');
        // the container element for the tab panel
        var tabPanel = $('<div class="tab-pane row-fluid" id="' + panelId + '" role="tabpanel" data-user-id="' + tabId + '" aria-labelledby="' + buttonId + '">');

        // set button caption
        tabButton.append(Forms.createCaptionMarkup(options));

        // insert the tab button and tab panel
        this._tabBarNode.append($('<li>').append(tabButton));
        this._tabPanelNode.append(tabPanel);
        this._tabPanelsMap[tabId] = tabPanel;
        this._tabPanelIds.push(tabId);

        // return the tab panel for further initialization
        return tabPanel;
    };

    /**
     * Returns the tab panel container element associated to the specified
     * identifier.
     *
     * @param {String} tabId
     *  The identifier of the tab panel, as passed to the method
     *  TabbedDialog.addTabPanel().
     *
     * @returns {jQuery}
     *  The container element of the specified tab panel. If the passed
     *  identifier is invalid, returns an empty jQuery collection.
     */
    TabbedDialog.prototype.getTabPanel = function (tabId) {
        return (tabId in this._tabPanelsMap) ? this._tabPanelsMap[tabId] : $();
    };

    /**
     * Returns the identifier of the tab panel that is currently visible.
     *
     * @returns {String|Null}
     *  The identifier of the tab panel that is currently visible, or null if
     *  no panel is visible yet.
     */
    TabbedDialog.prototype.getActiveTabId = function () {
        return this._activeId;
    };

    /**
     * Activates (displays) the specified tab panel.
     *
     * @param {String} tabId
     *  The identifier of the tab panel, as passed to the method
     *  TabbedDialog.addTabPanel().
     *
     * @returns {TabbedDialog}
     *  A reference to this instance.
     */
    TabbedDialog.prototype.activateTabPanel = function (tabId) {
        this._tabBarNode.find('[data-user-id="' + tabId + '"]').tab('show');
        return this;
    };

    // exports ================================================================

    return TabbedDialog;

});
