/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/format/characterstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom'
], function (Utils, Color, LineHeight, StyleCollection, AttributeUtils, DOM) {

    'use strict';

    var // definitions for character attributes
        DEFINITIONS = {

            fontName: {
                def: AttributeUtils.MINOR_FONT_KEY
            },

            fontNameSymbol: {
                def: ''
            },

            fontNameComplex: {
                def: ''
            },

            fontNameEastAsia: {
                def: ''
            },

            bold: {
                def: false
            },

            italic: {
                def: false
            },

            underline: {
                def: false
            },

            strike: {
                def: 'none'
            },

            caps: {
                def: 'none'
            },

            // baseline for Text
            vertAlign: {
                def: 'baseline'
            },

            // baseline for Presentation
            baseline: {
                def: 0
            },

            color: {
                def: Color.AUTO
            },

            fillColor: {
                def: Color.AUTO
            },

            language: {
                def: ''
            },

            url: {
                def: '',
                scope: 'element'
            },

            anchor: {
                def: '',
                scope: 'element'
            },

            autoDateField: {
                def: '',
                scope: 'element'
            },

            field: {
                def: '', // #45332, new odf filter sends field properties as character attribute
                scope: 'element'
            },

            // special attributes

            spellerror: {
                def: false,
                scope: 'element',
                special: true
            },

            highlight: {
                def: false,
                scope: 'element',
                special: true
            }
        },

        // definitions of character attributes specific for slide documents
        SLIDE_DEFINITIONS = {
            fontSize: {
                def: 18
            }
        },

        // definitions of character attributes specific for text documents
        TEXT_DEFINITIONS = {
            fontSize: {
                def: 11
            }
        },

        PARENT_RESOLVERS = {
            paragraph: function (span) { return span.closest(DOM.PARAGRAPH_NODE_SELECTOR); }
        };

    // class CharacterStyles ==================================================

    /**
     * Contains the style sheets for character formatting attributes. The CSS
     * formatting will be written to text span elements contained somewhere in
     * the paragraph elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextBaseModel} docModel
     *  The document model containing this instance.
     *
     * @param {Object} [initoptions]
     *  Optional parameters passed to the base class constructor.
     */
    var CharacterStyles = StyleCollection.extend({ constructor: function (docModel, initOptions) {

        // self reference
        var self = this;

        // the collection of paragraph style sheets
        var paraStyles = null;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'character', _.extend({}, initOptions, { parentResolvers: PARENT_RESOLVERS }));

        // private methods ----------------------------------------------------

        /**
         * Will be called for every text span whose character attributes have
         * been changed.
         *
         * @param {jQuery} textSpan
         *  The text span whose character attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateCharacterFormatting(textSpan, mergedAttributes) {
            if (DOM.isSpecialField(textSpan.parent())) { return; }

            var // the parent paragraph of the node (may be a grandparent)
                paragraph = textSpan.closest(DOM.PARAGRAPH_NODE_SELECTOR),
                // the closest cell node (if it exists)
                cell = textSpan.closest(DOM.TABLE_CELLNODE_SELECTOR),
                // the closest text frame content node (if it exists)
                textframeContent = textSpan.closest(DOM.TEXTFRAMECONTENT_NODE_SELECTOR),
                // the merged attributes of the paragraph
                paraAttrs = paraStyles.getElementAttributes(paragraph).paragraph,
                // the character attributes of the passed attribute map
                charAttrs = mergedAttributes.character,
                // effective text color, font size, and vertical offset
                textColor = charAttrs.color,
                fontSize = charAttrs.fontSize,
                textDecoration = AttributeUtils.getCssTextDecoration(charAttrs),
                position = 'static',
                bottom = null,
                // the background color of the cell and text frame
                cellBackgroundColor = null, textFrameBackgroundColor = null,
                // an array containing all affected fill colors of text, paragraph and cell
                allFillColors = [],
                // the effective line height, that might be reduced inside comment
                effectiveLineHeight = paraAttrs.lineHeight,
                // the maximum font size inside a comment
                commentMaxFontSize = 10,
                // the target chain, that is required to resolve the correct theme
                targets = null;

            // restriction of max font size for small devices (only for OX Text)
            if (Utils.SMALL_DEVICE && docModel.getApp().isTextApp() && fontSize > 16) {
                fontSize = 16;
            }

            // update formatting of the passed text span
            textSpan.attr({
                lang: charAttrs.language,
                'data-auto-date': charAttrs.autoDateField || null
            });
            textSpan.css({
                fontWeight: charAttrs.bold ? 'bold' : 'normal',
                fontStyle: charAttrs.italic ? 'italic' : 'normal',
                backgroundColor: docModel.getCssColor(charAttrs.fillColor, 'fill')
            });

            var url = charAttrs.url;
            if (!_.isString(url) || (url.length === 0)) { url = null; }
            var anchor = charAttrs.anchor;
            if (!_.isString(anchor) || (anchor.length === 0)) { anchor = null; }

            paragraph.attr({
                title: url,
                'data-hyperlink-url': url,
                'data-anchor-url': anchor
            });

            textSpan.toggleClass('spellerror', !!charAttrs.spellerror);
            textSpan.toggleClass('highlight', !!charAttrs.highlight);
            textSpan.toggleClass('uppercase', charAttrs.caps === 'all');
            textSpan.toggleClass('lowercase', charAttrs.caps === 'small');

            // Checking also the background-color of an affected cell, if there is one (28988).
            // For performance reasons, not using attributes (might be assigned to row or table, too),
            // but checking CSS style 'background-color' of the table directly.
            if (cell.length > 0) {
                cellBackgroundColor = Color.parseCSS(cell.css('background-color'), true);
                if (cellBackgroundColor) { allFillColors.push(cellBackgroundColor.toJSON()); }
            }

            // not only checking table cells, but also text frames, for their background-color (36385)
            if (textframeContent.length > 0) {
                textFrameBackgroundColor = Color.parseJSON(textframeContent.css('background-color'), true);
                if (textFrameBackgroundColor) { allFillColors.push(textFrameBackgroundColor.toJSON()); }
            }

            // adding fill color of paragraph and character (order is important)
            allFillColors.push(paraAttrs.fillColor);
            allFillColors.push(charAttrs.fillColor);

            // calculate effective text color, according to fill colors
            // or theme if it's hyperlink, #45263
            if (DOM.isHyperlinkNode(textSpan) && docModel.useSlideMode()) {
                textColor = '#' + docModel.getThemeModelForNode(textSpan).getSchemeColor('hyperlink', '00f');
                textDecoration = 'underline';
            } else {
                if (AttributeUtils.isColorThemed(textColor)) { targets = docModel.getThemeTargets(paragraph); }
                textColor = docModel.getCssTextColor(textColor, allFillColors, targets);
            }

            // calculate font height and vertical alignment (according to escapement)
            if ((charAttrs.vertAlign === 'sub') || (charAttrs.baseline < 0)) {
                position = 'relative';
                bottom = 0;
                fontSize = Utils.round(fontSize * 0.66, 0.1);
            } else if ((charAttrs.vertAlign === 'super') || charAttrs.baseline > 0) {
                position = 'relative';
                bottom = '0.5em';
                fontSize = Utils.round(fontSize * 0.66, 0.1);
            }

            // inside comments the paragraph bottom margin must be 0 (even if this was not applied via operation),
            // the font size is set to a maximum value and the line height is set to single.
            if (!docModel.getApp().isODF() && !docModel.getCommentLayer().isEmpty() && DOM.isNodeInsideComment(textSpan)) {
                if (fontSize > commentMaxFontSize) {
                    charAttrs.fontSize = commentMaxFontSize;
                    fontSize = commentMaxFontSize;
                }
                effectiveLineHeight = LineHeight.SINGLE;
            }

            if (!targets && AttributeUtils.isCharacterFontThemed(charAttrs)) {
                targets = docModel.getThemeTargets(paragraph);
            }

            // update CSS of the text span
            textSpan.css({
                fontFamily: self.resolveCssFontFamily(charAttrs, targets),
                fontSize: Utils.convertToEM(fontSize, 'pt'),
                color: textColor,
                textDecoration: textDecoration,
                position: position,
                bottom: bottom
            });

            // update line height due to changed font settings
            self.updateElementLineHeight(textSpan, effectiveLineHeight, charAttrs, targets);

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(textSpan, mergedAttributes);

            // TODO: set bullet character formatting according to paragraph attributes
            // Bug 30794: We want have a consistent list-label formatting, therefore only the
            // code in updateList in editor should be responsible to set character formatting.
            // This should be extended as the user is currently only able to change the list-label
            // formatting via character attributes in paragraph styles.
            //$(paragraph).find('> ' + DOM.LIST_LABEL_NODE_SELECTOR + ' > span').css('font-size', characterAttributes.fontSize + 'pt');
        }

        /**
         * Will be called for every text span used as preview in the UI.
         *
         * @param {jQuery} textSpan
         *  The text span preview node, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A complete attribute set, containing the effective attribute values
         *  merged from style sheets and explicit attributes.
         */
        function updatePreviewFormatting(textSpan, mergedAttributes) {

            // the merged character attributes
            var charAttrs = mergedAttributes.character;
            // the resolved text color
            var colorDesc = docModel.resolveColor(Color.parseJSON(charAttrs.color), 'text');

            // update formatting of the passed text span
            textSpan.css({
                fontFamily: self.resolveCssFontFamily(charAttrs),
                fontSize: Utils.minMax(Utils.round(10 + (charAttrs.fontSize - 10) / 1.5, 0.1), 6, 22) + 'pt',
                fontWeight: charAttrs.bold ? 'bold' : 'normal',
                fontStyle: charAttrs.italic ? 'italic' : 'normal',
                color: (colorDesc.y > 0.95) ? 'black' : colorDesc.css // bug 40872: use black if color is too light
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the effective CSS font family for the passed character
         * attributes.
         *
         * @param {Object} charAttributes
         *  A complete map with character attributes supported by this
         *  instance.
         *
         * @param {String|Array<String>} [targets]
         *  A single target name or an array of target names to specify the
         *  theme's target.
         *
         * @returns {String}
         *  The effective CSS font family for the passed character attributes.
         */
        this.resolveCssFontFamily = function (charAttributes, targets) {
            var fontFamily = docModel.getCssFontFamily(charAttributes.fontName, targets);
            if (charAttributes.fontNameSymbol) {
                if (fontFamily) { fontFamily += ','; }
                fontFamily += docModel.getCssFontFamily(charAttributes.fontNameSymbol, targets);
            }
            return fontFamily;
        };

        /**
         * Sets the text line height of the specified DOM element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose line height will be changed. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} lineHeight
         *  The new line height value. Must contain the properties 'type' and
         *  'value'.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the normal line height.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @param {String|Array<String>} [targets]
         *  A single target name or an array of target names to specify the
         *  theme's target.
         *
         */
        this.updateElementLineHeight = function (element, lineHeight, charAttributes, targets) {

            // effective line height in pixels (start with passed value, converted from 1/100 mm)
            var height = Utils.convertHmmToLength(lineHeight.value, 'px');
            // CSS unit string to be appended to the effective line height
            var unit = 'px';
            // value for the CSS attribute 'vertical-align'
            var vertAlign = null;

            // type 'fixed': use line height as passed, otherwise convert to percentage
            if (lineHeight.type !== 'fixed') {

                // workaround for bug 47960: get normal line height of enlarged font
                var normalLineHeight = docModel.getRenderFont(charAttributes, targets, 10).getNormalLineHeight() / 10;
                // workaround for Bug 47960 & Bug 45951
                var fontHeight = docModel.useSlideMode() ? normalLineHeight : Utils.convertLength(charAttributes.fontSize, 'pt', 'px');

                // calculate effective line height
                switch (lineHeight.type) {
                    case 'leading':
                        height += normalLineHeight;
                        break;
                    case 'atLeast':
                        height = Math.max(height, normalLineHeight);
                        break;
                    case 'percent':
                        // 'lineHeight.value' is a value in percent
                        height = normalLineHeight * lineHeight.value / 100;
                        break;
                    case 'normal':
                        height = normalLineHeight;
                        break;
                    default:
                        Utils.error('CharacterStyles.updateElementLineHeight(): invalid line height type');
                }

                // bug 47960: offset to pretend rounding error
                height =  Math.ceil(height) + 0.5;

                // convert to relative size in percent
                height = Utils.round(height / fontHeight * 100, 0.01);
                unit = '%';

                if (height < 100) {
                    //workaround for Bug 48695
                    vertAlign = '0.5em';
                }
            }

            $(element).first().css({
                lineHeight: height + unit,
                verticalAlign: vertAlign
            });
        };

        // initialization -----------------------------------------------------

        // deferred initialization
        docModel.getApp().onInit(function () {
            paraStyles = docModel.getStyleCollection('paragraph');
        }, this);

        // register the attribute definitions for the style family
        docModel.registerAttributes('character', DEFINITIONS);

        // app specific attributes (55728) (still avoiding subclass of CharacterStyles)
        if (docModel.getApp().isPresentationApp()) {
            docModel.registerAttributes('character', SLIDE_DEFINITIONS);
        } else {
            docModel.registerAttributes('character', TEXT_DEFINITIONS);
        }

        // register the formatting handlers for DOM elements
        this.registerFormatHandler(updateCharacterFormatting);
        this.registerPreviewHandler(updatePreviewFormatting);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = self = paraStyles = null;
        });

    } }); // class CharacterStyles

    // exports ================================================================

    return CharacterStyles;

});
