/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/rendererbase', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject'
], function (Utils, TriggerObject) {

    'use strict';

    // class RendererBase =====================================================

    /**
     * Base class for the layer renderers of a grid pane in a spreadsheet view.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this layer renderer.
     */
    var RendererBase = TriggerObject.extend({ constructor: function (gridPane) {

        // self reference
        var self = this;

        // the spreadsheet view containing this renderer
        var docView = gridPane.getDocView();

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, gridPane);

        // protected methods --------------------------------------------------

        /**
         * Registers an event handler at the specified event source object that
         * will only be invoked when the grid pane containing this renderer is
         * visible. See method BaseObject.listenTo() for details about the
         * parameters.
         *
         * @returns {RendererBase}
         *  A reference to this instance.
         */
        this.listenToWhenVisible = function (source, type, handler) {
            return this.listenTo(source, type, function () {
                if (gridPane.isVisible()) {
                    return handler.apply(self, arguments);
                }
            });
        };

        /**
         * Registers an event handler at the specified event source object that
         * will only be invoked when the grid pane containing this renderer is
         * activated. See method BaseObject.listenTo() for details about the
         * parameters.
         *
         * @returns {RendererBase}
         *  A reference to this instance.
         */
        this.listenToWhenActive = function (source, type, handler) {
            return this.listenToWhenVisible(source, type, function () {
                if (gridPane === docView.getActiveGridPane()) {
                    return handler.apply(self, arguments);
                }
            });
        };

        /**
         * Called after the layer range of the parent grid pane has been
         * changed. Subclasses MUST overwrite this method to refresh themselves
         * accordingly.
         *
         * @param {Object} layerSettings
         *  The settings of the new layer range of the grid pane, with the
         *  following properties:
         *  - {Range} range
         *      The new layer range.
         *  - {Rectangle} rectangle
         *      The new layer rectangle, in pixels.
         *  - {RangeArray} dirtyRanges
         *      The addresses of all cell ranges that need to be repainted.
         *  - {Number} zoomScale
         *      The ratio between old and new zoom factor. If this value is 1,
         *      the toom factor has not been changed.
         */
        this.setLayerRange = function (/*layerSettings*/) {
            Utils.error('RendererBase.setLayerRange(): missing implementation');
        };

        /**
         * Called after the parent grid pane hides itself. Subclasses MUST
         * overwrite this method and deinitialize this renderer accordingly.
         */
        this.hideLayerRange = function () {
            Utils.error('RendererBase.hideLayerRange(): missing implementation');
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = gridPane = docView = null;
        });

    } }); // class RendererBase

    // exports ================================================================

    return RendererBase;

});
