/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/labels', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/view/editlabels',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, EditLabels, SheetUtils, gt) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;

    // static class SpreadsheetLabels =========================================

    /**
     * Provides generic caption texts for control groups, menu headers, and
     * other GUI elements for the Spreadsheet application in a single map for
     * convenience.
     *
     * @extends EditLabels
     */
    var SpreadsheetLabels = _.extend({}, EditLabels);

    // header labels ----------------------------------------------------------

    /**
     * Header label for column/row operations.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.COL_ROW_HEADER_LABEL = /*#. Header label insert/delete/resize rows and columns in a spreadsheet */ gt.pgettext('menu-title', 'Rows/Columns');

    /**
     * Header label for cell operations.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.CELL_HEADER_LABEL = /*#. Header label for operations to change cells in a spreadsheet */ gt.pgettext('menu-title', 'Cells');

    /**
     * Header label for generic data operations.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.DATA_HEADER_LABEL = /*#. Header label for sorting, filtering, and other data operations */ gt.pgettext('menu-title', 'Data');

    /**
     * Header label for data filter operations.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.FILTER_HEADER_LABEL = /*#. Header label for data filter options */ gt.pgettext('menu-title', 'Filter');

    /**
     * Header label for data sort operations.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.SORT_HEADER_LABEL = /*#. Header label for data sort options */ gt.pgettext('menu-title', 'Sort');

    // control labels ---------------------------------------------------------

    /**
     * Label for an 'Insert sheet' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.INSERT_SHEET_LABEL = gt('Insert sheet');

    /**
     * Label for a 'Rename sheet' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.RENAME_SHEET_LABEL = gt('Rename sheet');

    /**
     * Label for a 'Copy sheet' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.COPY_SHEET_LABEL = gt('Copy sheet');

    /**
     * Label for a 'Delete sheet' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.DELETE_SHEET_LABEL = gt('Delete sheet');

    /**
     * Label for a 'Unprotect sheet' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.UNPROTECT_SHEET_LABEL = gt('Unprotect sheet');

    /**
     * Label for a 'Protect sheet' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.PROTECT_SHEET_LABEL = gt('Protect sheet');

    /**
     * Label for a 'Show sheet' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.SHOW_SHEET_LABEL = /*#. make a sheet visible that is currently hidden */ gt('Show sheet');

    /**
     * Label for a 'Hide sheet' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.HIDE_SHEET_LABEL = gt('Hide sheet');

    /**
     * Label for a 'Reorder sheets' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.REORDER_SHEETS_LABEL = gt('Reorder sheets');

    /**
     * Label for a 'Show all hidden sheets' button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.UNHIDE_SHEETS_LABEL = gt('Unhide sheets');

    /**
     * Label for a line-break toggle button.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.LINEBREAK_LABEL =
        //#. check box: automatic word wrapping (multiple text lines) in spreadsheet cells
        gt('Automatic word wrap');

    /**
     * A text label or tool tip for a 'Format Painter' control.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.FORMAT_PAINTER_LABEL =
        //#. copy formatting from one location and apply it to other cells
        gt('Format painter');

    /**
     * An map containing GUI labels for all function categories.
     *
     * @type Object<String,String>
     * @constant
     */
    SpreadsheetLabels.FUNCTION_CATEGORY_LABELS = {
        complex: /*#. Category for spreadsheet functions: complex numbers (e.g. IMSUM, IMSIN, IMPOWER) */ gt.pgettext('function-category', 'Complex'),
        conversion: /*#. Category for spreadsheet functions: conversion (e.g. DEC2HEX, ROMAN, CONVERT) */ gt.pgettext('function-category', 'Conversion'),
        database: /*#. Category for spreadsheet functions: database-like (e.g. DSUM, DCOUNT) */ gt.pgettext('function-category', 'Database'),
        datetime: /*#. Category for spreadsheet functions: date/time calculation (e.g. TODAY, WEEKDAY) */ gt.pgettext('function-category', 'Date and time'),
        engineering: /*#. Category for spreadsheet functions: engineering (e.g. BESSELI, ERF) */ gt.pgettext('function-category', 'Engineering'),
        financial: /*#. Category for spreadsheet functions: financial functions (e.g. ACCRINT, COUPDAYS) */ gt.pgettext('function-category', 'Financial'),
        information: /*#. Category for spreadsheet functions: information about cells (e.g. ISNUMBER, ISTEXT) */ gt.pgettext('function-category', 'Information'),
        logical: /*#. Category for spreadsheet functions: logical functions (e.g. AND, OR, NOT, IF) */ gt.pgettext('function-category', 'Logical'),
        math: /*#. Category for spreadsheet functions: common mathematical functions (e.g. SUM, SIN, ROUND) */ gt.pgettext('function-category', 'Math'),
        matrix: /*#. Category for spreadsheet functions: matrix functions (e.g. MMULT, MINVERSE, MDETERM) */ gt.pgettext('function-category', 'Matrix'),
        reference: /*#. Category for spreadsheet functions: cell references (e.g. ADDRESS, HLOOKUP, COLUMN) */ gt.pgettext('function-category', 'Reference'),
        statistical: /*#. Category for spreadsheet functions: statistical functions (e.g. COUNT, STDEV, HYPGEOMDIST) */ gt.pgettext('function-category', 'Statistical'),
        text: /*#. Category for spreadsheet functions: text functions (e.g. LEN, MID, REPLACE) */ gt.pgettext('function-category', 'Text'),
        web: /*#. Category for spreadsheet functions: web functions (e.g. ENCODEURL, HYPERLINK, WEBSERVICE) */ gt.pgettext('function-category', 'Web')
    };

    /**
     * A special value for the 'Custom' entry in the 'Number Format Code'
     * drop-down list, which must simply be different from any valid number
     * format.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.CUSTOM_FORMAT_VALUE = '\x00';

    /**
     * The label of the 'Custom' entry in the 'Number Format Code' drop-down
     * list.
     *
     * @type String
     * @constant
     */
    SpreadsheetLabels.CUSTOM_FORMAT_LABEL =
        //#. number format category in spreadsheets: all user-defined number formats
        gt.pgettext('format-code', 'Custom');

    // control options --------------------------------------------------------

    /**
     * Standard options for a line-break toggle button.
     *
     * @constant
     */
    SpreadsheetLabels.LINEBREAK_BUTTON_OPTIONS = {
        icon: 'docs-insert-linebreak',
        tooltip: SpreadsheetLabels.LINEBREAK_LABEL,
        toggle: true,
        dropDownVersion: { label: SpreadsheetLabels.LINEBREAK_LABEL }
    };

    /**
     * Standard options for a 'Format painter' button.
     *
     * @constant
     */
    SpreadsheetLabels.FORMAT_PAINTER_BUTTON_OPTIONS = {
        icon: 'docs-format-painter',
        tooltip: SpreadsheetLabels.FORMAT_PAINTER_LABEL,
        toggle: true,
        dropDownVersion: { label: SpreadsheetLabels.FORMAT_PAINTER_LABEL }
    };

    /**
     * Standard options for 'Sort ascending' buttons.
     *
     * @constant
     */
    SpreadsheetLabels.SORT_ASC_BUTTON_OPTIONS = {
        icon: 'docs-sort-up',
        label: gt.pgettext('sort', 'Ascending'),
        tooltip: gt.pgettext('sort', 'Sort from smallest to largest value')
    };

    /**
     * Standard options for 'Sort descending' buttons.
     *
     * @constant
     */
    SpreadsheetLabels.SORT_DESC_BUTTON_OPTIONS = {
        icon: 'docs-sort-down',
        label: gt.pgettext('sort', 'Descending'),
        tooltip: gt.pgettext('sort', 'Sort from largest to smallest value')
    };

    /**
     * Standard options for 'Custom sort' buttons.
     *
     * @constant
     */
    SpreadsheetLabels.SORT_CUSTOM_BUTTON_OPTIONS = {
        //#. sorting: label of the button opening a menu with custom sort options
        label: gt.pgettext('sort', 'Custom'),
        tooltip: gt.pgettext('sort', 'Settings for sorting')
    };

    /**
     * Standard options for 'Clear contents' buttons.
     *
     * @constant
     */
    SpreadsheetLabels.CLEAN_OPTIONS = {
        label: gt('Clear contents')
        //TODO: missing tooltip
    };

    // style sets for list controls -------------------------------------------

    /**
     * An list containing the default format codes, and the GUI labels for all
     * predefined number format categories.
     *
     * @constant
     */
    SpreadsheetLabels.FORMAT_CATEGORY_LIST = [
        { name: 'standard',   label: /*#. number format category in spreadsheets: no special format, show numbers as they are */ gt.pgettext('number-format', 'General') },
        { name: 'number',     label: /*#. number format category in spreadsheets: numbers with specific count of decimal places */ gt.pgettext('number-format', 'Number') },
        { name: 'currency',   label: /*#. number format category in spreadsheets: number with currency symbol */ gt.pgettext('number-format', 'Currency') },
        { name: 'date',       label: /*#. number format category in spreadsheets: date formats */ gt.pgettext('number-format', 'Date') },
        { name: 'time',       label: /*#. number format category in spreadsheets: time formats */ gt.pgettext('number-format', 'Time') },
        { name: 'datetime',   label: /*#. number format category in spreadsheets: combined date/time formats */ gt.pgettext('number-format', 'Date and time') },
        { name: 'percent',    label: /*#. number format category in spreadsheets: numbers with percent sign */ gt.pgettext('number-format', 'Percentage') },
        { name: 'scientific', label: /*#. number format category in spreadsheets: scientific notation (e.g. 1.23E+10) */ gt.pgettext('number-format', 'Scientific') },
        { name: 'fraction',   label: /*#. number format category in spreadsheets: fractional numbers (e.g. 3 1/4) */ gt.pgettext('number-format', 'Fraction') },
        { name: 'text',       label: /*#. number format category in spreadsheets: text only */ gt.pgettext('number-format', 'Text') },
        { name: 'custom',     label: /*#. number format category in spreadsheets: all user-defined number formats */ gt.pgettext('number-format', 'Custom') }
    ];

    // other spreadsheet specific labels --------------------------------------

    /**
     * @constant
     */
    SpreadsheetLabels.CALCULATING_LABEL =
        //#. A label shown in the top-bar of a spreadsheet while calculating the cell formulas
        gt('Calculating');

    /**
     * Returns the name of the passed column together with the word 'Column',
     * e.g. 'Column A'.
     *
     * @param {Number} col
     *  The zero-based column index to be converted to a column label.
     *
     * @returns {String}
     *  The label for the specified column.
     */
    SpreadsheetLabels.getColLabel = function (col) {
        //#. UI label for a spreadsheet column, e.g. "Column A", "Column B", etc.
        //#. %1$s is the name of the column
        //#, c-format
        return gt('Column %1$s', _.noI18n(Address.stringifyCol(col)));
    };

    /**
     * Returns the name of the passed row together with the word 'Row', e.g.
     * 'Row 1'.
     *
     * @param {Number} row
     *  The zero-based row index to be converted to a row label.
     *
     * @returns {String}
     *  The label for the specified row.
     */
    SpreadsheetLabels.getRowLabel = function (row) {
        //#. UI label for a spreadsheet row, e.g. "Row 1", "Row 2", etc.
        //#. %1$s is the name of the row
        //#, c-format
        return gt('Row %1$s', _.noI18n(Address.stringifyRow(row)));
    };

    /**
     * Returns the name of the passed series together with the word 'Series', e.g.
     * 'Series 1'.
     *
     * @param {Number} seriesIndex
     *  The zero-based series index to be converted to a series label.
     *
     * @returns {String}
     *  The label for the specified series.
     */
    SpreadsheetLabels.getSeriesLabel = function (seriesIndex) {
        //#. A data series in a chart object (a sequence of multiple data points with a title)
        //#. %1$d is the numeric index of the series (e.g. "Series 1", "Series 2", etc.)
        //#. This label should match the default title for data series in other spreadsheet
        //#. applications (OpenOffice/LibreOffice Calc, Microsoft Excel).
        //#, c-format
        return gt('Series %1$d', _.noI18n(seriesIndex + 1));
    };

    /**
     * Returns the name of the passed column or row together with the word
     * 'Column' or 'Row' respectively, e.g. 'Column A', or 'Row 1'.
     *
     * @param {Number} index
     *  The zero-based column or row index to be converted to a label.
     *
     * @param {Boolean} columns
     *  Whether to return a column label (true), or a row label (false).
     *
     * @returns {String}
     *  The label for the specified column or row.
     */
    SpreadsheetLabels.getColRowLabel = function (index, columns) {
        return columns ? SpreadsheetLabels.getColLabel(index) : SpreadsheetLabels.getRowLabel(index);
    };

    /**
     * Returns the localized name of the specified function category.
     *
     * @param {String} category
     *  The internal identifier of a function category.
     *
     * @returns {String}
     *  The localized name of the specified function category.
     */
    SpreadsheetLabels.getCategoryLabel = function (category) {

        // return labels for valid categories
        if (category in SpreadsheetLabels.FUNCTION_CATEGORY_LABELS) {
            return SpreadsheetLabels.FUNCTION_CATEGORY_LABELS[category];
        }

        // assert an unknown category, return it capitalized
        Utils.error('SpreadsheetLabels.getCategoryLabel(): unknown function category: "' + category + '"');
        return _.noI18n(Utils.capitalize(category));
    };

    // exports ================================================================

    return SpreadsheetLabels;

});
