/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/dialog/customsortdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/dialog/basedialog',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/controls',
    'gettext!io.ox/office/spreadsheet/main',
    'less!io.ox/office/spreadsheet/view/dialog/customsortdialog'
], function (Utils, KeyCodes, BaseDialog, Iterator, TriggerObject, Labels, Controls, gt) {

    'use strict';

    // convenience shortcuts
    var CheckBox = Controls.CheckBox;

    var customSortStr       = /*#. menu title: custom sort options */ gt.pgettext('sort', 'Custom sort'),
        directionStr        = /*#. sorting: the sort direction headline for 'top to bottom' or 'left to right' */ gt.pgettext('sort', 'Direction'),
        hasHeadersStr       = /*#. sorting: selected range has headers which shouldn't be sorted */ gt.pgettext('sort', 'Selection has headers'),
        expandSortingStr    = /*#. sorting: add new rule for which to sort */ gt.pgettext('sort', 'Add sort criteria'),
        ascStr              = /*#. sort order 'ascending' */ gt.pgettext('sort', 'Ascending'),
        descStr             = /*#. sort order 'descending' */ gt.pgettext('sort', 'Descending'),
        optionsStr          = /*#. sorting: the sort options */ gt.pgettext('sort', 'Options'),
        sortByStr           = /*#. sorting: sort range by (for example "column B" or "row 4") */ gt.pgettext('sort', 'Sort by'),
        orderStr            = /*#. sorting: the sort order (for example "ascending") */ gt.pgettext('sort', 'Order'),
        undefinedStr        = /*#. sorting: the sorting order is not yet defined */ gt.pgettext('sort', 'undefined'),

        error1              = /*#. sorting: error-case = col/row multiply selected */ gt.pgettext('sort', 'Some columns or rows are being sorted more than once.'),
        error2              = /*#. sorting: error-case = missing col/row selection in custom sort popup */ gt.pgettext('sort', 'All sort criteria must have a column or row specified.');

    var DIRECTIONS = {
        vertical: {
            name: /*#. sorting direction for spreadsheet data */ gt.pgettext('sort', 'Top to bottom'),
            value: 'vertical'
        },
        horizontal: {
            name: /*#. sorting direction for spreadsheet data */ gt.pgettext('sort', 'Left to right'),
            value: 'horizontal'
        }
    };

    /**
     * generates a dropdown-menu far away from the TK-Forms (bad way)
     *
     * @param  {Array<Object>} values
     *     The values which described the dropdown-entries.
     *     E.g.: { name: 'headline 1', value: '1' }
     *
     * @param  {Object} options
     *     Some options which can manipulate the dropdown.
     *  - {Function} [options.handler]
     *      If a handler is set, it will be invoked on each 'change' of the dropdown.
     *
     * @return {HTML}
     *     The generated dropdown-html.
     */
    function getDropdownMenu(values, options) {
        var menu        = null,
            jqList      = $('<ul class="dropdown-menu" role="menu">');

        if (values.length > 0) {
            // create menu anchor
            menu = $('<span class="dropdown">').
                append($('<a href="#">').
                    attr({ tabindex: 0, 'data-type': values[0].value, 'aria-haspopup': true, 'data-toggle': 'dropdown' }).
                    addClass('text').
                    text(values[0].name)).
                append(jqList);

            // fill dropdown menu with given values and assign callback handler, if given
            values.forEach(function (object) {

                var itemLink = $('<a class="dropdown-listentry" href="#" role="menuitem">').attr('data-value', object.value).text(object.name);

                var listItem = $('<li>').append(itemLink);
                listItem.on('click', function (evt) {
                    evt.preventDefault();
                    var curText = $(this).text(),
                        curValue = $(this).find('a').data('value');

                    // set currently selected entry as menu title
                    menu.find('a[data-toggle=dropdown]').text(curText);

                    if (_.isFunction(options.handler)) {
                        options.handler(curValue);
                    }
                });

                jqList.append(listItem);
            });
        }

        return menu;
    }

    // class RuleModel ========================================================

    /**
     * This is a single rule which will be used to sort.
     *
     * @param {CustomSortDialog} parentDialog
     *  The dialog instance containing this rule.
     */
    var Rule = TriggerObject.extend({ constructor: function (docView, parentDialog) {

        var self            = this,
            ruleCollection  = parentDialog.getRuleCollection(),

            index           = null,
            orderBy         = 'ascending',

            ruleHolder      = parentDialog.getContentNode().find('.rule-collection'),
            ruleLine        = null,
            remBtn          = null;

        TriggerObject.call(this, docView);

        // private ---------------------------------------------

        function selectEntry(element) {
            // set currently selected entry as menu title
            colRowSelector.find('a[data-toggle=dropdown]').text($(element).text()).css('font-style', '');

            // set the new choosen col/row-index
            index = $(element).find('a').data('value');

            // recheck problems after changing the value
            ruleCollection.hasProblem();
        }

        /**
         * Method to highlight the rule in case of an error.
         *
         * @param  {Event} event
         *     The trigger-event.
         *
         * @param  {Array<Number>}
         *     The array with dublicate-used-indexes.
         */
        function highlight(e, indexArray) {
            // first, remove the potential existing warning-icon
            colRowSelector.find('i.fa-warning').remove();

            // is the rules index 'null' or matches one of the dublicate-indexes
            if (index === null || indexArray.indexOf(index) !== -1) {
                // add the warning-icon and style the warning-icon 'red'
                colRowSelector.prepend($('<i>').addClass('fa fa-warning').attr('style', 'color: #fc0;'));
            }
        }

        /**
         * Method to update the col/row-dropdown labels.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Array<Object>} labels
         *  The array with the new labels for the col/row-dropdown.
         */
        function updateLabels(event, labels) {

            // get the dropdown-element
            var menu = colRowSelector.find('ul.dropdown-menu');

            // clear the dropdown
            menu.empty();

            // reset the dropdown-opener
            colRowSelector.find('>a')
                .attr('tabindex', 0)
                .attr('aria-haspopup', true)
                .attr('data-toggle', 'dropdown')
                .addClass('text')
                .text(undefinedStr)
                .css('font-style', 'italic');

            // refill col/row DorpDown with new label-values
            labels.forEach(function (entry) {
                var buttonNode = $('<a href="#" class="dropdown-listentry" role="menuitem" data-value="' + entry.index + '">').text(entry.name);
                if (entry.italic) { buttonNode.css('font-style', 'italic'); }
                menu.append(
                    $('<li>').append(buttonNode).on('click', function (evt) {
                        // prevent the default event
                        evt.preventDefault();
                        // select the dropdown-element
                        selectEntry(this);
                    })
                );
            });

            // when the rule has a defined index
            if (self.getIndex() !== null) {
                var txtEle = menu.find('a[data-value="' + self.getIndex() + '"]'),
                    txt = (txtEle.length > 0) ? txtEle.text() : undefinedStr;
                // preselect it after refill the dropdown
                colRowSelector.find('>a').text(txt).css('font-style', '');
            }
        }

        // public ----------------------------------------------

        /**
         * Preselect the first dropdown-value
         *
         * @return {Rule}
         *     Returns the self-reference.
         */
        this.preselectFirst = function (colRowIndex) {
            var menuItems = colRowSelector.find('ul.dropdown-menu li a[data-value="' + colRowIndex + '"]');
            selectEntry(menuItems.parent());
            return this;
        };

        /**
         * Returns the choosen index
         *
         * @return {Number}
         *     The chossen index.
         */
        this.getIndex = function () {
            return index;
        };

        /**
         * Set the index
         *
         * @param {integer} key
         *        The key
         *
         * @return {Rule}
         *     Returns the self-reference.
         */
        this.setIndex = function (key) {
            index = key;
            return this;
        };

        /**
         * Returns the chossen order-value.
         *
         * @return {String}
         *     'vertical' or 'horizontal'.
         */
        this.getOrderBy = function () {
            return orderBy;
        };

        /**
         * Set the order
         *
         * @param {string} order
         *        The order-direction
         *
         * @return {Rule}
         *     Returns the self-reference.
         */
        this.setOrderBy = function (string) {
            orderBy = string;
            $(orderSelector).find('>a').text((orderBy === 'ascending') ? ascStr : descStr);
            return this;
        };

        /**
         * Removes the rule from the UI.
         *
         * @return {Rule}
         *     Returns the self-reference.
         */
        this.remove = function () {
            ruleLine.remove();
            return this;
        };

        // initialize ------------------------------------------

        // prepares the col/row-dropdown (without the whole entries at this time)
        var colRowSelector = getDropdownMenu([{ name: undefinedStr }]);

        // prepares the direction (horizontal/vertical) dropdown
        var orderSelector = getDropdownMenu(
            [
                {
                    name: ascStr,
                    value: 'ascending'
                }, {
                    name: descStr,
                    value: 'descending'
                }
            ], {
                handler: function (val) {
                    orderBy = val;
                }
            }
        );

        // add this new rule to the ruleHolder-container
        ruleHolder.append(
            // prepares the rule-line with apending both dropdowns to itself
            ruleLine = $('<div>').append(
                colRowSelector,
                orderSelector
            )
        );

        // If this is not the first rule, add the 'remove-button' to the rule-line.
        // For tables, we can add a remove-button everytime
        if (parentDialog.getTableModel() || (ruleCollection.getRules().length > 0)) {
            remBtn = $('<a href="#" tabindex="0">').addClass('trash').append($('<i>').addClass('fa fa-trash-o'));
            remBtn.on('click keydown', function (e) {
                if ((e.type !== 'keydown') || (e.keyCode === KeyCodes.SPACE)) {
                    ruleCollection.removeRule(self);
                }
                return false;
            });
            ruleLine.append(remBtn);
        }

        // set some listener
        this.on('highlight', highlight);
        this.listenTo(ruleCollection, 'highlight', highlight);
        this.listenTo(parentDialog, 'refresh:labels', updateLabels);

        this.registerDestructor(function () {
            self = docView = parentDialog = ruleCollection = null;
        });

    } });

    // class RuleCollection ===================================================

    /**
     * The RuleCollection provides all rules which will be used to sort the
     * selected range.
     *
     * @param {CustomSortDialog} parentDialog
     *  The dialog instance containing this rule.
     */
    var RuleCollection = TriggerObject.extend({ constructor: function (docView, parentDialog) {

        // array with all rules
        var arrRules = [];

        TriggerObject.call(this, docView);

        // public methods -----------------------------------------------------

        /**
         * Method to add a new rule.
         *
         * @return {RuleCollection}
         *     Returns self-reference.
         */
        this.addRule = function () {
            var newRule = null;

            // check whether one more rule is allowed, or whether we reached
            // the count of cols/rows already
            if ((arrRules.length + 1) <= parentDialog.getRangeSize()) {
                // generate a new rule and push it to the rules-array of the ruleCollection
                arrRules.push(newRule = new Rule(docView, parentDialog));
                // trigger event 'addRule' to make dependent functions work
                this.trigger('addRule', newRule);
            }

            return this;
        };

        /**
         * Returns a single rule
         *
         * @param  {integer} key
         *         The index of the rule which should be returned
         *
         * @return {Rule}
         *         A rule object
         */
        this.getRule = function (key) {
            return (arrRules[key]) ? arrRules[key] : null;
        };

        /**
         * Preselect the first rule col/row
         *
         * @return {RuleCollection}
         *     Returns self-reference.
         */
        this.preselectFirst = function (columns) {
            var activeColRow = (columns) ? docView.getActiveColumnDescriptor().index : docView.getActiveRowDescriptor().index;
            // return the rule, if exists
            arrRules[0].preselectFirst(activeColRow);
            return this;
        };

        /**
         * Returns current rules prepared for the
         * cellCollections "generateSortOperations"-method.
         *
         * @return {Array}
         *     An array with objects containing the different sort-options.
         */
        this.getRules = function () {

            // iterate over all existing rules
            return arrRules.map(function (rule) {
                return { index: rule.getIndex(), orderBy: rule.getOrderBy() };
            });
        };

        /**
         * Remove a specific rule.
         *
         * @param  {Rule} rule
         *     The rule which should be deleted.
         *
         * @return {RuleCollection}
         *     Returns self-reference.
         */
        this.removeRule = function (rule) {
            // delete rule from ruleCollection-array
            arrRules.splice(arrRules.indexOf(rule), 1);
            // call delete-method of rule (to remove html from DOM)
            rule.remove();
            // trigger 'removeRule'-event to invoke dependent methods
            this.trigger('removeRule', rule);
            return this;
        };

        /**
         * Removes all rules.
         *
         * @return {RuleCollection}
         *     Returns self-reference
         */
        this.clear = function () {
            // reverse rule-order to delete rules from last to end
            arrRules.reverse();

            // set i-index
            var i = (arrRules.length - 1);
            while (i >= 0) {
                // call remove rule
                this.removeRule(arrRules[i]);
                i--;
            }
            return this;
        };

        /**
         * Detect if there were any problem(s) in sort-configuration.
         * For example: A col is selected twice.
         *
         * @return {Boolean}
         *     true = there is a problem; false = everything is ok
         */
        this.hasProblem = function (options) {
            var // initially we have no problems
                hasProblem  = false,
                // col/row indexes which were used already
                usedIndexes = [],
                // collect all dublicate used indexes
                dublicates  = [];

            // cleanup all hightlighty
            this.trigger('highlight', []);

            // iterate over all rules
            arrRules.forEach(function (rule) {
                // we have a problem, if a rule has no index
                if (rule.getIndex() === null) {
                    // activate problem
                    hasProblem = true;
                    // trigger this rule to highlight itselfs
                    rule.trigger('highlight', []);

                // if the index isn't used anywhere
                } else if (usedIndexes.indexOf(rule.getIndex()) === -1) {
                    // add it to the usedIndexes-array
                    usedIndexes.push(rule.getIndex());

                // otherwise the index is used
                } else {
                    // and we have a problem
                    hasProblem = true;
                    // add index to dublicates-array
                    dublicates.push(rule.getIndex());
                }
            });

            // when we detected a problem
            if (hasProblem) {
                // trigger all rules to hightlight themselfs, if they were
                // selected one of the dublicate indexes
                this.trigger('highlight', dublicates);

                if (Utils.getBooleanOption(options, 'final', false)) {
                    docView.yell({
                        type: 'warning',
                        message: (dublicates.length > 0) ? error1 : error2
                    });
                }
            }

            if (parentDialog.trigger) {
                parentDialog.trigger('problem', !hasProblem);
            }

            // return problem-boolean
            return hasProblem;
        };

        this.registerDestructor(function () {
            arrRules.forEach(function (rule) { rule.destroy(); });
            docView = parentDialog = null;
        });

    } });

    // class CustomSortDialog =================================================

    /**
     * A modal dialog with options for sorting.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    var CustomSortDialog = BaseDialog.extend(function (docView, tableModel, sortOptions) {

        var // self reference
            self = this,
            range = null,

            ruleCollection = new RuleCollection(docView, this),

            hasHeadersCheckbox = new CheckBox(docView, { label: hasHeadersStr, boxed: true }),
            directionGroup = null,
            direction = null,
            columns = null,
            labels = [];

        // base constructor ---------------------------------------------------

        BaseDialog.call(this, docView, { title: customSortStr, width: 400, okLabel: gt('Sort') });

        // private methods ----------------------------------------------------

        /**
         * iterate over the leading-col/-row and collect the labels for the dropdown.
         * If there is no header, collect the col/row names.
         */
        function updateLabelStrings() {

            var hasHeader       = tableModel ? tableModel.hasHeaderRow() : hasHeadersCheckbox.getValue();
            var headerRange     = (tableModel && hasHeader) ? tableModel.getHeaderRange() : range.lineRange(!columns, 0);
            var cellCollection  = docView.getCellCollection();

            // collect labels
            labels = Iterator.map(headerRange, function (address) {

                // the current column/row index
                var index = address.get(columns);
                // use display strings of the header cells if specified
                var display = hasHeader ? cellCollection.getDisplayString(address) : null;
                // empty or missing display string (e.g. format error): fall-back to column/row label
                var label = display || Labels.getColRowLabel(index, columns);
                // show fall-back column/row label in header mode in italic style
                var italic = hasHeader && !display;

                return { index: index, name: label, italic: italic };
            });

            // trigger 'refresh:labels' event to let dependent elements be refreshing
            self.trigger('refresh:labels', labels);
        }

        /**
         * Change the sort-direction (vertical/horizontal)
         *
         * @param  {String} val
         *     'vertical' or 'horizontal'.
         */
        function changeDirection(val) {
            // if nothing changes, get out here
            if (direction === val) { return; }

            // set some variables
            direction = val;
            columns = (direction === 'vertical');

            // set the direction-dropdown (value and title)
            var displayElement = directionGroup.find('a[data-type]');
            displayElement.attr('data-type', direction);
            displayElement.text(DIRECTIONS[direction].name);

            // update the labels-array
            updateLabelStrings();

            // remove current rules
            ruleCollection.clear();
            // add new (first) rule
            ruleCollection.addRule();

            ruleCollection.preselectFirst(columns);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the rule collection of the sort dialog.
         *
         * @returns {RuleCollection}
         *  The rule collection of the sort dialog.
         */
        this.getRuleCollection = function () {
            return ruleCollection;
        };

        /**
         * Returns the size of the possible rule-count.
         * It is not allowed to set more rules than columns/rows were selected.
         *
         * @return {Number}
         *     The number of the selected cols/rows.
         */
        this.getRangeSize = function () {
            return range.size(columns);
        };

        /**
         * Set the state of the "new Rule"-button.
         * Disable it in case of no more rules are allowed.
         */
        this.checkNewBtn = function () {
            addBtn.toggleClass('disabled', ((this.getRangeSize() === 1) || (ruleCollection.getRules().length === this.getRangeSize())));
        };

        /**
         * Returns the model of the table range this dialog is based on.
         *
         * @returns {TableModel|Null}
         *  The model of the table range this dialog is based on, or null for
         *  sorting an arbitrary cell range.
         */
        this.getTableModel = function () {
            return tableModel;
        };

        // initialization -----------------------------------------------------

        // close dialog when losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // add base menu items
        this.getPopup().addClass('io-ox-office-custom-sort-dialog');

        // initialze sort options from table model
        if (!sortOptions && tableModel) {
            sortOptions = { headers: true, orderRules: tableModel.getSortRules() };
        }

        // for tables, leave some options away
        if (!tableModel) {
            this.getBody().append(optionsStr, hasHeadersCheckbox.getNode());
        }

        // prepare direction (vertical/horizontal) dowpdown
        directionGroup = $('<div class="dropdown-link">').append(
            $('<label>').append(
                $('<span>').text(directionStr),
                $('<span>').text(_.noI18n(':') + ' ')
            ),
            getDropdownMenu(
                [DIRECTIONS.vertical, DIRECTIONS.horizontal],
                { handler: changeDirection }
            )
        );

        // prepare the holder-container (with headlines) for the rules
        var dropDownContainer = $('<div class="rule-collection">').append(
            $('<span>').text(sortByStr),
            $('<span>').text(orderStr)
        );

        // prepare the "plus"-button to be able to add more sort-rules
        var addBtn = $('<a class="newSortRuleBtn" href="#" tabindex="0">').text(expandSortingStr);
        addBtn.on('click keydown', function (e) {
            if ((e.type !== 'keydown') || (e.keyCode === KeyCodes.SPACE)) {
                ruleCollection.addRule();
            }
            return false;
        });

        // for tables, leave some options away
        if (!tableModel) {
            this.getBody().append(directionGroup, '<hr>');
        }
        // add all the stuff to the dialog
        this.getBody().append(
            dropDownContainer, '<hr>', $('<div class="btnHolder">').append(addBtn));

        // refill the column/row dropdown on changing the "hasHeadline"-option
        hasHeadersCheckbox.on('group:change', updateLabelStrings);

        this.on('show', function () {
            // set initial state of the "new"-btn
            $('a.newSortRuleBtn').toggleClass('disabled', (self.getRangeSize() === 1));

            ruleCollection.preselectFirst(columns);

            // fill the dialog with table sort options
            if (sortOptions) {
                if (sortOptions.direction) {
                    changeDirection(sortOptions.direction);
                }
                if (sortOptions.headers) {
                    hasHeadersCheckbox.setValue(sortOptions.headers);
                    updateLabelStrings();
                }
                if (sortOptions.orderRules) {
                    sortOptions.orderRules.forEach(function (rule, key) {
                        if (key > 0) {
                            ruleCollection.addRule();
                        }
                        var ruleObj = ruleCollection.getRule(key),
                            startCol = tableModel ? tableModel.getRange().start[0] : 0;
                        ruleObj.setIndex((rule.index >= 0) ? rule.index + startCol : null);
                        ruleObj.setOrderBy(rule.orderBy);
                    });
                    self.trigger('refresh:labels', labels);
                }
                ruleCollection.hasProblem();
            }
        });

        // beforeshow-handler to reset/update some values
        this.on('beforeshow', function () {
            // grab the data-range of tables, or grab the first range of the selection
            range = tableModel ? tableModel.getDataRange() : docView.getSelectedRanges().first();

            // reset direction
            changeDirection('vertical');

            // only try to detect a headline, when it's not a table
            if (!tableModel) {
                hasHeadersCheckbox.setValue(docView.getCellCollection().hasRangeHeadline(range, direction));
            }
            updateLabelStrings();
        });

        this.on('problem', function (e, state) {
            self.enableOkButton(state);
        });

        // action handler for the OK button
        this.setOkHandler(function () {

            // if there is a problem, keep this dialog open
            if (ruleCollection.hasProblem({ final: true })) {
                return new $.Deferred().reject();
            }

            return {
                direction:  direction,
                headers:    !tableModel && hasHeadersCheckbox.getValue(),
                orderRules: ruleCollection.getRules()
            };
        }, { keepOpen: 'fail' });

        // trigger label-update when a new rule was added
        ruleCollection.on('addRule', function () {
            // add labels (especially to new rule)
            self.trigger('refresh:labels', labels);
            // disable "new"-btn, if necessary
            self.checkNewBtn();
        });

        ruleCollection.on('removeRule', function () {
            // recheck possibly problems
            ruleCollection.hasProblem();
            // disable "new"-btn, if necessary
            self.checkNewBtn();
        });

        // destroy all class members
        this.registerDestructor(function () {
            hasHeadersCheckbox.destroy();
            ruleCollection.destroy();
            self = docView = tableModel = ruleCollection = null;
        });

    }); // class CustomSortDialog

    // exports ================================================================

    return CustomSortDialog;

});
