/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/sortmenubutton', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/control/compoundsplitbutton',
    'io.ox/office/spreadsheet/view/labels',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Button, CompoundSplitButton, Labels, gt) {

    'use strict';

    // class SortMenuButton ===================================================

    /**
     * Sorting control in toolbar.
     *
     * @constructor
     *
     * @extends CompoundSplitButton
     *
     * @param {SpreadsheetView} docView
     *  The document view containing this instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class
     *  CompoundSplitButton, except 'value', and 'updateCaptionMode'.
     */
    var SortMenuButton = CompoundSplitButton.extend({ constructor: function (docView, initOptions) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        CompoundSplitButton.call(this, docView, Utils.extendOptions({
            icon: Labels.SORT_ASC_BUTTON_OPTIONS.icon,
            //#. spreadsheet sorting: button tooltip for sorting
            label: gt.pgettext('sort', 'Sorting'),
            smallerVersion: { hideLabel: true }
        }, initOptions, {
            value: function () {
                return { orderRules: [{ orderBy: 'toggle' }] };
            }
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the icon according to the sort state.
         */
        function updateHandler(sortState) {

            var rules = sortState.orderRules,
                state = 'ascending';

            if (rules.length > 0 && rules[0].orderBy) {
                state = rules[0].orderBy;
            }

            switch (state) {
                case 'ascending':
                    self.setIcon(Labels.SORT_ASC_BUTTON_OPTIONS.icon);
                    break;
                case 'descending':
                    self.setIcon(Labels.SORT_DESC_BUTTON_OPTIONS.icon);
                    break;
                case 'custom':
                    self.setIcon('fa-cogs');
                    break;
            }
        }

        // initialization -----------------------------------------------------

        // create all drop-down menu items
        this.addGroup('cell/sort', new Button(docView, _.extend({ value: { orderRules: [{ orderBy: 'ascending' }] } }, Labels.SORT_ASC_BUTTON_OPTIONS)));
        this.addGroup('cell/sort', new Button(docView, _.extend({ value: { orderRules: [{ orderBy: 'descending' }] } }, Labels.SORT_DESC_BUTTON_OPTIONS)));
        this.addSeparator();
        this.addGroup('cell/sort/dialog', new Button(docView, Labels.SORT_CUSTOM_BUTTON_OPTIONS));

        // update the icon in the drop-down button
        this.registerUpdateHandler(updateHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = initOptions = null;
        });

    } }); // class SortMenuButton

    // exports ================================================================

    return SortMenuButton;

});
