/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/validationcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/extarray',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, ExtArray, ValueSet, ValueMap, ModelObject, AttributedModel, Operations, SheetUtils, FormulaUtils, TokenArray) {

    'use strict';

    // convenience shortcuts
    var UpdateMode = SheetUtils.UpdateMode;
    var Address = SheetUtils.Address;
    var RangeArray = SheetUtils.RangeArray;
    var RangeSet = SheetUtils.RangeSet;
    var Range3DArray = SheetUtils.Range3DArray;
    var MoveDescriptor = SheetUtils.MoveDescriptor;
    var FormulaType = FormulaUtils.FormulaType;

    // private global functions ===============================================

    /**
     * Returns the configuration flags for data validations dependent on the
     * file format of a document.
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model.
     *
     * @returns {Object}
     *  The configuration, as boolean flag set with the following properties:
     *  - {Boolean} indexedOps
     *      If true, integer indexes will be used to address validation models
     *      in operations (operation property "index"). Otherwise, the target
     *      range addresses will be used to identify operations (allows partial
     *      deletion and partial replacement on insertion).
     *  - {Boolean} customRef
     *      If true, the operations will contain a custom reference address
     *      (operation property "ref"). Otherwise, the reference address will
     *      be deduced automatically from the top-left cell of the bounding
     *      range of the target ranges.
     */
    function getConfigFlags(docModel) {
        var app = docModel.getApp();
        return {
            indexedOps: app.isOOXML(),  // OOXML uses indexes to address models
            customRef: app.isODF()      // ODF has explicit reference addresses
        };
    }

    /**
     * Transforms the specified target ranges.
     *
     * @param {RangeArray} targetRanges
     *  The addresses of the target ranges to be transformed.
     *
     * @param {Array<MoveDescriptor>} moveDescs
     *  An array of move descriptors that specify how to transform the passed
     *  target ranges.
     *
     * @param {Boolean} [reverse=false]
     *  If set to true, the move descriptors will be processed in reversed
     *  order, and the opposite move operation will be used to transform the
     *  target ranges.
     *
     * @returns {RangeArray}
     *  The transformed target ranges.
     */
    function transformTargetRanges(targetRanges, moveDescs, reverse) {
        // transform the passed ranges, expand the end of the range when inserting cells
        return MoveDescriptor.transformRanges(targetRanges, moveDescs, { expandEnd: true, reverse: reverse });
    }

    /**
     * Creates a token array representing the passed reference address of a
     * validation model as a formula expression.
     *
     * @param {SheetModel} sheetModel
     *  The model of the sheet containing the reference cell.
     *
     * @param {Address} refAddress
     *  The reference address of a validation model.
     *
     * @returns {TokenArray}
     *  The token array containing a formula expression that evaluates to the
     *  passed reference cell.
     */
    function createRefTokenArray(sheetModel, refAddress) {
        var tokenArray = new TokenArray(sheetModel, FormulaType.NAME);
        tokenArray.appendAddress(refAddress, { sheet: sheetModel.getIndex(), relSheet: true });
        return tokenArray;
    }

    // class ValidationModel ==================================================

    /**
     * Stores data validation settings for one or more cell ranges in a sheet.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this instance.
     *
     * @param {RangeArray} ranges
     *  The addresses of all cell ranges that contain the data validation
     *  settings.
     *
     * @param {Object} attrSet
     *  The data validation attribute set.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  - {String} [options.ref]
     *      The custom reference address, as formula expression.
     */
    var ValidationModel = AttributedModel.extend({ constructor: function (sheetModel, ranges, attrSet, initOptions) {

        // self reference
        var self = this;

        // the spreadsheet document model
        var docModel = sheetModel.getDocModel();

        // file-format dependent configuration
        var config = getConfigFlags(docModel);

        // the target cell ranges covered by the data validation
        var targetRanges = null;

        // the reference address for formulas with relative references
        var refAddress = null;

        // the token arrays representing the validation attributes 'value1' and 'value2'
        var tokenArrayMap = new ValueMap();

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, docModel, attrSet, { families: 'validation' });

        // private methods ----------------------------------------------------

        /**
         * Updates the token arrays after the validation attributes of this
         * validation model have been changed.
         */
        function changeAttributesHandler(event, newAttributes, oldAttributes) {

            // extract the validation attributes from the passed attribute sets
            newAttributes = (newAttributes || self.getMergedAttributeSet(true)).validation;
            oldAttributes = oldAttributes ? oldAttributes.validation : {};

            // parse a changed formula expression in attribute 'value1' or 'value2'
            tokenArrayMap.forEach(function (tokenArray, propName) {
                if (newAttributes[propName] !== oldAttributes[propName]) {
                    tokenArray.parseFormula('op', newAttributes[propName], { refAddress: refAddress });
                }
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this validation model for
         * the specified sheet.
         *
         * @internal
         *  Used by the class ValidationCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {ValidationModel}
         *  A clone of this validation model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            var attrSet = this.getExplicitAttributeSet(true);
            var refFormula = config.customRef ? createRefTokenArray(targetModel, refAddress).getFormula('op') : null;
            return new ValidationModel(targetModel, targetRanges, attrSet, { ref: refFormula });
        };

        /**
         * Tries to replace unresolved sheet names in the token arrays with
         * existing sheet indexes. Intended to be used after document import to
         * refresh all token arrays that refer to sheets that did not exist
         * during their creation.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.refreshAfterImport = function () {
            var options = { refAddress: refAddress };
            tokenArrayMap.forEach(function (tokenArray) {
                tokenArray.refreshAfterImport(options);
            });
            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the addresses of all target cell ranges of this validation
         * model.
         *
         * @returns {RangeArray}
         *  The addresses of all cell ranges.
         */
        this.getTargetRanges = function () {
            return targetRanges.clone(true);
        };

        /**
         * Returns whether the passed cell address is contained by the target
         * cell ranges of this validation model.
         *
         * @param {Address} address
         *  The address of the cell to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified cell is contained in this validation model.
         */
        this.containsCell = function (address) {
            return targetRanges.containsAddress(address);
        };

        /**
         * Returns the reference addresses of this validation model (the
         * top-left cell of the bounding range of all target cell ranges of
         * this validation model).
         *
         * @returns {Address}
         *  The address of the reference cell of this validation model.
         */
        this.getRefAddress = function () {
            return refAddress.clone();
        };

        /**
         * Returns the list contents for validations of type 'list' or 'source'
         * according to the formula expression contained in the attribute
         * 'value1'.
         *
         * @param {Address} targetAddress
         *  The target address used to resolve relative references in the
         *  formula expression.
         *
         * @returns {Array<Object>}
         *  The contents of the cells, if this model is a data validation of
         *  type 'list' or 'source', and the formula expression contained in
         *  the attribute 'value1' could be resolved successfully to a list;
         *  otherwise an empty array. Each cell content object will contain the
         *  following properties:
         *  - {Number|String|Boolean|ErrorCode|Null} [value]
         *      The cell value, or formula result (null for blank cells). Will
         *      be omitted for string lists (validation type 'list') to let the
         *      strings being parsed dynamically, resulting in floating-point
         *      numbers with a number format, booleans, or even formula cells.
         *  - {String|Null} display
         *      The display string for the list entry (will be the empty string
         *      for blank cells). The value null represents a cell result that
         *      cannot be formatted to a valid display string with the current
         *      number format of the cell.
         */
        this.queryListContents = function (targetAddress) {

            // resolve source data by validation type
            switch (this.getMergedAttributeSet(true).validation.type) {

                case 'list':
                    var stringList = tokenArrayMap.get('value1').resolveStringList();
                    // convert plain string array to objects with display properties (no values to force parsing the text later)
                    return stringList ? stringList.map(function (text) { return { display: text }; }) : [];

                case 'source':
                    // get the cell range addresses contained in the first token array
                    var result = tokenArrayMap.get('value1').interpretFormula('ref', { refAddress: refAddress, targetAddress: targetAddress });
                    // collect all cell values and display strings, restrict to 1000 entries
                    return (result.value instanceof Range3DArray) ? docModel.getRangeContents(result.value, { compressed: true, blanks: true, display: true, maxCount: 1000 }) : [];
            }

            // default (wrong validation type)
            return [];
        };

        /**
         * Changes the target cell ranges of this validation model.
         *
         * @param {RangeArray} newRanges
         *  The addresses of the new target cell ranges. The old target ranges
         *  of this validation model will be replaced completely.
         *
         * @returns {Boolean}
         *  Whether the current target ranges have changed.
         */
        this.setTargetRanges = function (newRanges) {

            // clone and merge the target ranges to prevent changes form external code
            newRanges = newRanges.merge();

            // nothing to do if the ranges will not change
            if (targetRanges && targetRanges.equals(newRanges, true)) {
                return false;
            }

            // set the new target ranges
            targetRanges = newRanges;

            // update implicit reference address
            if (!config.customRef && !targetRanges.empty()) {
                refAddress = targetRanges.boundary().start;
            }

            return true;
        };

        /**
         * Changes the reference address of this validation model.
         *
         * @param {String} refFormula
         *  A formula expression that contains a single reference with a single
         *  cell address.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.setRefFormula = function (refFormula) {

            // parse the passed reference address expression
            var tokenArray = new TokenArray(sheetModel, FormulaType.NAME);
            tokenArray.parseFormula('op', refFormula);

            // try to convert to a single reference address (do not pass a reference
            // sheet index, sheet name is expected to exist in the formula expression)
            var ranges = tokenArray.resolveRangeList();

            // if the expression resolves to a single cell address, use it as reference address
            var isSingle = (ranges.length === 1) && ranges.first().single();
            refAddress = isSingle ? ranges.first().start : Address.A1.clone();

            return this;
        };

        /**
         * Changes all formula expressions stored in the validation attributes
         * 'value1' and 'value2' with sheet references, after the sheet
         * collection in the document has been changed.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.transformSheet = function (fromSheet, toSheet) {
            tokenArrayMap.forEach('transformSheet', fromSheet, toSheet);
            return this;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operation to restore this validation model after
         * it has been deleted.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} index
         *  The index of this validation model to be inserted into the document
         *  operations.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {RangeArray} [options.ranges]
         *      The addresses of the cell ranges to be generated for the insert
         *      operations. If omitted, the current target ranges will be used.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, index, options) {

            // build the operation properties (add validation properties as top-level properties)
            var properties = config.indexedOps ? { index: index } : {};
            _.extend(properties, this.getExplicitAttributeSet(true).validation);

            // create the formula expression for the reference address
            if (config.customRef) {
                var tokenArray = createRefTokenArray(sheetModel, refAddress);
                properties.ref = tokenArray.getFormula('op');
            }

            // resolve custom target ranges
            var ranges = (options && options.ranges) || targetRanges;

            // generate the undo operation
            generator.generateRangesOperation(Operations.INSERT_VALIDATION, ranges, properties, { undo: true });
            return this;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of this data validation model.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} index
         *  The index of this validation model to be inserted into the document
         *  operations.
         *
         * @param {Object} updateDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on its 'type' property. See
         *  method TokenArray.resolveOperation() for more details.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.generateUpdateFormulaOperations = function (generator, index, updateDesc) {

            // the properties for the document operation
            var operProps = {};
            // the properties for the undo operation
            var undoProps = {};
            // restore the original target ranges on undo, if they cannot be restored implicitly
            var restoredRanges = null;

            // update the target ranges, if the moved cells are located in the own sheet
            if ((updateDesc.type === UpdateMode.MOVE_CELLS) && (sheetModel.getIndex() === updateDesc.sheet)) {

                // shortcut to the array of move descriptors
                var moveDescs = updateDesc.moveDescs;

                // restore the entire validation model, if it will be deleted implicitly
                var transformRanges = transformTargetRanges(targetRanges, moveDescs);
                if (transformRanges.empty()) {
                    return this.generateRestoreOperations(generator, index);
                }

                // restore the original target ranges on undo, if they cannot be restored implicitly
                restoredRanges = transformTargetRanges(transformRanges, moveDescs, true);

                // the resulting implicit reference address after undo (may differ from original
                // reference address, e.g. when deleting the first column/row of the bounding range)
                var restoredRefAddress = restoredRanges.boundary().start;

                // transform the explicit reference address
                if (config.customRef) {
                    var newRefAddress = MoveDescriptor.transformAddress(refAddress, moveDescs);
                    if (newRefAddress) {
                        // new reference address will always be restored to current, do not relocate formulas below
                        restoredRefAddress = refAddress;
                    } else {
                        newRefAddress = refAddress;
                        restoredRefAddress = MoveDescriptor.transformAddress(refAddress, moveDescs, { reverse: true });
                    }
                    if (newRefAddress.differs(refAddress)) {
                        operProps.ref = createRefTokenArray(sheetModel, newRefAddress).getFormula('op');
                        undoProps.ref = createRefTokenArray(sheetModel, refAddress).getFormula('op');
                    }
                }

                // if the reference address will be deleted, relocate the relative references in the formulas to the
                // reference address that remains valid during range transformation (top-left cell of restored ranges)
                if (restoredRefAddress.differs(refAddress)) {
                    updateDesc = _.extend({ relocate: { from: refAddress, to: restoredRefAddress } }, updateDesc);
                }
            }

            // calculate the new formula expressions for the validation model
            tokenArrayMap.forEach(function (tokenArray, propName) {
                tokenArray.resolveOperationProperty('op', updateDesc, propName, operProps, undoProps);
            });

            // ODF only: update the formula expression for the reference address for renamed sheets
            if (config.customRef && (updateDesc.type === UpdateMode.RENAME_SHEET)) {
                var tokenArray = createRefTokenArray(sheetModel, refAddress);
                tokenArray.resolveOperationProperty('op', updateDesc, 'ref', operProps, undoProps);
            }

            // generate the change operation with the new formula expressions
            if (!_.isEmpty(operProps)) {
                if (config.indexedOps) {
                    operProps.index = index;
                    generator.generateSheetOperation(Operations.CHANGE_VALIDATION, operProps);
                } else {
                    generator.generateRangesOperation(Operations.CHANGE_VALIDATION, targetRanges, operProps);
                }
            }

            // on which ranges to apply the change operations for undo
            var changedRanges = (!config.indexedOps && restoredRanges) ? restoredRanges : targetRanges;
            // whether to restore the original target ranges (auto-expansion for inserted rows/columns may change target ranges)
            var restoreTargetRanges = restoredRanges && !restoredRanges.equals(targetRanges, true);

            // ODF: generate insert/delete operations to restore missing ranges, or to delete surplus ranges
            if (restoreTargetRanges && !config.indexedOps) {

                // insert ranges that have been deleted while removing auto-expanded cells
                var missingRanges = targetRanges.difference(restoredRanges);
                if (!missingRanges.empty()) {
                    this.generateRestoreOperations(generator, index, { ranges: missingRanges });
                }

                // delete ranges that have been added while re-inserting deleted cells (auto-expansion)
                var surplusRanges = restoredRanges.difference(targetRanges);
                if (!surplusRanges.empty()) {
                    generator.generateRangesOperation(Operations.DELETE_VALIDATION, surplusRanges, null, { undo: true, prepend: true });
                    changedRanges = changedRanges.difference(surplusRanges);
                }
            }

            // generate the undo operation to restore the validation model if required
            if (restoreTargetRanges || !_.isEmpty(undoProps)) {
                if (config.indexedOps) { undoProps.index = index; }
                if (restoreTargetRanges || !config.indexedOps) {
                    generator.generateRangesOperation(Operations.CHANGE_VALIDATION, changedRanges, undoProps, { undo: true });
                } else {
                    generator.generateSheetOperation(Operations.CHANGE_VALIDATION, undoProps, { undo: true });
                }
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // initialize the target ranges
        this.setTargetRanges(ranges);

        // calculate the reference address
        if (config.customRef) {
            this.setRefFormula(Utils.getStringOption(initOptions, 'ref', ''));
        }

        // create the token arrays for the formula expressions in 'value1' and 'value2'
        tokenArrayMap.insert('value1', new TokenArray(sheetModel, FormulaType.VALIDATION));
        tokenArrayMap.insert('value2', new TokenArray(sheetModel, FormulaType.VALIDATION));

        // parse the source formulas after changing value attributes
        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // destroy all class members
        this.registerDestructor(function () {
            tokenArrayMap.clear();
            self = docModel = sheetModel = tokenArrayMap = null;
        });

    } }); // class ValidationModel

    // ValModelAndIndex =======================================================

    /**
     * Small helper structure carrying a validation model, and its current
     * index in the array container.
     *
     * @constructor
     *
     * @property {ValidationModel} model
     *  The validation model.
     *
     * @property {Number} index
     *  The array index of the validation model.
     */
    function ValModelAndIndex(valModel, index) {
        this.model = valModel;
        this.index = index;
    } // class ValModelAndIndex

    // class ValidationCollection =============================================

    /**
     * Stores data validation settings in a specific sheet. Data validation can
     * be used to restrict the values allowed to be entered in specific cell
     * ranges, and to show tooltips and error messages.
     *
     * Triggers the following events:
     *
     * - 'insert:validation'
     *      After a new validation rule has been inserted into this collection.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {ValidationModel} valModel
     *          The model of the new validation rule.
     * - 'delete:validation'
     *      Before an existing validation rule will be deleted from this
     *      collection. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {ValidationModel} valModel
     *          The model of the validation rule to be deleted.
     * - 'change:validation'
     *      After the attributes of a validation rule have been changed. Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {ValidationModel} valModel
     *          The model of the changed validation rule.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    var ValidationCollection = ModelObject.extend({ constructor: function (sheetModel) {

        // self reference
        var self = this;

        // the spreadsheet document model
        var docModel = sheetModel.getDocModel();

        // file-format dependent configuration
        var config = getConfigFlags(docModel);

        // all validation models as array (in operation/insertion order)
        var valModels = new ExtArray();

        // all model instances with validation settings, mapped by target ranges
        var targetRangeSet = new RangeSet();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Inserts the target ranges of the passed validation model into the
         * internal range set.
         *
         * @param {ValidationModel} valModel
         *  The validation model whose target ranges will be inserted into the
         *  range set.
         */
        function insertTargetRanges(valModel) {
            valModel.getTargetRanges().forEach(function (range) {
                range.valModel = valModel;
                targetRangeSet.insert(range, true);
            });
        }

        /**
         * Removes the target ranges of the passed validation model from the
         * internal range set.
         *
         * @param {ValidationModel} valModel
         *  The validation model whose target ranges will be removed from the
         *  range set.
         */
        function removeTargetRanges(valModel) {
            valModel.getTargetRanges().forEach(function (range) {
                targetRangeSet.remove(range);
            });
        }

        /**
         * Triggers a "delete:validation" event, removes the model from the
         * internal containers, and destroys the model.
         */
        function deleteValModel(valModel, index) {
            self.trigger('delete:validation', valModel);
            valModels.splice(index, 1);
            valModel.destroy();
        }

        /**
         * Returns the validation model addressed by the property "index" in
         * the passed document operation.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for validations,
         *  expected to contain an integer property "index".
         *
         * @returns {ValModelAndIndex}
         *  A descriptor for the validation model and its index.
         *
         * @throws {OperationError}
         *  If the operation does not contain a valid "index" property, or if
         *  the validation model with such an index does not exist.
         */
        function getModelDescByIndex(context) {

            // get the index from the operation
            var index = context.getInt('index');

            // check existence of the validation model
            var valModel = valModels[index];
            context.ensure(valModel, 'missing validation');

            return new ValModelAndIndex(valModel, index);
        }

        /**
         * Returns the validation models addressed by the property "ranges" in
         * the passed document operation.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for validations,
         *  expected to contain a range array property "ranges".
         *
         * @returns {Object}
         *  A result descriptor with the following properties:
         *  - {RangeArray} ranges
         *      The non-empty array of cell range addresses contained in the
         *      passed document operation.
         *  - {ExtArray<ValModelAndIndex>} models
         *      The validation models overlapping with the target ranges, and
         *      their array indexes. This array will be sorted by array index.
         *
         * @throws {OperationError}
         *  If the operation does not contain a valid "ranges" property.
         */
        function getModelDescsByRanges(context) {

            // the target rangesof the document operation
            var targetRanges = context.getRangeArray().merge();
            // the validation models to be returned
            var valModelSet = new ValueSet('getUid()');

            // process all target ranges contained in the operation
            targetRanges.forEach(function (targetRange) {
                // get all ranges from the set covered by a validation model
                targetRangeSet.findRanges(targetRange).forEach(function (range) {
                    valModelSet.insert(range.valModel);
                });
            });

            // construct the array of descriptors with model and array index
            var modelDescs = new ExtArray();
            // do not process the array of all models, if there is no overlapping model
            if (!valModelSet.empty()) {
                valModels.forEach(function (valModel, index) {
                    if (valModelSet.has(valModel)) {
                        modelDescs.push(new ValModelAndIndex(valModel, index));
                    }
                });
            }

            return { ranges: targetRanges, models: modelDescs };
        }

        /**
         * Reduces the target ranges of all existing validation models so that
         * they do not overlap with the ranges in the passed document operation
         * anymore. If a validation model is covered completely by the ranges
         * in the operation, it will be deleted from this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for validations,
         *  expected to contain a range array property "ranges".
         *
         * @returns {RangeArray}
         *  The target ranges contained in the document operation.
         *
         * @throws {OperationError}
         *  If the operation does not contain a valid "ranges" property, or if
         *  no validation models overlapping with the ranges exist.
         */
        function reduceModelsByRanges(context) {

            var modelsData = getModelDescsByRanges(context);
            modelsData.models.forEachReverse(function (modelDesc) {

                // the current validation model
                var valModel = modelDesc.model;
                // the current target ranges of the validation model
                var oldTargetRanges = valModel.getTargetRanges();
                // the reminaing target ranges for the validation model
                var newTargetRanges = oldTargetRanges.difference(modelsData.ranges);

                // delete the model
                if (newTargetRanges.empty()) {
                    removeTargetRanges(valModel);
                    deleteValModel(valModel, modelDesc.index);
                    return;
                }

                // shorten the target ranges
                removeTargetRanges(valModel);
                valModel.setTargetRanges(newTargetRanges);
                insertTargetRanges(valModel);
                self.trigger('change:validation', valModel);
            });

            // return the target ranges from the operation
            return modelsData.ranges;
        }

        /**
         * Recalculates the target ranges of all validation models, after cells
         * have been moved (including inserted/deleted columns or rows) in the
         * sheet.
         */
        function moveCellsHandler(event, moveDesc) {

            // simply fill a new target range set while transforming the ranges
            targetRangeSet = new RangeSet();

            // process all validation models in revered order to be able to delete array elements
            valModels.forEachReverse(function (valModel, index) {
                var targetRanges = transformTargetRanges(valModel.getTargetRanges(), [moveDesc]);
                if (targetRanges.empty()) {
                    deleteValModel(valModel, index);
                } else {
                    var changed = valModel.setTargetRanges(targetRanges);
                    insertTargetRanges(valModel);
                    if (changed) { self.trigger('change:validation', valModel); }
                }
            });
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { valModels: valModels };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * contents from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {ValidationCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            // clone the contents of the source collection
            cloneData.valModels.forEach(function (valModel) {
                valModel = valModel.clone(sheetModel);
                valModels.push(valModel);
                insertTargetRanges(valModel);
            });
        };

        /**
         * Callback handler for the document operation 'insertValidation'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertValidation' document operation.
         */
        this.applyInsertOperation = function (context) {

            // the target ranges for the data validation
            var targetRanges = context.getRangeArray();
            // the validation attribute set (mapped by attribute family)
            var attributeSet = { validation: context.operation };
            // the custom reference address (as formula expression)
            var refFormula = config.customRef ? context.getStr('ref') : null;

            // the array insertion index for the new validation model
            var modelIndex = (function () {

                // OOXML: validation model will be addressed by an index
                if (config.indexedOps) {
                    var index = context.getInt('index');
                    context.ensure((index >= 0) && (index <= valModels.length), 'invalid index');
                    return index;
                }

                // ODF: shorten target ranges of existing validation models
                reduceModelsByRanges(context);
                // append new model to array
                return valModels.length;
            }());

            // create and insert the new validation model
            var valModel = new ValidationModel(sheetModel, targetRanges, attributeSet, { ref: refFormula });
            valModels.splice(modelIndex, 0, valModel);
            insertTargetRanges(valModel);

            // notify event listeners
            this.trigger('insert:validation', valModel);
        };

        /**
         * Callback handler for the document operation 'deleteValidation'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteValidation' document operation.
         */
        this.applyDeleteOperation = function (context) {

            // OOXML: delete an existing validation model addressed by an index
            if (config.indexedOps) {

                // resolve the validation model addressed by the operation (throws on error)
                var modelDesc = getModelDescByIndex(context);
                var valModel = modelDesc.model;

                // destroy and remove the validation model
                removeTargetRanges(valModel);
                deleteValModel(valModel, modelDesc.index);
                return;
            }

            // ODF: shorten target ranges of existing validation models (throws on error)
            reduceModelsByRanges(context);
        };

        /**
         * Callback handler for the document operation 'changeValidation'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeValidation' document operation.
         */
        this.applyChangeOperation = function (context) {

            // the validation model to be changed
            var valModel = null;
            // whether the target ranges have been changed by the operation (OOXML only)
            var changedRanges = false;

            // OOXML: change an existing validation model addressed by an index
            if (config.indexedOps) {

                // resolve the validation model addressed by the operation (throws on error)
                var modelDesc = getModelDescByIndex(context);
                valModel = modelDesc.model;

                // resolve optional 'ranges' property (throws on error), change target ranges
                var newTargetRanges = context.getOptRangeArray();
                if (newTargetRanges) {
                    removeTargetRanges(valModel);
                    changedRanges = valModel.setTargetRanges(newTargetRanges);
                    insertTargetRanges(valModel);
                }

            } else {

                // ODF: range addresses in operation must refer exactly to an existing model
                var modelsData = getModelDescsByRanges(context);
                context.ensure(modelsData.models.length === 1, 'multiple validations covered');
                valModel = modelsData.models.first().model;

                // target ranges must be exactly equal
                context.ensure(valModel.getTargetRanges().equals(modelsData.ranges, true), 'target ranges do not match');
            }

            // resolve optional 'ref' property (throws on error), change reference address
            var changedRef = false;
            var refFormula = config.customRef ? context.getOptStr('ref') : null;
            if (refFormula) {
                valModel.setRefFormula(refFormula);
                changedRef = true;
            }

            // change validation attributes
            var changedAttrs = valModel.setAttributes({ validation: context.operation });

            // notify event listeners
            if (changedRanges || changedRef || changedAttrs) {
                this.trigger('change:validation', valModel);
            }
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the validation model that covers the specified cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {ValidationModel|Null}
         *  The validation for the specified cell. If the cell is not validated
         *  at all, returns null.
         */
        this.getModelByAddress = function (address) {
            var targetRanges = targetRangeSet.findByAddress(address);
            // result is an array with at most one element (target ranges cannot overlap)
            return targetRanges.empty() ? null : targetRanges.first().valModel;
        };

        /**
         * Creates an iterator that visits the models of the specified data
         * validations in this collection.
         *
         * @param {RangeArray|Range} [ranges]
         *  If specified, the iterator will visit data validations overlapping
         *  with that cell ranges. By default, all data validations in this
         *  collection will be visited.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the validation
         *  model (class ValidationModel) as value property.
         */
        this.createModelIterator = function (ranges) {

            // visit all data validations
            if (!ranges) { return valModels.iterator(); }

            // the validation models to be visited (prevent multiple hits)
            var valModelSet = new ValueSet('getUid()');

            // process all target ranges contained in the operation
            RangeArray.forEach(ranges, function (range) {
                // get all ranges from the set covered by a validation model
                targetRangeSet.findRanges(range).forEach(function (range) {
                    valModelSet.insert(range.valModel);
                });
            });

            // create an iterator that visits the models
            return valModelSet.iterator();
        };

        /**
         * Returns the validation settings for the specified cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Object|Null}
         *  The validation settings of the specified cell, in the following
         *  properties:
         *  - {RangeArray} ranges
         *      The addresses of all related cell ranges containing the same
         *      validation settings.
         *  - {Object} attributes
         *      All validation attributes.
         *  If the cell is not validated at all, returns null.
         */
        this.getValidationSettings = function (address) {
            var valModel = this.getModelByAddress(address);
            return valModel ? {
                ranges: valModel.getTargetRanges(),
                attributes: valModel.getMergedAttributeSet(true).validation
            } : null;
        };

        /**
         * Returns the list contents for a data validation of type 'list' or
         * 'source' at the specified cell address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Array<Object>}
         *  The contents of the cells, if the specified cell contains a data
         *  validation model of type 'list' or 'source'; otherwise an empty
         *  array. See method ValidationModel.queryListContents() for details.
         */
        this.queryListContents = function (address) {
            var valModel = this.getModelByAddress(address);
            return valModel ? valModel.queryListContents(address) : [];
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the data validations in this collection.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} updateDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on its 'type' property. See
         *  method TokenArray.resolveOperation() for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateUpdateFormulaOperations = function (generator, updateDesc) {
            return this.iterateArraySliced(valModels, function (valModel, index) {
                valModel.generateUpdateFormulaOperations(generator, index, updateDesc);
            }, 'ValidationCollection.generateFormulaOperations');
        };

        // initialization -----------------------------------------------------

        // additional processing and event handling after the document has been imported
        this.waitForImportSuccess(function (alreadyImported) {

            // refresh all formulas after import is finished (formulas may refer to sheets not yet imported)
            if (!alreadyImported) {
                valModels.forEach(function (valModel) {
                    valModel.refreshAfterImport();
                });
            }

            // update the sheet indexes after the sheet collection has been manipulated
            this.listenTo(docModel, 'transform:sheet', function (event, fromSheet, toSheet) {
                valModels.forEach(function (valModel) {
                    valModel.transformSheet(fromSheet, toSheet);
                });
            });

            // update target ranges after moving cells, or inserting/deleting columns or rows
            this.listenTo(sheetModel, 'move:cells', moveCellsHandler);

        }, this);

        // destroy all class members
        this.registerDestructor(function () {
            targetRangeSet.clear();
            valModels.destroyElements();
            self = docModel = sheetModel = valModels = targetRangeSet = null;
        });

    } }); // class ValidationCollection

    // exports ================================================================

    return ValidationCollection;

});
