/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/usedrangecollection', [
    'io.ox/office/baseframework/model/modelobject'
], function (ModelObject) {

    'use strict';

    // class UsedRangeCollection ==============================================

    /**
     * The base class for sheet collections based on cells that want to track a
     * bounding range of their used area.
     *
     * Triggers the following events:
     *  - 'change:usedrange'
     *      After the bounding range of this instance has changed. Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Range|Null} usedRange
     *          The address of the bounding range of this collection; or null,
     *          if this collection is empty.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    var UsedRangeCollection = ModelObject.extend({ constructor: function (sheetModel, usedRangeResolver, initOptions) {

        // the bounding range of this collection (or null for empty collection)
        var usedRange = null;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, sheetModel.getDocModel(), initOptions);

        // protected methods --------------------------------------------------

        /**
         * Recalculates the used range of this collection, and triggers a
         * 'change:usedrange' event if necessary.
         */
        this.updateUsedRange = function () {

            // the new bounding range
            var newUsedRange = usedRangeResolver.call(this);
            // whether the used range does not change
            var equalRanges = (!usedRange && !newUsedRange) || (usedRange && newUsedRange && usedRange.equals(newUsedRange));

            // trigger change event if the used range has changed, or if this collection was or becomes empty
            if (!equalRanges) {
                usedRange = newUsedRange;
                this.trigger('change:usedrange', usedRange);
            }
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether this collection is empty.
         *
         * @returns {Boolean}
         *  Whether this collection is empty.
         */
        this.isUsedRangeEmpty = function () {
            return !usedRange;
        };

        /**
         * Returns the address of the bounding range of the used area in this
         * collection.
         *
         * @returns {Range|Null}
         *  The address of the bounding range of the used area in this
         *  collection; or null, if this collection is empty.
         */
        this.getUsedRange = function () {
            return usedRange ? usedRange.clone() : null;
        };

        /**
         * Returns the address of the bounding range of the used area in this
         * collection, shrinked to its visible parts. See method
         * SheetModel.shrinkRangeToVisible() for details.
         *
         * @returns {Range|Null}
         *  The address of the visible part of the bounding range of the used
         *  area in this collection; or null, if this collection is empty, or
         *  its used are is completely hidden.
         */
        this.getVisibleUsedRange = function () {
            return usedRange ? sheetModel.shrinkRangeToVisible(usedRange) : null;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            sheetModel = usedRangeResolver = null;
        });

    } }); // class UsedRangeCollection

    // exports ================================================================

    return UsedRangeCollection;

});
