/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Peter Seliger <peter.seliger@open-xchange.com>
 */

define('io.ox/office/presentation/view/controls', [
    'io.ox/office/tk/utils',
    'io.ox/office/textframework/view/controls',
    'io.ox/office/presentation/view/control/layoutslidepicker',
    'gettext!io.ox/office/presentation/main'
], function (Utils, Controls, LayoutSlidePicker, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var CheckBox = Controls.CheckBox;
    var CompoundButton = Controls.CompoundButton;

    // static class PresentationControls ======================================

    /**
     * Provides different classes for GUI form controls for presentation
     * documents collected within a single map for convenience.
     */
    var PresentationControls = _.extend({}, Controls);

    // class InsertLayoutSlidePicker ==========================================

    PresentationControls.InsertLayoutSlidePicker = LayoutSlidePicker.extend({ constructor: function (docView, initOptions) {

        LayoutSlidePicker.call(this, docView, Utils.extendOptions({
            label: gt('Slide'),
            tooltip: gt('Insert new slide'),
            icon: 'fa-plus',
            splitValue: 'blank_slide',
            isOdfLayout: docView.getApp().isODF()
        }, initOptions));

    } }); // class InsertLayoutSlidePicker

    // class ChangeLayoutSlidePicker ==========================================

    PresentationControls.ChangeLayoutSlidePicker = LayoutSlidePicker.extend({ constructor: function (docView, initOptions) {

        LayoutSlidePicker.call(this, docView, Utils.extendOptions({
            label: gt('Layout'),
            tooltip: gt('Change layout'),
            icon: 'fa-columns',
            updateCaptionMode: 'none',
            boundingBox: docView.getApp().getRootNode(),
            isOdfLayout: docView.getApp().isODF()
        }, initOptions));

    } }); // class ChangeLayoutSlidePicker

    // class SwitchSlideEditModeButton ========================================

    //#. start to edit master/layout slides of a presentation
    var EDIT_MASTER_LABEL = gt('Edit master');
    //#. start to (or return to) edit normal slides of a presentation
    var EDIT_NORMAL_LABEL = gt('Edit normal');

    PresentationControls.SwitchSlideEditModeButton = Button.extend({ constructor: function (docView, initOptions) {

        var self = this;

        // base constructor ---------------------------------------------------

        Button.call(this, docView, Utils.extendOptions({
            label: EDIT_MASTER_LABEL,
            icon: 'docs-slide-master',
            tooltip: gt('Switch between editing master/layout slides and normal slides'),
            smallerVersion: { hideLabel: true }
        }, initOptions));

        // initialization -----------------------------------------------------

        this.listenTo(docView.getDocModel(), 'change:activeView:after', function (event, options) {
            self.setLabel(options.isMasterView ? EDIT_NORMAL_LABEL : EDIT_MASTER_LABEL);
        });

    } }); // class SwitchSlideEditModeButton

    // class PlaceholderTypePicker ============================================

    /**
     * Control to pick placeholder drawing from dropdown menu
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    PresentationControls.PlaceholderTypePicker = CompoundButton.extend({ constructor: function (docView, options) {

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, Utils.extendOptions(options, {
            label: /*#. button to insert placeholder objects */ gt('Placeholder'),
            tooltip: gt('Insert placeholder'),
            icon: 'docs-placeholder',
            smallerVersion: { hideLabel: true }
        }));

        // initialization -----------------------------------------------------

        if (docView.getApp().isODF()) {

            this.addGroup('placeholder/odfdatetime', new CheckBox(docView, {
                label: /*#. check box label: show/hide date/time placeholder object */ gt('Date/Time object'),
                tooltip: /*#. check box tooltip: show/hide date/time placeholder object */ gt('Show or hide date and time placeholder object'),
                boxed: true
            }));

            this.addGroup('placeholder/odffooter', new CheckBox(docView, {
                label: /*#. check box label: show/hide footer placeholder object */ gt('Footer object'),
                tooltip: /*#. check box tooltip: show/hide footer placeholder object */ gt('Show or hide footer placeholder objects'),
                boxed: true
            }));

            this.addGroup('placeholder/odfslidenum', new CheckBox(docView, {
                label: /*#. check box label: show/hide slide number placeholder object */ gt('Slide number object'),
                tooltip: /*#. check box tooltip: show/hide slide number placeholder object */ gt('Show or hide slide number object'),
                boxed: true
            }));

        } else {

            this.addGroup('placeholder/title', new CheckBox(docView, {
                label: /*#. check box label: show/hide the title placeholder object */ gt('Title'),
                tooltip: /*#. check box tooltip: show/hide title placeholder object  */ gt('Show or hide title placeholder object'),
                boxed: true
            }));

            this.addGroup('placeholder/body', new CheckBox(docView, {
                label: /*#. check box label: show/hide text body placeholder object  */ gt('Text body'),
                tooltip: /*#. check box tooltip: show/hide text body placeholder object */ gt('Show or hide text body placeholder object'),
                boxed: true
            }));

            this.addGroup('placeholder/footers', new CheckBox(docView, {
                label: /*#. check box label: show/hide footer placeholder objects */ gt('Footer objects'),
                tooltip: /*#. check box tooltip: show/hide footer placeholder objects */ gt('Show or hide footer placeholder objects'),
                boxed: true
            }));

            this.addSeparator()
                .addGroup('placeholder/insertcustom', new Button(docView, { icon: 'docs-placeholder-content', value: 'content', label: /*#. insert content placeholder */ gt('Content') }))
                .addGroup('placeholder/insertcustom', new Button(docView, { icon: 'docs-placeholder-content-vertical', value: 'contentv', label: /*#. insert vertical content placeholder */ gt('Content vertical') }))
                .addGroup('placeholder/insertcustom', new Button(docView, { icon: 'docs-placeholder-text', value: 'text', label: /*#. insert text placeholder */ gt('Text') }))
                .addGroup('placeholder/insertcustom', new Button(docView, { icon: 'docs-placeholder-text-vertical', value: 'textv', label: /*#. insert vertical text placeholder */ gt('Text vertical') }))
                .addGroup('placeholder/insertcustom', new Button(docView, { icon: 'docs-placeholder-picture', value: 'picture', label: /*#. insert picture placeholder */ gt('Image') }))
                .addGroup('placeholder/insertcustom', new Button(docView, { icon: 'docs-placeholder-table', value: 'table', label: /*#. insert table placeholder */ gt('Table') }));

        }

    } }); // class PlaceholderTypePicker

    // exports ================================================================

    return PresentationControls;

});
