/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/edittoolpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/toolpane',
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/editframework/utils/editconfig',
    'io.ox/office/editframework/view/editlabels',
    'io.ox/office/editframework/view/editcontrols',
    'gettext!io.ox/office/editframework/main'
], function (Utils, ToolPane, ToolBar, Config, Labels, Controls, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var CompoundButton = Controls.CompoundButton;
    var CompoundSplitButton = Controls.CompoundSplitButton;

    // class EditToolPane =====================================================

    /**
     * Represents the main tool pane in OX Document applications shown below
     * the top-level view pane, containing the currently active tool bar, and
     * the 'View' drop-down menu.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {EditView} docView
     *  The document view instance containing this tool pane.
     *
     * @param {ToolBarTabCollection} toolBarTabs
     *  The collection of all registered tool bar tabs.
     */
    function EditToolPane(docView, toolBarTabs, initOptions) {

        var // self reference
            self = this,

            // the tool bars, mapped by tab identifier
            toolBarsMap = {},

            // the wrapper nodes for the center tool bar area
            centerWrapperNode = null,
            centerAreaNode = null,

            // state of "paneLayoutHandler" (to prevent multiple calls)
            reLayout = false,
            // current pane width after last layout
            paneWidth = 0,
            // possibility to force invoking the paneLayoutHandler
            forceLayoutHandler = false,

            // the new menu including the tab buttons to activate the different edit tool bars, as drop-down list
            flatTabView = null,
            // new document close button for flat-pane
            closeBtn = null,
            // new search button for the new flat-pane-menu
            searchBtn = null,

            // reference to the topPane
            topPane = docView.getTopPane(),
            // reference to the View menu group
            viewMenuGroup = topPane.getViewMenuGroup(),
            // reference to the view menu
            viewMenu = viewMenuGroup.getMenu();

        // base constructor ---------------------------------------------------

        ToolPane.call(this, docView, Utils.extendOptions({
            position: 'top',
            classes: 'edit-pane standard-design',
            cursorNavigate: true
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Shows or hides the tool bars associated to the specified tab
         * identifier, according to the own visibility of the tool bars if
         * bound to a controller item.
         */
        function toggleToolBars(tabId, state) {
            if (_.isString(tabId)) {
                _.each(toolBarsMap[tabId], function (toolBarInfo) {
                    if (_.isObject(toolBarInfo.shrinkToolBar)) {
                        toolBarInfo.toolBar.toggle(state && toolBarInfo.visible);

                        if (toolBarInfo.shrunken && toolBarInfo.shrinkMenu.hasVisibleGroups()) {
                            toolBarInfo.shrinkToolBar.toggle(state && toolBarInfo.visible);
                        } else {
                            toolBarInfo.shrinkToolBar.hide();
                        }

                    } else {
                        toolBarInfo.toolBar.toggle(state && toolBarInfo.visible);
                    }
                });
            }
        }

        /**
         * Updates the layout of tool bars according to available space in the
         * tool pane.
         */
        var paneLayoutHandler = this.createDebouncedMethod('EditToolPane.paneLayoutHandler', null, function () {

            // check if another process/trigger is running this already.
            // cancel in this case, to prevent flashing controls
            if (reLayout) { return; }
            reLayout = true;

            // pane width
            var newPaneWidth = centerWrapperNode.width();
            // tool bars width
            var toolbarsWidth = centerAreaNode.outerWidth();
            // overflow size of the center tool bar area
            var overflowWidth = toolbarsWidth - newPaneWidth;
            // all tool bars of the active tab
            var toolbars = _.sortBy(toolBarsMap[toolBarTabs.getActiveTabId()], 'priority');
            // shrinkable tool bars
            var shrinkableToolbars = toolbars.filter(function (bar) { return bar.shrinkToolBar; });
            // get the number of shrunken tool bars
            var countShrunken = toolbars.filter(function (bar) { return bar.shrunken; }).length;
            // hideable tool bars
            var hideableToolbars = toolbars.filter(function (bar) { return bar.hideable; });
            // get the number of hidden tool bars
            var countHidden = toolbars.filter(function (bar) { return bar.hidden; }).length;
            // stop acting
            var forceStop = false;

            // if the width hasn't changed and no control is out of the visible range
            // and no force is requested - get out here. Do not repaint the panes
            if (!forceLayoutHandler && (paneWidth === newPaneWidth) && (overflowWidth < 0)) {
                reLayout = false;
                return;
            }
            paneWidth = newPaneWidth;

            // catch new widths
            function resetDimensions() {
                // go on, if the node isn't visible
                if (centerAreaNode.outerWidth() === 0) { return false; }

                // pane width
                newPaneWidth = centerWrapperNode.width();
                // tool bars width
                toolbarsWidth = centerAreaNode.outerWidth();
                // overflow size of the center tool bar area
                return toolbarsWidth - newPaneWidth;
            }

            /**
             * add/remove given class to/from toolbar
             *
             * @param {String} addClass
             *  class which should be add/remove
             *
             * @param {Boolean} visible
             *  whether add or remove the classes
             */
            function toolbarClass(addClass, visible) {
                var node = docView.getToolPane().getNode();
                node.toggleClass(addClass, visible);
            }

            /**
             * add/remove class "vertical-list" to/from pop-up
             *
             * @param {Integer} i
             *  which of the shrinkable tool bars
             *
             * @param {Boolean} state
             *  whether add or remove the classes
             */
            function makeVerticalList(i, state) {
                var popupWrapper = shrinkableToolbars[i].shrinkMenu.getNode();
                popupWrapper.toggleClass('vertical-list drop-down', state);
            }

            /**
             * moves the toolbar from pane to drop-down or reversed
             *
             * @param {Integer} i
             *  which of the shrinkable tool bars
             *
             * @param {String} type
             *  whether shrink or expand = move to default pane or to small drop-down
             */
            function moveToolBar(i, type) {

                if (type === 'shrink' && self.hasViewComponent(shrinkableToolbars[i].toolBar)) {
                    // remove component from pane
                    self.removeViewComponent(shrinkableToolbars[i].toolBar);
                    // add component to shrunken (drop-down) menu
                    shrinkableToolbars[i].shrinkMenu.replaceComponent(shrinkableToolbars[i].toolBar);

                    // go through all groups
                    shrinkableToolbars[i].toolBar.iterateGroups(function (group) {
                        // and, if exists a menu
                        if (_.isFunction(group.getMenu)) {
                            // add the components to the focusable nodes (to prevent closing the pop-up)
                            shrinkableToolbars[i].shrinkMenu.registerFocusableNodes(group.getMenu().getContentNode());

                            // for US 102078806: gives the moved group a reference the root from the
                            // menu tree (the root anchor from its parent) and a reference to its parent menu
                            if (Utils.SMALL_DEVICE) {
                                group.setParentAnchorNode(shrinkableToolbars[i].shrinkToolBar.getNode());
                                group.setParentMenu(shrinkableToolbars[i].shrinkMenu);
                            }
                        }
                        // show controls which should only be shown in drop-down version
                        group.toggleDropDownMode(true);
                    });

                    // maximize all groups
                    shrinkableToolbars[i].toolBar.iterateGroups(function (group) {
                        group.deactivateSmallVersion();
                    });

                    // add class for vertical design
                    makeVerticalList(i, true);
                    return true;

                } else if (type === 'unshrink' && !self.hasViewComponent(shrinkableToolbars[i].toolBar)) {

                    // remove component from shrunken drop-down menu
                    var component = shrinkableToolbars[i].shrinkMenu.releaseComponent();
                    // add view component to default pane
                    self.addViewComponent(component, { insertBefore: shrinkableToolbars[i].shrinkToolBar });

                    // minimize all groups
                    if (shrinkableToolbars[i].shrunken) {
                        shrinkableToolbars[i].toolBar.iterateGroups(function (group) {

                            // for US 102078806: set root anchor and parent menu to null when expanded
                            // important for landscape/portrait mode on mobile
                            if (Utils.SMALL_DEVICE) {
                            // and, if exists a menu
                                if (_.isFunction(group.getMenu)) {
                                    group.setParentAnchorNode(null);
                                    group.setParentMenu(null);
                                }
                            }

                            group.activateSmallVersion();
                            // hide controls which should only be shown in drop-down version
                            group.toggleDropDownMode(false);
                        });
                    }

                    // remove vertical-design
                    makeVerticalList(i, false);
                    return true;

                } else {
                    return false;
                }
            }

            docView.lockPaneLayout(function () {
                // returns whether the toolbar(s) are too big for the pane
                function tooBig() {
                    overflowWidth = resetDimensions();
                    if (overflowWidth > 0) {
                        // reset force-option to stop invoking paneLayoutHandler here
                        forceLayoutHandler = false;
                        return true;
                    }
                    return false;
                }
                // returns whether the toolbar(s) fit in the pane
                function fitIn() {
                    overflowWidth = resetDimensions();
                    return (overflowWidth < 0);
                }
                // returns the sum of all shrunken tool bars of this pane
                function getCountShrunken() {
                    countShrunken = _.filter(toolBarsMap[toolBarTabs.getActiveTabId()], function (bar) { return bar.shrunken; }).length;
                    return countShrunken;
                }
                // returns the sum of all hidden tool bars of this pane
                function getCountHidden() {
                    countHidden = _.filter(toolBarsMap[toolBarTabs.getActiveTabId()], function (bar) { return bar.hidden; }).length;
                    return countHidden;
                }

                // Controls not fit in the pane
                if (overflowWidth > 0) {
                    // shrink the space between the controls
                    toolbarClass('small-distance', true);

                    // activate small versions of groups
                    if (tooBig()) {
                        Utils.iterateArray(toolbars, function (toolbar) {
                            // only activate the small version of the groups, if their were not shrunken
                            if (!toolbar.shrunken) {
                                toolbar.toolBar.iterateGroups(function (group) {
                                    group.activateSmallVersion();
                                });
                                if (fitIn()) { return Utils.BREAK; }
                            }
                        }, { reverse: true });

                        // shrink complete toolbar to dropdown-menu
                        if (tooBig()) {
                            // iterate over all toolbars (reversed)
                            Utils.iterateArray(shrinkableToolbars, function (toolbar, i) {
                                if (toolbar.shrunken) { return; }

                                if ((i === shrinkableToolbars.length - 1) || ((i < shrinkableToolbars.length - 1) && shrinkableToolbars[i + 1].shrunken)) {
                                    if (moveToolBar(i, 'shrink')) {
                                        toolbar.shrunken = true;
                                        if (fitIn()) { return Utils.BREAK; }
                                    }
                                }
                            }, { reverse: true });

                            // hide toolbars (last step to fit in the given area)
                            if (tooBig()) {
                                // iterate backwards through all hideable toolbars
                                Utils.iterateArray(hideableToolbars, function (toolbar, i) {
                                    if (toolbar.hidden) { return; }

                                    if ((i === hideableToolbars.length - 1) || ((i < hideableToolbars.length - 1) && hideableToolbars[i + 1].hidden)) {
                                        (toolbar.shrinkToolBar || toolbar.toolBar).getNode().hide();
                                        toolbar.hidden = true;
                                        if (fitIn()) { return Utils.BREAK; }
                                    }
                                }, { reverse: true });
                            }
                        }
                    }

                // Controls have free space in the pane
                } else {

                    // show hidden toolbars
                    Utils.iterateArray(hideableToolbars, function (toolbar, i) {
                        if (!toolbar.hidden) { return; }

                        if ((i === 0) || ((i > 0) && !hideableToolbars[i - 1].hidden)) {
                            var toolbarNode = (toolbar.shrinkToolBar || toolbar.toolBar).getNode().css('display', '');
                            toolbar.hidden = false;

                            // if it was too much, revert to last state
                            if (tooBig()) {
                                toolbarNode.hide();
                                toolbar.hidden = true;
                                forceStop = true;
                                return Utils.BREAK;
                            }
                        }
                    });

                    if (fitIn() && forceStop === false && getCountHidden() === 0) {

                        // unshrink toolbars
                        Utils.iterateArray(shrinkableToolbars, function (toolbar, i) {
                            if (!toolbar.shrunken) { return; }

                            // toolbar have to be shrunken
                            // and the toolbar before have to be expanded (or do not exists "===0")
                            if ((i === 0) || ((i > 0) && !shrinkableToolbars[i - 1].shrunken)) {
                                if (moveToolBar(i, 'unshrink')) {
                                    toolbar.shrunken = false;
                                    overflowWidth = resetDimensions();
                                    forceStop = true;

                                    // if "overflow" minus "shrunken-toolbar" positive
                                    // reshrink the toolbar
                                    if ((overflowWidth - toolbar.shrinkToolBar.getNode().outerWidth()) > 0) {
                                        if (moveToolBar(i, 'shrink')) {
                                            toolbar.shrunken = true;
                                        }
                                        overflowWidth = resetDimensions();
                                        return Utils.BREAK;
                                    }
                                }
                            }
                        });

                        // maximize groups
                        if (fitIn() && forceStop === false && getCountShrunken() === 0) {
                            // iterate through all groups of all toolbars
                            Utils.iterateArray(toolbars, function (toolbar) {
                                toolbar.toolBar.iterateGroups(function (group) {
                                    group.deactivateSmallVersion();
                                });

                                // if it was too much, revert to last state
                                if (tooBig()) {
                                    toolbar.toolBar.iterateGroups(function (group) {
                                        group.activateSmallVersion();
                                    });
                                    forceStop = true;
                                    return Utils.BREAK;
                                }
                            });
                        }
                    }
                }
            });

            // check visibility of toolbars
            toggleToolBars(toolBarTabs.getActiveTabId(), true);

            // reset status (actually not in layout-progess)
            reLayout = false;
        }, { delay: 500 });

        /**
         *  Checks the height of the window-node and (de-)activates the flat-pane
         */
        function setPaneHeight() {

            var isStandAlone = docView.getApp().isStandalone();

            self.getNode().toggleClass('flat-pane', true);  // add special flat-pane css-class

            viewMenu.iterateGroups(function (group) {

                // add sub-menus to focusable nodes, so that the parent-menu don't close automatically
                if (_.isFunction(group.getMenu)) {
                    flatTabView.getMenu().registerFocusableNodes(group.getMenu().getNode());
                }

                // check whether something should happen with this view-menu-group if its moved to the flatPane
                if (Utils.getStringOption(group.getOptions(), 'flatPane') === 'hide') {
                    group.hide();
                }
            });

            topPane.hide();                                 // hide top pane (tab bar)
            flatTabView.show();                             // show new tab menu (in leading container of tool pane)
            closeBtn.toggle(!isStandAlone);                 // show new close button (in trailing container of tool pane) if standAloneMode is off
            self.show();                                    // make tool pane visible (maybe the user has made the pane invisible)
        }

        function tabCreateHandler(event, tabId, options) {

            var tabButton = new Button(docView, Utils.extendOptions(options, {
                label: Utils.getStringOption(options, 'label', ''),
                value: tabId,
                highlight: function (value) { return value === tabId; }
            }));

            flatTabView.addGroup('view/toolbars/tab', tabButton, {
                visibleKey: options.visibleKey,
                insertBefore: searchBtn.getNode(),
                saveSeparator: true
            });
        }

        /**
         *  Sets the correct label to the dynamic filled "drawing-tab"
         */
        function tabLabelHandler(event, tabId, label) {
            flatTabView.getMenu().iterateGroups(function (group) {
                var options = group.getOptions();
                if (options.value === tabId) {
                    group.setLabel(label);
                    return Utils.BREAK;
                }
            });
        }

        /**
         * Updates the visibility of the tool bars according to the active tab.
         */
        function tabActivateHandler(event, newTabId, oldTabId) {
            docView.lockPaneLayout(function () {
                toggleToolBars(oldTabId, false);
                if (_.isString(newTabId)) { self.show(); }
                toggleToolBars(newTabId, true);
                // the changing of the tab invokes paneLayoutHandler by force
                forceLayoutHandler = true;
                paneLayoutHandler();
            });
        }

        /**
         * Updates the visibility of the active tool bars that are bound to the
         * enabled state of a controller item.
         */
        function controllerChangeHandler(changedItems) {
            _.each(toolBarsMap, function (toolBarInfos) {
                _.each(toolBarInfos, function (toolBarInfo) {
                    var key = toolBarInfo.visibleKey,
                        enabled = (_.isString(key) && (key in changedItems)) ? changedItems[key].enabled : null;
                    if (_.isBoolean(enabled)) {
                        toolBarInfo.visible = enabled;
                    }
                });
            });
            toggleToolBars(toolBarTabs.getActiveTabId(), true);
        }

        // public methods -----------------------------------------------------

        /**
         * Inserts a new tool bar into this tool pane.
         *
         * @param {String} tabId
         *  The identifier of the tab button the tool bar will be associated
         *  with.
         *
         * @param {ToolBar} toolBar
         *  The tool bar to be inserted into this tool pane. The tool pane
         *  takes ownership of the tool bar!
         *
         * @param {Object} [options]
         *  Optional parameters. All of the following options may be passed to
         *  the constructor of the tool bar and will  be used from there in
         *  this case. Options passed with this parameter will override the
         *  tool bar options though:
         *  - {String} [options.visibleKey]
         *      The key of the controller item that controls the visibility of
         *      the tool bar while it is active according to the specified tab
         *      identifier. The visibility will be bound to the 'enabled' state
         *      of the respective controller item. If omitted, the tool bar
         *      will always be visible.
         *  - {Number} [options.priority]
         *      The priority of this tool bar among its siblings in this tool
         *      pane. Used when shrinking the contents of this tool pane due to
         *      missing space. The higher the value of this option, the lower
         *      is the priority of the tool bar. Tool bars with lower priority
         *      will be shrunken earlier.
         *  - {Boolean} [options.hideable=false]
         *      If set to true, the entire tool bar will be hidden if there is
         *      not enough space in this tool pane (and even not enough space
         *      to shrink the tool bar into a drop-down menu).
         *  - {Object} [options.shrinkToMenu]
         *      An object with all settings needed to prepare a drop-down menu
         *      button. If there is not enough free space for the new toolbar,
         *      all its controls will be moved into the drop-down menu, and it
         *      will be shown instead of the toolbar. The value of this option
         *      is an object that contains the initialization options to be
         *      passed to the constructor of the class CompoundButton or
         *      CompountSplitButton. If this object contains the string option
         *      'splitKey', an instance of the class CompountSplitButton will
         *      be created, and the split button will be bound to the specified
         *      controller key.
         *
         * @returns {ToolBar}
         *  The tool bar instance passed to this method, for convenience.
         */
        this.addToolBar = function (tabId, toolBar, options) {

            // set the tool bar to hidden state initially
            this.addViewComponent(toolBar.hide());

            // use the options passed to the tool bar, override with the options passed to this method
            options = _.extend({}, toolBar.getOptions(), options);

            // initialize the runtime settings of the toolbar
            var toolBarInfo = {
                toolBar: toolBar,                                                   // normal tool bar
                shrinkToolBar: null,                                                // shrunken tool bar with a drop-down menu
                shrinkMenu: null,                                                   // the drop-down menu from the shrunken tool bar
                visibleKey: Utils.getStringOption(options, 'visibleKey'),           // toolbar visible at
                priority: Utils.getIntegerOption(options, 'priority', null),        // toolbar priority (1: high, 10: low)
                hideable: Utils.getBooleanOption(options, 'hideable', null),        // is toolbar hideable
                visible: true,                                                      // toolbar visibility
                hidden: false,                                                      // is toolbar hidden
                shrunken: false                                                     // is toolbar shrunken
            };

            // move toolbar contents to a drop-down menu if there is not enough space
            var shrinkToMenu = Utils.getObjectOption(options, 'shrinkToMenu', null);
            if (shrinkToMenu) {

                // create the additional tool bar for the drop-down button
                toolBarInfo.shrinkToolBar = new ToolBar(docView, options).hide();
                self.addViewComponent(toolBarInfo.shrinkToolBar);

                // controller key for a split button
                var splitKey = Utils.getStringOption(shrinkToMenu, 'splitKey', null);
                // create the compound button with the drop-down menu
                var menuGroup = splitKey ? new CompoundSplitButton(docView, shrinkToMenu) : new CompoundButton(docView, shrinkToMenu);
                toolBarInfo.shrinkMenu = menuGroup.getMenu();
                // insert the drop-down button into the tool bar
                toolBarInfo.shrinkToolBar.addGroup(splitKey, menuGroup);
            }

            // insert the tool bar settings
            (toolBarsMap[tabId] || (toolBarsMap[tabId] = [])).push(toolBarInfo);

            return toolBar;
        };

        /**
         *  return the flat-pane-menu for small device gui
         * @returns {CompoundButton}
         */
        this.getFlatTabView = function () {
            return flatTabView;
        };

        // initialization -----------------------------------------------------

        // new menu for flat-pane (in leading-container of toolbar)
        flatTabView = new CompoundButton(docView, { icon: 'fa-bars', caret: false, anchorPadding: -3 });

        // add static controls to new flat-pane-menu
        flatTabView
            .addGroup('view/searchpane', searchBtn = new Button(docView, { icon: 'fa-search', label: gt('Search'), tooltip: gt('Toggle search'), toggle: true }), { visibleKey: 'app/valid' })
            .addSeparator()
            .addGroup('document/undo', new Button(docView, { icon: 'docs-undo', tooltip: gt('Revert last operation') }), { visibleKey: 'document/editable', inline: true })
            .addGroup('document/redo', new Button(docView, { icon: 'docs-redo', tooltip: gt('Restore last operation') }), { visibleKey: 'document/editable', inline: true });

        // create the tool bar containing the tab buttons and add it to the leading-container of the toolBar
        this.addViewComponent(new ToolBar(docView)
            .addGroup(null, flatTabView.hide()),
        { targetArea: 'leading' });

        // create the toolbar containing the close button and add it to the trailing-container of the toolBar
        this.addViewComponent(new ToolBar(docView)
            .addGroup('app/quit', closeBtn = new Button(docView, Labels.QUIT_BUTTON_OPTIONS).hide()),
        { targetArea: 'trailing' });

        // update layout of tool bars according to available space
        this.on('pane:layout', paneLayoutHandler);

        // listen to tab events, update visibility of the tool bars
        this.listenTo(toolBarTabs, 'tab:create', tabCreateHandler);
        this.listenTo(toolBarTabs, 'tab:label', tabLabelHandler);
        this.listenTo(toolBarTabs, 'tab:activate', tabActivateHandler);

        // handle changed controller items
        this.handleChangedControllerItems(controllerChangeHandler);

        // bug 39071: reactivate last tool bar tab, if the tool pane becomes visible
        this.on('pane:show', function () {
            toolBarTabs.restoreActiveTab();
        });

        // get the container nodes for the center tool bar area
        centerWrapperNode = this.getAreaWrapperNode('center');
        centerAreaNode = centerWrapperNode.find('>.area-container');

        // decide whether to show the combined tool/tab bar
        if (Config.COMBINED_TOOL_PANES) {
            setPaneHeight();
        }

        docView.getApp().registerWindowResizeHandler(function () {
            // only remove the 'small-distance' marker, if the width has changed
            if (centerWrapperNode.width() !== paneWidth) {
                docView.getToolPane().getNode().removeClass('small-distance');
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = toolBarTabs = initOptions = toolBarsMap = null;
            centerWrapperNode = centerAreaNode = null;
        });

    } // class ToolPane

    // exports ================================================================

    // derive this class from class ToolPane
    return ToolPane.extend({ constructor: EditToolPane });

});
