/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.impl;

import static com.openexchange.imageconverter.impl.ImageConverterUtils.LOG;
import java.awt.Dimension;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.imageconverter.api.ElementLock;
import com.openexchange.imageconverter.api.ElementLocker;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.imageconverter.api.IFileItemWriteAccess;
import com.openexchange.imageconverter.api.IMetadata;
import com.openexchange.imageconverter.api.IMetadataReader;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.api.ImageFormat;
import com.openexchange.imageconverter.api.MetadataException;
import com.openexchange.imageconverter.api.MetadataImage;
import com.openexchange.imagetransformation.ScaleType;

/**
 * {@link ImageConverterRunnableImpl}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
class ImageConverterRunnable implements Runnable, Closeable {

    private static File DEFAULT_TEMP_DIR = new File("/tmp");

    /**
     * Initializes a new {@link ImageConverterRunnableImpl}.
     */
    ImageConverterRunnable(@NonNull ImageConverterQueue queue,
        @NonNull final IFileItemService fileItemService,
        @NonNull final IMetadataReader metadataReader,
        @NonNull final ImageProcessor imageProcessor,
        @NonNull final List<ImageFormat> imageFormats,
        @NonNull final String imageKey,
        @NonNull final InputStream srcImageStm,
        @Nullable final String context) throws ImageConverterException {

        super();

        m_queue =  queue;
        m_fileItemService = fileItemService;
        m_metadataReader = metadataReader;
        m_imageProcessor = imageProcessor;
        m_imageKey = imageKey;
        m_context = context;
        m_inputFile = implCreateImageInputFile(srcImageStm);

        try {
            if ((null == m_inputFile) || (m_inputFile.length() < 1)) {
                throw new ImageConverterException("IC Runnable creation of conversion Runnable not possible due to missing or empty input file: " + imageKey);
            }

            m_imageMetadata = implCreateKeyMetadata(m_imageKey, m_inputFile, m_context);

            if (null == m_imageMetadata) {
                throw new ImageConverterException("IC Runnable error while trying to read image metadata from input file: " + imageKey);
            }

            m_imageFormats = implCreateImageFormatList(m_imageMetadata, imageFormats);

            implFillProcessMap();
        } catch (MetadataException e) {
            throw new ImageConverterException("IC Runnable error while creating metadata entry for input file", e);
        } finally {
            // ensure correct cleanup in error case (no input file or metadata available)
            if ((null == m_inputFile) || (null == m_imageMetadata)) {
                close();
            }
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see java.io.Closeable#close()
     */
    @Override
    public void close() {
        FileUtils.deleteQuietly(m_inputFile);
        m_inputFile = null;

        m_processFormatMap.clear();
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {
        ImageFormat curTargetFormat = null;
        boolean finished = m_queue.isTerminated();

        ElementLocker.lock(m_imageKey, ElementLock.LockMode.PROCESSING);

        m_queue.startProcessing(m_imageKey, this);

        try {
            for (final Iterator<ImageFormat> iter = m_processFormatMap.keySet().iterator(); iter.hasNext() && !finished;) {
                curTargetFormat = iter.next();

                if (LOG.isTraceEnabled()) {
                    LOG.trace(new StringBuilder("IC Runnable worker thread asynchronously processing ").
                        append(m_imageKey).append(": ").
                        append(curTargetFormat.getFormatString()).toString());
                }

                processMapKey(curTargetFormat, true);
                iter.remove();
            }

            finished = !m_queue.isTerminated();
        } catch (Exception e) {
            LOG.error(new StringBuilder("IC Runnable error while asynchronously processing ").
                append(m_imageKey).append(": ").
                append((null != curTargetFormat) ? curTargetFormat.getFormatString() :"null").toString(), e);
        } finally {
            m_queue.closeRunnable(m_imageKey, this, !finished);
            m_queue.endProcessing(m_imageKey);

            ElementLocker.unlock(m_imageKey, true);

            ImageConverterUtils.IC_MONITOR.incrementProcessedKeys(m_processingTime);
        }
    }

    /**
     *
     */
    protected void initMeasurementTime() {
        m_measurementStartTimeMillis = System.currentTimeMillis();
    }

    /**
     * @return
     */
    protected long getCurrentMeasurementTimeMillis() {
        return (System.currentTimeMillis() - m_measurementStartTimeMillis);
    }

    /**
     *
     */
    protected IMetadata getImageMetadata() {
        return m_imageMetadata;
    }

    /**
     * @param targetFormat
     * @return
     * @throws ImageConverterException
     */
    protected void processMapKey(@NonNull final ImageFormat targetFormat, final boolean isAsync) throws ImageConverterException {
        ImageFormat processingFormat = m_processFormatMap.get(targetFormat);
        final long subProcessingStartTimeMillis = System.currentTimeMillis();

        if (null == processingFormat) {
            // check first, if we need to create the reference format entry
            if ((null != m_referenceFormat) && (null != (processingFormat = m_processFormatMap.get(m_referenceFormat)))) {
                implCreateThumbnail(m_imageKey, m_inputFile, m_imageMetadata, m_referenceFormat, processingFormat, m_context, isAsync);
                m_processFormatMap.remove(m_referenceFormat);
            }

            implCreatReferenceThumbnail(m_imageKey, m_imageMetadata, targetFormat, m_referenceFormat, m_context, isAsync);
        } else {
            implCreateThumbnail(m_imageKey, m_inputFile, m_imageMetadata, targetFormat, processingFormat, m_context, isAsync);
        }

        m_processingTime += System.currentTimeMillis() - subProcessingStartTimeMillis;
    }

    /**
     * @param targetFormat
     * @return
     * @throws ImageConverterException
     */
    protected MetadataImage processRequestFormat(@NonNull final String requestFormat) throws ImageConverterException {
        ImageFormat requestTargetFormat = null;
        MetadataImage ret = null;

        try {
            requestTargetFormat = ImageConverterUtils.getBestMatchingFormat(
                m_imageFormats,
                m_imageMetadata,
                ImageFormat.parseImageFormat(requestFormat));

            if (null == requestTargetFormat) {
                throw new ImageConverterException("IC Runnable is not able to parse requested image format: " + requestFormat);
            }

            processMapKey(requestTargetFormat, false);
            m_processFormatMap.remove(requestTargetFormat);

            ret = new MetadataImage(ImageConverterUtils.readImageByTargetFormat(m_fileItemService, m_imageKey, requestTargetFormat), m_imageMetadata);
        } catch (Exception e) {
            throw new ImageConverterException(new StringBuilder("IC Runnable error while processing: ").
                append(m_imageKey).
                append(" / ").append(requestFormat).
                append(" / ").append((null != requestTargetFormat) ?
                    requestTargetFormat.getFormatString() :
                        "null").toString(), e);
        }

        return ret;
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param imageInputStm
     * @return
     */
    private File implCreateImageInputFile(@NonNull final InputStream imageInputStm) {
        File inputFile = null;

        try {
            inputFile = File.createTempFile("oxic", ".tmp",
                !m_queue.isTerminated() ?
                    ImageConverterConfig.IMAGECONVERTER_SPOOLDIR :
                        DEFAULT_TEMP_DIR);

            FileUtils.copyInputStreamToFile(imageInputStm, inputFile);
        } catch (IOException e) {
            LOG.trace("IC Runnable error while creating image input file from input stream", e);
        }

        return inputFile;
    }

    /**
    *
    */
    private void implFillProcessMap() {
        // The process map is filled with 3 different kinds of key/value pairs:
        // 1. key and value are (!=null) and same: do a real image processing with the key format and store under key format
        // 2. key and value are (!=null) but different: do real image processing with the value format, but store under key format
        // 3. key is (!=null) and value is (==null): set only the reference property and store under key format without real image processing

        final Dimension imageDimension = m_imageMetadata.getImageDimension();

        for (final ImageFormat curImageFormat : m_imageFormats) {
            if (null != m_referenceFormat) {
                // if a reference format is set, we need to check, if
                // the current format image result matches the
                // reference format result, in order to properly
                // use the reference format
                if (ImageConverterUtils.shouldUseReferenceFormat(imageDimension, curImageFormat)) {
                    m_processFormatMap.put(curImageFormat, null);
                } else {
                    m_processFormatMap.put(curImageFormat, curImageFormat);
                }

            } else if (ImageConverterUtils.shouldUseReferenceFormat(imageDimension, curImageFormat)) {
                // the imageFormat at the first reference format position is substituted with the original size format
                final ImageFormat originalFormat = ImageFormat.createFrom(curImageFormat.getFormatShortName(),
                    true,
                    imageDimension.width,
                    imageDimension.height, ScaleType.AUTO,
                    false,
                    curImageFormat.getQuality());

                m_processFormatMap.put(m_referenceFormat = curImageFormat, originalFormat);
            } else {
                m_processFormatMap.put(curImageFormat, curImageFormat);
            }
        }
    }

    /**
     * @return
     * @throws MetadataException
     */
    protected IMetadata implCreateKeyMetadata(@NonNull final String imageKey, @NonNull final File inputFile, @Nullable final String context) throws MetadataException {
        IMetadata ret = m_metadataReader.readMetadata(inputFile);

        if (null != ret) {
            try (final IFileItemWriteAccess fileWriteAcess = m_fileItemService.getWriteAccess(
                ImageConverterUtils.IMAGECONVERTER_GROUPID,
                imageKey,
                ImageConverterUtils.IMAGECONVERTER_METADATA_FILEID)) {

                if (null != fileWriteAcess) {
                    try (final PrintWriter printWriter = new PrintWriter(fileWriteAcess.getOutputStream())) {
                        printWriter.write(ret.getJSONObject().toString());
                    }

                    implSetProperties(fileWriteAcess, context, null);

                    if (LOG.isTraceEnabled()) {
                        final Dimension imageDimension = ret.getImageDimension();
                        final String formatName = ret.getImageFormatName();

                        LOG.trace(new StringBuilder("IC created metada for ").
                            append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] ").
                            append(formatName).append(" image: ").
                            append(imageKey).toString());
                    }
                }
            } catch (Exception e) {
                LOG.error("IC error while creating metadata entry: " + imageKey, e);

                implRemoveFile(imageKey, ImageConverterUtils.IMAGECONVERTER_METADATA_FILEID);
                ret = null;
            }
        }

        return ret;
    }

    /**
     * @param writeAccess
     * @param referencePropertyValue
     * @throws FileItemException
     */
    protected static void implSetProperties(@NonNull final IFileItemWriteAccess writeAccess, @Nullable final String context, @Nullable final String referencePropertyValue) throws FileItemException {
        if (null != context) {
            writeAccess.setKeyValue(ImageConverterUtils.IMAGECONVERTER_KEY_CONTEXT, context);
        }

        if (null != referencePropertyValue) {
            writeAccess.setKeyValue(ImageConverterUtils.IMAGECONVERTER_KEY_REFERENCE, referencePropertyValue);
        }
    }

    /**
     * @param targetFormat
     * @param processingFormat
     * @throws ImageConverterException
     */
    protected void implCreateThumbnail(@NonNull final String imageKey, @NonNull final File inputFile,
        @NonNull final IMetadata imageMetadata,
        @NonNull final ImageFormat targetFormat,
        @NonNull final ImageFormat processingFormat,
        @Nullable final String context,
        final boolean isAsync) throws ImageConverterException {

        final String targetFormatStr = targetFormat.getFormatString();

        try (final IFileItemWriteAccess fileWriteAccess =
            m_fileItemService.getWriteAccess(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey, targetFormatStr)) {

            final ImageFormat usedProcessingFormat = (null != processingFormat) ? processingFormat : targetFormat;

            if ((null != fileWriteAccess) &&
                (m_imageProcessor.scale(inputFile, fileWriteAccess.getOutputFile(), usedProcessingFormat))) {

                implSetProperties(fileWriteAccess, context, null);
            }

            if (LOG.isTraceEnabled()) {
                final Dimension imageDimension = imageMetadata.getImageDimension();
                final String imageFormatShortName = imageMetadata.getImageFormatName();

                LOG.trace(new StringBuilder("IC ").append(isAsync ? "a" : "").
                    append("synchronously created ").append(imageFormatShortName).append(" thumbnail for ").
                    append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] ").
                    append(imageFormatShortName).append(" image: ").
                    append(targetFormat.getFormatString()).toString());
            }
        } catch (Exception e) {
            implRemoveFile(imageKey, targetFormatStr);

            if (LOG.isTraceEnabled()) {
                final Dimension imageDimension = imageMetadata.getImageDimension();
                final String imageFormatShortName = imageMetadata.getImageFormatName();

                LOG.error(new StringBuilder("IC error while ").append(isAsync ? "a" : "").
                    append("synchronously creating ").append(imageFormatShortName).append(" thumbnail for ").
                    append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] image: ").
                    append(imageKey).append(" / ").
                    append(targetFormatStr).toString(), e);
            }
        }
    }

    /**
     * @param targetFormat
     * @throws ImageConverterException
     */
    protected void implCreatReferenceThumbnail(@NonNull final String imageKey,
        @NonNull final IMetadata imageMetadata,
        @NonNull final ImageFormat targetFormat,
        @NonNull final ImageFormat referenceFormat,
        @Nullable final String context,
        final boolean isAsync) throws ImageConverterException {

        // we create a reference entry in case the shrinkOnly Flag
        // is set and the scaling would expand the source format
        final String targetFormatStr = targetFormat.getFormatString();
        final String referenceFormatStr = referenceFormat.getFormatString();

        try (final IFileItemWriteAccess fileWriteAccess = m_fileItemService.getWriteAccess(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey, targetFormatStr)) {
            implSetProperties(fileWriteAccess, context, referenceFormatStr);

            if (LOG.isTraceEnabled()) {
                final Dimension imageDimension = imageMetadata.getImageDimension();
                final String imageFormatShortName = imageMetadata.getImageFormatName();

                LOG.trace(new StringBuilder("IC ").append(isAsync ? "a" : "").
                    append("synchronously created reference entry for ").
                    append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] ").
                    append(imageFormatShortName).append(" image: ").
                    append(imageKey).append(" / ").append(targetFormatStr).
                    append(" (=> ").append(referenceFormatStr).append(')').toString());
            }
        } catch (Exception e) {
            implRemoveFile(imageKey, targetFormatStr);

            final Dimension imageDimension = imageMetadata.getImageDimension();
            final String imageFormatShortName = imageMetadata.getImageFormatName();

            LOG.error(new StringBuilder("IC error while ").append(isAsync ? "a" : "").
                append("synchronously creating reference entry for: ").
                append("[").append(imageDimension.width).append('x').append(imageDimension.height).append("] ").
                append(imageFormatShortName).append(" image: ").
                append(imageKey).append(" / ").append(targetFormatStr).
                append(" (=> ").append(referenceFormatStr).append(')').toString(), e);
        }
    }

    /**
     * @param imageKey
     * @param fileId
     */
    protected void implRemoveFile(@NonNull final String imageKey, @NonNull final String fileName) {
        try {
            m_fileItemService.remove(ImageConverterUtils.IMAGECONVERTER_GROUPID, imageKey, fileName);
        } catch (FileItemException e) {
            if (LOG.isTraceEnabled()) {
                LOG.trace(new StringBuilder("IC error while removing file for ").
                    append(imageKey).append(": ").
                    append(fileName).toString(), Throwables.getRootCause(e));
            }
        }
    }

    /**
     * @param metadata
     * @param imageFormats
     * @return
     */
    protected static List<ImageFormat> implCreateImageFormatList(@NonNull final IMetadata metadata, @NonNull final List<ImageFormat> imageFormats) {
        final List<ImageFormat> ret = new ArrayList<>();
        final String imageFormatShortName = metadata.getImageFormatName();
        final boolean isAnimated = imageFormatShortName.startsWith("gif");
        final boolean isTransparent = isAnimated || imageFormatShortName.startsWith("png") || imageFormatShortName.startsWith("tif");

        for (final ImageFormat curFormat : imageFormats) {
            ImageFormat.ImageType targetFormatImageType = null;


            switch (curFormat.getFormatShortName()) {
                case "jpg": {
                    targetFormatImageType = ImageFormat.ImageType.JPG;
                    break;
                }

                case "png": {
                    targetFormatImageType = ImageFormat.ImageType.PNG;
                    break;
                }

                case "auto":
                default: {
                    targetFormatImageType = isAnimated ?
                        // TODO (KA) add GIF format to possible target types
                        ImageFormat.ImageType.PNG :
                            (isTransparent  ? ImageFormat.ImageType.PNG : ImageFormat.ImageType.JPG);
                    break;
                }
            }

            final ImageFormat targetImageFormat = new ImageFormat(targetFormatImageType);

            targetImageFormat.setAutoRotate(curFormat.isAutoRotate());
            targetImageFormat.setWidth(curFormat.getWidth());
            targetImageFormat.setHeight(curFormat.getHeight());
            targetImageFormat.setScaleType(curFormat.getScaleType());
            targetImageFormat.setShrinkOnly(curFormat.isShrinkOnly());


            ret.add(targetImageFormat);
        }

        return ret;
    }

    // - Members ---------------------------------------------------------------

    final private ImageConverterQueue m_queue;

    final private ImageProcessor m_imageProcessor;

    final private IFileItemService m_fileItemService;

    final private IMetadataReader m_metadataReader;

    final private List<ImageFormat> m_imageFormats;

    final private Map<ImageFormat, ImageFormat> m_processFormatMap = new LinkedHashMap<>();

    final private String m_imageKey;

    private File m_inputFile = null;

    private String m_context = null;

    private IMetadata m_imageMetadata = null;

    private ImageFormat m_referenceFormat = null;

    private long m_measurementStartTimeMillis = 0;

    private long m_processingTime = 0;
}
