/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.fileitem.impl;

import static org.apache.commons.lang.ArrayUtils.isEmpty;
import static org.apache.commons.lang.ArrayUtils.isNotEmpty;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Array;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.exception.OXException;
import com.openexchange.imageconverter.api.FileItemException;

/**
 * {@link FileItemUtils}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
/**
 * {@link FileItemUtils}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
/**
 * {@link FileItemUtils}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
/**
 * {@link FileItemUtils}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItemUtils {

    /**
     * FILEITEM_TEMP_DIRNAME
     */
    final private static String FILEITEM_TEMP_DIRNAME = "oxfitmp";

    /**
     * FILEITEM_DATA_DIRNAME
     */
    final private static String FILEITEM_DATA_DIRNAME = "oxfidata";

    /**
     * Initializes a new {@link FileItemUtils}.
     */
    private FileItemUtils() {
        super();

    }

    // - Misc ------------------------------------------------------------------

    /**
     *
     */
    public static void shutdown() {
        if (IS_TERMINATED.compareAndSet(false, true)) {
            FileUtils.deleteQuietly(TEMP_DIR);
        }
    }

    /**
     * @param spoolDir
     */
    public static void setSpoolDir(final File spoolDir) {
       if (null != spoolDir) {
           TEMP_DIR = implCreateDirectory(spoolDir, FILEITEM_TEMP_DIRNAME, true);
           DATA_DIR = implCreateDirectory(spoolDir, FILEITEM_DATA_DIRNAME, false);
       }
    }

    /**
     * @return
     */
    public static File getDataDir() {
        return DATA_DIR;
    }

    /**
     * @param baseDirectory
     * @param directoryName
     * @param clean
     * @return
     */
    private static File implCreateDirectory(@NonNull final File baseDirectory, @NonNull final String directoryName, boolean clean) {
        File targetDirectory = new File(baseDirectory, directoryName);

        // use or create given spool directory, use /tmp as fallback parent directory
        if ((targetDirectory.exists() && targetDirectory.canWrite() && targetDirectory.isDirectory()) ||
            targetDirectory.mkdirs() || (targetDirectory = new File("/tmp", directoryName)).mkdirs()) {

            if (clean) {
                try {
                    FileUtils.cleanDirectory(targetDirectory);
                } catch (IOException e) {
                    logExcp(e);
                }
            }
        }

        return targetDirectory;
    }

    /**
     * @param id
     * @return
     */
    public static boolean isValid(final String id) {
        return isNotEmpty(id);
    }

    /**
     * @param id
     * @return
     */
    public static boolean isValid(final String[] ids) {
        return isNotEmpty(ids);
    }

    /**
     * @param id
     * @return
     */
    public static boolean isInvalid(final String id) {
        return isEmpty(id);
    }

    /**
     * @param id
     * @return
     */
    public static boolean isInvalid(final String[] ids) {
        return isEmpty(ids);
    }

    /**
     * @param fileItem
     * @return
     */
    public static boolean isValid(final FileItem fileItem) {
        return ((null != fileItem) && fileItem.isValid());
    }

    /**
     * @param fileItem
     * @return
     */
    public static boolean isInvalid(final FileItem fileItem) {
        return ((null == fileItem) || fileItem.isInvalid());
    }

    /**
     * @param fileStoreData
     * @return
     */
    public static boolean isValid(final FileStoreData fileStoreData) {
        return ((null != fileStoreData) && fileStoreData.isValid());
    }

    /**
     * @param fileStoreData
     * @return
     */
    public static boolean isValid(final FileStoreData[] fileStoreData) {
        return isNotEmpty(fileStoreData);
    }

    /**
     * @param fileStoreData
     * @return
     */
    public static boolean isInvalid(final FileStoreData fileStoreData) {
        return ((null == fileStoreData) || fileStoreData.isInvalid());
    }

    /**
     * @param fileStoreData
     * @return
     */
    public static boolean isInvalid(final FileStoreData[] fileStoreData) {
        return isEmpty(fileStoreData);
    }

    /**
     * @param list
     * @param T
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T> T[] listToArray(@NonNull final List<T> list, Class<?> type) {
        return list.toArray((T[]) Array.newInstance(type, list.size()));
    }

    /**
     * @return
     * @throws IOException
     */
    public static File createTempFile() throws IOException {
        return File.createTempFile("oxfi", ".tmp", !IS_TERMINATED.get() ? TEMP_DIR : DEFAULT_TEMP_DIR);
    }

    /**
     * @param startString
     * @param aliasPrefix
     * @param queryProperties
     * @param end
     * @return
     */
    public static String createSqlWithPropertyVariables(@NonNull final String startString,
        @NonNull Properties properties, @NonNull String combineString, final String... appendStrings) {

        final Set<Object> keySet = properties.keySet();
        final StringBuilder sql = new StringBuilder(startString);
        boolean following = false;

        for(final Object curKey : keySet) {
            if (isNotEmpty(curKey.toString())) {
                if (following) {
                    sql.append(combineString);
                } else {
                    following = true;
                }

                sql.append(' ').append(curKey).append("=?");
            }
        }

        if (null != appendStrings) {
            for (final String curStr : appendStrings) {
                if (isNotEmpty(curStr)) {
                    sql.append(' ').append(curStr);
                }
            }
        }

        return sql.toString();
    }

    /**
     * @param stmt
     * @param values
     * @param startIndex
     * @return The updated index after setting all given values
     * @throws SQLException
     */
    public static int setStatementValues(@NonNull final PreparedStatement stmt, final Object[] values, final int startIndex) throws SQLException {
        int curIndex = startIndex;

        if (isNotEmpty(values) && (startIndex > 0)) {
            for(final Object curValue : values) {
                if (null != curValue) {
                    if (curValue instanceof String) {
                        stmt.setString(curIndex++, curValue.toString());
                    } else if (curValue instanceof Long) {
                        stmt.setLong(curIndex++, ((Long) curValue).longValue());
                    } else if (curValue instanceof Integer) {
                        stmt.setInt(curIndex++, ((Integer) curValue).intValue());
                    } else if (curValue instanceof String) {
                        stmt.setString(curIndex++, curValue.toString());
                    } else if (curValue instanceof Double) {
                        stmt.setDouble(curIndex++, ((Double) curValue).doubleValue());
                    } else if (curValue instanceof Float) {
                        stmt.setFloat(curIndex++, ((Float) curValue).floatValue());
                    } else if (curValue instanceof Boolean) {
                        stmt.setBoolean(curIndex++, ((Boolean) curValue).booleanValue());
                    } else  {
                        stmt.setString(curIndex++, curValue.toString());
                    }
                } else {
                    stmt.setNull(curIndex++, Types.NULL);
                }
            }
        }

        return curIndex;
    }

    /**
     * @param fileStoreData
     * @throws IOException
     */
    public static void deleteFileStoreFile(@NonNull FileItemService fileItemService, @NonNull final FileStoreData fileStoreData) throws FileItemException {
        if (isValid(fileStoreData)) {
            try {
                fileItemService.getFileStorage(fileStoreData).deleteFile(fileStoreData.getFileStoreId());
            } catch (OXException e) {
                throw new FileItemException("Delete of FileStore file failed: " + fileStoreData.toString(), e);
            }
        }
    }

    /**
     * @param fileStoreDatas
     * @return
     * @throws FileItemException
     */
    public static int deleteFileStoreFiles(@NonNull FileItemService fileItemService, @NonNull final List<FileStoreData> fileStoreDataList) throws FileItemException {
        FileItemException rethrowable = null;
        int ret = 0;

        final LinkedList<FileStoreData> fileStoreDataWorkList = new LinkedList<>();
        final List<String> workList = new ArrayList<>(fileStoreDataList.size());

        fileStoreDataWorkList.addAll(fileStoreDataList);

        while (!fileStoreDataWorkList.isEmpty()) {
            final FileStoreData firstData = fileStoreDataWorkList.get(0);

            if (null != firstData) {
                final int curFileStoreNumber = firstData.getFileStoreNumber();

                workList.clear();

                // collect all items with current fileStore number in worklist and remove from source list
                for (final ListIterator<FileStoreData> iter = fileStoreDataWorkList.listIterator(); iter.hasNext();) {
                    final FileStoreData curData = iter.next();

                    if (curData.getFileStoreNumber() == curFileStoreNumber) {
                        workList.add(curData.getFileStoreId());
                        iter.remove();
                    }
                }

                // remove collected files and iuncrease delete count
                final int workListSize = workList.size();

                if (workListSize > 0) {
                    try {
                        final Set<String> leftOverFiles = fileItemService.getFileStorage(firstData).deleteFiles(workList.toArray(new String[workListSize]));

                        ret += (workListSize - leftOverFiles.size());

                        if (!leftOverFiles.isEmpty()) {
                            logWarn(new StringBuilder(128).
                                append("FileStorageService was not able to delete ").
                                append(leftOverFiles.size()).
                                append(" files").toString());
                        }
                    } catch (OXException e) {
                        rethrowable = new FileItemException("Error while removing multiple files from FileStore", Throwables.getRootCause(e));
                    }
                }
            } else {
                fileStoreDataWorkList.remove();
            }
        }

        if (null != rethrowable) {
            throw rethrowable;
        }

        return ret;
    }

    // - Logging ---------------------------------------------------------------

    /**
     * @return
     */
    public static boolean isLogError() {
        return implGetLogger().isErrorEnabled();
    }

    /**
     * @param msg
     */
    public static void logError(final String msg) {
        if (isNotEmpty(msg)) {
            implGetLogger().error(msg);
        }
    }

    /**
     * @return
     */
    public static boolean isLogWarn() {
        return implGetLogger().isWarnEnabled();
    }

    /**
     * @param msg
     */
    public static void logWarn(final String msg) {
        if (isNotEmpty(msg)) {
            implGetLogger().warn(msg);
        }
    }

    /**
     * @return
     */
    public static boolean isLogInfo() {
        return implGetLogger().isInfoEnabled();
    }

    /**
     * @param msg
     */
    public static void logInfo(final String msg) {
        if (isNotEmpty(msg)) {
            implGetLogger().info(msg);
        }
    }

    /**
     * @return
     */
    public static boolean isLogDebug() {
        return implGetLogger().isDebugEnabled();
    }

    /**
     * @param msg
     */
    public static void logDebug(final String msg) {
        if (isNotEmpty(msg)) {
            implGetLogger().debug(msg);
        }
    }

    /**
     * @return
     */
    public static boolean isLogTrace() {
        return implGetLogger().isTraceEnabled();
    }

    /**
     * @param msg
     */
    public static void logTrace(final String msg) {
        if (isNotEmpty(msg)) {
            implGetLogger().trace(msg);
        }
    }

    /**
     * @param msg
     */
    public static void logTrace(final String msg, final Throwable e) {
        if (isNotEmpty(msg)) {
            implGetLogger().trace(msg, e);
        }
    }

    /**
     * @param e
     */
    public static void logExcp(Throwable e) {
        if (implGetLogger().isErrorEnabled()) {
            String msg = null;

            if (null != e) {
                final Throwable cause = Throwables.getRootCause(e);

                msg = (null != cause) ? cause.getMessage() : null;

                if (null != msg) {
                    if (msg.length() > ERROR_MESSAGE_TOTAL_LENGTH) {
                        final int beginLength = ERROR_MESSAGE_TOTAL_LENGTH - ERROR_MESSAGE_END_LENGTH - ERROR_MESSAGE_FILL_STRING.length();
                        final int endLength = Math.min(msg.length() - beginLength, ERROR_MESSAGE_END_LENGTH);
                        msg = msg.substring(0, beginLength) + ((endLength > 0) ? ("..." + msg.substring(msg.length() - endLength)) : StringUtils.EMPTY);
                    }
                } else {
                    msg = ((null != cause) ? cause : e).getClass().getName();
                }
            } else {
                msg = DEF_EXCEPTION_MSG;
            }

            implGetLogger().error(msg);
        }
    }

    // - Implementation --------------------------------------------------------

    static Logger implGetLogger() {
        return FileItemService.getLogger();
    }

    // - Statics ---------------------------------------------------------------

    final private static AtomicBoolean IS_TERMINATED = new AtomicBoolean(false);

    final private static  String DEF_EXCEPTION_MSG = "Unspecified FileItemService exception occured";

    final private static  String ERROR_MESSAGE_FILL_STRING = "...";

    final private static int ERROR_MESSAGE_TOTAL_LENGTH = 2048;

    final private static int ERROR_MESSAGE_END_LENGTH = 256;

    final private static File DEFAULT_TEMP_DIR = new File("/tmp");

    private static File TEMP_DIR = new File("/tmp");

    private static File DATA_DIR = TEMP_DIR;
}
