/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter.client.impl;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import com.openexchange.documentconverter.JobError;
import com.openexchange.documentconverter.JobPriority;
import com.openexchange.documentconverter.LogData;
import com.openexchange.documentconverter.Properties;
import com.openexchange.documentconverter.ServerType;
import com.openexchange.documentconverter.TransferHelper;
import com.openexchange.documentconverter.TransferObject;
import com.openexchange.documentconverter.TransferResponseProcessor;

/**
 * {@link ManagedClientJob}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
class ManagedClientJob {

    /**
     * Initializes a new {@link ManagedClientJob}.
     *
     * @param clientManager
     */
    ManagedClientJob(ClientManager clientManager, ServerType serverType) {
        super();

        m_clientManager = clientManager;
        m_serverType = serverType;

        final boolean hasRemoteServer = clientManager.hasRemoteServer(serverType);

        if (hasRemoteServer) {
            switch (m_serverType) {
                case DOCUMENTCONVERTER: {
                    m_remoteUrl = ClientManager.getClientConfig().REMOTEURL_DOCUMENTCONVERTER;
                    break;
                }

                case CACHESERVER:
                default: {
                    break;
                }
            }
        }
    }

    /**
     * @param jobType
     * @param jobProperties
     * @param resultProperties
     * @return
     */
    boolean queryAvailability(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        boolean ret = false;

        if (checkJobPreconditions(jobProperties, resultProperties, 1)) {
            if (prepareRemoteCall(jobProperties, RCALL_QUERY_AVAILABILITY)) {
                jobProperties.put(Properties.PROP_JOBTYPE, jobType);

                doRemoteCall(jobProperties, resultProperties);

                ret = resultProperties.containsKey(Properties.PROP_RESULT_ERROR_CODE) &&
                    (JobError.NONE == JobError.fromObject(resultProperties.get(Properties.PROP_RESULT_ERROR_CODE)));
            }
        }

        return ret;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.documentconverter.impl.ManagedJob#convert(java.lang.String, java.util.HashMap, java.util.HashMap)
     */
    InputStream process(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        InputStream ret = null;
        final Boolean asyncObject = (Boolean) jobProperties.get(Properties.PROP_ASYNC);

        if ((null != asyncObject) && asyncObject.booleanValue()) {
            if (checkJobInputFileSize(jobProperties, resultProperties) &&
                checkSupportedJobInputType(jobProperties, resultProperties)) {

                m_clientManager.triggerAsyncConvert(jobType, jobProperties, resultProperties);
            }
        } else if (checkJobPreconditions(jobProperties, resultProperties, 1)) {
            if (prepareRemoteCall(jobProperties, RCALL_CONVERT)) {
                jobProperties.put(Properties.PROP_JOBTYPE, jobType);
                ret = (InputStream) doRemoteCall(jobProperties, resultProperties);
            }
        }

        return ret;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.documentconverter.impl.ManagedJob#beginPageConversion(java.lang.String, java.util.HashMap, java.util.HashMap)
     */
    String beginPageConversion(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        String ret = null;

        if (checkJobPreconditions(jobProperties, resultProperties, 2)) {
            // check if a remote conversion is requested and if we're not called remotely by ourselves
            if (prepareRemoteCall(jobProperties, RCALL_BEGINPAGECONVERSION)) {
                jobProperties.put(Properties.PROP_JOBTYPE, jobType);
                doRemoteCall(jobProperties, resultProperties);
                ret = (String) resultProperties.get(Properties.PROP_RESULT_JOBID);

                // save cookie for begin of stateful call in global cookie map, if possible
                final String converterJobId = (String) resultProperties.get(Properties.PROP_RESULT_JOBID);
                final String converterCookie = (String) resultProperties.get(Properties.PROP_RESULT_CONVERTER_COOKIE);

                if (StringUtils.isNotEmpty(converterJobId) && StringUtils.isNotEmpty(converterCookie)) {
                    m_converterCookieMap.put(converterJobId, converterCookie);
                }
            }
        }

        return ret;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.documentconverter.impl.ManagedJob#getConversionPage(java.lang.String, int, java.util.HashMap)
     */
    InputStream getConversionPage(String jobIdStr, int pageNumber, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        InputStream ret = null;

        if (checkJobPreconditions(jobProperties, resultProperties, 2)) {
            jobProperties.put(Properties.PROP_JOBID, jobIdStr);
            jobProperties.put(Properties.PROP_PAGE_NUMBER, Integer.valueOf(pageNumber));

            // check if a remote conversion is requested and if we're not called remotely by ourselves;
            // also add a possibly stored cookie for this job id to current job properties
            if (prepareRemoteCall(jobProperties, RCALL_GETCONVERSIONPAGE)) {
                ret = (InputStream) doRemoteCall(jobProperties, resultProperties);
            }
        }

        return ret;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.documentconverter.impl.ManagedJob#endPageConversion(java.lang.String)
     */
    void endPageConversion(String jobIdStr, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        if (checkJobPreconditions(jobProperties, resultProperties, 2)) {
            jobProperties.put(Properties.PROP_JOBID, jobIdStr);

            // check if a remote conversion is requested and if we're not called remotely by ourselves
            if (prepareRemoteCall(jobProperties, RCALL_ENDPAGECONVERSION)) {
                doRemoteCall(jobProperties, resultProperties);

                // delete cookie for end of stateful call from global cookie map, if possible
                final String converterJobId = (String) resultProperties.get(Properties.PROP_RESULT_JOBID);

                if (StringUtils.isNotEmpty(converterJobId)) {
                    m_converterCookieMap.remove(converterJobId);
                }
            }
        }
    }

    // - Protected Methodds ----------------------------------------------------

    /**
     * @param jobProperties
     * @param resultProperties
     * @return
     */
    static protected byte[] getInputBuffer(final HashMap<String, Object> jobProperties, final HashMap<String, Object> resultProperties) {
        byte[] ret = null;

        try (final InputStream inputStm = (jobProperties.containsKey(Properties.PROP_INPUT_FILE) ?
            new FileInputStream((File) jobProperties.get(com.openexchange.documentconverter.Properties.PROP_INPUT_FILE)) :
                (InputStream) jobProperties.get(Properties.PROP_INPUT_STREAM))) {

            if (null != inputStm) {
                ret = IOUtils.toByteArray(inputStm);
            }
        } catch (final Exception e) {
            ClientManager.logExcp(e);
        }

        // check, if length of byte array is larger than specified config source file size
        if ((null != ret) && (setSourceFileSizeError(ret.length, jobProperties, resultProperties))) {
            ret = null;
        }

        return ret;
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param jobProperties
     */
    /**
     * @param jobProperties
     * @param resultProperties
     * @param minRemoteAPIVersion
     * @return true, if job processing can proceed
     */
    protected boolean checkJobPreconditions(final HashMap<String, Object> jobProperties, final HashMap<String, Object> resultProperties, int minRemoteAPIVersion) {
        boolean ret = false;

        if (null != jobProperties) {
            // job is valid by default, if job properties exists at all
            // and source file size fits into the given constraints
            ret = checkJobInputFileSize(jobProperties, resultProperties) &&
                checkSupportedJobInputType(jobProperties, resultProperties);
        } else {
            resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.GENERAL.errorCode()));
        }

        if (ret && (null != m_remoteUrl)) {
            ret = m_clientManager.getRemoteAPIVersion(m_serverType) >= minRemoteAPIVersion;
        }

        return ret;
    }

    /**
     * @param jobProperties
     * @param resultProperties
     * @return true, if source file size is unlimited or below/equal
     *  to the given maximum source file size
     */
    protected static boolean checkJobInputFileSize(final HashMap<String, Object> jobProperties, final HashMap<String, Object> resultProperties) {
        boolean ret = false;

        if (null != jobProperties) {
            // check max source file size, if source file is given
            final File inputFile = (File) jobProperties.get(Properties.PROP_INPUT_FILE);

            if (null != inputFile) {
                // check maxSourceFile
                ret = !setSourceFileSizeError(inputFile.length(), jobProperties, resultProperties);
            } else {
                // job is valid by default, if job properties exists
                // at all and source file is given as input stream
                ret = true;
            }
        } else {
            resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.GENERAL.errorCode()));
        }

        return ret;
    }

    /**
     * @param jobProperties
     * @param resultProperties
     * @return true, if source file size is unlimited or below/equal
     *  to the given maximum source file size
     */
    protected static boolean checkSupportedJobInputType(final HashMap<String, Object> jobProperties, final HashMap<String, Object> resultProperties) {
        boolean ret = false;

        if (null != jobProperties) {
            // check max source file size, if source file is given
            String inputType = (String) jobProperties.get(Properties.PROP_INPUT_TYPE);

            if (StringUtils.isEmpty(inputType)) {
                final String fileName = (String) jobProperties.get(Properties.PROP_INFO_FILENAME);

                if (StringUtils.isNotEmpty(fileName)) {
                    inputType = FilenameUtils.getExtension(fileName);

                    if (StringUtils.isNotEmpty(inputType)) {
                        inputType = inputType.trim().toLowerCase();
                    }
                }
            }

            if (StringUtils.isEmpty(inputType)) {
                inputType = "bin";
            }

            ret = !setSupportedTypeError(inputType, jobProperties, resultProperties);
        }

        return ret;
    }

    /**
     * @param sourceSize
     * @param maxSize
     * @param resultProperties
     * @return true, if source size is larger than max size
     */
    protected static boolean setSourceFileSizeError(long sourceSize, final HashMap<String, Object> jobProperties, final HashMap<String, Object> resultProperties) {
        final long maxSize = ClientManager.getClientConfig().MAX_DOCUMENT_SOURCEFILESIZE;
        boolean ret = false;

        if (sourceSize > maxSize) {
            ret = true;

            if (null != resultProperties) {
                resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.MAX_SOURCESIZE.errorCode()));

                if (ClientManager.isLogError()) {
                    ClientManager.logError("DC detected source file size larger than configured size",
                        jobProperties,
                        new LogData("sourcesize", (new Long(sourceSize)).toString()),
                        new LogData("maxsize", (new Long(maxSize)).toString()));
                }
            }
        }

        return ret;
    }

    /**
     * @param sourceSize
     * @param maxSize
     * @param resultProperties
     * @return true, if source size is larger than max size
     */
    protected static boolean setSupportedTypeError(String inputType, final HashMap<String, Object> jobProperties, final HashMap<String, Object> resultProperties) {
        final Set<String> supportedTypes = ClientManager.getClientConfig().SUPPORTED_TYPES;
        final boolean ret = (supportedTypes.size() > 0) &&
            SUPPORTED_TYPE_CHECK_LIST.contains(inputType) &&
            !supportedTypes.contains(inputType);

        if (ret) {
            if (null != resultProperties) {
                resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.UNSUPPORTED.errorCode()));
            }

            if (ClientManager.isLogError()) {
                ClientManager.logError("DC detected unsupported input file extension",
                    jobProperties, new LogData("inputype", inputType));
            }
        }

        return ret;
    }

    /**
     * @param jobProperties
     * @param method
     * @return
     */
    private boolean prepareRemoteCall(HashMap<String, Object> jobProperties, String method) {
        final String calledRemoteUrlString = (String) jobProperties.get(Properties.PROP_ENGINE_REMOTEURL);
        URL calledRemoteUrl = null;
        boolean ret = false;

        if (!StringUtils.isEmpty(calledRemoteUrlString)) {
            try {
                calledRemoteUrl = new URL(calledRemoteUrlString);
            } catch (final MalformedURLException e) {
                ClientManager.logExcp(e);
            }
        }

        // check if a remote conversion is requested and if we're not remotely called by ourselves
        if ((null != m_remoteUrl) && ((null == calledRemoteUrl) || !calledRemoteUrl.equals(m_remoteUrl))) {
            jobProperties.put(Properties.PROP_REMOTE_METHOD, method);
            jobProperties.put(Properties.PROP_ENGINE_REMOTEURL, m_remoteUrl.toString());
            ret = true;
        }

        // get stored cookie from previous beginPage call and add to job properties, if possible
        if (method.equals(RCALL_GETCONVERSIONPAGE) || method.equals(RCALL_ENDPAGECONVERSION)) {
            final String converterJobId = (String) jobProperties.get(Properties.PROP_JOBID);

            if (StringUtils.isNotEmpty(converterJobId)) {
                final String converterCookie = m_converterCookieMap.get(converterJobId);

                if (StringUtils.isNotEmpty(converterCookie)) {
                    jobProperties.put(Properties.PROP_CONVERTER_COOKIE, converterCookie);
                }
            }
        }

        return ret;
    }

    /**
     * @param jobProperties
     * @param resultProperties
     * @param jobType
     * @return
     */
    private Object doRemoteCall(final HashMap<String, Object> jobProperties, final HashMap<String, Object> resultProperties) {
        Object ret = null;
        final boolean isMultipartRequest = (jobProperties.containsKey(Properties.PROP_INPUT_FILE) || jobProperties.containsKey(Properties.PROP_INPUT_STREAM));
        final Boolean asyncJob = (Boolean) jobProperties.get(Properties.PROP_ASYNC);
        final Boolean remoteAsyncJob = (Boolean) jobProperties.get(Properties.PROP_REMOTE_ASYNC);
        final Boolean cacheOnlyJob = (Boolean) jobProperties.get(Properties.PROP_CACHE_ONLY);
        final String remoteCacheHash = (String) jobProperties.get(Properties.PROP_REMOTE_CACHE_HASH);
        final boolean isAsync = ((null != asyncJob) && asyncJob.booleanValue()) || ((null != remoteAsyncJob) && remoteAsyncJob.booleanValue());
        final boolean isCacheOnly = (null != cacheOnlyJob) && cacheOnlyJob.booleanValue();
        final String remoteParams = isMultipartRequest ? getRemoteParams(jobProperties, resultProperties, false, "returntype=binary") : getRemoteParams(jobProperties, resultProperties, true);
        final boolean logTrace = ClientManager.isLogTrace();
        final String remoteMethod = jobProperties.containsKey(Properties.PROP_REMOTE_METHOD) ? (String) jobProperties.get(Properties.PROP_REMOTE_METHOD) : "";
        final String logTitle = (remoteMethod.equals("transformimage")) ?
            ("ImageServer " + (isAsync ? "asynchronous " : "") + "remote transformation") :
                ("DC " + (isAsync ? "asynchronous " : "") + "remote conversion");

            if ((null != m_remoteUrl) && (null != remoteParams)) {
                final String callCookie = (String) jobProperties.get(Properties.PROP_CONVERTER_COOKIE);
                URL remoteUrl = null;

                if (logTrace) {
                    ClientManager.logTrace(logTitle + " initiated", jobProperties, new LogData("method", remoteMethod));
                }

                try {
                    remoteUrl = new URL(m_remoteUrl.toString() + "?action=rconvert");
                } catch (final MalformedURLException e) {
                    ClientManager.logExcp(e);
                }

                if ((null != remoteUrl) && isMultipartRequest) {
                    final String inputType = (String) jobProperties.get(Properties.PROP_INPUT_TYPE);
                    final String inputMimeType = (null != inputType) && (inputType.equals("pdf")) ? "application/pdf" : "application/octet-stream";
                    final Serializable contentSerializable =  jobProperties.containsKey(Properties.PROP_INPUT_FILE) ?
                        (File) jobProperties.get(Properties.PROP_INPUT_FILE) :
                            getInputBuffer(jobProperties, resultProperties);


                    if (null != contentSerializable) {
                        final TransferObject<Serializable> transferObject = new TransferObject<>(
                            remoteParams,
                            contentSerializable,
                            "ByteArray.bin",
                            inputMimeType,
                            callCookie);
                        final TransferHelper.PostFormDataMode postFormDataMode =
                            ((contentSerializable instanceof File) && (m_clientManager.getRemoteAPIVersion(ServerType.DOCUMENTCONVERTER) >= 6)) ?
                            TransferHelper.PostFormDataMode.FILE_LENGTH_AND_DATA :
                                TransferHelper.PostFormDataMode.BYTEARRAY_OBJECT;

                        TransferHelper.postFormData(m_clientManager, remoteUrl, m_serverType, transferObject, postFormDataMode, new TransferResponseProcessor() {

                            @SuppressWarnings("unchecked")
                            @Override
                            public void processResponse(int responseCode, InputStream responseInputStream, String resultCookie) {
                                try {
                                    // evaluate result only for synchronous calls with valid result stream
                                    if (!isAsync && (null != responseInputStream)) {
                                        try (ObjectInputStream objInputStream = new ObjectInputStream(new BufferedInputStream(responseInputStream))) {
                                            final Object readObject = objInputStream.readObject();

                                            if ((null != readObject) && (readObject instanceof HashMap<?, ?>)) {
                                                resultProperties.clear();
                                                resultProperties.putAll((HashMap<? extends String, ?>) readObject);

                                                if (null != resultCookie) {
                                                    resultProperties.put(Properties.PROP_RESULT_CONVERTER_COOKIE, resultCookie);
                                                }

                                                // check for old style property and convert to result error
                                                final Boolean passwordProtected = (Boolean) resultProperties.get("PasswordProtected");

                                                if ((null != passwordProtected) && passwordProtected.booleanValue()) {
                                                    resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.PASSWORD.errorCode()));
                                                }
                                            }
                                        } catch (final Exception e) {
                                            resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.TIMEOUT.errorCode()));
                                            ClientManager.logExcp(e);
                                        }
                                    }
                                } finally {
                                    IOUtils.closeQuietly(responseInputStream);
                                }
                            }
                        });

                        final byte[] resultBuffer = (byte[]) resultProperties.get(Properties.PROP_RESULT_BUFFER);

                        if (null != resultBuffer) {
                            ret = new ByteArrayInputStream(resultBuffer);
                        }
                    }
                } else {
                    final TransferObject<byte[]> transferObject = new TransferObject<>(remoteParams, callCookie);
                    final StringBuilder response = new StringBuilder();

                    TransferHelper.postRequest(m_clientManager, remoteUrl, m_serverType, transferObject, new TransferResponseProcessor() {

                        @Override
                        public void processResponse(int responseCode, InputStream responseInputStream, String resultCookie) {
                            try {
                                // evaluate result only for synchronous calls with valid result stream
                                if (!isAsync && (null != responseInputStream)) {
                                    try (BufferedReader inputReader = new BufferedReader(new InputStreamReader(responseInputStream))) {
                                        for (String readLine = null; (readLine = inputReader.readLine()) != null;) {
                                            response.append(readLine).append('\n');
                                        }

                                        if (null != resultCookie) {
                                            resultProperties.put(Properties.PROP_RESULT_CONVERTER_COOKIE, resultCookie);
                                        }
                                    } catch (final Exception e) {
                                        resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.TIMEOUT.errorCode()));
                                        ClientManager.logExcp(e);
                                    }
                                }
                            } finally {
                                IOUtils.closeQuietly(responseInputStream);
                            }
                        }
                    });

                    // parse JSON response only on case of synchronous jobs,
                    // since we won't get a valid answer for async requests
                    if (!isAsync) {
                        try {
                            ret = parseJSONResponse(new JSONObject(new JSONTokener(response.toString())), resultProperties);
                        } catch (final JSONException e) {
                            ClientManager.logExcp(e);
                        }
                    }
                }
            }

            if (isAsync) {
                if (logTrace) {
                    ClientManager.logTrace(logTitle + " implicitly succeeded", jobProperties);
                }
            } else if (JobError.NONE == JobError.fromObject(resultProperties.get(Properties.PROP_RESULT_ERROR_CODE))) {
                if (logTrace) {
                    ClientManager.logTrace(logTitle + " succeeded", jobProperties);
                }
            } else if (ClientManager.isLogWarn()) {
                // don't log warnings for cacheOnly jobs /
                // don't log warnings for async jobs /
                // don't log warnings for remote cachehash jobs
                // ------------------------------------------------------
                // all above mentioned types of jobs may return without a
                // valid or with an error result. This should not be logged
                // as an error/warning at all here, since there'll either be a
                // fallback conversion or the caller is not interested in the
                // the current result of these 'try and ignore' conversions
                if (!isCacheOnly && !isAsync && StringUtils.isEmpty(remoteCacheHash)) {
                    final StringBuilder errorMessageBuilder = new StringBuilder("DC received a remote conversion ");
                    final JobError jobError = JobError.fromObject(resultProperties.get(Properties.PROP_RESULT_ERROR_CODE));

                    ClientManager.logWarn(errorMessageBuilder.append(
                        (JobError.TIMEOUT == jobError) ? "timeout error" :
                            (JobError.NO_CONTENT == jobError) ? "no content error" :
                                (JobError.PASSWORD == jobError) ? "password error" :
                                    (JobError.MAX_SOURCESIZE == jobError) ? "max source size error" :
                                        "general error").toString());
                }
            }

            return ret;
    }

    /**
     * @param jobType
     * @param jobProperties
     * @return
     */
    private String getRemoteParams(final HashMap<String, Object> jobProperties, final HashMap<String, Object> resultProperties, boolean includeDataUrl, String... additionalParams) {
        StringBuilder paramBuilder = new StringBuilder("action=rconvert");
        Object curParam = null;

        if (null != jobProperties) {
            // MethodName
            if (null != (curParam = jobProperties.get(Properties.PROP_REMOTE_METHOD))) {
                paramBuilder.append("&method=").append((String) curParam);
            }

            // JobType
            if (null != (curParam = jobProperties.get(Properties.PROP_JOBTYPE))) {
                paramBuilder.append("&jobtype=").append((String) curParam);
            }

            // JobId
            if (null != (curParam = jobProperties.get(Properties.PROP_JOBID))) {
                paramBuilder.append("&jobid=").append((String) curParam);
            }

            // JobPriority
            if (null != (curParam = jobProperties.get(Properties.PROP_PRIORITY))) {
                paramBuilder.append("&priority=").append(((JobPriority) curParam).toString());
            }

            // Locale
            if (null != (curParam = jobProperties.get(Properties.PROP_LOCALE))) {
                paramBuilder.append("&locale=").append((String) curParam);
            }

            // CacheHash
            if (null != (curParam = jobProperties.get(Properties.PROP_CACHE_HASH))) {
                paramBuilder.append("&cachehash=").append((String) curParam);
            }

            // RemoteCacheHash
            if (null != (curParam = jobProperties.get(Properties.PROP_REMOTE_CACHE_HASH))) {
                paramBuilder.append("&remotecachehash=").append((String) curParam);
            }

            // FilterShortName
            if (null != (curParam = jobProperties.get(Properties.PROP_FILTER_SHORT_NAME))) {
                paramBuilder.append("&filtershortname=").append((String) curParam);
            }

            // InputType
            if (null != (curParam = jobProperties.get(Properties.PROP_INPUT_TYPE))) {
                paramBuilder.append("&inputtype=").append((String) curParam);
            }

            // PixelX
            if (null != (curParam = jobProperties.get(Properties.PROP_PIXEL_X))) {
                paramBuilder.append("&pixelx=").append(((Integer) curParam).toString());
            }

            // PixelY
            if (null != (curParam = jobProperties.get(Properties.PROP_PIXEL_Y))) {
                paramBuilder.append("&pixely=").append(((Integer) curParam).toString());
            }

            // PixelWidth
            if (null != (curParam = jobProperties.get(Properties.PROP_PIXEL_WIDTH))) {
                paramBuilder.append("&pixelwidth=").append(((Integer) curParam).toString());
            }

            // PixelHeight
            if (null != (curParam = jobProperties.get(Properties.PROP_PIXEL_HEIGHT))) {
                paramBuilder.append("&pixelheight=").append(((Integer) curParam).toString());
            }

            // MimeType
            if (null != (curParam = jobProperties.get(Properties.PROP_MIME_TYPE))) {
                paramBuilder.append("&mimetype=").append((String) curParam);
            }

            // PageRange
            if (null != (curParam = jobProperties.get(Properties.PROP_PAGE_RANGE))) {
                paramBuilder.append("&pagerange=").append((String) curParam);
            }

            // PageNumber
            if (null != (curParam = jobProperties.get(Properties.PROP_PAGE_NUMBER))) {
                paramBuilder.append("&pagenumber=").append(((Integer) curParam).toString());
            }

            // ZipArchive
            if (null != (curParam = jobProperties.get(Properties.PROP_ZIP_ARCHIVE))) {
                paramBuilder.append("&ziparchive=").append(((Boolean) curParam).toString());
            }

            // PROP_INFO_FILENAME
            if (null != (curParam = jobProperties.get(Properties.PROP_INFO_FILENAME))) {
                final String infoFileName = (String) curParam;

                try {
                    paramBuilder.append("&infofilename=").append(URLEncoder.encode(infoFileName, "UTF-8"));
                } catch (final UnsupportedEncodingException e) {
                    ClientManager.logExcp(e);
                }
            }

            // Feature
            if (null != (curParam = jobProperties.get(Properties.PROP_FEATURES_ID))) {
                paramBuilder.append("&featuresid=").append(((Integer) curParam).toString());
            }

            // CacheOnly
            if (null != (curParam = jobProperties.get(Properties.PROP_CACHE_ONLY))) {
                paramBuilder.append("&cacheonly=").append(((Boolean) curParam).toString());
            }

            // HideChanges
            if (null != (curParam = jobProperties.get(Properties.PROP_HIDE_CHANGES))) {
                paramBuilder.append("&hidechanges=").append(((Boolean) curParam).toString());
            }

            // HideComments
            if (null != (curParam = jobProperties.get(Properties.PROP_HIDE_COMMENTS))) {
                paramBuilder.append("&hidecomments=").append(((Boolean) curParam).toString());
            }

            // Async
            if (null != (curParam = jobProperties.get(Properties.PROP_ASYNC))) {
                paramBuilder.append("&async=").append(((Boolean) curParam).toString());
            }

            // RemoteAsync (treated as async for remote side)
            if (null != (curParam = jobProperties.get(Properties.PROP_REMOTE_ASYNC))) {
                paramBuilder.append("&async=").append(((Boolean) curParam).toString());
            }

            // IsAvailable
            if (null != (curParam = jobProperties.get(Properties.PROP_QUERY_AVAILABILITY))) {
                paramBuilder.append("&queryavailability=").append(((Boolean) curParam).toString());
            }

            // ImageResolution
            if (null != (curParam = jobProperties.get(Properties.PROP_IMAGE_RESOLUTION))) {
                paramBuilder.append("&imageresolution=").append(((Integer) curParam).toString());
            }

            // ScaleType
            if (null != (curParam = jobProperties.get(Properties.PROP_IMAGE_SCALE_TYPE))) {
                paramBuilder.append("&imagescaletype=").append((String) curParam);
            }

            // RemoteUrl
            paramBuilder.append("&remoteurl=").append(m_remoteUrl.toString());

            // add additional params, if available
            if (null != additionalParams) {
                for (final String curParamStr : additionalParams) {
                    if ((null != curParamStr) && (curParamStr.indexOf('=') > 0)) {
                        paramBuilder.append('&').append(curParamStr);
                    }
                }
            }

            if (includeDataUrl) {
                // add base64 encoded data from InputFile or InputStream as data Url
                final byte[] inputBuffer = getInputBuffer(jobProperties, resultProperties);

                if (null != inputBuffer) {
                    final String inputType = (String) jobProperties.get(Properties.PROP_INPUT_TYPE);
                    final String inputMimeType = (null != inputType) && (inputType.equals("pdf")) ? "application/pdf" : "application/octet-stream";

                    try {
                        final String base64String = Base64.encodeBase64URLSafeString(inputBuffer);

                        if (null != base64String) {
                            paramBuilder.append("&url=data:").append(inputMimeType).append(";base64,").append(base64String);
                        }
                    } catch (final Exception e) {
                        ClientManager.logExcp(e);
                        paramBuilder = null;
                    }
                }
            }
        }

        return ((null != paramBuilder) ? paramBuilder.toString() : null);
    }

    /**
     * @param jsonObject
     * @return
     */
    static private Object parseJSONResponse(JSONObject jsonObject, HashMap<String, Object> resultProperties) {
        Object ret = null;

        try {
            if (null != jsonObject) {
                final Set<String> keys = jsonObject.keySet();

                if (keys.contains("errorcode")) {
                    resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(jsonObject.getInt("errorcode")));
                }

                if (keys.contains("cachehash")) {
                    resultProperties.put(Properties.PROP_RESULT_CACHE_HASH, jsonObject.getString("cachehash"));
                }

                if (keys.contains("inputfilehash")) {
                    resultProperties.put(Properties.PROP_RESULT_INPUTFILE_HASH, jsonObject.getString("inputfilehash"));
                }

                if (keys.contains("locale")) {
                    resultProperties.put(Properties.PROP_RESULT_LOCALE, jsonObject.getString("locale"));
                }

                if (keys.contains("jobid")) {
                    resultProperties.put(Properties.PROP_RESULT_JOBID, jsonObject.getString("jobid"));
                }

                if (keys.contains("pagecount")) {
                    resultProperties.put(Properties.PROP_RESULT_PAGE_COUNT, Integer.valueOf(jsonObject.getInt("pagecount")));
                }

                if (keys.contains("originalpagecount")) {
                    resultProperties.put(Properties.PROP_RESULT_ORIGINAL_PAGE_COUNT, Integer.valueOf(jsonObject.getInt("originalpagecount")));
                }

                if (keys.contains("pagenumber")) {
                    resultProperties.put(Properties.PROP_RESULT_PAGE_NUMBER, Integer.valueOf(jsonObject.getInt("pagenumber")));
                }

                if (keys.contains("mimetype")) {
                    resultProperties.put(Properties.PROP_RESULT_MIME_TYPE, jsonObject.getString("mimetype"));
                }

                if (keys.contains("extension")) {
                    resultProperties.put(Properties.PROP_RESULT_EXTENSION, jsonObject.getString("extension"));
                }

                if (keys.contains("passwordprotected") && jsonObject.getBoolean("passwordprotected")) {
                    resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.PASSWORD.errorCode()));
                }

                if (keys.contains("maxsourcesize") && jsonObject.getBoolean("maxsourcesize")) {
                    resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.MAX_SOURCESIZE.errorCode()));
                }

                if (keys.contains("unsupported") && jsonObject.getBoolean("unsupported")) {
                    resultProperties.put(Properties.PROP_RESULT_ERROR_CODE, Integer.valueOf(JobError.UNSUPPORTED.errorCode()));
                }

                if (keys.contains("result")) {
                    String resultDataUrl = jsonObject.getString("result");

                    // we're done with the JSON object, so close it ASAP
                    // in order to save memory for the upcoming decoding process
                    jsonObject.reset();

                    if (resultDataUrl.startsWith("data:")) {
                        int pos = resultDataUrl.indexOf(BASE64_PATTERN);

                        if ((-1 != pos) && (pos > 5)) {
                            if (resultDataUrl.substring(5, 5 + ZIP_MIMETYPE.length()).equals(ZIP_MIMETYPE)) {
                                resultProperties.put(Properties.PROP_RESULT_ZIP_ARCHIVE, Boolean.TRUE);
                            }
                        }

                        if ((-1 != pos) && ((pos += BASE64_PATTERN.length()) < (resultDataUrl.length() - 1))) {
                            final byte[] data = Base64.decodeBase64(resultDataUrl.substring(pos));

                            // reset early since we're done with this string
                            resultDataUrl = null;

                            if ((null != data) && (data.length > 0)) {
                                resultProperties.put(Properties.PROP_RESULT_BUFFER, data);
                                ret = new ByteArrayInputStream(data);
                            }
                        }
                    }
                }
            }
        } catch (final JSONException e) {
            ClientManager.logExcp(e);
        }

        return ret;
    }

    // - Members ---------------------------------------------------------------

    protected ClientManager m_clientManager = null;

    protected ServerType m_serverType = null;

    protected volatile boolean m_curJobDone = false;

    protected volatile boolean m_jobRunningStateReached = false;

    protected volatile long m_jobStartExecTimeMillis = 0;

    protected URL m_remoteUrl = null;

    // - Static members ---------------------------------------------------------

    protected static final String ZIP_MIMETYPE = "application/zip";

    protected static final String BASE64_PATTERN = "base64,";

    protected static final String RCALL_CONVERT = "convert";

    protected static final String RCALL_QUERY_AVAILABILITY = "queryavailability";

    protected static final String RCALL_BEGINPAGECONVERSION = "beginpageconversion";

    protected static final String RCALL_GETCONVERSIONPAGE = "getconversionpage";

    protected static final String RCALL_ENDPAGECONVERSION = "endpageconversion";

    protected static Map<String, String> m_converterCookieMap = new ConcurrentHashMap<>();

    protected static final Set<String> SUPPORTED_TYPE_CHECK_LIST = new HashSet<>();

    {
        // List of all supported types
        // csv,doc,docm,docx,docxm,dot,dotm,dotx,dotxm,fodp,fods,fodt,html,odg,odgm,odp,odpm,ods,odsm,odt,odtm,otg,otgm,otm,ots,otsm,odtm,otgm,otp,otpm,ott,ottm,pot,potm,potx,pps,ppt,pptm,pptx,rtf,txt,xlm,xls,xlsm,xlst,xlstm,xlt,xltm,xlsb,xlsx,xlsxm,xltx,xltxm

        // static initialization
        SUPPORTED_TYPE_CHECK_LIST.add("csv");
        SUPPORTED_TYPE_CHECK_LIST.add("doc");
        SUPPORTED_TYPE_CHECK_LIST.add("docm");
        SUPPORTED_TYPE_CHECK_LIST.add("docx");
        SUPPORTED_TYPE_CHECK_LIST.add("docxm");
        SUPPORTED_TYPE_CHECK_LIST.add("dot");
        SUPPORTED_TYPE_CHECK_LIST.add("dotm");
        SUPPORTED_TYPE_CHECK_LIST.add("dotx");
        SUPPORTED_TYPE_CHECK_LIST.add("dotxm");
        SUPPORTED_TYPE_CHECK_LIST.add("fodp");
        SUPPORTED_TYPE_CHECK_LIST.add("fods");
        SUPPORTED_TYPE_CHECK_LIST.add("fodt");
        SUPPORTED_TYPE_CHECK_LIST.add("html");
        SUPPORTED_TYPE_CHECK_LIST.add("odg");
        SUPPORTED_TYPE_CHECK_LIST.add("odgm");
        SUPPORTED_TYPE_CHECK_LIST.add("odp");
        SUPPORTED_TYPE_CHECK_LIST.add("odpm");
        SUPPORTED_TYPE_CHECK_LIST.add("ods");
        SUPPORTED_TYPE_CHECK_LIST.add("odsm");
        SUPPORTED_TYPE_CHECK_LIST.add("odt");
        SUPPORTED_TYPE_CHECK_LIST.add("odtm");
        SUPPORTED_TYPE_CHECK_LIST.add("otg");
        SUPPORTED_TYPE_CHECK_LIST.add("otgm");
        SUPPORTED_TYPE_CHECK_LIST.add("otm");
        SUPPORTED_TYPE_CHECK_LIST.add("ots");
        SUPPORTED_TYPE_CHECK_LIST.add("otsm");
        SUPPORTED_TYPE_CHECK_LIST.add("otp");
        SUPPORTED_TYPE_CHECK_LIST.add("otpm");
        SUPPORTED_TYPE_CHECK_LIST.add("ott");
        SUPPORTED_TYPE_CHECK_LIST.add("ottm");
        SUPPORTED_TYPE_CHECK_LIST.add("pot");
        SUPPORTED_TYPE_CHECK_LIST.add("potm");
        SUPPORTED_TYPE_CHECK_LIST.add("potx");
        SUPPORTED_TYPE_CHECK_LIST.add("potxm");
        SUPPORTED_TYPE_CHECK_LIST.add("pps");
        SUPPORTED_TYPE_CHECK_LIST.add("ppt");
        SUPPORTED_TYPE_CHECK_LIST.add("pptm");
        SUPPORTED_TYPE_CHECK_LIST.add("pptx");
        SUPPORTED_TYPE_CHECK_LIST.add("pptxm");
        SUPPORTED_TYPE_CHECK_LIST.add("rtf");
        SUPPORTED_TYPE_CHECK_LIST.add("txt");
        SUPPORTED_TYPE_CHECK_LIST.add("xlm");
        SUPPORTED_TYPE_CHECK_LIST.add("xls");
        SUPPORTED_TYPE_CHECK_LIST.add("xlsb");
        SUPPORTED_TYPE_CHECK_LIST.add("xlsm");
        SUPPORTED_TYPE_CHECK_LIST.add("xlst");
        SUPPORTED_TYPE_CHECK_LIST.add("xlstm");
        SUPPORTED_TYPE_CHECK_LIST.add("xlt");
        SUPPORTED_TYPE_CHECK_LIST.add("xltm");
        SUPPORTED_TYPE_CHECK_LIST.add("xlsx");
        SUPPORTED_TYPE_CHECK_LIST.add("xlsxm");
        SUPPORTED_TYPE_CHECK_LIST.add("xltx");
        SUPPORTED_TYPE_CHECK_LIST.add("xltxm");
    }
}
