/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter;

import java.io.File;
import java.util.HashMap;

/**
 * {@link ICache}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public interface ICache {

    /**The UID of the cache can be used to distinguish between
     *  different cache instances in order ro decide, if
     *  e.g. remote requests should be performed or not.
     * @return uid The UID of the cache implementation
     */
    public String getUID();

    /**
     * Creates a new ICacheEntry instance to work with and to
     * possibly add to the cache via <code>addCacheEntry</code>.
     * After finishing work with this new entry, the caller
     * must call <code>endNewEntry</code> in order to free
     * all internally allocated resources
     *
     * @param hash The unique hash for the new entry.
     * @return The new  <code>ICacheEntry</code> instance or null
     *  in case of an error.
     *  In case of success, the returned <code>ICacheEntry</code> instance
     *  is already locked.
     */
    public ICacheEntry startNewEntry(String hash);

    /**
     * Creates a new ICacheEntry instance from given Data and makes the
     * new entry persistent.
     * After finishing work with this new entry, the caller
     * must call <code>endNewEntry</code> in order to free
     * all internally allocated resources
     *
     * @param hash
     * @param resultProperties
     * @param resultContent
     * @return
     */
    public ICacheEntry startNewEntry(String hash, HashMap<String, Object> resultProperties, byte[] resultContent);

    /**
     * Finishes working with a previously acquired new entry by
     * unlocking the new object and freeing all internally allocated
     * resources.
     * This method must be finally called after previously creating
     * a new entry via <code>startNewEntry</code>
     *
     * @param newCacheEntry The <code>ICacheEntry</code> instance to be released
     */
    public void endNewEntry(ICacheEntry newCacheEntry);

    /**
     * Add a new entry to the cache. If there isn't enough free space or free entries, we need to remove
     * one or more of the oldest entries until the entry count and the current size of the cache+the size
     * of the new entry is smaller than the maximum size of the cache
     *
     * @param entry The new cache entry
     * @param force If set to true, add the entry in any case,
     *      regardless of any cache config restrictions (e.g. size or entry count)
     * @return true, if the entry was added to the cache
     */
    public boolean addCacheEntry(ICacheEntry entry, boolean force);

    /**
     * @param entry The cach entry that should be removed.
     * @param force If set to true, remove the entry and all associated resources in any case.
     *              If set to false, the entry is just removed from the cache if it is not locked
     */
    public void removeEntry(ICacheEntry entry, boolean force);

    /**
     * Acquires an entry with a given hash from the cache. After a successful call to <code>acquireEntry</code>
     * the returned resource needs to be released with a call to <code>releaseEntry</code> after processing!
     *
     * @param hash The hash value to search for within the cache.
     * @return The entry with the given hash value or null if no such object exists within the cache.
     */
    /**
     * @param hash The hash value to search for within the cache
     * @param args String parameters, whose usage is depending on the implementation
     * @return
     */
    public ICacheEntry acquireEntry(String hash, boolean synchronize, String... args);

    /**
     * Releases an entry and all connected resources from a previous call to <code>acquireEntry</code>.
     *
     * @param entry The entry to be released.
     */
    public void releaseEntry(ICacheEntry entry, boolean synchronize);

    /**
     * @param entry The entry that should be put at the last position of the cache.
     * @return The object with the given hash value or null if no such object exists within the cache.
     */
    public void touchEntry(ICacheEntry entry);

    /**
     * @return The working directory for the cache
     */
    public File getPersistentCacheDirectory();

    /**
     * Removes all entries within the cache and calls the clear method of entries that are not persistent
     */
    public void end();

    /**
    * Tells the cache to perform internal management tasks, e.g. update statistics, cleanup etc.
    */
    public void update(boolean logInfoAtNextTimeslot);
}
