/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.ant.tasks;

import static com.openexchange.ant.tasks.ComputeBuildOrder.calculateModulesByFilename;
import static com.openexchange.ant.tasks.ComputeBuildOrder.calculateModulesByPackage;
import static com.openexchange.ant.Tools.calculateModulesByName;
import static com.openexchange.ant.Tools.readClasspathModules;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.types.Path;
import com.openexchange.ant.data.AbstractModule;
import com.openexchange.ant.data.DirModule;
import com.openexchange.ant.data.JarModule;
import com.openexchange.ant.data.SrcDirModule;

/**
 * {@link SetBundleProperties} refreshes the bundle properties after some bundles of the same package are built. This is used to compile the
 * current bundle always against the already built bundles and not against some source bundles.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public class SetBundleProperties extends Task {

    private File dir;
    private String currentBundle;
    private String buildType;
    private Path classpath;

    public SetBundleProperties() {
        super();
    }

    public void setDir(File dir) {
        this.dir = dir;
    }

    public void setCurrentBundle(String currentBundle) {
        this.currentBundle = currentBundle;
    }

    public void setBuildType(String buildType) {
        this.buildType = buildType;
    }

    public Path createClasspath() {
        classpath = new Path(getProject());
        return classpath;
    }

    @Override
    public void execute() throws BuildException {
        log("using classpath: " + classpath, Project.MSG_DEBUG);
        // Parse JAR files given by the classpath definition. This are normally binaries (mostly JARs or bundle directories containing JARs).
        final List<AbstractModule> classpathModules = readClasspathModules(getProject(), classpath);

        // Read bundle to build
        final DirModule module = readModule(getProject(), classpathModules, new File(dir, currentBundle));

        final List<AbstractModule> allModules = new ArrayList<AbstractModule>(classpathModules.size() + 1);
        allModules.addAll(classpathModules);
        allModules.add(module);

        // Compute dependencies in the bundles.
        for (AbstractModule tmpModule : allModules) {
            tmpModule.computeDependencies(calculateModulesByName(allModules), calculateModulesByPackage(allModules), calculateModulesByFilename(allModules), true);
        }
        for (AbstractModule tmpModule : allModules) {
            tmpModule.computeDependenciesForFragments();
        }

        // Set properties for bundle.
        setProjectProperties(getProject(), module, buildType);
    }

    static DirModule readModule(Project project, List<AbstractModule> classpathModules, File bundleDirectory) {
        final DirModule module;
        module = new SrcDirModule(bundleDirectory);
        module.readLocalFiles(project);
        // Parse the bundles jars directory maybe containing 3rd bundle JARs.
        final File jarsDir = new File(bundleDirectory, "jars");
        if (jarsDir.exists() && jarsDir.isDirectory()) {
            File[] files = jarsDir.listFiles();
            if (null != files) {
                for (final File osgiJar : files) {
                    if (!osgiJar.isFile()) {
                        continue;
                    }
                    if (!osgiJar.getName().endsWith(".jar")) {
                        continue;
                    }
                    if (osgiJar.getName().startsWith(bundleDirectory.getName())) {
                        // The temporarily build JAR of this bundle.
                        continue;
                    }
                    // These JARs need to be added to the previously parsed classpath.
                    classpathModules.add(new JarModule(osgiJar));
                }
            }
        }
        return module;
    }

    static void setProjectProperties(Project project, AbstractModule module, String buildType) {
        final StringBuilder dependencies = new StringBuilder();
        for (AbstractModule dependency : module.getDependencies()) {
            dependencies.append(dependency.getName());
            dependencies.append(',');
        }
        if (dependencies.length() > 0) {
            dependencies.setLength(dependencies.length() - 1);
        }
        project.log(module.getName() + " requires " + dependencies, Project.MSG_DEBUG);
        // First direct dependencies for compiling the classes.
        final StringBuilder requiredClasspath = new StringBuilder();
        for (final String classpathEntry : module.getRequiredClasspath()) {
            requiredClasspath.append(classpathEntry);
            requiredClasspath.append(',');
        }
        if (requiredClasspath.length() > 0) {
            // first package will not have dependencies.
            requiredClasspath.setLength(requiredClasspath.length() - 1);
        }
        project.log(module.getName() + ".requiredClasspath: " + requiredClasspath, Project.MSG_DEBUG);
        project.setInheritedProperty(module.getName() + ".requiredClasspath", requiredClasspath.toString());
        // deep dependencies for executing the classes to run tests
        final StringBuilder runtimeClasspath = new StringBuilder();
        for (final String classpathEntry : module.getDeepRuntimeClasspath()) {
            runtimeClasspath.append(classpathEntry);
            runtimeClasspath.append(',');
        }
        if (runtimeClasspath.length() > 0) {
            // first package will not have dependencies.
            runtimeClasspath.setLength(runtimeClasspath.length() - 1);
        }
        project.log(module.getName() + ".runtimeClasspath: " + runtimeClasspath, Project.MSG_DEBUG);
        project.setInheritedProperty(module.getName() + ".runtimeClasspath", runtimeClasspath.toString());
        // Bundles may have multiple source directories for easier adoption to incompatible APIs.
        final StringBuilder sources = new StringBuilder();
        final Set<String> sourceDirs;
        if ("bundle".equals(buildType)) {
            sourceDirs = module.getSourceDirs();
        } else if ("tests".equals(buildType)) {
            sourceDirs = module.getTestSourceDirs();
        } else if ("osgiTests".equals(buildType)) {
            sourceDirs = module.getSourceDirs();
        } else {
            throw new BuildException("Unknown build type \"" + buildType + "\".");
        }
        for (final String sourceDir : sourceDirs) {
            sources.append(sourceDir);
            sources.append(',');
        }
        if (sources.length() > 0) {
            sources.setLength(sources.length() - 1);
        }
        project.log(module.getName() + ".sourceDirs: " + sources, Project.MSG_DEBUG);
        project.setInheritedProperty(module.getName() + ".sourceDirs", sources.toString());
    }
}
