/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.ant.tasks;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.types.Path;
import com.openexchange.ant.data.AbstractModule;
import com.openexchange.ant.data.BinDirModule;
import com.openexchange.ant.data.DirModule;
import com.openexchange.ant.data.JarModule;
import com.openexchange.build.utils.Tools;

/**
 * {@link RemoveFromListByClasspathCompare}
 *
 * @author <a href="mailto:felix.marx@open-xchange.com">Felix Marx</a>
 */
public class RemoveFromListByClasspathCompare extends Task {

    private File dir;

    private String propertyName;

    private String projectList;

    private Path classpath;

    public RemoveFromListByClasspathCompare() {
        super();
    }

    public void setProjectList(String projectList) {
        this.projectList = projectList;
    }

    public void setPropertyName(String propertyName) {
        this.propertyName = propertyName;
    }

    public void setDir(File dir) {
        this.dir = dir;
    }

    public Path createClasspath() {
        classpath = new Path(getProject());
        return classpath;
    }

    @Override
    public void execute() throws BuildException {
        log("using classpath: " + classpath, Project.MSG_INFO);
        // Parse JAR files given by the classpath definition. This are normally binaries (mostly JARs or bundle directories containing
        // JARs).
        final List<AbstractModule> classpathModules = readClasspathModules(getProject(), classpath);

        // All bundles from projectList are source bundles.
        final List<AbstractModule> projectlistModules = new ArrayList<AbstractModule>();
        for (String split : Tools.split(projectList, ',')) {
            projectlistModules.add(SetBundleProperties.readModule(getProject(), classpathModules, new File(dir, split)));
        }

        // Remove already present modules
        log("projectlist modules: " + projectlistModules, Project.MSG_INFO);

        final Map<String, AbstractModule> classpathModulesByName = calculateModulesByName(classpathModules);
        List<AbstractModule> removeModules = new ArrayList<AbstractModule>();
        for (AbstractModule module : projectlistModules) {
            if (classpathModulesByName.containsKey(module.getName())) {
                removeModules.add(module);
            }
        }
        projectlistModules.removeAll(removeModules);

        log("projectlist modules without classpath: " + projectlistModules, Project.MSG_INFO);

        // Set properties for other Ant tasks.
        String moduleNamesList = getModuleNamesList(projectlistModules, ',');
        getProject().setInheritedProperty(propertyName, moduleNamesList);
    }

    static List<AbstractModule> readClasspathModules(Project project, Path classpath) {
        final String[] classpathFiles = classpath.list();
        final List<AbstractModule> retval = new ArrayList<AbstractModule>(classpathFiles.length);
        for (final String classpathFilename : classpathFiles) {
            final File classpathFile = new File(classpathFilename);
            if (classpathFile.isFile()) {
                retval.add(new JarModule(classpathFile));
            } else {
                DirModule classpathModule = new BinDirModule(classpathFile);
                classpathModule.readLocalFiles(project);
                retval.add(classpathModule);
            }
        }
        return retval;
    }

    static Map<String, AbstractModule> calculateModulesByFilename(final List<AbstractModule> allModules) {
        final Map<String, AbstractModule> modulesByFilename;
        modulesByFilename = new HashMap<String, AbstractModule>();
        for (AbstractModule module : allModules) {
            modulesByFilename.put(module.getFile().getName(), module);
        }
        return modulesByFilename;
    }

    static Map<String, AbstractModule> calculateModulesByName(final List<AbstractModule> allModules) {
        final Map<String, AbstractModule> modulesByName;
        modulesByName = new HashMap<String, AbstractModule>();
        for (final AbstractModule module : allModules) {
            modulesByName.put(module.getName(), module);
        }
        return modulesByName;
    }

    static Map<String, Set<AbstractModule>> calculateModulesByPackage(final List<AbstractModule> allModules) {
        final Map<String, Set<AbstractModule>> modulesByPackage;
        modulesByPackage = new HashMap<String, Set<AbstractModule>>();
        for (AbstractModule module : allModules) {
            for (String exportedPackage : module.getExportedPackages()) {
                Set<AbstractModule> exportingModules = modulesByPackage.get(exportedPackage);
                if (exportingModules == null) {
                    exportingModules = new HashSet<AbstractModule>();
                    modulesByPackage.put(exportedPackage, exportingModules);
                }
                exportingModules.add(module);
            }
        }
        return modulesByPackage;
    }

    private static String getModuleNamesList(Collection<AbstractModule> modules, char delimiter) {
        final StringBuffer buffer = new StringBuffer();
        for (AbstractModule module : modules) {
            buffer.append(module.getName());
            buffer.append(delimiter);
        }
        if (buffer.length() > 0) {
            buffer.setLength(buffer.length() - 1);
        }
        return buffer.toString();
    }
}
